/******************************************************************************
**
**  COPYRIGHT (C) 2000, 2001, 2002 Intel Corporation.
**
**  The information in this file is furnished for informational use 
**  only, is subject to change without notice, and should not be construed as 
**  a commitment by Intel Corporation. Intel Corporation assumes no 
**  responsibility or liability for any errors or inaccuracies that may appear 
**  in this document or any software that may be provided in association with 
**  this document. 
**
**  FILENAME:       Jflash.cpp
**
**  PURPOSE:        A utility to program Intel flash devices from a PC parallel port.
**
**  LAST MODIFIED:  $Modtime: 6/18/02 2:54p $
******************************************************************************/

#include <stdio.h>
#include <time.h>

#ifndef __linux__
#include <windows.h>
#include <conio.h>
#endif

#include "Compile_switches.h"
#include "Jflash.h"
#include "cotullajtag.h"

/*
* Linux compatibility definitions
*/

#ifdef __linux__

#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/io.h>
#include <errno.h>
#include <string.h>

#define FALSE	0
#define TRUE	1

typedef int BOOL;
typedef long DWORD;
typedef short WORD;

#define _inp(a)  inb(a)
#define _outp(a,d)  outb(d,a)
#define _getche getchar

#endif


/*
*******************************************************************************
Globals
*******************************************************************************
*/

int lpt_address;	// Global variable assigned to parallel port address

FILE *in_file;

int out_dat[MAX_DR_LENGTH];

#ifdef SANDGATE_PLATFORM
int WORKBUF[SANDGATE_WORKBUF_SIZE];
#endif

#ifdef LUBBOCK_PLATFORM
int WORKBUF[LUBBOCK_WORKBUF_SIZE];
#endif

#ifdef ASSABET_PLATFORM
#ifndef LUBBOCK_SA1110

int WORKBUF[ASSABET_WORKBUF_SIZE];

#else

int WORKBUF[ASSABET_WORKBUF_SIZE -2];

#endif
#endif



/*
*******************************************************************************
Forward declarations
*******************************************************************************
*/

int putp(int,int, int); // writes the JTAG data on the parallel port
void id_command(void);	// issues the JTAG command to read the device ID for all 3 chips
void bypass_all(void);	// issues the JTAG command to put all 3 device in bypass mode
void extest(void);		// issues the JTAG command EXTEST to the Processor


#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
DWORD access_rom(int, DWORD, DWORD, int);	// Passes read/write/setup data for the Flash memory
#else
WORD access_rom(int, DWORD, WORD, int);	// Passes read/write/setup data for the Flash memory
#endif  

DWORD access_bus(int, DWORD, DWORD, int);	// Read/write access to the Processor pins

int test_port(void);	// Looks for and finds a valid parallel port address
int check_id(char*);	// Compares the device IDs for the 3 JTAG chips to expected values
void error_out(char*);	// Prints error and exits program
void erase_flash(DWORD, DWORD, DWORD, DWORD, int);
void program_flash(DWORD, DWORD, DWORD);
void verify_flash(DWORD, DWORD);
void test_logic_reset(void);
void test_lock_flash(DWORD, DWORD, DWORD, DWORD, int);
void set_lock_flash(DWORD, DWORD, DWORD, DWORD, int);
void set_address (DWORD);
int PZ_scan_code(int, int, int);
int controller_scan_code(int, int, int);
void pre_DRSCAN(void);
void post_DRSCAN(void);
void pre_IRSCAN(void);
void post_IRSCAN(void);
void mem_rw_mode(int);
void mem_data_driver(int);
void mem_write_enable(int);
void mem_output_enable(int);
void clear_chip_selects(void);
void set_chip_select(DWORD);
DWORD shift_data(int);
void set_data(DWORD);
void jtag_test(void);
void dump_chain(void);
void init_workbuf(void);
void invert_workbuf(void);
void set_pin_chip_select(DWORD);

void check_file_info(DWORD *fsize , DWORD *last_non_zero, DWORD *last_non_ff, DWORD rom_size);
void check_rom_info(DWORD *max_erase_time, DWORD * dsize, DWORD * max_write_buffer, DWORD * block_size, int * nblocks );
/*
*******************************************************************************
*
* FUNCTION:         main
*
* DESCRIPTION:      Entry point for utility
*
* INPUT PARAMETERS: uses optional input parameters:
*                       argv[1] = filename to flash (binary files only)
*                       argv[2] = program options, currently only 'P' for Program
*                       argv[3] = block number (used to compute base_address)    
*
* RETURNS:          void
*
*******************************************************************************
*/

void main( int argc, char *argv[] )
{
    time_t start;
	DWORD fsize = 0;
	DWORD last_non_zero = 0 ;
	DWORD last_non_ff = 0;
//	DWORD li;
	int block_number = 0;
    DWORD i;

    DWORD max_erase_time, dsize, max_write_buffer, block_size;
	int   nblocks;

    
    printf("JFLASH Version %s\n",VERSION);
    printf("COPYRIGHT (C) 2000, 2001 Intel Corporation\n");

#ifndef __linux__
    //Test operating system, if WinNT or Win2000 then get device driver handle
	OSVERSIONINFO osvi;
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(&osvi);
	if(osvi.dwPlatformId == VER_PLATFORM_WIN32_NT)
	{
        HANDLE h;

		h = CreateFile("\\\\.\\giveio", GENERIC_READ, 0, NULL,
					OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
		if(h == INVALID_HANDLE_VALUE)
			error_out("Couldn't access giveio device");
		CloseHandle(h);
	}
#endif

	lpt_address = test_port();	// find a valid parallel port address
	if(!lpt_address)
		error_out("Error, unable to find parallel port");

    test_logic_reset();
    jtag_test();
    
	char filename[MAX_IN_LENGTH];
	if(argc >= 2)
		strcpy(filename,argv[1]);
	else
		{
		printf("enter binary file name: ");
		fgets(filename, sizeof(filename), stdin);
		}

	test_logic_reset();

	id_command();
	bypass_all();
    //init_workbuf();

    test_logic_reset();
  
    check_rom_info(&max_erase_time, &dsize, &max_write_buffer, &block_size, &nblocks);
	
	if( (in_file = fopen(filename, "rb" )) == NULL)
    {
		error_out("error, can not open binary input file");
    }

	check_file_info(&fsize , &last_non_zero, &last_non_ff, dsize);
    
    // Don't wast time programming ff's at the end of the file this is the erase state
    fsize = last_non_ff;  

	if(argc >= 4)
    {
		block_number = atoi(argv[3]);
	}
    
    DWORD base_address = block_number * block_size; //the block_size is byte number, 
	                                                //the base_address is 16-bit word address for each device

    #if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
    printf("There are two 16-bit Flash devices in parallel\n\n");
    #else
    printf("There is one 16-bit Flash device\n\n");
    #endif
    printf("Characteristics for one device:\n");
	printf(" Number of blocks in device = %ld\n",nblocks);
	printf(" Block size = %ld 0x%lx word(16-bit)\n",block_size,block_size);
	printf(" Device size = %ld 0x%lx word(16-bit)\n\n",dsize,dsize);
    printf("Sample block to address list:\n\n");
    for (i = 0; i < 129; i += 40)
    { 
        #if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
        printf(" Block %d = hex address: %08X \n", i, i * block_size * 4); 
        #else
        printf(" Block %d = hex address: %08X \n", i, i * block_size * 2); 
        #endif 
	
    }
    if(block_number >= nblocks || block_number < 0)
	{
    	error_out("error specifying block number");
    }

	if(100 - (last_non_zero * 100)/last_non_ff > 20)
	{
		printf("The last %2ld percent of image file is all zeros\n",100 - (last_non_zero * 100)/last_non_ff);
		printf("Would you like to save time by not programming that area? [y/n]: ");
		if(toupper(_getche()) == 'Y')
			fsize = last_non_zero;
	}

	printf("\n");
	char option = 'P';
	if(argc >= 3)
	{
    	option = toupper(*argv[2]);
    }
	
    if(option == 'P')
	{

		test_lock_flash(base_address, fsize, block_size, max_erase_time, block_number);
		erase_flash(base_address, fsize, block_size, max_erase_time, block_number);
		program_flash(max_write_buffer, base_address, fsize);
	}

	access_rom(SETUP, 0, F_READ_ARRAY, IGNORE_PORT); // put back into read mode
	access_rom(WRITE, 0, F_READ_ARRAY, IGNORE_PORT);
	access_rom(HOLD, 0, F_READ_ARRAY, IGNORE_PORT);

	access_rom(READ, base_address, 0x0L, IGNORE_PORT); //extra read to get the pipeline going

	time(&start);

	verify_flash(base_address, fsize);

	fclose(in_file);

	test_logic_reset();
}

/*
*******************************************************************************
*
* FUNCTION:         get_rom_info
*
* DESCRIPTION:      get the flash information such as max erase time, flash size, 
*                   max write buffer, block number               
*                   
*
* INPUT PARAMETERS: DWORD *max_erase_time	: max erase time, number of seconds
*					DWORD *dsize			: each flash size, number of 16-bit word
*					DWORD *max_write_buffer : each flash max write buffer, number of 16-bit word
					DWORD *block_size		: each block size, number of 16-bit word
*					DWORD *nblocks			: number of erase blocks in each Flash*
					
* RETURNS:          None
*
* GLOBAL EFFECTS:   None
*
*******************************************************************************
*/
void check_rom_info(DWORD *max_erase_time, DWORD * dsize, DWORD * max_write_buffer, DWORD * block_size, int * nblocks )
{	access_rom(SETUP, 0, F_CLEAR_STATUS, IGNORE_PORT); // clear status register
	access_rom(WRITE, 0, F_CLEAR_STATUS, IGNORE_PORT);
	access_rom(HOLD, 0, F_CLEAR_STATUS, IGNORE_PORT);

	access_rom(SETUP, 0, F_READ_QUERY, IGNORE_PORT); // read query
	access_rom(WRITE, 0, F_READ_QUERY, IGNORE_PORT);
	access_rom(HOLD, 0, F_READ_QUERY, IGNORE_PORT);

	// To read data from the Flash Memory you must first fill the processor JTAG chain
	// with the Address, then pump the entire chain out.
	// however while pumping data out you can be pumping the next cycle's Address in
	// Therefore the JTAG chain looks like a pipeline, valid read data always coming
	// out one cycle late.

	// Note that both Flash Memory devices are accessed in parallel (i.e. 32 data bits)
    
    // Test to see if the Flash supports Query Structured Output
	access_rom(READ, 0x10, 0, IGNORE_PORT);  
	
    #ifdef DEBUG
    printf("about to read flash attributes....\n");
    #endif
    
    if(access_rom(READ, 0x11, 0, READ_PORT) != F_ATTR_Q) //Q
    {
        // We failed to successfully query the flash. We can't determine
        // whether the target flash is J3 or K3. At this point, we'll try to
        // set flash to asynchronous mode and hope for the best.
        access_rom(SETUP, 0x8000, F_SET_READ_CFG_REG, IGNORE_PORT);
        access_rom(WRITE, 0x8000, F_SET_READ_CFG_REG, IGNORE_PORT);
        access_rom(HOLD, 0x8000, F_SET_READ_CFG_REG, IGNORE_PORT);

        access_rom(SETUP, 0x8000, F_SET_READ_CFG_REG_2ND, IGNORE_PORT);
        access_rom(WRITE, 0x8000, F_SET_READ_CFG_REG_2ND, IGNORE_PORT);
        access_rom(HOLD, 0x8000, F_SET_READ_CFG_REG_2ND, IGNORE_PORT);

    	access_rom(SETUP, 0, F_CLEAR_STATUS, IGNORE_PORT); // clear status register
    	access_rom(WRITE, 0, F_CLEAR_STATUS, IGNORE_PORT);
    	access_rom(HOLD, 0, F_CLEAR_STATUS, IGNORE_PORT);

    	access_rom(SETUP, 0, F_READ_QUERY, IGNORE_PORT); // read query
    	access_rom(WRITE, 0, F_READ_QUERY, IGNORE_PORT);
    	access_rom(HOLD, 0, F_READ_QUERY, IGNORE_PORT);

        // Test to see if the Flash supports Query Structured Output
    	access_rom(READ, 0x10, 0, IGNORE_PORT);  

        if(access_rom(READ, 0x11, 0, READ_PORT) != F_ATTR_Q) //Q
        {
    		error_out("error reading flash attribute space\ncheck cables, power and flash sockets");
        }
    }

	if(access_rom(READ, 0x12, 0, READ_PORT) != F_ATTR_R) //R
	{
    	error_out("error reading flash attribute space R");
	}
    // Toggle chip select for K3 flash.
    access_rom(RS, 0x25, 0, READ_PORT);
    if(access_rom(READ, 0x25, 0, READ_PORT) != F_ATTR_Y) //Y
	{
    	error_out("error reading flash attribute space Y\n");
    }

            
    // Toggle chip select for K3 flash.
    access_rom(RS, 0x27, 0, READ_PORT);
    // "n" such that the max block erase time = 2^n
	*max_erase_time = access_rom(READ, 0x27, 0, READ_PORT);  

#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
	if((*max_erase_time & 0xffffL) != (*max_erase_time >> 16)) // both devices should be the same time
	{
    	error_out("error, max erase time of E11 and E12 are different");
    }
#endif	
    // convert erase time 2^n to the number of seconds
    *max_erase_time = 1 << (*max_erase_time & 0xffffL);  

    // Toggle chip select for K3 flash.
   access_rom(RS, 0x2a, 0, READ_PORT);
    // "n" such that the device size = 2^n in number of bytes
	*dsize = access_rom(READ, 0x2a, 0, READ_PORT);  
#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
	if((*dsize & 0xffffL) != (*dsize >> 16)) // both devices should be the same size
	{
    	error_out("error, device size of E11 and E12 are different");
	}
#endif
    // convert data size from 2^n to the number of bytes
    *dsize = 1 << (*dsize & 0xffffL);  


    // Toggle chip select for K3 flash.
    access_rom(RS, 0x2d, 0, READ_PORT);
    // "n" such that the max num of bytes in write buffer = 2^n
	*max_write_buffer = access_rom(READ, 0x2d, 0, READ_PORT);  
#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
	if((*max_write_buffer & 0xffffL) != (*max_write_buffer >> 16)) // both devices should have the same write buffer size
	{
    	error_out("error, write buffer size of E11 and E12 are different");
	}
#endif
    // convert from 2^n bytes to the number of byte
    *max_write_buffer = (1 << (*max_write_buffer & 0xffffL));  

    // Toggle chip select for K3 flash.
    access_rom(RS, 0x2e, 0, READ_PORT);
    // get the number of erase blocks in Flash - 1
	*nblocks = access_rom(READ, 0x2e, 0, READ_PORT);
#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
	if((*nblocks & 0xffffL) != (*nblocks >> 16))	// both devices should have the same number
	{
    	error_out("error, number of blocks of E11 and E12 are different");
	}
#endif
    *nblocks = (*nblocks & 0xffffL) + 1L;  // need to add '1' for true number

	*block_size = (*dsize) / (*nblocks);

	//change the dsize, max_write_buffer, block_size from byte number to 16-bit word number
	*dsize = *dsize / 2;
	*max_write_buffer = *max_write_buffer / 2;
	*block_size = *block_size / 2;
}

/*
*******************************************************************************
*
* FUNCTION:         check_file_info
*
* DESCRIPTION:      get the file information and check with rom size
*                   
*
* INPUT PARAMETERS: DWORD *fsize			: file size
					DWORD *last_non_zero	: the point where only 0 or FF remain
					DWORD *last_non_ff		: the point where only ff remain
										
										    if LUBBOCK_SABINAL or
										    LUBBOCK_DALHART_16 is defined,
										    the upper 3 parameters 
											are in 16-bit word number
											else they are in 32-bit DWORD number

					DWORD rom_size			: the rom size in 
*					
					
* RETURNS:          None
*
* GLOBAL EFFECTS:   None
*
*******************************************************************************
*/
void check_file_info(DWORD *fsize , DWORD *last_non_zero, DWORD *last_non_ff, DWORD rom_size)
{	BUS_GRAIN li;
		
	for(;;)
	{
		int n = fread((BUS_GRAIN *)&li, sizeof(BUS_GRAIN) , 1, in_file);

    	if(feof(in_file))break; // Any bytes not on a 4 byte boundry at end-of-file will be ignored
		{
            (*fsize)++;
        }
		if(li != 0 && li != -1) // Find point in file were only 0's and ff's remain
		{						// For 32 bit data bus, -1 is 0xffffffff, for 16 bit data bus -1 is 0xffff
        	*last_non_zero = *fsize;
        }
		if(li != -1)  // Find point in file were only ff's remain
		{
        	*last_non_ff = *fsize;
        }
	}
	

	rewind(in_file);

#if defined(LUBBOCK_SABINAL) || defined(LUBBOCK_DALHART_16)
	// if SABINAL 16-bit data width used, it assume only one 16-bit flash installed
		if ((*fsize) > rom_size)
			error_out("error, file size is bigger than device size");
#else
	// if 32-bit data width used. It assume 2 16-bit flash installed. each flash size=fsize
		if ((*fsize) * 2 > rom_size * 2)
			error_out("error, file size is bigger than device size");
#endif

}


/*
*******************************************************************************
*
* FUNCTION:         putp
*
* DESCRIPTION:      Drives TCK, TDI, and TMS signals and reads TDO signal 
*                   via _outp and _inp calls.
*                   
*                   Cable used had 100 ohm resistors between the following pins
*                   (Cable shipped as part of SA-1110 Development Kit)
*                   Output pins (LPT driving)
*                       LPT D0 Pin 2 and TCK J10 Pin 4
*                       LPT D1 Pin 3 and TDI J10 Pin 11
*                       LPT D2 Pin 4 and TMS J10 Pin 9
*
*                   Input pin (SA-1110 board drives)
*                       LPT Busy Pin 11 and TDO J10 Pin 13
*                   
*                   
*
* INPUT PARAMETERS: int tdi - test data in
*
* RETURNS:          int - TDO (Test Data Out)
*
* GLOBAL EFFECTS:   None
*
*******************************************************************************
*/

int putp(int tdi, int tms, int rp)
{
#ifdef PARALLEL_JTAG
	int tdo = -1;
	// TMS is D2, TDI is D1, and TCK is D0, so construct an output by creating a 
    // rising edge on TCK with TMS and TDI data set.
    _outp(lpt_address, tms*4+tdi*2);	// TCK low
	_outp(lpt_address, tms*4+tdi*2+1);	// TCK high
	
    // if we want to read the port, set TCK low because TDO is sampled on the 
    // TCK falling edge.
    if(rp == READ_PORT)
        _outp(lpt_address, tms*4+tdi*2);	// TCK low
	if(rp == READ_PORT)
        tdo = !((int)_inp(lpt_address + 1) >> 7);	// get TDO data
#endif



#ifdef INSIGHT_JTAG
// There's some bit clearing here that isn't needed. It should make this 
// code easier to understand.

	//defines for the INSIGHT IJC-1 JTAG cable

	/* the output port (lpt_address) */
	#define INSIGHT_CLK		0x02
	#define INSIGHT_DIN		0x01
	#define nINSIGHT_PROG	0x10  /* This causes the TDO line to be driven. We'll leave it high.*/
	#define INSIGHT_TMS_IN  0x04  
	#define nINSIGHT_CTRL	0x08  /* Output Enable for the standard JTAG outputs 
		                             (not TDO since this is an output from the 
			                         chip we want to talk to                    */

	/*the input port (lpt_address + 1)*/
	#define TDO_INPUT			0x10  
	#define TDO_INPUT_BITPOS	4  



	int tdo = -1;
	int lpt_data;
	
	//form the data we want to write to the parallel port
	lpt_data = nINSIGHT_PROG;   //Output to TDO off
	lpt_data &= ~nINSIGHT_CTRL;	//Enable the outputs

	if(tms == 1) lpt_data |= INSIGHT_TMS_IN;
	if(tdi == 1) lpt_data |= INSIGHT_DIN;


   	// construct an output by creating a 
    // rising edge on TCK with TMS and TDI data set.
	lpt_data &= ~INSIGHT_CLK;
	_outp(lpt_address, lpt_data);	// TCK low
	
	lpt_data |= INSIGHT_CLK;
	_outp(lpt_address, lpt_data);	// TCK high
	

    // if we want to read the port, set TCK low because TDO is sampled on the 
    // TCK falling edge.
    if(rp == READ_PORT){
		lpt_data &= ~INSIGHT_CLK;
		_outp(lpt_address, lpt_data);	// TCK high
		tdo = ((int)_inp(lpt_address + 1) & TDO_INPUT) >> TDO_INPUT_BITPOS;	// get TDO data
	}
#endif

//	#ifdef DEBUG
//    printf("TDI = %d, TMS = %d, TDO = %d\n",tdi,tms,tdo);
//  #endif
    return tdo;


}
/*
*******************************************************************************
*
* FUNCTION:         id_command
*
* DESCRIPTION:      extract and verify the id codes of the devices in the chain
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/

void id_command(void)
{
    pre_IRSCAN();

#ifdef SANDGATE_PLATFORM

    controller_scan_code(COT_IDCODE, IGNORE_PORT, CONTINUE);
    PZ_scan_code(PZ_IDCODE, IGNORE_PORT, CONTINUE);    // PZ3128 IDCODE
    PZ_scan_code(PZ_IDCODE, IGNORE_PORT, TERMINATE);   // PZ3032 IDCODE

#endif // SANDGATE_PLATFORM

#ifdef LUBBOCK_PLATFORM
    controller_scan_code(COT_IDCODE, IGNORE_PORT, TERMINATE); 

#endif // LUBBOCK_PLATFORM

#ifdef ASSABET_PLATFORM
#ifndef LUBBOCK_SA1110
    controller_scan_code(SA_IDCODE, IGNORE_PORT, CONTINUE);
    PZ_scan_code(PZ_IDCODE, IGNORE_PORT, CONTINUE);    // PZ3128 IDCODE
    PZ_scan_code(PZ_IDCODE, IGNORE_PORT, TERMINATE);    // PZ3032 IDCODE
#else
    controller_scan_code(SA_IDCODE, IGNORE_PORT, TERMINATE);


#endif
#endif // ASSABET_PLATFORM

    post_IRSCAN();

    pre_DRSCAN();
    
#ifdef SANDGATE_PLATFORM 

    if(check_id(COTULLAID))
		error_out("failed to read device ID for the DBPXA250");
	if(check_id(PZ3128ID))
		error_out("failed to read device ID for the PZ3128");
	if(check_id(PZ3032SID))
		error_out("failed to read device ID for the PZ3032");
        
#endif// SANDGATE_PLATFORM

#ifdef ASSABET_PLATFORM
#ifndef LUBBOCK_SA1110
    if(check_id(SA1110ID))
		error_out("failed to read device ID for the SA1110");
	if(check_id(PZ3128ID))
		error_out("failed to read device ID for the PZ3128");
	if(check_id(PZ3032SID))
		error_out("failed to read device ID for the PZ3032");
#else
    if(check_id(SA1110ID))
		error_out("failed to read device ID for the SA1110");
#endif        
#endif //ASSABET_PLATFORM


#ifdef LUBBOCK_PLATFORM
#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)

    if(check_id(COTULLAID))
	{
		error_out("failed to read device ID for the DBPXA250");
	}
#endif

// There may still be some A1 Sabinal chips out there. Let's not break when we see one.
#if defined(LUBBOCK_SABINAL) || defined(LUBBOCK_DALHART_16)
    if((check_id(SABINALID)) && (check_id(COTULLAID)))
	{
#ifndef LUBBOCK_DALHART_16
		error_out("failed to read device ID for the DBPXA210");
#else
		error_out("failed to read device ID for the DALHART 16");
#endif
	}

#endif
        
#endif //LUBBOCK_PLATFORM

    
    post_DRSCAN();
}
/*
*******************************************************************************
*
* FUNCTION:         bypass_all
*
* DESCRIPTION:      put all devices into bypass mode
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/

void bypass_all(void)
{
    pre_IRSCAN();

#ifdef SANDGATE_PLATFORM 

    controller_scan_code(COT_BYPASS, IGNORE_PORT, CONTINUE);
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, CONTINUE);    // PZ3128
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, TERMINATE);    // PZ3032
    
#endif //SANDGATE_PLATFORM

#ifdef ASSABET_PLATFORM 
#ifndef LUBBOCK_SA1110
    controller_scan_code(SA_BYPASS, IGNORE_PORT, CONTINUE);
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, CONTINUE);    // PZ3128
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, TERMINATE);    // PZ3032
#else
    controller_scan_code(SA_BYPASS, IGNORE_PORT, TERMINATE);

#endif    
#endif //SANDGATE_PLATFORM

#ifdef LUBBOCK_PLATFORM 

    controller_scan_code(COT_BYPASS, IGNORE_PORT, TERMINATE);
    
#endif //LUBBOCK_PLATFORM
 
    post_IRSCAN();
}
/*
*******************************************************************************
*
* FUNCTION:         extest
*
* DESCRIPTION:      put the processor into extest (drive pins) mode, and all 
*                   others into bypass mode.
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/

void extest(void)
{
    pre_IRSCAN();

#ifdef SANDGATE_PLATFORM 

    controller_scan_code(COT_EXTEST, IGNORE_PORT, CONTINUE);
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, CONTINUE);    // PZ3128
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, TERMINATE);    // PZ3032
    
#endif //SANDGATE_PLATFORM

#ifdef ASSABET_PLATFORM 
#ifndef LUBBOCK_SA1110
    controller_scan_code(SA_EXTEST, IGNORE_PORT, CONTINUE);
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, CONTINUE);    // PZ3128
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, TERMINATE);    // PZ3032
#else
    controller_scan_code(SA_EXTEST, IGNORE_PORT, TERMINATE);

#endif    
#endif //SANDGATE_PLATFORM

#ifdef LUBBOCK_PLATFORM 

    controller_scan_code(COT_EXTEST, IGNORE_PORT, TERMINATE);
    
#endif //LUBBOCK_PLATFORM

    post_IRSCAN();

}
/*
*******************************************************************************
*
* FUNCTION:         access_rom
*
* DESCRIPTION:      This is just an access-bus with the address multiplied by 4
*
* INPUT PARAMETERS: int - rw, or access mode
*                   DWORD - address
*                   DWORD - data
*                   int rp - whether to read or ignore the parallel port
*
* RETURNS:          DWORD - returned data
*
*******************************************************************************
*/

DWORD access_rom_32(int rw, DWORD address, DWORD data, int rp)
{
	DWORD returnvalue;
    
    // Shift Flash address making A2 the LSB
    
	#ifdef DEBUG
    printf("ACCESS_ROM: inp addr = %X, inp data = %X\n", address, data);
    #endif
    
    returnvalue = access_bus(rw, address << 2, data, rp);
    
    #ifdef DEBUG
    printf("ACCESS_ROM Returns %X\n", returnvalue);
    #endif
    
    return returnvalue;
}

WORD access_rom_16(int rw, DWORD address, WORD data16, int rp)
{
	WORD returnvalue;
	DWORD data;

	data=(DWORD)data16;
    
    // Shift Flash address making A2 the LSB
    
	#ifdef DEBUG
    printf("ACCESS_ROM_16: inp addr = %X, inp data = %X\n", address, data);
    #endif
    
    returnvalue =(WORD)access_bus(rw, address << 1, data, rp);
    
    #ifdef DEBUG
    printf("ACCESS_ROM_16 Returns %X\n", returnvalue);
    #endif
    
    return returnvalue;
}

/*
*******************************************************************************
*
* FUNCTION:         access_bus
*
* DESCRIPTION:      This routine manipulates the memory bus to do reads 
*                   and writes to any memory location.
*
* INPUT PARAMETERS: int rw - mode of READ, WRITE, SETUP, HOLD, or RS
*                   DWORD - address of access
*                   DWORD - data to write
*                   int rp - read or ignore port data
*
* RETURNS:          DWORD - returned data
*
*******************************************************************************
*/

DWORD access_bus(int rw, DWORD address, DWORD data, int rp)
{
	int i;

	// Preset SA-1110 or Cotulla pins to default values (all others set in Cotullajtag.h)
    
    clear_chip_selects();
	mem_output_enable(ENABLE);
	mem_write_enable(DISABLE);
	mem_rw_mode(WRITE);
    mem_data_driver(HIZ); 

    #ifdef DEBUG
    printf("Chain initial state\n");
    dump_chain();
    #endif
    
    set_address(address);

    //----------------------------------------------
    if(rw == READ)
	{
		#ifdef DEBUG
        printf("Read Mode\n");
        #endif
		mem_rw_mode(READ);
        set_pin_chip_select(address);
	}
    
    //----------------------------------------------
	if(rw == WRITE)
	{
		#ifdef DEBUG
        printf("Write Mode\n");
        #endif
        mem_write_enable(ENABLE);
		mem_output_enable(DISABLE);
        mem_data_driver(DRIVE);
        set_pin_chip_select(address);
        set_data(data);
	}
    
    //----------------------------------------------
	if(rw == SETUP || rw == HOLD)	// just like a write except WE, WE needs setup time
	{
		#ifdef DEBUG
        printf("Setup or Hold Mode\n");
        #endif
		mem_output_enable(DISABLE);
        mem_data_driver(DRIVE);
		set_data(data);
	}
    //----------------------------------------------
	if(rw == RS)	// setup prior to RD_nWR_OUT
	{
		#ifdef DEBUG
        printf("RS Mode\n");
        #endif
        mem_output_enable(DISABLE);
	}
	
    // Common finish
        
    putp(1,0,IP);	//Run-Test/Idle
	putp(1,0,IP);	//Run-Test/Idle
	putp(1,0,IP);	//Run-Test/Idle
	putp(1,0,IP);	//Run-Test/Idle
	putp(1,1,IP);	//select DR scan
	putp(1,0,IP);	//capture DR
	putp(1,0,IP);	//shift IR


    #ifdef ASSABET_PLATFORM

	int out_dat[300];
	for(i = 0; i < 291; i++)	// shift write data in to JTAG port and read data out
		out_dat[i] = putp(pin[i],0,rp);
    #endif

    #ifndef ASSABET_PLATFORM

	int out_dat[400];
	for(i = 1; i < 384; i++)	// shift write data in to JTAG port and read data out
		out_dat[i] = putp(pin[i-1],0,rp);
        
        // fudge factor because chain appears to be shifted by 1
        // DEBUG LATER
        
        putp(pin[0], 0, rp);
    #endif

    // if ASSABET defined and NOT LUBBOCK_SA1110
    #ifdef ASSABET_PLATFORM
        #ifndef LUBBOCK_SA1110
            putp(0,0,IP);   // extra clicks for the CPLDs
	        putp(0,0,IP);
        #endif
    #endif

    #ifdef SANDGATE_PLATFORM
            putp(0,0,IP);   // extra clicks for the CPLDs
	        putp(0,0,IP);
    #endif
    
	putp(0,1,IP);	//Exit1-DR
	putp(1,1,IP);	//Update-DR
	putp(1,0,IP);	//Run-Test/Idle
	putp(1,0,IP);	//Run-Test/Idle
	putp(1,0,IP);	//Run-Test/Idle

	DWORD busdat = 0;

	for(i = 0; i < 32; i++)	// convert serial data to single DWORD
	{

    #ifdef ASSABET_PLATFORM

    busdat = busdat | (DWORD)(out_dat[dat_order[i] - 2] << i);

    #endif

    #ifdef LUBBOCK_PLATFORM

    busdat = busdat | ((DWORD)(out_dat[input_dat_order[i]] << i));

    #endif

    #ifdef SANDGATE_PLATFORM
    
    // mysterious behavior here. There should be a "-2" compensation 
    // here like the ASSABET_PLATFORM. DEBUG THIS LATER BUT IT WORKS FOR NOW.
    
    busdat = busdat | ((DWORD)(out_dat[input_dat_order[i] ] << i));

    #endif

	}
	
    extest();

#ifdef DEBUG
    printf("just before return\n");
    dump_chain();
#endif
    
    return(busdat);

}




/*
*******************************************************************************
*
* FUNCTION:         test_port
*
* DESCRIPTION:      Searches for a valid parallel port
*
* INPUT PARAMETERS: void
*
* RETURNS:          int - Address of the port or zero if none available
*
*******************************************************************************
*/

#ifdef __linux__

int io_access_on( unsigned long port )
{
	if (ioperm (port, 3, 1)) {
		perror ("ioperm()");
		return 0;
	}
	if (ioperm (0x80, 1, 1)) {
		perror ("ioperm()");
		return 0;
	}
	return 1;
}

void io_access_off( unsigned long port )
{
	ioperm (port, 3, 0);
	ioperm (0x80, 1, 0);
}

#else
#define io_access_on(x) (1)
#define io_access_off(x)
#endif


int test_port(void)
{
	// search for valid parallel port
	if(io_access_on(LPT1))
	{
		_outp(LPT1, 0x55);
		if((int)_inp(LPT1) == 0x55)
		{
			#ifdef DEBUG
			printf("Parallel Com port found at I/O address: %X\n", LPT1);
			#endif
			return LPT1;
		}
		io_access_off(LPT1);
	}
	if(io_access_on(LPT2))
	{
                _outp(LPT2, 0x55);
		if((int)_inp(LPT2) == 0x55)
		{
			#ifdef DEBUG
			printf("Parallel Com port found at I/O address: %X\n", LPT2);
			#endif
			return LPT2;
		}
		io_access_off(LPT2);
	}
	if(io_access_on(LPT3))
	{
		_outp(LPT3, 0x55);
	if((int)_inp(LPT3) == 0x55)
		{
			#ifdef DEBUG
			printf("Parallel Com port found at I/O address: %X\n", LPT3);
			#endif
			return LPT3;
		}
		io_access_off(LPT3);
	}

	return(0);	// return zero if none found
}
/*
*******************************************************************************
*
* FUNCTION:         check_id
*
* DESCRIPTION:      Compare an ID string returned from the device with the expected string.
*
* INPUT PARAMETERS: char * device_id - a pointer to the string returned from the device
*
* RETURNS:          int - 0 if ID matches expected, -1 if a match fails
*
*******************************************************************************
*/

int check_id(char *device_id)
{
	// compare passed device ID to the one returned from the ID command
	char in_id[40];
	BOOL error_flag = FALSE;


	for(int i = 34; i >= 0; i--)
	{
		// skip over the spaces in the ID string
        if(i == 4 || i == 21 || i == 33)
		{
			in_id[i] = ' ';
			i--;
		}
        
		if(putp(1,0,READ_PORT) == 0)
			in_id[i] = '0';
		else
			in_id[i] = '1';

		if((in_id[i] != *(device_id + i)) && (*(device_id + i) != '*'))
		{
			error_flag = TRUE;
			
		}
	}
	in_id[35] = 0;

	
#ifdef DEBUG

	if(error_flag)
	{
		printf("error, failed to read device ID\n");
		printf("check cables and power\n");
		printf("ACT: %s\n",in_id);
		printf("EXP: %s\n\n",device_id);
		return -1;
	}
#endif    
#ifdef ASSABET_PLATFORM

	if(!strcmp(device_id,SA1110ID))	// print SA-1110 device revision
	{
		int sa_rev =
			(int)(in_id[0] - '0') * 8 +
			(int)(in_id[1] - '0') * 4 +
			(int)(in_id[2] - '0') * 2 +
			(int)(in_id[3] - '0');
		switch(sa_rev)
		{
			case 0: printf("SA-1110 revision A0\n"); break;
			case 4: printf("SA-1110 revision B0\n"); break;
			case 5: printf("SA-1110 revision B1\n"); break;
			case 6: printf("SA-1110 revision B2\n"); break;
			case 8: printf("SA-1110 revision B4\n"); break;
			default: printf("SA-1110 revision B4 + %d\n",sa_rev - 8);
		}

	}	
#endif

#ifdef SANDGATE_PLATFORM

	if(!strcmp(device_id, COTULLAID))	// print Cotulla device revision
	{
		int sa_rev =
			(int)(in_id[0] - '0') * 8 +
			(int)(in_id[1] - '0') * 4 +
			(int)(in_id[2] - '0') * 2 +
			(int)(in_id[3] - '0');
		switch(sa_rev)
		{
			case 0: printf("PXA250 revision A0\n\n"); break;
			case 1: printf("PXA250 revision A1\n\n"); break;
			case 2: printf("PXA250 revision B0\n\n"); break;
			case 3: printf("PXA250 revision B1\n\n"); break;
			case 4: printf("PXA250 revision ??\n\n"); break;
			case 5: printf("PXA250 revision ??\n\n"); break;
			case 6: printf("PXA250 revision ??\n\n"); break;
			case 8: printf("PXA250 revision ??\n\n"); break;
			default: printf("PXA250 revision ?? + %d\n\n",sa_rev - 8);
		}

	}
#endif

#ifdef LUBBOCK_PLATFORM
#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)

	if(!strcmp(device_id, COTULLAID))	// print Cotulla device revision
	{
		int sa_rev =
			(int)(in_id[0] - '0') * 8 +
			(int)(in_id[1] - '0') * 4 +
			(int)(in_id[2] - '0') * 2 +
			(int)(in_id[3] - '0');
		switch(sa_rev)
		{
			case 0: printf("PXA250 revision A0\n\n"); break;
			case 1: printf("PXA250 revision A1\n\n"); break;
			case 2: printf("PXA250 revision B0\n\n"); break;
			case 3: printf("PXA250 revision B1\n\n"); break;
			case 4: printf("PXA250 revision B2\n\n"); break;
			case 5: printf("PXA250 revision ??\n\n"); break;
			case 6: printf("PXA250 revision ??\n\n"); break;
			case 8: printf("PXA250 revision ??\n\n"); break;
			default: printf("PXA250 revision ?? + %d\n\n",sa_rev - 8);
		}

	}
#endif
#endif

#if defined(LUBBOCK_SABINAL) || defined(LUBBOCK_DALHART_16)

	if((!strcmp(device_id, SABINALID))||(!strcmp(device_id, COTULLAID))) // print SABINAL device revision
	{
		int sa_rev =
			(int)(in_id[0] - '0') * 8 +
			(int)(in_id[1] - '0') * 4 +
			(int)(in_id[2] - '0') * 2 +
			(int)(in_id[3] - '0');
		switch(sa_rev)
		{
#ifdef LUBBOCK_SABINAL
			case 0: printf("PXA210 revision A0\n\n"); break;
			case 1: printf("PXA210 revision A1\n\n"); break;
			case 2: printf("PXA210 revision B0\n\n"); break;
			case 3: printf("PXA210 revision B1\n\n"); break;
			case 4: printf("PXA210 revision B2\n\n"); break;
			case 5: printf("PXA210 revision ??\n\n"); break;
			case 6: printf("PXA210 revision ??\n\n"); break;
			case 8: printf("PXA210 revision ??\n\n"); break;
			default: printf("PXA210 revision ?? + %d\n\n",sa_rev - 8);
#else
			case 0: printf("DALHART 16 revision A0\n\n"); break;
			case 1: printf("DALHART 16 revision A1\n\n"); break;
			case 2: printf("DALHART 16 revision B0\n\n"); break;
			case 3: printf("DALHART 16 revision B1\n\n"); break;
			case 4: printf("DALHART 16 revision B2\n\n"); break;
			case 5: printf("DALHART 16 revision ??\n\n"); break;
			case 6: printf("DALHART 16 revision ??\n\n"); break;
			case 8: printf("DALHART 16 revision ??\n\n"); break;
			default: printf("DALHART 16 revision ?? + %d\n\n",sa_rev - 8);
#endif
		}

	}
#endif


	return 0;
}
/*
*******************************************************************************
*
* FUNCTION:         error_out
*
* DESCRIPTION:      generic error printout and program exit.
*
* INPUT PARAMETERS: char * error_string to print before exit
*
* RETURNS:          void
*
* GLOBAL EFFECTS:   Exits the program
*******************************************************************************
*/

void error_out(char *error_string)
{
	printf("%s\n",error_string);
	exit(0);
}
/*
*******************************************************************************
*
* FUNCTION:         erase_flash
*
* DESCRIPTION:      erases a selected number of flash blocks
*
* INPUT PARAMETERS: DWORD base_address of flash device
*                   DWORD fsize - size of flash device
*                   DWORD block_size - block size of the flash device
*                   DWORD max_erase_time - for timeout value
*                   int block_number - number of blocks to erase
*
* RETURNS:          void
*
*******************************************************************************
*/

void erase_flash(DWORD base_address, DWORD fsize, DWORD block_size, DWORD max_erase_time, int block_number)
{
	time_t start, now;

	printf("Starting erase\n");

	for(DWORD lj = base_address; lj < fsize + base_address; lj = lj + block_size)  // Erase only blocks to be programmed
		{
		access_rom(SETUP, 0, F_BLOCK_ERASE, IGNORE_PORT); // Erase block command
		access_rom(WRITE, 0, F_BLOCK_ERASE, IGNORE_PORT);
		access_rom(HOLD, 0, F_BLOCK_ERASE, IGNORE_PORT);

		access_rom(SETUP, lj, F_BLOCK_ERASE_2ND, IGNORE_PORT); // Erase confirm at the block address
		access_rom(WRITE, lj, F_BLOCK_ERASE_2ND, IGNORE_PORT);
		access_rom(HOLD, lj, F_BLOCK_ERASE_2ND, IGNORE_PORT);

		time(&start);
		printf("Erasing block %3d   \r",block_number++);
		while(access_rom(RS, 0, 0, READ_PORT) != F_STATUS_READY)	// Loop until successful status return
			{
			access_rom(READ, 0, 0, READ_PORT);
			time(&now);
			if(difftime(now,start) > max_erase_time + 1)	// Check for erase timeout
				error_out("Error, Block erase timed out");
			}
		}
	printf("Erasing done                                           \n");
}
/*
*******************************************************************************
*
* FUNCTION:         program_flash
*
* DESCRIPTION:      program the flash using buffered writes
*
* INPUT PARAMETERS: DWORD max_write_buffer derived from the flash query
*                   DWORD base_address
*                   DWORD fsize (flash size)
*
* RETURNS:          void
*
*******************************************************************************
*/

void program_flash(DWORD max_write_buffer, DWORD base_address, DWORD fsize)
{
	time_t start, now;
	BUS_GRAIN li;

	printf("Starting programming\n");


	// "Write Buffer" flow.
	// This uses almost half the cycles required by "word programming" flow
	// Status register is not read to save time.  There is also no checking to see
	// if maximum "Write Buffer Program Time" is violated.  However even with the
	// fastest parallel port bus speed this should not be a problem
	// (i.e. 16 words * 300 JTAG chain length * 4 parallel port cycles * 1uS fast
	// parallel port cycle = 19mS, typical write buffer program times are in the 200uS range).

#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
	DWORD write_word_count = (max_write_buffer - 1) + ((max_write_buffer - 1) << 16);
#else
	WORD write_word_count = (WORD)(max_write_buffer - 1);
#endif

	time(&start);

	for(DWORD lj = base_address; lj < fsize + base_address; lj = lj + max_write_buffer)
		{
		access_rom(WRITE, lj, F_WRITE_BUFFER, IGNORE_PORT); // write buffer command
		access_rom(HOLD, lj, F_WRITE_BUFFER, IGNORE_PORT);

		access_rom(WRITE, lj, write_word_count, IGNORE_PORT); // write word count (max write buffer size)
		access_rom(HOLD, lj, write_word_count, IGNORE_PORT);

		time(&now);
		if(difftime(now,start) > STATUS_UPDATE)	// Update status every 2 seconds
			{
			printf("Writing flash at hex address %8lx, %5.2f%% done    \r"
				,lj * 4,(float)(lj - base_address)/(float)fsize*100.0);
#ifdef __linux__
			  fflush(0);
#endif
			time(&start);
			}

		for(DWORD lk = 0; lk < max_write_buffer; lk++)
			{
			fread((BUS_GRAIN *)&li, sizeof(BUS_GRAIN) , 1, in_file);
			access_rom(WRITE, lj+lk, li, IGNORE_PORT);  // Write buffer data
			access_rom(HOLD, lj+lk, li, IGNORE_PORT);  // New
			}
	
#if !defined(LUBBOCK_SABINAL) && !defined(LUBBOCK_DALHART_16)
		access_rom(WRITE, 0, 0xd000d0L, IGNORE_PORT); // Program Buffer to Flash Confirm
		access_rom(HOLD, 0, 0xd000d0L, IGNORE_PORT);  //New
#else
		access_rom(WRITE, 0, 0x00d0L, IGNORE_PORT); // Program Buffer to Flash Confirm
		access_rom(HOLD, 0, 0x00d0L, IGNORE_PORT);  //New
#endif
		}

	printf("Programming done\n");

	rewind(in_file);
}
/*
*******************************************************************************
*
* FUNCTION:         verify_flash
*
* DESCRIPTION:      compares data programmed in flash with the original binary file.
*
* INPUT PARAMETERS: DWORD base_address
*                   DWORD flash_size
*
* RETURNS:          void
*
*******************************************************************************
*/

void verify_flash(DWORD base_address, DWORD fsize)
{
	time_t start, now;
	BUS_GRAIN li, li1;

	    printf("Starting Verify\n");

		time(&start);
		for(DWORD lj = base_address + 1; lj <= fsize + base_address; lj++)
			{
			fread((BUS_GRAIN *)&li, sizeof(BUS_GRAIN) , 1, in_file);
            // Toggle chip select for K3 flash.
			access_rom(RS, lj, 0x0L, READ_PORT);
			li1 = access_rom(READ, lj, 0x0L, READ_PORT);
			time(&now);
			if(difftime(now,start) > STATUS_UPDATE)	// Update status every 2 seconds
				{
				printf("Verifying flash at hex address %8lx, %5.2f%% done    \r"
					,lj * 4,(float)(lj - base_address)/(float)fsize*100.0);
#ifdef __linux__
				fflush(0);
#endif
				time(&start);
				}
			if(li != li1)
				{
				printf("verify error at address = %lx exp_dat = %lx act_dat = %lx\n",lj - 1,li,li1);
				exit(1);
				}
			}


	printf("Verification successful!                                                    \n");
}
/*
*******************************************************************************
*
* FUNCTION:         test_logic_reset
*
* DESCRIPTION:      initializes the JTAG state machine to a known state
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
 
void test_logic_reset(void)
{
    #ifdef DEBUG
    printf("begin test logic reset\n");
    #endif
	
    putp(1,1,IGNORE_PORT);	// keep TMS set to 1 force a test logic reset
	putp(1,1,IGNORE_PORT);	// no matter where you are in the TAP controller
	putp(1,1,IGNORE_PORT);
	putp(1,1,IGNORE_PORT);
	putp(1,1,IGNORE_PORT);
	putp(1,1,IGNORE_PORT);
    
    #ifdef DEBUG
    printf("finish test logic reset\n");
    #endif
    
    
}
/*
*******************************************************************************
*
* FUNCTION:         test_lock_flash
*
* DESCRIPTION:      Tests whether a specified block is locked and if it is, asks
*                   if you would like to unlock it. It will perform an unlock 
*                   procedure if requested.
*
* INPUT PARAMETERS: DWORD base_address of flash (16-bit address for each device)
*                   DWORD fsize - size of flash (if Sabinal 16-bit data bus used, this is the 16-bit
												 word number, if 32-bit data bus used, this is the 32
												 -bit DWORD number)
*                   DWORD block_size - block size of flash (1-bit word number)
*                   DWORD max_erase_time - used for a timeout (s number)
*                   int block_number - block number of interest
*
* RETURNS:          void
*
*
*******************************************************************************
*/


void test_lock_flash(DWORD base_address, DWORD fsize, DWORD block_size, 
                     DWORD max_erase_time, int block_number)
{
	time_t start, now;
    DWORD lockstatus;

	for(DWORD lj = base_address; lj < fsize + base_address; lj = lj + block_size)  // Test only blocks to be programmed
		{
		access_rom(SETUP, 0, F_READ_IDCODES, IGNORE_PORT); // Read Identifier Codes
		access_rom(WRITE, 0, F_READ_IDCODES, IGNORE_PORT);
		access_rom(HOLD, 0, F_READ_IDCODES, IGNORE_PORT);

		access_rom(READ, lj + 2, 0, IGNORE_PORT); // read lock configuration (extra read to get JTAG pipeline going)
		lockstatus = access_rom(READ, lj + 2, 0, READ_PORT);
        
        if((lockstatus == 0x10001) || (lockstatus == 0x10000) || (lockstatus == 0x00001))
			{
			printf("Block of Flash Memory is Write Locked, would you like to unlock it? [y/n]: ");
			if(toupper(_getche()) == 'Y')
				{
			
				printf("\nblock is locked\n");
	
				access_rom(SETUP, 0, F_CLEAR_BLOCK_LOCK, IGNORE_PORT); // Clear block lock bit command
				access_rom(WRITE, 0, F_CLEAR_BLOCK_LOCK, IGNORE_PORT);
				access_rom(HOLD, 0, F_CLEAR_BLOCK_LOCK, IGNORE_PORT);

				access_rom(SETUP, lj, F_CLEAR_BLOCK_LOCK_2ND, IGNORE_PORT); // Confirm
				access_rom(WRITE, lj, F_CLEAR_BLOCK_LOCK_2ND, IGNORE_PORT);
				access_rom(HOLD, lj, F_CLEAR_BLOCK_LOCK_2ND, IGNORE_PORT);

				time(&start);
				printf("Unlocking block %3d   \r",block_number++);
				while(access_rom(RS, 0, 0, READ_PORT) != F_STATUS_READY)	// Loop until successful status return 0x0080
					{
					access_rom(READ, 0, 0, READ_PORT);
					time(&now);
					if(difftime(now,start) > max_erase_time + 1)	// Check for status timeout
						error_out("\nError, Clear lock timed out");
					}
				}
			else
				error_out("\nUnable to program Write Locked Flash Memory Block");
			}
		}
}
/*
*******************************************************************************
*
* FUNCTION:         set_lock_flash
*
* DESCRIPTION:      sets locks bits in specified block
*
* INPUT PARAMETERS: DWORD base_address of flash
*                   DWORD fsize - size of flash
*                   DWORD block_size - block size of flash
*                   DWORD max_erase_time - used for a timeout 
*                   int block_number - block number of interest
*
* RETURNS:          void
*
*******************************************************************************
*/


void set_lock_flash(DWORD base_address, DWORD fsize, DWORD block_size, DWORD max_erase_time, int block_number)
{
	time_t start, now;

	printf("Starting set block lock bit\n");

	for(DWORD lj = base_address; lj < fsize + base_address; lj = lj + block_size)  // locks only blocks to be programmed
		{
		access_rom(SETUP, 0, F_SET_BLOCK_LOCK, IGNORE_PORT); //  block lock bit command
		access_rom(WRITE, 0, F_SET_BLOCK_LOCK, IGNORE_PORT);
		access_rom(HOLD, 0, F_SET_BLOCK_LOCK, IGNORE_PORT);

		access_rom(SETUP, lj, F_SET_BLOCK_LOCK_2ND, IGNORE_PORT); // Confirm
		access_rom(WRITE, lj, F_SET_BLOCK_LOCK_2ND, IGNORE_PORT);
		access_rom(HOLD, lj, F_SET_BLOCK_LOCK_2ND, IGNORE_PORT);

		time(&start);
		printf("Erasing block %3d   \r",block_number++);
		while(access_rom(RS, 0, 0, READ_PORT) != 0x800080L)	// Loop until successful status return
			{
			access_rom(READ, 0, 0, READ_PORT);
			time(&now);
			if(difftime(now,start) > max_erase_time + 1)	// Check for status timeout
				error_out("Error, Clear lock timed out");
			}
		}
	printf("Set lock bit done                                           \n");
}

/*
*******************************************************************************
*
* FUNCTION:         set_address
*
* DESCRIPTION:      Loads the address into the address bits
*
* INPUT PARAMETERS: address
*
* RETURNS:          void
*
* GLOBAL EFFECTS:   None
*
* ASSUMPTIONS:      None
*
* CALLS:            None
*
* CALLED BY:        Anyone
*
* PROTOTYPE:        void set_address(unsigned int address);
*
*******************************************************************************
*/
void set_address (DWORD address)
{
    unsigned int i;
    
    for (i = 0; i < 26; i++)
    {
        pin[addr_order[i]] = (int)((address >> i) & 1);
    }
}

/*
*******************************************************************************
*
* FUNCTION:         set_data
*
* DESCRIPTION:      Fills the chain with the data bits
*
* INPUT PARAMETERS: DWORD data
*
* RETURNS:          void
*
*******************************************************************************
*/
void set_data(DWORD data)
{
    DWORD i;
    
    for(i = 0; i < 32; i++)
    {
        pin[dat_order[i]] = (int)((data >> i) & 1);	// set data pins
    }
}
/*
*******************************************************************************
*
* FUNCTION:         shift_data (Not used at this time)
*
* DESCRIPTION:      Extracts the data from the chain 
*
* INPUT PARAMETERS: Void
*
* RETURNS:          DWORD data
*
*******************************************************************************
*/
DWORD shift_data(int rp)
{
    int i;
    DWORD busdat = 0;
    int tms = 0;

    #ifdef SANDGATE_PLATFORM
    int dr_length = SANDGATE_WORKBUF_SIZE;
    #endif

    #ifdef LUBBOCK_PLATFORM
    int dr_length = LUBBOCK_WORKBUF_SIZE;
    #endif

    #ifdef ASSABET_PLATFORM
    #ifndef LUBBOCK_SA1110
    int dr_length = ASSABET_WORKBUF_SIZE;
    #else
    int dr_length = ASSABET_WORKBUF_SIZE - 2;
    #endif
    #endif




    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,1,IGNORE_PORT);	//select DR scan
	putp(1,0,IGNORE_PORT);	//capture DR
	putp(1,0,IGNORE_PORT);	//shift IR
    

	for(i = 0; i < dr_length; i++)	// shift write data into JTAG port and read data out
	{
        out_dat[i] = putp((int)WORKBUF[i], 0, rp);     // fill the global out_dat array
    }
    
    // replace the work buffer with the captured data
    
    
    for(i = 0; i < dr_length; i++)
    {
        WORKBUF[i] = out_dat[i];
    
    }
    
	putp(0,1,IGNORE_PORT);	//Exit1-DR
	putp(1,1,IGNORE_PORT);	//Update-DR
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle



	for(i = 0; i < 32; i++)	// convert serial data to single DWORD
	{
		busdat = busdat | (DWORD)((int)WORKBUF[input_dat_order[i] + DEVICES_AFTER ] << i);
        
	}
    #ifdef DEBUG
    printf("Boundary Chain After Scan out\n");
    dump_chain();
    #endif
    return busdat;
    
}
/*
*******************************************************************************
*
* FUNCTION:         set_pin_chip_select
*
* DESCRIPTION:      Sets chip selects depending on the address and the platform
*
* INPUT PARAMETERS: DWORD address
*
* RETURNS:          void
*
*******************************************************************************
*/

void set_pin_chip_select(DWORD address)
{
#ifdef ASSABET_PLATFORM
    switch(address >> 27)
    {
	    case 0:{pin[nCS0_OUT] = 0; break;}
	    case 1:{pin[nCS1_OUT] = 0; break;}
	    case 2:{pin[nCS2_OUT] = 0; break;}
	    case 3:{pin[nCS3_OUT] = 0; break;}
	    case 4:{pin[nCS4_OUT] = 0; break;}
	    case 5:{pin[nCS5_OUT] = 0; break;}
    }
#endif

#ifdef SANDGATE_PLATFORM
    if(address  <  0x04000000)                            pin[nCS0_OUT] = 0;
    if((address >= 0x04000000) && (address < 0x08000000)) pin[nCS1_OUT] = 0;
    if((address >= 0x08000000) && (address < 0x0c000000)) pin[nCS2_OUT] = 0;
    if((address >= 0x0c000000) && (address < 0x10000000)) pin[nCS3_OUT] = 0;
    if((address >= 0x10000000) && (address < 0x14000000)) pin[nCS4_OUT] = 0;
    if((address >= 0x14000000) && (address < 0x18000000)) pin[nCS5_OUT] = 0;
#endif

#ifdef LUBBOCK_PLATFORM
    if(address  <  0x02000000)                            pin[nCS0_OUT] = 0;
    if((address >= 0x04000000) && (address < 0x06000000)) pin[nCS1_OUT] = 0;
    if((address >= 0x08000000) && (address < 0x08000100)) pin[nCS2_OUT] = 0;
    if((address >= 0x0A000000) && (address < 0x0A100000)) pin[nCS2_OUT] = 0;
    if((address >= 0x0c000000) && (address < 0x0E100000)) pin[nCS3_OUT] = 0;
    if((address >= 0x10000000) && (address < 0x10400000)) pin[nCS4_OUT] = 0;
    if((address >= 0x14000000) && (address < 0x18000000)) pin[nCS5_OUT] = 0;

#endif

#ifdef LUBBOCK_SA1110
    switch(address >> 27)
    {
	    case 0:{pin[nCS0_OUT] = 0; break;}
	    case 1:{pin[nCS1_OUT] = 0; break;}
	    case 2:{pin[nCS2_OUT] = 0; break;}
	    case 3:{pin[nCS3_OUT] = 0; break;}
	    case 4:{pin[nCS4_OUT] = 0; break;}
	    case 5:{pin[nCS5_OUT] = 0; break;}
    }

#endif



}

/*
*******************************************************************************
*
* FUNCTION:         set_chip_select (not used at this time)
*
* DESCRIPTION:      Sets the chip select lines relative to the address required.
*
* INPUT PARAMETERS: DWORD address
*
* RETURNS:          void
*
*******************************************************************************
*/

void set_chip_select(DWORD address)
{

    // memory has 26 bits of byte address and 6 chip selects. This means 64 Mb
    // for each bank and we can have 6 banks for a total of 384 Mb
    switch(address >> 27)
    {
        case 0:{WORKBUF[nCS0_OUT + DEVICES_AFTER] = 0; break;}
        case 1:{WORKBUF[nCS1_OUT + DEVICES_AFTER] = 0; break;}
        case 2:{WORKBUF[nCS2_OUT + DEVICES_AFTER] = 0; break;}
        case 3:{WORKBUF[nCS3_OUT + DEVICES_AFTER] = 0; break;}
        case 4:{WORKBUF[nCS4_OUT + DEVICES_AFTER] = 0; break;}
        case 5:{WORKBUF[nCS5_OUT + DEVICES_AFTER] = 0; break;}
    }
}

/*
*******************************************************************************
*
* FUNCTION:         clear_chip_selects
*
* DESCRIPTION:      reset all chip selects
*
* INPUT PARAMETERS: None
*
* RETURNS:          none
*
*******************************************************************************
*/

void clear_chip_selects()
{

    // Preset to default values (all others set in cotullajtag.h)
    pin[nCS0_OUT] = 1;
    pin[nCS1_OUT] = 1;
    pin[nCS2_OUT] = 1;
    pin[nCS3_OUT] = 1;
    pin[nCS4_OUT] = 1;
    pin[nCS5_OUT] = 1;

}
/*
*******************************************************************************
*
* FUNCTION:         mem_output_enable
*
* DESCRIPTION:      enable or disable memory output. This pin is connected to 
*                   the output enables of the memory device.
*
* INPUT PARAMETERS: int - enable or disable
*
* RETURNS:          void
*
*******************************************************************************
*/
void mem_output_enable(int endis)
{
    if (endis == ENABLE)
    {
	    pin[nOE_OUT] = 0;
    } 
    else
    {
	    pin[nOE_OUT] = 1;
    }
}

/*
*******************************************************************************
*
* FUNCTION:         mem_write_enable
*
* DESCRIPTION:      enable or disable memory writes. This pin is connected to 
*                   the write enables of the memory device.
*
* INPUT PARAMETERS: int - enable or disable
*
* RETURNS:          void
*
*******************************************************************************
*/
void mem_write_enable(int endis)
{
    if (endis == ENABLE)
    {
	    pin[nWE_OUT] = 0;
    } 
    else
    {
	    pin[nWE_OUT] = 1;
    }
}

/*
*******************************************************************************
*
* FUNCTION:         mem_data_driver
*
* DESCRIPTION:      Sets memory data pins to DRIVE or HIZ. 
*
* INPUT PARAMETERS: int - drive or highz
*
* RETURNS:          void
*
*******************************************************************************
*/

void mem_data_driver(int df)
{
#ifndef ASSABET_PLATFORM
    if (df == DRIVE)
    {
	    pin[mdupper_ctrl] = 1;  
        pin[mdlower_ctrl] = 1;
    } 
    else
    {
	    pin[mdupper_ctrl] = 0;  
        pin[mdlower_ctrl] = 0;
    }
#else
    if (df == DRIVE)
    {
	    pin[D31_0_EN] = 0;  
    } 
    else
    {
	    pin[D31_0_EN] = 1;  
    }
#endif // !ASSABET_PLATFORM

}

/*
*******************************************************************************
*
* FUNCTION:         mem_rw_mode
*
* DESCRIPTION:      Sets memory mode to READ or WRITE. 
*
* INPUT PARAMETERS: int - READ or WRITE
*
* RETURNS:          void
*
*******************************************************************************
*/
void mem_rw_mode(int rw)
{
    if (rw == WRITE)
    {
        pin[RD_nWR_OUT] = 0;
    } 
    else
    {
        pin[RD_nWR_OUT] = 1;
    }
}

/*
*******************************************************************************
*
* FUNCTION:         pre_IRSCAN
*
* DESCRIPTION:      Sets up the state machine to accept an IR SCAN
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
void pre_IRSCAN()
{
    #ifdef DEBUG
    printf("begin pre-IR scan code\n");
    #endif

    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,1,IGNORE_PORT);
    putp(1,1,IGNORE_PORT);	//select IR scan
    putp(1,0,IGNORE_PORT);	//capture IR
    putp(1,0,IGNORE_PORT);	//shift IR
}

/*
*******************************************************************************
*
* FUNCTION:         post_IRSCAN
*
* DESCRIPTION:      Get back to IDLE after scanning in the instruction
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
void post_IRSCAN()
{
    #ifdef DEBUG
    printf("begin post-IR scan code\n");
    #endif

   	//putp(1,1,IGNORE_PORT);	//Exit1-IR
	putp(1,1,IGNORE_PORT);	//Update-IR
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
}

/*
*******************************************************************************
*
* FUNCTION:         pre_DRSCAN
*
* DESCRIPTION:      Sets up the state machine to accept an DR SCAN
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
void pre_DRSCAN()
{
    #ifdef DEBUG
    printf("begin pre-DR scan code\n");
    #endif
    
    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,0,IGNORE_PORT);	//Run-Test/Idle
    putp(1,1,IGNORE_PORT);  //select DR scan
    putp(1,0,IGNORE_PORT);	//capture DR
 //   putp(1,0,IGNORE_PORT);	//shift DR
}

/*
*******************************************************************************
*
* FUNCTION:         post_DRSCAN
*
* DESCRIPTION:      Get back to IDLE after scanning in the data register
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
void post_DRSCAN()
{
    #ifdef DEBUG
    printf("begin post-DR scan code\n");
    #endif
	
    putp(1,1,IGNORE_PORT);	//Exit1-DR
	putp(1,1,IGNORE_PORT);	//Update-DR
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
	putp(1,0,IGNORE_PORT);	//Run-Test/Idle
}

/*
*******************************************************************************
*
* FUNCTION:         controller_scan_code
*
* DESCRIPTION:      clocks in a specified cotulla IR scan code
*
* INPUT PARAMETERS: int instruction code
*                   int read or ignore port
*                   int continue or terminate instruction stream
*
* RETURNS:          void
*
*******************************************************************************
*/
int controller_scan_code(int code, int rp, int ct)
{
    int i;
    int outval = 0;
    int tms = 0;
    
    #ifdef DEBUG
    printf("begin controller scan code\n");
    #endif
    
    for (i = 0; i < COTULLA_IRLENGTH; i++)
    {
        if (ct == TERMINATE) 
        {
            if (i == COTULLA_IRLENGTH -1)
            {
                tms = 1;
            }
        }
        outval |= putp(((code & (1 << i)) >> i), tms, rp) << (COTULLA_IRLENGTH - i - 1);
        
    }
    #ifdef DEBUG
    printf("Controller IR value: %X\n", outval);
    #endif
    return outval;
}

/*
*******************************************************************************
*
* FUNCTION:         PZ_scan_code
*
* DESCRIPTION:      clocks in a specified PZxxxx IR scan code
*
* INPUT PARAMETERS: int code
*
* RETURNS:          void
*
*******************************************************************************
*/
int PZ_scan_code(int code, int rp, int ct)
{
    int i;
    int outval = 0;
    int tms = 0;
    
    #ifdef DEBUG
    printf("begin PZ scan code\n");
    #endif
    
    for (i = 0; i < PZ_IRLENGTH; i++)
    {
        if (ct == TERMINATE) 
        {
            if (i == PZ_IRLENGTH -1)
            {
                tms = 1;
            }
        }
        
        outval |= putp(((code & (1 << i)) >> i), tms, rp) << (PZ_IRLENGTH - i - 1);
    }
    return outval;
}

/*
*******************************************************************************
*
* FUNCTION:         jtag_test
*
* DESCRIPTION:      tests the JTAG connection by reading the steady state 
*                   instruction register.
*
* INPUT PARAMETERS: void
*
* RETURNS:          int - 0 if passed
*
*******************************************************************************
*/
void jtag_test()
{
    // set all devices into bypass mode as a safe instruction

    pre_IRSCAN();

#ifdef SANDGATE_PLATFORM 

    if (controller_scan_code(COT_BYPASS, READ_PORT, CONTINUE) != 0x1)
    {
		error_out("Jtag test failure. Check connections and power.\n");    
    }
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, CONTINUE);    // PZ3128
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, TERMINATE);    // PZ3032
    
#endif //SANDGATE_PLATFORM

#ifdef ASSABET_PLATFORM 
#ifndef LUBBOCK_SA1110
    if (controller_scan_code(SA_BYPASS, READ_PORT, CONTINUE) != 0x1)
    {
		error_out("Jtag test failure. Check connections and power.\n");    
    }
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, CONTINUE);    // PZ3128
    PZ_scan_code(PZ_BYPASS, IGNORE_PORT, TERMINATE);    // PZ3032
#else
    if (controller_scan_code(SA_BYPASS, READ_PORT, TERMINATE) != 0x1)
    {
		error_out("Jtag test failure. Check connections and power.\n");    
    }

#endif    
#endif //SANDGATE_PLATFORM

#ifdef LUBBOCK_PLATFORM 

    if (controller_scan_code(COT_BYPASS, READ_PORT, TERMINATE) != 0x1)
    {
        error_out("Jtag test failure. Check connections and power.\n");    
    }
    
#endif //LUBBOCK_PLATFORM
 
    post_IRSCAN();
    
    printf("JTAG Test Passed\n");

}
/*
*******************************************************************************
*
* FUNCTION:         dump_chain
*
* DESCRIPTION:      This is a debug routine that dumps the contents of the 
*                   current boundary chain to the standard I/O.
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
void dump_chain()
{
    DWORD addrdat = 0;
    DWORD obusdat = 0;
    DWORD ibusdat = 0;

    int i;
    
    
    printf("------------------------------------------------------\n");

	for(i = 0; i < 32; i++)	// convert serial data to single DWORD
	{
		obusdat = obusdat | (DWORD)((int)pin[dat_order[i]] << i);
    }
    printf("Data Bus (Output) = %X\n", obusdat);
    
	for(i = 0; i < 32; i++)	// convert serial data to single DWORD
	{
		ibusdat = ibusdat | (DWORD)((int)pin[input_dat_order[i]] << i);
    }
    printf("Data Bus (Input) = %X\n", ibusdat);

    for(i = 0; i < 26; i++) // convert address data into single DWORD
    {
        addrdat = addrdat | (DWORD)((int)pin[addr_order[i]] << i);
    }
    
    
    printf("Address Bus = %X\n", addrdat);
    printf("Flash address is: %X\n",addrdat >> 2);
    
    printf("Chip Select 0 = %d\n", (int)pin[nCS0_OUT]);
    printf("Chip Select 1 = %d\n", (int)pin[nCS1_OUT]);
    printf("Chip Select 2 = %d\n", (int)pin[nCS2_OUT]);
    printf("Chip Select 3 = %d\n", (int)pin[nCS3_OUT]);
    printf("Chip Select 4 = %d\n", (int)pin[nCS4_OUT]);
    printf("Chip Select 5 = %d\n", (int)pin[nCS5_OUT]);

    printf("Mem Out Enable      = %d\n", (int)pin[nOE_OUT]);    
    printf("Mem Write Enable    = %d\n", (int)pin[nWE_OUT]);    
    printf("Mem READ/WRITE Mode = %d\n", (int)pin[RD_nWR_OUT]);    
    printf("------------------------------------------------------\n");

}   



/*
*******************************************************************************
*
* FUNCTION:         invert_workbuf (not used at this time)
*
* DESCRIPTION:      inverts the workbuffer 
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
void invert_workbuf()
{
    int i;
    
    #ifdef SANDGATE_PLATFORM
    int tempbuf[SANDGATE_WORKBUF_SIZE];
	int size = SANDGATE_WORKBUF_SIZE;
    #endif

    #ifdef LUBBOCK_PLATFORM
    int tempbuf[LUBBOCK_WORKBUF_SIZE];
	int size = LUBBOCK_WORKBUF_SIZE;
    #endif

    #ifdef ASSABET_PLATFORM
    #ifndef LUBBOCK_SA1110
    int tempbuf[ASSABET_WORKBUF_SIZE];
	int size = ASSABET_WORKBUF_SIZE;
    #else
    int tempbuf[ASSABET_WORKBUF_SIZE - 2];
	int size = ASSABET_WORKBUF_SIZE - 2;
    
    #endif
    #endif

    for (i = 0; i < size; i++)
    {
        tempbuf[i] = WORKBUF[size - i - 1];    
    }

    // now copy the temp buffer back
    
    for (i = 0; i < size; i++)
    {
        WORKBUF[i] = tempbuf[i];
    }

}
/* 
*******************************************************************************
*
* FUNCTION:         init_workbuf (not used at this time)
*
* DESCRIPTION:      initializes the work buffer with the resting pin states.
*                   Note that the BSDL file gives offsets from TDO and not TDI.
*
* INPUT PARAMETERS: void
*
* RETURNS:          void
*
*******************************************************************************
*/
void init_workbuf()
{
    int i;
    int next = 0;
    
    #ifdef SANDGATE_PLATFORM
    int size = SANDGATE_WORKBUF_SIZE;
    #endif

    #ifdef LUBBOCK_PLATFORM
    int size = LUBBOCK_WORKBUF_SIZE;
    #endif

    #ifdef ASSABET_PLATFORM
    #ifndef LUBBOCK_SA1110
    int size = ASSABET_WORKBUF_SIZE;
    #else
    int size = ASSABET_WORKBUF_SIZE - 2;
    
    #endif
    #endif
    
   
    // devices found after the processor
    if(DEVICES_AFTER != 0)
    {
        for (i = 0; i < DEVICES_AFTER; i++)
        {
            next = i + 1;
            WORKBUF[i] = 0;
        }
    }
   
    // the processor
    for (i = next; i < CHAIN_LENGTH + DEVICES_AFTER; i++)
    {
        WORKBUF[i] = pin[i - DEVICES_AFTER];
        next = i + 1;
    }
   
    // for devices before the processor
    if(DEVICES_BEFORE != 0)
    {
       
        for (i = next; i < CHAIN_LENGTH + DEVICES_BEFORE + DEVICES_AFTER; i++)
        {
            WORKBUF[i] = 0;
        } 
    }
	
    invert_workbuf();

}




