#!/bin/sh

abort () {
	echo "$@"
	exit 1
}

##########################################################
#
#  Input validation functions.
#
##########################################################

set_default_values () {
	: ${PATCHES_DIR="$(pwd)/patches"}
	: ${PATCHED_SOURCES_DIR="$(pwd)/patched_sources"}
	: ${BUILD_DIR="$(pwd)/build"}
	: ${OUTPUT_DIR="$(pwd)/output"}
}

validate_all_input () {

	verify_mandatory_values

	verify_existing_files \
		${GCC_SRC} \
		${GLIBC_SRC} \
		${GLIBC_LINUXTHREADS_SRC} \
		${BINUTILS_SRC} \
		${KERNEL_SRC} \
		${PATCHES_DIR}

	verify_output_directories \
		${PATCHED_SOURCES_DIR} \
		${BUILD_DIR} \
		${OUTPUT_DIR}
}

verify_mandatory_values () {

	: ${GCC_SRC?"error, unset variable"}
	: ${GLIBC_SRC?"error, unset variable"}
	: ${GLIBC_LINUXTHREADS_SRC?"error, unset variable"}
	: ${BINUTILS_SRC?"error, unset variable"}
	: ${LINUX_SRC?"error, unset variable"}
	: ${TARGET?"error, unset variable"}

	echo "Mandatory input values verified." >&2
}

verify_existing_files () {
	for obj in $* ; do
		test -e $obj || abort "Error, required input object $obj does not exist, exiting."
	done

	echo "Existing files verified." >&2
}

verify_output_directories () {
	for dir in $* ; do
		if test -d $dir ; then
			test -w $dir || abort "Directory $dir is not writable by you, exiting."
			continue
		fi
		mkdir -p $dir || abort "Non-existent output dir $dir can't be created, exiting."
	done

	echo "Output directories exist or have been created." >&2
}

##########################################################
#
#  Lots of helper variables.
#
##########################################################

set_helper_variables () {

	GCC=$(basename $(strip_archive_suffix "$GCC_SRC"))
	GLIBC=$(basename $(strip_archive_suffix "$GLIBC_SRC"))
	GLIBC_LINUXTHREADS=$(basename $(strip_archive_suffix "$GLIBC_LINUXTHREADS_SRC"))
	BINUTILS=$(basename $(strip_archive_suffix "$BINUTILS_SRC"))
	LINUX=$(basename $(strip_archive_suffix "$LINUX_SRC"))

	GCC_DIR=${PATCHED_SOURCES_DIR}/${GCC}
	GLIBC_DIR=${PATCHED_SOURCES_DIR}/${GLIBC}
	BINUTILS_DIR=${PATCHED_SOURCES_DIR}/${BINUTILS}
	LINUX_DIR=${PATCHED_SOURCES_DIR}/${LINUX}

	GCC_CORE_BUILD_DIR=${BUILD_DIR}/gcc-core
	GCC_BUILD_DIR=${BUILD_DIR}/${GCC}
	GLIBC_BUILD_DIR=${BUILD_DIR}/${GLIBC}
	GLIBC_HEADERS_BUILD_DIR=${BUILD_DIR}/glibc-headers
	BINUTILS_BUILD_DIR=${BUILD_DIR}/${BINUTILS}
	HEADERS_DIR=${BUILD_DIR}/${LINUX}/include	# ??????

	TOOLCOMBO=${GCC}-${GLIBC}
	PREFIX=${OUTPUT_DIR}/${TARGET}/${TOOLCOMBO}
	ARCH=$(map_target_to_arch $TARGET)

	echo "All helper variables set, some of the more important:" >&2
	echo "TOOLCOMBO = ${TOOLCOMBO}" >&2
	echo "PREFIX = ${PREFIX}" >&2
	echo "ARCH = ${ARCH}" >&2

}

map_target_to_arch () {
	case $1 in
		alpha*)   arch=alpha ;;
		arm*)     arch=arm ;;
		cris*)    arch=cris ;;
		hppa*)    arch=parisc ;;
		i*86*)    arch=i386 ;;
		ia64*)    arch=ia64 ;;
		mips*)    arch=mips ;;
		m68k*)    arch=m68k ;;
		powerpc*) arch=ppc ;;
		ppc*)     echo "Target $TARGET incompatible with binutils and gcc regression tests; use target powerpc-* instead" ; exit 1 ;;
		s390*)    arch=s390 ;;
		sh*)      arch=sh ;;
		sparc64*) arch=sparc64 ;;
		sparc*)   arch=sparc ;;
		x86_64*)  arch=x86_64 ;;
		*) echo "Bad target $TARGET" ; exit 1
	esac
	echo "$arch"
}

##########################################################
#
#  Unpacking and patching sources.
#
##########################################################

	
unpack_and_patch_sources () {

	rm -rf ${PATCHED_SOURCES_DIR}
	mkdir -p ${PATCHED_SOURCES_DIR}

	for src in $GCC_SRC $GLIBC_SRC $BINUTILS_SRC $LINUX_SRC ; do
		BASE=$(basename $(strip_archive_suffix "$src"))
		echo "Unpacking $BASE ..."
		unpack_archive_into_directory $src ${PATCHED_SOURCES_DIR}

		echo "Patching $BASE ..."
		CURR_PATCH_DIR=${PATCHES_DIR}/${BASE}

		test -d ${CURR_PATCH_DIR} || continue

		apply_patches_to_directory  \
			${PATCHED_SOURCES_DIR}/${BASE} \
			${CURR_PATCH_DIR}/*.patch
	done
}

apply_glibc_addons () {

	for addon in $* ; do
		echo "Applying glibc addon $addon..."
		unpack_archive_into_directory $addon ${GLIBC_DIR}

		CURR_PATCH_DIR=${PATCHES_DIR}/${GLIBC_LINUXTHREADS}
		test -d ${CURR_PATCH_DIR} || continue

#		apply_patches_to_directory \
#			${GLIBC_DIR}
#			${CURR_PATCH_DIR}/*.patch
	done
}

strip_archive_suffix () {
	echo $1 | sed 's,\.tar\.gz$,, ; s,\.tar\.bz2$,, ; s,\.tgz$,,;'
}

unpack_archive_into_directory () {

	ARCHIVE=$1
	DIR=$2

	test -w $DIR || abort "Trying to unpack into non-writable directory $DIR, exiting."

	case $ARCHIVE in 
		*tar.gz)	tar xzf $ARCHIVE -C $DIR ;;
		*tgz)		tar xzf $ARCHIVE -C $DIR ;;
		*tar.bz2)	tar xjf $ARCHIVE -C $DIR ;;
	esac
}

apply_patches_to_directory () {

	DIR=$1 ; shift

	for patch in $* ; do
		echo "Applying $patch to directory $DIR"
		patch -d $DIR -p1 --force < $patch
	done
}

##########################################################
#
#  Some of the build functions.
#
##########################################################

prepare_kernel_headers () {

	save_cd=$(pwd)
	cd ${LINUX_DIR}

	make mrproper			# just to play it safe
	make ARCH=$ARCH symlinks include/linux/version.h

	INCLUDE_DIR=${PREFIX}/${TARGET}/include
	mkdir -p ${INCLUDE_DIR}
	rm -rf ${INCLUDE_DIR}/*		# clean that directory first

	cp -r include/linux ${INCLUDE_DIR}
	cp -r include/asm-${ARCH} ${INCLUDE_DIR}/asm
	cp -r include/asm-generic ${INCLUDE_DIR}/asm-generic

	cd ${save_cd}
}

build_binutils () {

echo "**********************************************"
echo "Starting build_binutils."
echo "**********************************************"

	rm -rf ${BINUTILS_BUILD_DIR}
	mkdir ${BINUTILS_BUILD_DIR}

	save_cd=$(pwd)
	cd ${BINUTILS_BUILD_DIR}
	
	${BINUTILS_DIR}/configure --target=$TARGET --prefix=$PREFIX --disable-nls

	make all
	make install

	echo "Contents of bin directory: " >&2
	ls ${PREFIX}/bin

	cd ${save_cd}

echo "**********************************************"
echo "Leaving build_binutils."
echo "**********************************************"

}

install_glibc_headers () {

echo "**********************************************"
echo "Starting install_glibc_headers."
echo "**********************************************"



	rm ${PREFIX}/${TARGET}/include/features.h	# to force re-do




	if grep -q gcc-3 ${GCC_DIR}/ChangeLog && \
	   test '!' -f ${PREFIX}/${TARGET}/include/features.h; then

		echo "Installing glibc headers ..." >&2

		rm -rf ${GLIBC_HEADERS_BUILD_DIR}
		mkdir ${GLIBC_HEADERS_BUILD_DIR}
		save_cd=$(pwd)
		cd ${GLIBC_HEADERS_BUILD_DIR}

    		if test '!' -f Makefile; then
        # The following three things have to be done to build glibc-2.3.x, but they don't hurt older versions.
        # 1. override CC to keep glibc's configure from using $TARGET-gcc. 
        # 2. disable linuxthreads, which needs a real cross-compiler to generate tcb-offsets.h properly
        # 3. build with gcc 3.2 or later
        # Compare these options with the ones used when building glibc for real below - they're different.
	# As of glibc-2.3.2, to get this step to work for hppa-linux, you need --enable-hacker-mode
	# so when configure checks to make sure gcc has access to the assembler you just built...
	# Alternately, we could put ${PREFIX}/${TARGET}/bin on the path.
        # Set --build so maybe we don't have to specify "cross-compiling=yes" below (haven't tried yet)
        		CC=gcc \
            			${GLIBC_DIR}/configure \
				--host=$TARGET \
				--prefix=/usr \
	    			--build=$BUILD \
            			--without-cvs \
				--disable-sanity-checks \
				--with-headers=${PREFIX}/${TARGET}/include \
	    			--enable-hacker-mode
		fi

		if grep -q GLIBC_2.3 ${GLIBC_DIR}/ChangeLog ; then
        # glibc-2.3.x passes cross options to $(CC) when generating errlist-compat.c, which fails without a real cross-compiler.
        # Fortunately, we don't need errlist-compat.c, since we just need .h files, 
        # so work around this by creating a fake errlist-compat.c and satisfying its dependencies.
        # Another workaround might be to tell configure to not use any cross options to $(CC).
        # The real fix would be to get install-headers to not generate errlist-compat.c.
			make sysdeps/gnu/errlist.c
			mkdir -p stdio-common
			touch stdio-common/errlist-compat.c
    		fi

		make cross-compiling=yes install_root=${PREFIX}/${TARGET} prefix="" install-headers

    # Two headers -- stubs.h and features.h -- aren't installed by install-headers,
    # so do them by hand.  We can tolerate an empty stubs.h for the moment.
    # See e.g. http://gcc.gnu.org/ml/gcc/2002-01/msg00900.html

		mkdir -p ${PREFIX}/${TARGET}/include/gnu
		touch ${PREFIX}/${TARGET}/include/gnu/stubs.h
		cp ${GLIBC_DIR}/include/features.h ${PREFIX}/${TARGET}/include/features.h

		cd ${save_cd}
	fi

echo "**********************************************"
echo "Leaving install_glibc_headers."
echo "**********************************************"

}

build_gcc_core () {

echo "**********************************************"
echo "Starting build_gcc_core."
echo "**********************************************"

	rm -rf ${GCC_CORE_BUILD_DIR}
	mkdir ${GCC_CORE_BUILD_DIR}
	save_cd=$(pwd)
	cd ${GCC_CORE_BUILD_DIR}

	if test '!' -f Makefile; then
		${GCC_DIR}/configure \
		--target=$TARGET \
		--host=$HOST \
		--prefix=$PREFIX \
		--with-local-prefix=${PREFIX}/${TARGET} \
		--disable-multilib \
		--with-newlib \
        	${GCC_EXTRA_CONFIG} \
        	--without-headers \
		--disable-nls \
		--enable-threads=no \
		--enable-symvers=gnu \
		--enable-__cxa_atexit \
        	--enable-languages=c \
		--disable-shared
	fi

	make all-gcc install-gcc 
	cd ${save_cd}

	test -x ${PREFIX}/bin/${TARGET}-gcc || abort "gcc-core build failed."

echo "**********************************************"
echo "Leaving build_gcc_core."
echo "**********************************************"

}

build_glibc () {

echo "**********************************************"
echo "Starting build_glibc."
echo "**********************************************"

	rm -rf ${GLIBC_BUILD_DIR}
	mkdir ${GLIBC_BUILD_DIR}
	save_cd=$(pwd)
	cd ${GLIBC_BUILD_DIR}

# sh4 really needs to set configparms as of gcc-3.4/glibc-2.3.2
# note: this is awkward, doesn't work well if you need more than one line in configparms
	echo ${GLIBC_CONFIGPARMS} > configparms

	if test '!' -f Makefile ; then
    # Configure with --prefix the way we want it on the target...
    # There are a whole lot of settings here.  You'll probably want
    # to read up on what they all mean, and customize a bit.
    # e.g. I picked --enable-kernel=2.4.3 here just because it's the kernel Bill 
    # used in his example gcc2.95.3 script.  That means some backwards compatibility 
    # stuff is turned on in glibc that you may not need if you're using a newer kernel.
    # Compare these options with the ones used when installing the glibc headers above - they're different.
    # Adding "--without-gd" option to avoid error "memusagestat.c:36:16: gd.h: No such file or directory" 
    # See also http://sources.redhat.com/ml/libc-alpha/2000-07/msg00024.html. 
    # The --enable-clocale=gnu is recomended by LFS; see http://bugs.linuxfromscratch.org/show_bug.cgi?id=411
    # Set BUILD_CC, or you won't be able to build datafiles
    # Set --build, else glibc-2.3.2 will think you're not cross-compiling, and try to run the test programs

		BUILD_CC=gcc \
		CFLAGS="$TARGET_CFLAGS" \
		CC=${TARGET}-gcc \
		AR=${TARGET}-ar \
		RANLIB=${TARGET}-ranlib \
        	${GLIBC_DIR}/configure  \
			--host=$TARGET \
			--prefix=/usr \
			--build=$BUILD \
        		${GLIBC_EXTRA_CONFIG} \
        		--without-tls \
			--without-__thread \
			--enable-kernel=2.4.3 \
        		--without-cvs \
			--disable-profile \
			--disable-debug \
			--without-gd \
			--enable-clocale=gnu \
        		--enable-add-ons=linuxthreads \
			--with-headers=${PREFIX}/${TARGET}/include
	fi

# If this fails with an error like this:
# ...  linux/autoconf.h: No such file or directory 
# then you need to set the KERNELCONFIG variable to point to a .config file for this arch.
# The following architectures are known to need kernel .config: alpha, arm, ia64, s390, sh, sparc
	make
	make install install_root=${PREFIX}/${TARGET} prefix=""

# Fix problems in linker scripts.
# 
# 1. Remove absolute paths
# Any file in a list of known suspects that isn't a symlink is assumed to be a linker script.
# FIXME: test -h is not portable
# FIXME: probably need to check more files than just these three...
# Need to use sed instead of just assuming we know what's in libc.so because otherwise alpha breaks
# But won't need to do this at all once we use --with-sysroot (available in gcc-3.3.3 and up)
#
# 2. Remove lines containing BUG per http://sources.redhat.com/ml/bug-glibc/2003-05/msg00055.html,
# needed to fix gcc-3.2.3/glibc-2.3.2 targeting arm
#
# To make "strip *.so.*" not fail (ptxdist does this), rename to .so_orig rather than .so.orig
	for file in libc.so libpthread.so libgcc_s.so; do
		if test -f ${PREFIX}/${TARGET}/lib/$file && \
		   test ! -h ${PREFIX}/${TARGET}/lib/$file; then
			mv ${PREFIX}/${TARGET}/lib/$file ${PREFIX}/${TARGET}/lib/${file}_orig
			sed 's,/lib/,,g;/BUG in libc.scripts.output-format.sed/d' < ${PREFIX}/${TARGET}/lib/${file}_orig > ${PREFIX}/${TARGET}/lib/$file
		fi
	done

	cd ${save_cd}

	test -f ${PREFIX}/${TARGET}/lib/libc.a || abort "Building glibc failed." >&2

echo "**********************************************"
echo "Leaving build_glibc."
echo "**********************************************"

}

build_gcc () {

echo "**********************************************"
echo "Starting build_gcc."
echo "**********************************************"

	rm -rf ${GCC_BUILD_DIR}
	mkdir ${GCC_BUILD_DIR}
	save_cd=$(pwd)
	cd ${GCC_BUILD_DIR}

	if test '!' -f Makefile; then
    # --enable-symvers=gnu really only needed for sh4 to work around a detection problem
    # only matters for gcc-3.2.x and later, I think
    # --disable-nls to work around crash bug on ppc405, but also because embedded
    # systems don't really need message catalogs...
    # Use --with-headers, else it will define disable_glibc while building libgcc, and you'll have no profiling
		${GCC_DIR}/configure \
			--target=$TARGET \
			--host=$HOST \
			--prefix=$PREFIX \
			${GCC_EXTRA_CONFIG} \
			--with-headers=${PREFIX}/${TARGET}/include \
			--disable-nls \
			--enable-threads=posix \
			--enable-symvers=gnu \
			--enable-__cxa_atexit \
			--enable-languages=c,c++ \
			--enable-shared \
			--enable-c99 \
			--enable-long-long

	fi

	make all 
	make install 

	echo "If the chip does not have a floating point unit, and there are shared libraries in /lib/nof, copy them to /lib"
	echo "We check GLIBC_EXTRA_CONFIG ($GLIBC_EXTRA_CONFIG) to see if it contains --without-fp to decide."
	case "$GLIBC_EXTRA_CONFIG" in
   		*--without-fp*)
      			cp -a ${PREFIX}/${TARGET}/lib/nof/*.so* ${PREFIX}/${TARGET}/lib ;;
	esac

	cd ${save_cd}

	test -x ${PREFIX}/bin/${TARGET}-gcc || Build failed during final gcc 

echo "**********************************************"
echo "Leaving build_gcc."
echo "**********************************************"

}


##########################################################
#
#  MAIN
#
##########################################################

set_default_values
validate_all_input
set_helper_variables

PATH=${PREFIX}/bin:$PATH

#
#  Handle all processing/build options.
#

while [ $# -gt 0 ]; do
	case "$1" in
		--unpackandpatch)		opt_unpack_and_patch=1	;;
		--preparekernelheaders)		opt_prepare_kernel_headers=1 ;;
		--buildbinutils)		opt_build_binutils=1 ;;
		--installglibcheaders)		opt_install_glibc_headers=1 ;;
		--buildgcccore)			opt_build_gcc_core=1 ;;
		--buildglibc)			opt_build_glibc=1 ;;
		--buildgcc)			opt_build_gcc=1 ;;
		--builduserland)		opt_build_userland=1 ;;
		--test)				opt_test=1 ;;
		*) 
			echo "Usage error" >&2 ; exit 1 ;;
	esac
	shift
done

############################################################
#
#  Unpack (or copy) sources into $PATCHED_SOURCES directory.
#
############################################################


if test "$opt_unpack_and_patch" ; then
	echo "Starting unpack and patch of sources." >&2
	unpack_and_patch_sources
	apply_glibc_addons ${GLIBC_LINUXTHREADS_SRC}
	echo "Finished unpacking and patching sources, exiting." >&2
fi

BUILD=$(${GCC_DIR}/config.guess)
HOST=$(echo $BUILD | sed s/-/-host_/)

echo "BUILD = ${BUILD}" >&2
echo "HOST = ${HOST}" >&2
echo "TARGET = ${TARGET}" >&2

############################################################
#
#  Start processing all the user-selected steps.
#
############################################################

if test "$opt_prepare_kernel_headers" ; then
	echo "Preparing kernel headers ..." >&2
	prepare_kernel_headers
fi

if test "$opt_build_binutils" ; then
	echo "Building binutils ..." >&2
	build_binutils
fi

if test "$opt_install_glibc_headers" ; then
	echo "Installing glibc headers ..." >&2
	install_glibc_headers
fi

if test "$opt_build_gcc_core" ; then
	echo "Building gcc core ..." >&2
	build_gcc_core
fi

if test "$opt_build_glibc" ; then
	echo "Building glibc ..." >&2
	build_glibc
fi

if test "$opt_build_gcc" ; then
	echo "Building gcc ..." >&2
	build_gcc
fi

exit 0

#############################################################
#
#  rewritten up to here at the moment.
#
############################################################







    sh $TOP_DIR/crosstool.sh
    cd $TOP_DIR

    # Cute little compile test
    sh testhello.sh
fi

if test "$opt_builduserland" = "1"; then
    # Build /bin/sh and any other non-toolchain things configured in ptx.config
    # Only needed if you can't run the target's normal /bin/sh with the new toolchain
    cd $BUILD_DIR
    sh $TOP_DIR/ptx.sh
fi

if test "$opt_no_test" = ""; then
    # Beefy test that lasts for hours
    cd $BUILD_DIR
    sh $TOP_DIR/crosstest.sh 
fi

