/*
 * Jornada 720 MCU keyboard/mouse driver
 *
 * Created 1/1/2001 by John Ankcorn, MIT LCS
 *
 * Changelog:
 *
 * This driver is based on linux/drivers/char/sa1111_keyb.c
 * (2000 by VASARA RESEARCH INC.)
 *
 */

#include <linux/module.h>
#include <linux/version.h>

#include <linux/config.h>
#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/mm.h>
#include <linux/signal.h>
#include <linux/init.h>
#include <linux/kbd_ll.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/kbd_kern.h>
#include <asm/bitops.h>
#include <asm/uaccess.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/io.h>
#include <linux/pc_keyb.h>
#include <asm/keyboard.h>
#include <asm/hardware.h>
#include <asm/arch/h3600_hal.h> /* for TS_RET */

#define TS_MODULE_NAME "ts"
#define MAX_READ_SIZE	256
#define AUX_DEV ((void *)queue)
#define SC_LIM 128
#define INCBUF(x,mod) (((x)+1) & ((mod) - 1))

spinlock_t kbd_controller_lock = SPIN_LOCK_UNLOCKED;
static struct aux_queue *queue;        /* Mouse data buffer. */
static int aux_count = 0;
static unsigned char keyboard_data[MAX_READ_SIZE];
MODULE_AUTHOR("John Ankcorn");
MODULE_DESCRIPTION("Touchscreen and keyboard drivers for the Jornada720");

struct h3600_ts_general_device {
    unsigned int          head, tail;        /* Position in the event buffer */
    struct fasync_struct *async_queue;       /* Asynchronous notification    */
    wait_queue_head_t     waitq;             /* Wait queue for reading       */
    struct semaphore      lock;              /* Mutex for reading            */
    unsigned int          usage_count;       /* Increment on each open       */
};

#define KEYBUF_SIZE  4
struct h3600_ts_key_device {
    struct h3600_ts_general_device d;        /* Include first so we can cast to the general case */
    unsigned char                  buf[KEYBUF_SIZE];
};

#define MOUSEBUF_SIZE 8
struct h3600_ts_mouse_device {
    struct h3600_ts_general_device d;
    struct h3600_ts_event          buf[MOUSEBUF_SIZE];
};

enum pen_state {
    PEN_UP = 0,
    PEN_DISCARD,
    PEN_DOWN
};

#define TS_FILTER_LENGTH 8
struct pen_data {
    enum pen_state state;
    unsigned short x[TS_FILTER_LENGTH];  // Unfiltered data points
    unsigned short y[TS_FILTER_LENGTH];
    unsigned short count;   // Number of points recorded in this "DOWN" or "DISCARD" series
    unsigned short index;   // Location in ring buffer of last stored data value
};

struct jornada720_ts_device {
    struct pen_data               pen;
    struct h3600_ts_calibration   cal;          /* ts calibration parameters */
    struct h3600_ts_mouse_device  raw;
    struct h3600_ts_mouse_device  filtered;     /* and calibrated */
    struct h3600_ts_key_device    key;
};

struct jornada720_ts_device g_touchscreen;
static devfs_handle_t devfs_ts, devfs_ts_dir, devfs_tsraw;
static int gMajor;      /* Dynamic major for now */

static int touchScreenSilenced       = 0;
static int discard_initial_touch     = 1;
static int touch_filter_delay        = 0;
static int touch_filter[TS_FILTER_LENGTH] = { 1, 1, 1, 1, 0, 0, 0, 0 };

static int jornada720_setkeycode(unsigned int scancode, unsigned int keycode)
{
	return -EINVAL;
}

static int jornada720_getkeycode(unsigned int scancode)
{
	return -EINVAL;
}

static int jornada720_translate(unsigned char scancode, unsigned char *keycode, char raw_mode)
{
	scancode &= 0x7f;
	*keycode = scancode;
	return 1;
}

static char jornada720_unexpected_up(unsigned char keycode)
{
	return 0200;
}

static void jornada720_leds(unsigned char leds)
{
/* dummy called from keyboard.c (keyb_leds() ) */
}

/*
 * Initialize driver. (called from drivers/char/keyboard.c
 * -- it calls kbd_init_hw(), a macro defined in include/asm/arch/keyboard.h)
 */
void __init jornada720_kbd_init_hw(void)
{
    printk(KERN_INFO "Jornada720 keyboard driver v1.0\n");
	k_setkeycode = jornada720_setkeycode;
	k_getkeycode = jornada720_getkeycode;
	k_translate = jornada720_translate;
	k_unexpected_up = jornada720_unexpected_up;
	k_leds = jornada720_leds;
}

static void keyboard_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
unsigned long flags;
int this_size;
unsigned char *p;
unsigned char scancode;

    kbd_pt_regs = regs;
    spin_lock_irqsave(&kbd_controller_lock, flags);
	this_size = jornada720_getkey(keyboard_data, sizeof(keyboard_data));
    p = keyboard_data;
    while (this_size-- > 0) {
		scancode = *p++ & 0xff;
        handle_scancode(scancode, !(scancode & 0x80));
        tasklet_schedule(&keyboard_tasklet);
	}
    spin_unlock_irqrestore(&kbd_controller_lock, flags);
}

static void h3600_ts_add_queue( struct h3600_ts_mouse_device *dev, 
				unsigned short x, unsigned short y, int down )
{
#if 0
	struct h3600_ts_event *event = &dev->buf[dev->d.head];
	unsigned int          nhead = INCBUF( dev->d.head, MOUSEBUF_SIZE );
	
	/* Store the character only if there is room */
	if ( nhead != dev->d.tail ) {
		event->x        = x;
		event->y        = y;
		event->pressure = down;

		dev->d.head = nhead;

		if ( dev->d.async_queue )
			kill_fasync( &dev->d.async_queue, SIGIO, POLL_IN );

		wake_up_interruptible( &dev->d.waitq );
	}
#else 
union {
	TS_RET ret;
	char c[sizeof(TS_RET)];
} temp;
int i;
int ok_for_fasync;

   if (aux_count) {
       int head = queue->head;
       ok_for_fasync = 1;
temp.ret.x = x;
temp.ret.y = y;
temp.ret.pressure = down;
       for (i = 0; i < sizeof(TS_RET); i++) {
           queue->buf[head] = temp.c[i];
           head = (head + 1) & (AUX_BUF_SIZE-1);
           if (head == queue->tail)
               ok_for_fasync = 0;
       }
       if (ok_for_fasync) {
           queue->head = head;
           if (queue->fasync)
               kill_fasync(&queue->fasync, SIGIO, POLL_IN);
           wake_up_interruptible(&queue->proc_list);
       }
   }
#endif
}

static int h3600_ts_apply_filter( unsigned short data[] )
{
	struct pen_data *pen = &g_touchscreen.pen;

	int i;
	unsigned long data_sum = 0;
	unsigned long filter_sum = 0;
	
	for ( i = 0 ; i < TS_FILTER_LENGTH && i < pen->count; i++ ) {
		int index = ( pen->index - i + TS_FILTER_LENGTH ) % TS_FILTER_LENGTH;
		filter_sum += touch_filter[i];
		data_sum += touch_filter[i] * data[index];
	}
	if ( filter_sum <= 0 ) {
		printk(KERN_ERR __FUNCTION__ ": unable to apply imaginary filter\n");
		return data[pen->index];
	}
	return data_sum / filter_sum;
}


static void h3600_ts_touchpanel_event( unsigned short x, unsigned short y, int down )
{
	struct pen_data *pen = &g_touchscreen.pen;
	struct h3600_ts_calibration *cal = &g_touchscreen.cal;

	if ( touchScreenSilenced )
		return;

	switch ( g_touchscreen.pen.state ) {
	case PEN_UP: 
		if ( !down )
			return;        // No events to report

		if ( discard_initial_touch > 0 ) {
			pen->state = PEN_DISCARD;
			pen->count = 1;
			return;        // No events to report
		}

		pen->state = PEN_DOWN;
		pen->count = 1;
		break;

	case PEN_DISCARD:
		if ( !down ) {
			pen->state = PEN_UP;
			return;        // No events to report
		}

		pen->count++;
		if ( pen->count <= discard_initial_touch )
			return;

		pen->state = PEN_DOWN;
		pen->count = 1;
		break;

	case PEN_DOWN:
		if ( !down )
			pen->state = PEN_UP;
		else
			pen->count++;
		break;
	}

#if 0
	// If I get this far, I need to record a DOWN or an UP raw event
	// and possibly a down or up filtered event
	h3600_ts_add_queue( &g_touchscreen.raw, x, y, down ); 
#endif

	pen->index = (pen->index + 1) % TS_FILTER_LENGTH;
	pen->x[pen->index] = x;
	pen->y[pen->index] = y;

	// Only process and record filtered pen up/down events if we have exceeded our delay
	// This could be an up or a down event
	if ( pen->count > touch_filter_delay ) {
		int x1 = ((h3600_ts_apply_filter(pen->x) * cal->xscale)>>8) + cal->xtrans;
		int y1 = ((h3600_ts_apply_filter(pen->y) * cal->yscale)>>8) + cal->ytrans;

		x = ( x1 < 0 ? 0 : x1 );
		y = ( y1 < 0 ? 0 : y1 );

		h3600_ts_add_queue( &g_touchscreen.filtered, x, y, down );
	}
}

/*    
      Called whenever the touchscreen is blanked
*/
static void h3600_ts_reset_filters( void )
{
	if (0) printk(__FUNCTION__ "\n");
	h3600_ts_touchpanel_event(0,0,0);   // Send a "mouse up"
}

static void mouse_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
unsigned long flags;
int this_x, this_y, this_pressure;

    kbd_pt_regs = regs;
    spin_lock_irqsave(&kbd_controller_lock, flags);
    if ((this_pressure = jornada720_gettouch(&this_x, &this_y)) >=0
        && aux_count) {
      h3600_ts_touchpanel_event(this_x, this_y, this_pressure);
    }
    
	spin_unlock_irqrestore(&kbd_controller_lock, flags);
}

static unsigned char get_from_queue(void)
{
       unsigned char result;
       unsigned long flags;

       spin_lock_irqsave(&kbd_controller_lock, flags);
       result = queue->buf[queue->tail];
       queue->tail = (queue->tail + 1) & (AUX_BUF_SIZE-1);
       spin_unlock_irqrestore(&kbd_controller_lock, flags);
       return result;
} 

static inline int queue_empty(void)
{
       return queue->head == queue->tail;
}

static int aux_ioctl(struct inode * inode, struct file *filp,
        unsigned int cmd , unsigned long arg)
{
//printk ("[%s:%d] %s ENTRY\n", __FILE__, __LINE__, __FUNCTION__);
    printk("tsIoctl: cmd %04x  arg=%x\n",cmd, arg);
    switch(cmd) {
    case TS_GET_CAL:
        printk("TS_GET_CAL\n");
        __copy_to_user((char *)arg, (char *)&g_touchscreen.cal, sizeof(TS_CAL) );
        break;
    case TS_SET_CAL:
        printk("\n\nTS_SET_CAL: ");
        __copy_from_user( (char *)&g_touchscreen.cal,(char *)arg ,sizeof(TS_CAL) );
    
        printk("xscale=%d xtrans=%d yscale=%d ytrans=%d\n\n",
               g_touchscreen.cal.xscale,g_touchscreen.cal.xtrans,
               g_touchscreen.cal.yscale,g_touchscreen.cal.ytrans);
        break;
    default:
        printk("IOCTL: unknowncmd %04x\n",cmd);
        return -EINVAL;
    }
    return 0;
}

static int fasync_aux(int fd, struct file *filp, int on)
{
       int retval;

       retval = fasync_helper(fd, filp, on, &queue->fasync);
       if (retval < 0)
               return retval;
       return 0;
} 

/*
 * Random magic cookie for the aux device
 */
static int release_aux(struct inode * inode, struct file * file)
{
       fasync_aux(-1, file, 0);
       if (--aux_count)
               return 0;
       free_irq(GPIO_JORNADA720_KEYBOARD_IRQ, AUX_DEV);
       free_irq(GPIO_JORNADA720_MOUSE_IRQ, AUX_DEV);
       return 0;
}

static int open_aux(struct inode * inode, struct file * file)
{
//kdev_t dev = inode->i_rdev;

	if (aux_count++)
		return 0;
	mdelay(50);
	queue->head = queue->tail = 0;          /* Flush input queue */
	set_GPIO_IRQ_edge( GPIO_JORNADA720_MOUSE, GPIO_FALLING_EDGE );
	set_GPIO_IRQ_edge( GPIO_JORNADA720_MOUSE, GPIO_RISING_EDGE );
	set_GPIO_IRQ_edge( GPIO_JORNADA720_KEYBOARD, GPIO_FALLING_EDGE );
	if (request_irq(GPIO_JORNADA720_KEYBOARD_IRQ, keyboard_interrupt,
		SA_SHIRQ, "Jornada720 Keyboard", AUX_DEV)) {
		aux_count--;
		return -EBUSY;
	}
	if (request_irq(GPIO_JORNADA720_MOUSE_IRQ, mouse_interrupt,
		SA_SHIRQ, "Jornada720 Mouse", AUX_DEV)) {
		aux_count--;
		return -EBUSY;
	}
	return 0;
}

/*
 * Put bytes from input queue to buffer.
 */
static ssize_t read_aux(struct file * file, char * buffer,
                       size_t count, loff_t *ppos)
{
       DECLARE_WAITQUEUE(wait, current);
       ssize_t i = count;
       unsigned char c;
int minor;
kdev_t dev = file->f_dentry->d_inode->i_rdev;

	minor = MINOR(dev);
       if (queue_empty()) {
               if (file->f_flags & O_NONBLOCK)
                       return -EAGAIN;
               add_wait_queue(&queue->proc_list, &wait);
repeat:
               set_current_state(TASK_INTERRUPTIBLE);
               if (queue_empty() && !signal_pending(current)) {
                       schedule();
                       goto repeat;
               }
               current->state = TASK_RUNNING;
               remove_wait_queue(&queue->proc_list, &wait);
       }
       while (i > 0 && !queue_empty()) {
               c = get_from_queue();
               put_user(c, buffer++);
               i--;
       }
       if (count-i) {
               file->f_dentry->d_inode->i_atime = CURRENT_TIME;
               return count-i;
       }
       if (signal_pending(current))
               return -ERESTARTSYS;
       return 0;
}
#if 0
/*
 * Write to the aux device.
 */
static ssize_t write_aux(struct file * file, const char * buffer,
                        size_t count, loff_t *ppos)
{
ssize_t retval = 0;
//kdev_t dev = file->f_dentry->d_inode->i_rdev;

       if (count) {
               ssize_t written = 0;
               if (count > 32)
                       count = 32; /* Limit to 32 bytes. */
               do {
                       char c;
                       get_user(c, buffer++);
                       printk ("[%s]: write_aux 0x%x\n", __FILE__, c);
                       written++;
               } while (--count);
               retval = -EIO;
               if (written) {
                       retval = written;
                       file->f_dentry->d_inode->i_mtime = CURRENT_TIME;
               }
       }
       return retval;
}
#endif

static unsigned int aux_poll(struct file *file, poll_table * wait)
{
int minor;
kdev_t dev = file->f_dentry->d_inode->i_rdev;
int ret;

	minor = MINOR(dev);
       poll_wait(file, &queue->proc_list, wait);
       ret = ( queue_empty() ) ? 0 : (POLLIN | POLLRDNORM);
       return ret;
}

struct file_operations psaux_fops = {
       read:           read_aux,
       poll:           aux_poll,
       ioctl:    aux_ioctl,
       open:           open_aux,
       release:        release_aux,
       fasync:         fasync_aux,
};

int __init jornada720_kbd_init_module(void)
{
    queue = (struct aux_queue *) kmalloc(sizeof(*queue), GFP_KERNEL);
    memset(queue, 0, sizeof(*queue));
    queue->head = queue->tail = 0;
#if 0 /* jca */
    g_touchscreen.penStatus=PEN_DOWN;
#endif
    g_touchscreen.cal.xscale = -178;
    g_touchscreen.cal.xtrans = 697;
    g_touchscreen.cal.yscale = 72;
    g_touchscreen.cal.ytrans = -26;
    init_waitqueue_head(&queue->proc_list);

	gMajor = devfs_register_chrdev(0, TS_MODULE_NAME, &psaux_fops);
	if (gMajor < 0) {
		printk(__FUNCTION__ ": can't get major number\n");
		return gMajor;
	}

	devfs_ts_dir = devfs_mk_dir(NULL, "touchscreen", NULL);
	if ( !devfs_ts_dir ) return -EBUSY;

	devfs_ts = devfs_register( devfs_ts_dir, "0", DEVFS_FL_DEFAULT,
				gMajor, TS_MINOR, S_IFCHR | S_IRUSR | S_IWUSR,
				&psaux_fops, &g_touchscreen.filtered );
	devfs_tsraw  = devfs_register( devfs_ts_dir, "0raw", DEVFS_FL_DEFAULT,
			   gMajor, TSRAW_MINOR, S_IFCHR | S_IRUSR | S_IWUSR,
			   &psaux_fops, &g_touchscreen.raw );
	return 0;
}

void jornada720_kbd_cleanup_module(void)
{
	printk(__FUNCTION__ ": shutting down touchscreen\n");

	flush_scheduled_tasks();
	devfs_unregister(devfs_ts);
	devfs_unregister(devfs_tsraw);
	devfs_unregister(devfs_ts_dir);
	devfs_unregister_chrdev(gMajor, TS_MODULE_NAME);
}

module_init(jornada720_kbd_init_module);
module_exit(jornada720_kbd_cleanup_module);
