//-------------------------------------------------------------------
//  Copyright  1999-2001 Symbol Technologies, Inc.
//   All rights reserved.
//
//  FACILITY:   Scorch Driver Interface
//
//  ABSTRACT:
//      This module contains public definitions used to interface
//      to the Spectrum24 driver IOCTL interface.
//
//-------------------------------------------------------------------

#ifndef _SCORCH_DRIVER_INTERFACE_H_
#define _SCORCH_DRIVER_INTERFACE_H_

/*
 * Define this macro if you want to use the /proc interface for this driver.
 * Primarily its used for Access Point mode.
 */
/* #define S24_PROC_SYS 1 */

#include <linux/types.h>

#ifdef __KERNEL__
    #include <linux/ioctl.h>
#else
    #include <sys/ioctl.h>
#endif

#define _SCORCH_IOCTL_CONNECT       SIOCDEVPRIVATE+0
#define _SCORCH_IOCTL_DISCONNECT    SIOCDEVPRIVATE+1
#define _SCORCH_IOCTL_WRITE_RID     SIOCDEVPRIVATE+2
#define _SCORCH_IOCTL_READ_RID      SIOCDEVPRIVATE+3
#define _SCORCH_IOCTL_ASSOC_STATE   SIOCDEVPRIVATE+4
#define _SCORCH_IOCTL_BCAST_RATE    SIOCDEVPRIVATE+5

#define  S24T_PROC_NAME_ROOT  "spectrum24t_"
#define  S24T_PROCSYS_NAME_FORMAT S24T_PROC_NAME_ROOT "%s"

#define S24T_KRB_KEY_LEN             13

#define MAX_802_11_PKT_LEN          2300

#define SNAP_WLAP_BPDU              0x8782
#define SNAP_BPDU                   0x4242

/*
 * All Symbol signal quality metrics are in the range 0-31.
 */
#define S24_MAX_COMMS               31

#define RID_INFO_HANDOVER_ADDR      0xF000
#define RID_INFO_COMM_TALLIES       0xF100
#define RID_INFO_SCAN_RESULTS       0xF101
#define RID_INFO_GROUP_ORD			0xF103
#define RID_INFO_HOST_SCAN_RESULTS  0xF104
#define RID_INFO_LINK_STATUS        0xF200
#define RID_INFO_ASSOCIATION_STATUS 0xF201

#define RID_CFG_PORT_TYPE           0xFC00
#define RID_CFG_MAC_ADDRESS         0xFC01
#define RID_CFG_SSID                0xFC02
#define RID_CFG_CHANNEL             0xFC03
#define RID_CFG_OWN_SSID            0xFC04
#define RID_CFG_OWN_ATIM            0xFC05
#define RID_CFG_SYSTEM_SCALE        0xFC06      // AP Density
#define RID_CFG_MAX_DATA_LENGTH     0xFC07
#define RID_CFG_WDS_ADDRESS         0xFC08
#define RID_CFG_PM_ENABLE           0xFC09
#define RID_CFG_PM_EPS              0xFC0A
#define RID_CFG_MULTICAST_RCV       0xFC0B
#define RID_CFG_MAX_SLEEP_TIME      0xFC0C
#define RID_CFG_PM_HOLDOVER_TIME    0xFC0D
#define RID_CFG_NAME                0xFC0E
#define RID_CFG_MKK_FLAG            0xFC0F
#define RID_CFG_PREFERRED_BSS_ID    0xFC20
#define RID_CFG_MANDATORY_BSS_ID    0xFC21
#define RID_CFG_AUTHENTICATION      0xFC22      // Symbol
#define RID_CFG_ENCRYPT_INDEX       0xFC23      // Symbol
#define RID_CFG_KEY1                0xFC24      // Symbol
#define RID_CFG_KEY2                0xFC25      // Symbol
#define RID_CFG_KEY3                0xFC26      // Symbol
#define RID_CFG_KEY4                0xFC27      // Symbol
#define RID_CFG_ENCRYPT_FLAG        0xFC28      // Symbol
#define RID_CFG_AUTH_TYPE           0xFC2A      // Symbol
#define RID_CFG_KEY_LENGTH          0xFC2B      // Symbol

#define RID_CFG_ENABLE_KERBEROS     0xFC2F
#define RID_CFG_ADD_SESSION_KEY     0xFC30
#define RID_CFG_DEL_SESSION_KEY     0xFC31

#define RID_GROUP_ADDR              0xFC80
#define RID_CREATE_IBSS             0xFC81
#define RID_FRAG_THRESHOLD          0xFC82
#define RID_RTS_THRESHOLD           0xFC83

#define RID_TX_RATE                 0xFC84 // rate bit map, any or all; 1 = 1mbps, 2 = 2mbps, 4 = 5.5mbps, 8 = 11mbps
#define RID_PROMISCUOUS_MODE        0xFC85

#define RID_DIVERSITY               0xFC87      // Symbol
#define RID_OPTIONS                 0xFC88      // Symbol
#define RID_IF_FREQUENCY            0xFC89      // Symbol
#define RID_BASIC_RATES             0xFC8A      // Symbol

#define RID_TX_INHIBIT              0xFCA5      // Symbol
#define RID_CLEAR_STATS             0xFCA6      // Symbol
#define RID_ACK_TIMEOUT             0xFCAD      // Symbol

#define RID_BDCST_SCAN              0xFCAB
#define RID_HOST_SCAN_OPTIONS_5_SEC     (1 << 0)
#define RID_HOST_SCAN_OPTIONS_ONE_TIME  (1 << 1)
#define RID_HOST_SCAN_OPTIONS_PASSIVE   (1 << 6)
#define RID_HOST_SCAN_OPTIONS_BCAST     (1 << 7)

#define RID_TICK_TIME               0xFCE0

#define RID_MAX_DOWNLOAD_TIME       0xFD00
#define RID_DOWNLOAD_BUFFER         0xFD01
#define RID_PRI_IDENTITY            0xFD02
#define RID_PRI_SUP_RANGE           0xFD03
#define RID_PRIMARY_VERSION         0xFD05
#define RID_PRIMARY_DATE            0xFD06

#define RID_NIC_SERIAL              0xFD0A
#define RID_NIC_FW_ID               0xFD0B

#define RID_CHANNEL_LIST            0xFD10
#define RID_REGULATORY              0xFD11  // regulatory domains
#define RID_TEMP_TYPE               0xFD12  // temperature type
#define RID_CIS                     0xFD13
#define RID_COUNTRY_CODE            0xFD14
#define RID_COUNTRY_NAME            0xFD15
#define RID_MKK_CALL_SIGN           0xFD16
#define RID_MKK_RSSI                0xFD17
#define RID_ORDINALS                0xFD18

#define RID_STA_IDENTITY            0xFD20
#define RID_STA_SUP_RANGE           0xFD21
#define RID_MFI_ACT_RANGE           0xFD22
#define RID_CFI_ACT_RANGE           0xFD23
#define RID_SECONDARY_VERSION       0xFD24  // Symbol
#define RID_SECONDARY_DATE          0xFD25  // Symbol

#define RID_PORT_STATUS             0xFD40
#define RID_CURRENT_SSID            0xFD41
#define RID_CURRENT_BSSID           0xFD42
#define RID_COMMS_QUALITY           0xFD43
#define RID_CURR_TX_RATE            0xFD44

#define RID_SHORT_RETRY             0xFD48  // short retry limit (no RTS)
#define RID_LONG_RETRY              0xFD49  // long retry limit (RTS/CTS)

#define RID_PRIVACY                 0xFD4F  // privacy availability

#define RID_PHY_TYPE                0xFDC0  // phy type
#define RID_CURRENT_CHANNEL         0xFDC1  // current channel
#define RID_POWER_MODE              0xFDC2  // power save mode
#define RID_CCA_MODE                0xFDC3  // CCA mode

#define RID_SUPPORTED_DATA_RATES    0xFDC6  // supported rates

#define RID_FW_VERSION              0xFFFF  // file names

typedef struct
{
    __u16   WordLen;
    __u16   Rid;
}  __attribute__ ((packed)) S24tRidType, *PS24tRidType;

#define MAX_INFO_REC_DATA ((512*2)-sizeof(S24tRidType))
typedef struct
{
    S24tRidType Rec;
    __u8        Data[MAX_INFO_REC_DATA];
}  __attribute__ ((packed)) S24tInfoRecType, *PS24tInfoRecType;

typedef struct _Trilogy_TX_CTL
{
    __u16      usStatus;
    __u16      usReserved0;
    __u16      usReserved1;
    __u8       CwRange;
    __u8       Slot;
    __u8       RetriesThisFrag;
    __u8       MaxAttempts;

    /*
     * Filled out after the packet is sent.
     */
    __u8       TotalRetries;
    __u8       FragsSent;

#define         TX_CTL_PORT0    0x0000
#define         TX_CTL_PORT1    0x0100
#define         TX_CTL_PORT2    0x0200
#define         TX_CTL_PORT3    0x0300
#define         TX_CTL_PORT4    0x0400
#define         TX_CTL_PORT5    0x0500
#define         TX_CTL_PORT6    0x0600
#define         TX_CTL_802_3    0x0000          // Use 802.3 portion of header
#define         TX_CTL_802_11   0x0008          // Use 802.11 portion of header
#define         TX_CTL_TX_INT   0x0002          // Generate event on successfull TX
#define         TX_CTL_TX_EX    0x0004          // Generate event on TX error
    __u16      usTxControl;          // Type of TX (bits defined below)
}  __attribute__ ((packed)) Trilogy_TX_CTL, *PTrilogy_TX_CTL;

typedef struct
{
    unsigned    FcsErr:1;
    unsigned    Undecrypted:1;
    unsigned    Reserved1:6;
    unsigned    MacPort:3;
    unsigned    Reserved2:2;

#define      RX_MSG_TYPE_NORMAL  (RX_STATUS_TYPE_NORMAL>>13)
#define      RX_MSG_TYPE_RFC1042 (RX_STATUS_TYPE_RFC1042>>13)
#define      RX_MSG_TYPE_BRIDGE  (RX_STATUS_TYPE_BRIDGE_TUNNEL>>13)
#define      RX_MSG_TYPE_WMP     (RX_STATUS_TYPE_WMP>>13)
    unsigned    MsgType:3;
} __attribute__ ((packed)) Trilogy_RX_Status, *PTrilogy_RX_Status;

typedef struct _Trilogy_RX_CTL
{
#define         RX_STATUS_TYPE_NORMAL           0x0000
#define         RX_STATUS_TYPE_RFC1042          0x2000
#define         RX_STATUS_TYPE_BRIDGE_TUNNEL    0x4000
#define         RX_STATUS_TYPE_WMP              0x6000
#define         RX_STATUS_RCV_PORT0             0x0000
#define         RX_STATUS_RCV_PORT1             0x0100
#define         RX_STATUS_RCV_PORT2             0x0200
#define         RX_STATUS_RCV_PORT3             0x0300
#define         RX_STATUS_RCV_PORT4             0x0400
#define         RX_STATUS_RCV_PORT5             0x0500
#define         RX_STATUS_RCV_PORT6             0x0600
#define         RX_STATUS_UNDECRYPT_ERR         0x0002  // Only in promiscuous mode
#define         RX_STATUS_FCS_ERR               0x0001  // Only in promiscuous mode
#define         RX_TUNNELED (RX_STATUS_TYPE_RFC1042|RX_STATUS_TYPE_BRIDGE_TUNNEL)
    __u16       usStatus;               // RX status

    __u32      ulTime;
    __u8       ucSignal;
    __u8       ucSilence;
    __u8       ucReserved0;
    __u8       ucRate;
    __u16      usReserved1;
    __u16      usReserved2;
}  __attribute__ ((packed)) Trilogy_RX_CTL, *PTrilogy_RX_CTL;

typedef struct _HDR_802_3
{
    __u8    ucDestAddr[6];
    __u8    ucSrcAddr[6];
    __u16   usLength;               // Big Endian format
}  __attribute__ ((packed)) HDR_802_3, *PHDR_802_3;

#define ETHERNET_HEADER_SIZE 14
#define MAX_ETHERNET_BODY_SIZE 1500
#define RFC1042_SIZE            8

#define MAX_ETHERNET_PKT_SIZ (MAX_ETHERNET_BODY_SIZE+ETHERNET_HEADER_SIZE+RFC1042_SIZE)

typedef struct _TRILOGY_MAC_HDR
{
#define TX_STATUS_FORM_ERR              0x0008
#define TX_STATUS_DISCON                0x0004
#define TX_STATUS_AGED_ERR              0x0002
#define TX_STATUS_RETRY_ERR             0x0001

#define RX_STATUS_TYPE_NORMAL           0x0000
#define RX_STATUS_TYPE_RFC1042          0x2000
#define RX_STATUS_TYPE_BRIDGE_TUNNEL    0x4000
#define RX_STATUS_TYPE_WMP              0x6000
#define RX_STATUS_RCV_PORT0             0x0000
#define RX_STATUS_RCV_PORT1             0x0100
#define RX_STATUS_RCV_PORT2             0x0200
#define RX_STATUS_RCV_PORT3             0x0300
#define RX_STATUS_RCV_PORT4             0x0400
#define RX_STATUS_RCV_PORT5             0x0500
#define RX_STATUS_RCV_PORT6             0x0600
#define RX_STATUS_UNDECRYPT_ERR         0x0002  // Only in promiscuous mode
#define RX_STATUS_FCS_ERR               0x0001  // Only in promiscuous mode

#define RX_TUNNELED (RX_STATUS_TYPE_RFC1042|RX_STATUS_TYPE_BRIDGE_TUNNEL)

    union
    {
        Trilogy_TX_CTL tx;
        Trilogy_RX_CTL rx;
    } __attribute__ ((packed)) u;

    // 802.11 Format header information
#define FCTL_DATA                       0x0008
#define FCTL_TO_DS                      0x0100
#define FCTL_FROM_DS                    0x0200
#define FCNTL_BRIDGED (FCTL_DATA|FCTL_TO_DS|FCTL_FROM_DS)

    __u16   usFrameControl;
    __u16   usDuration_id;
    __u8    Dst[6];
    __u8    Src[6];
    __u8    BssId[6];

    __u16   usSequenceControl;

    __u8    ucAddress4[6];
    __u16   usDataLen;              // Little-Endian format

    // 802.3 Format header information

    __u8    Dot3Dest[6];
    __u8    Dot3Src[6];
    __u16   Dot3Len;               // Big Endian format
} __attribute__ ((packed)) TRILOGY_MAC_HDR, *PTRILOGY_MAC_HDR;

#if S24_PROC_SYS

/*
 * AP mode data structures.
 */
/*
 * Incoming transmit packets are all of this form.
 */
typedef struct
{
    /*
     * The transmit rate is in bits/sec.
     */
    __u32     TransmitRate;

    /*
     * Number of transmit attempts.
     */
    __u32     AttemptsThreshHold;
} __attribute__ ((packed)) TRILOGY_XMIT_HEADER, *PTRILOGY_XMIT_HEADER;

typedef struct
{
    __u32   Unused[2];
} __attribute__ ((packed)) TRILOGY_RCV_HEADER, *PTRILOGY_RCV_HEADER;

typedef struct
{
    union
    {
        TRILOGY_XMIT_HEADER Xmt;
        TRILOGY_RCV_HEADER  Rcv;
    } __attribute__ ((packed)) u;
    TRILOGY_MAC_HDR Hdr;
} __attribute__ ((packed)) TRILOGY_HEADER, *PTRILOGY_HEADER;

typedef struct
{
    __u32       *Next[2];

    /*
     * If Ap is non-zero, then DeviceInterface is ignored. Otherwise,
     * DeviceInterface is an index to the Delivery System interface
     * for the packet. Beacon is only valid if Ap is non-zero. If Ap
     * is AP_XMIT_TIMEOUT, then its a packet being received with transmit
     * timeout feedback. Usually this information is used to throttle back
     * on transmit rate.
     */
#define DS_INTERFACE         0x00 /* Should never see one of these in the Linux driver */
#define AP_RECV              0x01
#define AP_XMIT              0x02
#define AP_XMIT_ERR          0x10
#define AP_XMIT_RETRY_ERR    0x20
    __u8  Ap;
    __u8  Beacon;

    /*
     * Only valid if (Ap & (AP_XMIT_TIMEOUT|AP_XMIT_SUCCESS))
     */
    __u8   ApTransmitDest[6];

    /*
     * Delivery System device interface index from SCORCH_OPEN_NDIS. Only valid
     * if Ap is zero.
     */
    __u16  DeviceInterface; /* 0-AP, 1-4 are DS interfaces. */

    /*
     * Data frame length.
     */
    __u32   Len;
} __attribute__((packed)) SCORCH_DEVICE_HEADER, *PSCORCH_DEVICE_HEADER;

/*
 * This is the structure that is sent/received on the read/write proc interface to the
 * Spectrum24 driver. The union members Eth[] and Tr[] are inherited from the Win32
 * driver, but are never used for Linux. All Linux Spectrum24 read/write buffers are of the
 * form:
    SCORCH_DEVICE_HEADER Hdr;
    TRILOGY_HEADER       Trilogy
 */
typedef struct
{
    SCORCH_DEVICE_HEADER Hdr;
    union
    {
        __u8            Eth[1514];
        __u8            Tr[1514];
        __u8            Ap[1]; /* Variable size */
        TRILOGY_HEADER  Trilogy;
    } Data;
} __attribute__((packed)) SCORCH_RW, *PSCORCH_RW;

/*
 * This is the actual RID structure that the CW10 FW expects. wlen does
 * not include itself in the total length of the RID..
 */
    #define SCORCH_RID_DATA_LEN 512
typedef struct
{
    __u16   wlen; /* (sizeof(rid) + data length)/2 */
    __u16   rid; /* Host byte order */
    __u8    data[SCORCH_RID_DATA_LEN];
} __attribute__((packed)) SCORCH_RID, *PSCORCH_RID;

    #define RID_DATA_LEN(_R) ((((__u32)(_R)->wlen) - 1) * 2)

/*
 * This structure is used to add or remove an MU association. It can also be
 * used to update the transmit speed.
 */
typedef struct 
{
    __u8    MuAddress[6];
    /*
     * Non-zero if associating. 0 if unassociating.
     */
    __u16   Assoc;

    /*
     * If associating, then this is the power saving mode state.
     */
    __u32 PowerSavingMode;

    /*
     * If associating, then this is the association ID assigned to the MU.
     */
    __u32   Aid;

    /*
     * Default transmit rate and thresh hold attempts.
     */
    __u32   TransmitRate;
    __u32   AttemptsThreshHold;
} __attribute__((packed)) SCORCH_MU_ASSOC, *PSCORCH_MU_ASSOC;
#endif

typedef struct
{
    __u16   ProtocolId;
    __u8    VersionId;
    __u8    Type;
    __u8    Flag;
    __u8    Status;

    /*
     * Root ID.
     */
    __u16   RootPriority;
    __u8    RootAddr[6];

    /*
     * Root path cost.
     */
    __u16   SlotCount;
    __u16   RootPathCost;

    /*
     * Bridge ID.
     */
    __u16   WlapPriority;
    __u8    WlapAddr[6];

    /*
     * Port ID.
     */
    __u16   InterfaceId;

    __u16   MessageAge;
    __u16   MaxAge;
    __u16   HelloTime;
    __u16   ForwardDelay;
} __attribute__((packed)) WLAP_BPDU_TYPE, *PWLAP_BPDU_TYPE;

#define SIZEOF_WLAP_BPDU_TYPE 36

typedef struct
{
    __u8    Dest[6];
    __u8    Src[6];
    __u16   Dix;

    WLAP_BPDU_TYPE  Bpdu;
} __attribute__((packed)) ENCAPSULATED_WLAP_BPDU_TYPE, *PENCAPSULATED_WLAP_BPDU_TYPE;

#define SIZEOF_ENCAPSULATED_WLAP_BPDU_TYPE (SIZEOF_WLAP_BPDU_TYPE+ETHERNET_HEADER_SIZE)

/*
 * Used for SIOCDEVPRIVATE + 0x0 (get scan results)
 */
typedef struct
{
    __u16   bss_channel_id;
    __u16   average_noise_level;    /* range 0..31 */
    __u16   signal_level;           /* range 0..31 */
    __u8    mac_address[6];
    __u16   beacon_interval;        /* units of K usec */
    __u16   bss_capability;
    /*
     * Note that the SSID is a counted string that is not neccessarily NULL terminated.
     */
#define  SCAN_MAX_SSID_LEN 32
    __u16   service_set_id_len;
    __u8    service_set_id[SCAN_MAX_SSID_LEN];
    __u16   rates[5];
    __u16   basic;
    __u16   mucnt;
    __u16   byte_count;
    __u16   pkt_count;
} __attribute__((packed)) DS_SCAN_RESULT;

#define MAX_SCAN_RESULTS 100
typedef struct
{
    __u32           num_scan_results;
    DS_SCAN_RESULT  scan_results[MAX_SCAN_RESULTS];
} DS_SCAN_RESULTS;

/*
 * Used for SIOCDEVPRIVATE + 0x1 (get statistics)
 */
// Fixed size data:

#define	S24_ORD_STAT_TX_HOST_REQUESTS			1		// # of requested Host Tx's (MSDU)
#define	S24_ORD_STAT_TX_HOST_COMPLETE			2		// # of successful Host Tx's (MSDU)
#define	S24_ORD_STAT_TX_DIR_DATA				3		// # of successful Directed Tx's (MSDU)
#define	S24_ORD_STAT_TX_DIR_DATA1				4		// # of successful Directed Tx's (MSDU) @ 1MB
#define	S24_ORD_STAT_TX_DIR_DATA2				5		// # of successful Directed Tx's (MSDU) @ 2MB
#define	S24_ORD_STAT_TX_NONDIR_DATA				6		// # of successful Non-Directed Tx's (MSDU)
#define	S24_ORD_STAT_TX_NONDIR_DATA1			7		// # of successful Non-Directed Tx's (MSDU) @ 1MB
#define	S24_ORD_STAT_TX_NONDIR_DATA2			8		// # of successful Non-Directed Tx's (MSDU) @ 2MB
#define	S24_ORD_STAT_TX_BEACON					9		// # of successful Beacon Tx's
#define	S24_ORD_STAT_TX_POLL					10		// # of successful Poll Tx's
#define	S24_ORD_STAT_TX_WNMP					11		// # of successful WNMP Tx's
#define	S24_ORD_STAT_TX_TOTAL_BYTES				12		// Total # of successful Data Tx Bytes
#define	S24_ORD_STAT_TX_RETRIES					13		// # of Tx retries
#define	S24_ORD_STAT_TX_RETRIES1				14		// # of Tx retries @ 1MB
#define	S24_ORD_STAT_TX_RETRIES2				15		// # of Tx retries @ 2MB
#define	S24_ORD_STAT_TX_FAILURES				16		// # of failed Tx's, even after retries
#define	S24_ORD_STAT_TX_ERR_ACK					17		// # of Tx errors due to Ack
#define	S24_ORD_STAT_TX_ERR_ACK_TIMEOUT			18		// # of Tx errors due timeout waiting for Ack
#define	S24_ORD_STAT_TX_ERR_ACK_TYPE			19		// # of Tx errors due to missing Ack
#define	S24_ORD_STAT_TX_ERR_ACK_ADDRESS			20		// # of Tx errors due to dest. address in Ack
#define	S24_ORD_STAT_TX_ERR_ACK_CRC				21		// # of Tx errors due to CRC error
#define	S24_ORD_STAT_POLL_ERR					22		// # of Poll errors due to Data frame
#define	S24_ORD_STAT_POLL_ERR_PLCP				23		// # of Poll errors due to Data frame PLCP timeouts
#define	S24_ORD_STAT_POLL_ERR_TYPE				24		// # of Poll errors due to bad type for Data frame
#define	S24_ORD_STAT_POLL_ERR_ADDRESS			25		// # of Poll errors due to dest. address mismatch
#define	S24_ORD_STAT_TX_ABORT_AT_HOP			26		// # of Tx's aborted at hop time
#define	S24_ORD_STAT_TX_MAX_TRIES_IN_HOP		27		// # of times all allowed tries in a given hop failed
#define	S24_ORD_STAT_TX_ABORT_LATE_DMA			28		// # of Tx aborts due to late DMA switch
#define	S24_ORD_STAT_LATE_RX_TO_TX				29		// # of times RxToTx was set too late
#define	S24_ORD_STAT_TX_RETRY_THRESHOLD			30		// # of  Tx's where Tx retry threshold was exceeded
#define	S24_ORD_STAT_TX_ABORT_STX				31		// # of times backoff was aborted
#define	S24_ORD_STAT_TX_ABORT_STX_MAP			32		// # of times backoff for BCN was aborted
#define	S24_ORD_STAT_RX_HOST					33		// # of Rx packets passed to host
#define	S24_ORD_STAT_RX_DIR_DATA				34		// # of Directed Data Rx's
#define	S24_ORD_STAT_RX_DIR_DATA1				35		// # of Directed Data Rx's @ 1MB
#define	S24_ORD_STAT_RX_DIR_DATA2				36		// # of Directed Data Rx's @ 2MB
#define	S24_ORD_STAT_RX_NONDIR_DATA				37		// # of Non-Directed Data Rx's
#define	S24_ORD_STAT_RX_NONDIR_DATA1			38		// # of Non-Directed Data Rx's @ 1MB
#define	S24_ORD_STAT_RX_NONDIR_DATA2			39		// # of Non-Directed Data Rx's @ 2MB
#define	S24_ORD_STAT_RX_BEACON					40		// # of received Beacons
#define	S24_ORD_STAT_RX_POLL					41		// # of received Polls
#define	S24_ORD_STAT_RX_WNMP					42		// # of received WNMP
#define	S24_ORD_STAT_RX_TOTAL_BYTES				43		// Total # of successful MPDU Rx bytes
#define	S24_ORD_STAT_RX_ERR_CRC					44		// # of Rx CRC errors
#define	S24_ORD_STAT_RX_ERR_CRC1				45		// # of Rx CRC errors @ 1MB
#define	S24_ORD_STAT_RX_ERR_CRC2				46		// # of Rx CRC errors @ 2MB
#define	S24_ORD_STAT_RX_DUPLICATE				47		// # of duplicate Rx MPDU's
#define	S24_ORD_STAT_RX_TOO_LATE				48		// # of frames w/PLCP read too late
#define	S24_ORD_STAT_RX_ABORTED					49		// # of frames aborted due to CRUX anomaly
#define	S24_ORD_STAT_RX_INVALID_PLCP			50		// # of frames Rx'ed w/CRC error in PLCP
#define	S24_ORD_STAT_RX_PLCP_TOO_BIG			51		// # of frames with PLCP length too big
#define	S24_ORD_STAT_RX_PLCP_TOO_SMALL			52		// # of frames with PLCP length too small
#define	S24_ORD_STAT_RX_INVALID_PROTOCOL		53		// # of frames Rx'ed w/invalid protocol version
#define	S24_ORD_STAT_RX_INVALID_TYPE			54		// # of invalid types after SFD
#define	S24_ORD_STAT_RX_NO_BUFFER				55		// # of frames dropped due to no buffers
#define	S24_ORD_STAT_RX_ABORT_LATE_DMA			56		// # of times Rx aborted due to returning from
														// suspension too late to switch DMA buffers
#define	S24_ORD_STAT_RX_ABORT_AT_HOP			57		// # of Rx's aborted at hop time
#define	S24_ORD_STAT_RX_CRUX_ERROR2				58		// Rx error: TC reached, CRC32G=0,RXCNT=0
#define	S24_ORD_STAT_RX_CRUX_ERROR3				59		// Rx error: TC reached, CRC32G=1, but more bytes
														// DMA'ed than expected
#define	S24_ORD_STAT_MISSING_FRAGMENTS			60		// # of dropped Rx fragments
#define	S24_ORD_STAT_ORPHAN_FRAGMENTS			61		// # of non-sequential fragments
#define	S24_ORD_STAT_ORPHAN_FRAME				62		// # of unmatched non-first frames
#define	S24_ORD_STAT_FRAGMENT_AGEOUT			63		// # of unmatched branches aged out
#define	S24_ORD_STAT_RX_BAD_SRC_ADR				64		// Src_Adr not in Association Table
#define	S24_ORD_STAT_RX_BAD_SSID				65		// ESSID does not match MAP's ESSID
#define	S24_ORD_STAT_RX_BAD_STATION_ID			66		// Station ID mismatch in Poll
#define	S24_ORD_STAT_RX_MULTIPLE_POLLS			67		// Multiple Polls Rx'ed for same packet
#define	S24_ORD_STAT_RX_POLL_NO_DATA			68		// Poll but no data queued
#define	S24_ORD_STAT_RX_POLL_TIMEOUT			69		// MAP timeout waiting for Poll
#define	S24_ORD_STAT_PSP_SUSPENSIONS			70		// # of PSP adapter suspensions
#define	S24_ORD_STAT_PSP_RX_TIMS				71		// # of PSP TIM's received
#define	S24_ORD_STAT_PSP_RX_DTIMS				72		// # of PSP DTIM's received
#define	S24_ORD_STAT_PSP_BEACON_TIMEOUTS		73		// # of timeouts waiting for Beacons
#define	S24_ORD_STAT_PSP_POLL_TIMEOUTS			74		// # of timeouts waiting for Poll response
#define	S24_ORD_STAT_PSP_BCAST_TIMEOUT			75		// # of timeouts waiting for the last broadcast/
														// multicast msg. when in PSP mode
#define	S24_ORD_STAT_FIRST_ASSOC_TIME			76		// Time of first AP association
#define	S24_ORD_STAT_LAST_ASSOC_TIME			77		// Time of last AP association
#define	S24_ORD_STAT_PERCENT_MISSED_BEACONS		78		// % missed Beacons in the last 5 seconds
#define	S24_ORD_STAT_PERCENT_RETRIES			79		// % Tx retries in the last 2.5 seconds
#define	S24_ORD_STAT_PERCENT_CRC_ERRORS			80		// % CRC errors in the last 2.5 seconds
#define	S24_ORD_STAT_ASSOCIATED_AP				81		// ptr. to the AP Table entry for the currently associated AP
#define	S24_ORD_STAT_AVAIL_AP_CNT				82		// # of AP's described in the AP Table
#define	S24_ORD_STAT_AP_LIST					83		// ptr. to the linked list of available AP's
#define	S24_ORD_STAT_AP_ASSOCIATIONS			84		// # of AP associations
#define	S24_ORD_STAT_ASSOCIATE_FAIL				85		// # of failed AP associations
#define	S24_ORD_STAT_ASSOC_RESPONSE_FAIL		86		// # of failed association responses
#define	S24_ORD_STAT_FULL_SCANS					87		// # of full scans performed
#define	S24_ORD_STAT_PARTIAL_SCANS				88		// # of partial scans performed
#define	S24_ORD_STAT_ROAM_INHIBIT				89		// *** TBD ***
#define	S24_ORD_STAT_RSSI_PEAK					90		// Peak RSSI for the  associated AP (LSB=1/4 counts)
#define	S24_ORD_STAT_RSSI_TRIGGER				91		// RSSI value at which a PSS is triggered (LSB=1/4 counts)
#define	S24_ORD_STAT_RSSI_TRIGGER_RESETS		92		// # of RSSI_Trigger resets
#define	S24_ORD_STAT_ASSOC_CAUSE1				93		// Assoc. cause: No Tx seen from the AP in the last N hops
														// or no Probe Response seen in 3 minutes
#define	S24_ORD_STAT_ASSOC_CAUSE2				94		// Assoc. cause: AP RSSI fell below the eligble threshold
#define	S24_ORD_STAT_ASSOC_CAUSE3				95		// Assoc. cause: Association dropped by the AP
														// (reported in a Probe Response)
#define	S24_ORD_STAT_ASSOC_CAUSE4				96		// Assoc. cause: Poor Rx/Tx quality
#define	S24_ORD_STAT_ASSOC_CAUSE5				97		// Assoc. cause: AP load leveling
#define	S24_ORD_STAT_ASSOC_CAUSE6				98		// Assoc. cause: Power mode change
#define	S24_ORD_STAT_PSS_CAUSE1					99		// Partial scan series cause: Unassociated
#define	S24_ORD_STAT_PSS_CAUSE2					100		// Partial scan series cause: RSSI dropped 8 counts from the peak
#define	S24_ORD_STAT_PSS_CAUSE3					101		// Partial scan series cause: RSSI dropped an additional 3 counts
#define	S24_ORD_STAT_PSS_CAUSE4					102		// Partial scan series cause: Poor Rx/Tx quality
#define	S24_ORD_STAT_PSS_CAUSE5					103		// Partial scan series cause: AP load leveling
#define	S24_ORD_STAT_PSS_CUM_CAUSE1				104		// Cum. Partial scan series cause: Unassociated
#define	S24_ORD_STAT_PSS_CUM_CAUSE2				105		// Cum. Partial scan series cause: RSSI dropped 8
														// counts from the peak
#define	S24_ORD_STAT_PSS_CUM_CAUSE3				106		// Cum. Partial scan series cause: RSSI dropped an
														// additional 3 counts
#define	S24_ORD_STAT_PSS_CUM_CAUSE4				107		// Cum. Partial scan series cause: Poor Rx/Tx quality
#define	S24_ORD_STAT_PSS_CUM_CAUSE5				108		// Cum. Partial scan series cause: AP load leveling
#define	S24_ORD_STAT_STEST_RESULTS_CURR			109		// Results of the last self-test
#define	S24_ORD_STAT_STEST_RESULTS_CUM			110		// Cumulative results of the self-test
#define	S24_ORD_STAT_SELFTEST_STATUS			111		// Current self-test status
#define	S24_ORD_STAT_POWER_MODE					112		// Current power mgt. Mode 0=CAM, 10h=PSP
#define	S24_ORD_STAT_TIM_ALGORITHM				113		// TIM interval algorithm number
#define	S24_ORD_STAT_TIM_MINIMUM				114		// Minimum TIM listen interval
#define	S24_ORD_STAT_TIM_MAXIMUM				115		// Maximum TIM listen interval
#define	S24_ORD_STAT_TX_DISABLED				116		// Transmitter disable state: 0=enabled, 1=disabled
#define	S24_ORD_STAT_COUNTRY_CODE				117		// Country code: 1=US, 2=Japan, 3=Fra, etc.
#define	S24_ORD_STAT_SCRAMBLE_KEY				118		// Data scramble key (default=0)
#define	S24_ORD_STAT_STATION_ID					119		// Station ID for PSP mode
#define	S24_ORD_STAT_RESET_COUNT				120		// Resets since power up
#define	S24_ORD_STAT_BEACON_INTERVAL			121		// Beacon interval (LSB = 1K us)
#define	S24_ORD_STAT_CRUX_VERSION				122		// CRUX config. Register version
#define	S24_ORD_STAT_MOON_VERSION				123		// Moon config. Register version
#define	S24_ORD_STAT_ANTENNA_DIVERSITY			124		// Antenna diversity state: 0=enabled, 1=disabled
#define	S24_ORD_STAT_MKK_RSSI					125		// Manufacturing RSSI threshold
#define	S24_ORD_STAT_FLASH_UPDATE				126		// # of times FLASH was updated
#define	S24_ORD_STAT_CW_TX						127		// # of times Tx started on top of CW
#define	S24_ORD_STAT_HOP_SET					128		// Hop set number
#define	S24_ORD_STAT_HOP_PATTERN				129		// Hop pattern number
#define	S24_ORD_STAT_HOP_INDEX					130		// Hop index
#define	S24_ORD_STAT_DWELL_TIME					131		// Hop interval (LSB = 1K us)
#define	S24_ORD_STAT_DTIM_PERIOD				132		// # of Beacon intervals between DTIM's
#define	S24_ORD_STAT_OUR_HOP_SEQUENCE			133		// Hop set/hop set index
#define	S24_ORD_STAT_OUR_HOP_DELTA				134		// Hop delta from hop  table
#define	S24_ORD_STAT_OUR_FREQUENCY				135		// Current frequency
#define S24_ORD_STAT_TX_NULL_DATA				136		// Successful null data tx's
#define S24_ORD_STAT_TX_AUTHENTICATION			137		// Successful authentication tx's
#define S24_ORD_STAT_TX_ASSOCIATION				138		// Successful association tx's
#define S24_ORD_STAT_TX_ASSOC_RESP				139		// Successful association response tx's
#define S24_ORD_STAT_TX_DEAUTHENTICATION		140		// Successful deauthentic tx's
#define S24_ORD_STAT_TX_DISASSOCIATION			141		// Successful disassoc tx's
#define S24_ORD_STAT_TX_PROBE					142		// Successful probe tx
#define S24_ORD_STAT_TX_PROBE_RESP				143		// Successful probe response tx
#define S24_ORD_STAT_TX_RTS						144		// Successful RTS
#define S24_ORD_STAT_TX_CTS						145		// Successful CTS
#define S24_ORD_STAT_TX_CFEND					146		// Successful CF end
#define S24_ORD_STAT_RX_NULL_DATA				147		// Receive null data msg
#define S24_ORD_STAT_RX_AUTHENTICATION			148		// Receive authentication msg
#define S24_ORD_STAT_RX_ASSOCIATION				149		// Receive association msg
#define S24_ORD_STAT_RX_ASSOC_RESP				150		// Receive assoc resp msg
#define S24_ORD_STAT_RX_DEAUTHENTICATION		151		// Receive deauthentic msg
#define S24_ORD_STAT_RX_DISASSOCIATION			152		// Receive disassoc msg
#define S24_ORD_STAT_RX_PROBE					153		// Receive probe msg
#define S24_ORD_STAT_RX_PROBE_RESP				154		// Receive probe resp msg
#define S24_ORD_STAT_RX_RTS						155		// Receive RTS frame
#define S24_ORD_STAT_RX_CTS						156		// Receive CTS frame
#define S24_ORD_STAT_RX_CFEND					157		// Receive CF end
#define S24_ORD_STAT_RX_DUPLICATE1				158		// Duplicate msgs at 1 mbps
#define S24_ORD_STAT_RX_DUPLICATE2				159		// Duplicate msgs at 2 mbps
#define S24_ORD_STAT_AUTHENTICATE_FAIL			160		// Authentication phase fail
#define S24_ORD_STAT_AUTHENTICATE_RESP_FAIL		161		// Authentication resp phase fail
#define S24_ORD_STAT_ASSOC_CAUSE7				162		// Assoc. cause: ESSID change
#define S24_ORD_STAT_ASSOC_CAUSE8				163		// Assoc. cause: Preferred/Mandatory change
#define S24_ORD_RTC_TIME						164		// 30.5 usec RTC time
#define S24_ORD_OP_MODE							165		// Operating mode
#define S24_ORD_ASSOCIATION_EVENTS				166		// Association events/status
#define S24_ORD_ASSOC_TABLE_COUNT				167		// Number of associated MUs
#define S24_ORD_DEBUG_PROGRAM_COUNTER			168		// Diagnostic program counter
#define S24_ORD_ADAPTER_POWER_TYPE				169		// Adapter power (100/500)
#define S24_ORD_TX_RATE							170		// Current tx rate
#define S24_ORD_MAX_TX_RATE						171		// Maximum tx rate for mu
#define S24_ORD_MAX_ASSOC_TX_RATE				172		// Max tx rate for associated ap
#define S24_ORD_CAPABILITY_INFO					173		// 
#define S24_ORD_STAT_TX_HW_FAIL					174		// 
#define S24_ORD_BASIC_RATES_BITMAP				175		// 
#define S24_ORD_STAT_DISASSOCIATION_FAIL		176		// Times disassociation failed
#define	S24_ORD_ACL_ENTRIES						177		// Number of ACL entries
#define	S24_ORD_AUTHENTICATION_TYPE				178		// Open system / shared key
#define S24_ORD_STAT_ICV_ERRORS					179		// 
#define S24_ORD_RADIO_TYPE						180		// Sirius / duo / trilogy
#define S24_ORD_VARIABLE_CPU_SPEED				181		// Capable or not
#define S24_ORD_STAT_ERR_CTS					182		// Number of cts errors
#define S24_ORD_CTS_MSW							183		// # of CTS errors - MSW
#define S24_ORD_RTS_THRESHOLD					184		// RSSI threshold for rts
#define S24_ORD_DISCOVER_MODE_COUNT				189		// Discover Mode Count

// From here up is Trilogy only.
#define SYM_ORD_STAT_ASSOC_MU_COUNT				211		// count of associated mus
#define SYM_ORD_STAT_LOWEST_MU_COUNT			212		// lowest count of associated mus
#define	SYM_ORD_STAT_LOAD_LEVELING_SCANS		213		// # of load leveling scans
#define	SYM_ORD_STAT_PERIODIC_SCANS				214		// # of periodic scans
#define	SYM_ORD_STAT_SRTSEQ_PTR					215		// Pointer to ordered ap list
#define	SYM_ORD_STAT_RATE_LOG_INDEX				216		// 'Next' entry in rate log
#define	SYM_ORD_STAT_RATE_LOG_WRAP				217		// Rate log has wrapped or not
#define SYM_ORD_STAT_ENABLE_COUNT				218		// # of adapter enables
#define	SYM_ORD_STAT_DISABLE_COUNT				219		// # of adapter disables
#define	SYM_ORD_STAT_TX_DIR_DATA5_5				220		// # of ok Directed Tx's (MSDU) @ 5.5MB
#define	SYM_ORD_STAT_TX_DIR_DATA11				221		// # of ok Directed Tx's (MSDU) @ 11MB
#define	SYM_ORD_STAT_TX_NONDIR_DATA5_5			222		// # of ok Nondirected Tx's (MSDU) @ 5.5MB
#define	SYM_ORD_STAT_TX_NONDIR_DATA11			223		// # of ok Nondirected Tx's (MSDU) @ 11MB
#define	SYM_ORD_STAT_TX_RETRIES5_5				224		// # of Tx retries @ 5.5MB
#define	SYM_ORD_STAT_TX_RETRIES11				225		// # of Tx retries @ 11MB
#define	SYM_ORD_STAT_RX_DIR_DATA5_5				226		// # of ok Directed Rx's (MSDU) @ 5.5MB
#define	SYM_ORD_STAT_RX_DIR_DATA11				227		// # of ok Directed Rx's (MSDU) @ 11MB
#define	SYM_ORD_STAT_RX_NONDIR_DATA5_5			228		// # of ok Nondirected Rx's (MSDU) @ 5.5MB
#define	SYM_ORD_STAT_RX_NONDIR_DATA11			229		// # of ok Nondirected Rx's (MSDU) @ 11MB
#define	SYM_ORD_STAT_RX_ERR_CRC5_5				230		// # of Rx CRC errors @ 5.5MB
#define	SYM_ORD_STAT_RX_ERR_CRC11				231		// # of Rx CRC errors @ 11MB
#define	SYM_ORD_STAT_RX_DUPLICATE5_5			232		// Duplicate msgs at 5.5 mbps
#define	SYM_ORD_STAT_RX_DUPLICATE11				233		// Duplicate msgs at 11 mbps
#define	SYM_ORD_STAT_TIME						234		// Location of firmware ms clock
#define	SYM_ORD_STAT_MAX_BASIC_RATE				235
#define	SYM_ORD_STAT_CURRENT_RATES				236
#define	SYM_ORD_STAT_LOG_INDEX					237		// 'Next' entry in roam log
#define	SYM_ORD_ROAM_LOG_WRAP					238

#define SYM_ORD_STAT_FREE_BUFFERS				239
#define SYM_ORD_STAT_TX_NOR_Q_COUNT				240
#define SYM_ORD_STAT_TX_EXP_Q_COUNT				241
#define SYM_ORD_STAT_TX_PRND_Q_COUNT			242
#define SYM_ORD_STAT_TX_ALLOC_Q_COUNT			243
#define SYM_ORD_STAT_TX_COMP_Q_COUNT			244
#define SYM_ORD_STAT_RX_ACTIV_Q_COUNT			245
#define SYM_ORD_STAT_RX_Q_COUNT					246
#define SYM_ORD_STAT_AP_TABLE_START				247
#define	SYM_ORD_STAT_CHANNEL_QUALITY			248
#define	SYM_ORD_STAT_AVERAGE_SIGNAL_LEVEL		249
#define	SYM_ORD_STAT_AVERAGE_NOISE_LEVEL		250

#define	LAST_FIXED_LENGTH_ORDINAL				250

/*
 * This structure is sent by an application with the offset set for the ordinals that it
 * wishes to be returned. For example, Ord[SYM_ORD_STAT_AVERAGE_NOISE_LEVEL]=1 will cause
 * a value to be returned in Val[SYM_ORD_STAT_AVERAGE_NOISE_LEVEL]. 
 * Uses SIOCDEVPRIVATE + 0x5.
 */
typedef struct {
    __u8    Ord[LAST_FIXED_LENGTH_ORDINAL+1];
    __u32   Val[LAST_FIXED_LENGTH_ORDINAL+1];
} S24T_ORD_ARRAY;

typedef struct _GROUP_ORD 
{
    __u32 tx_host_complete;
    __u32 tx_dir;
    __u32 tx_dir1;
    __u32 tx_dir2;
    __u32 tx_non_dir;
    __u32 tx_non_dir1;
    __u32 tx_non_dir2;
    __u32 tx_bytes;
    __u16 tx_failures;
    __u32 rx_host;
    __u32 rx_dir;
    __u32 rx_dir1;
    __u32 rx_dir2;
    __u32 rx_nondir;
    __u32 rx_nondir1;
    __u32 rx_nondir2;
    __u32 rx_beacon;
    __u32 rx_bytes;
    __u16 percent_missed_beacons;
    __u16 percent_retries;
    __u16 percent_crc_errors;
    __u16 avail_ap_cnt;
    __u16 ap_associations;
    __u16 full_scans;
    __u16 partial_scans;
    __u16 assoc_cause1;
    __u16 assoc_cause2;
    __u16 assoc_cause3;
    __u16 assoc_cause4;
    __u16 assoc_cause5;
    __u16 assoc_cause6;
    __u16 stest_results_curr;
    __u16 selftest_status;
    __u16 tx_disabled;
    __u16 country_code;
    __u16 channel;
    __u16 curr_tx_rate;
    __u32 tx_dir5;
    __u32 tx_dir11;
    __u32 tx_nondir5;
    __u32 tx_nondir11;
    __u32 rx_dir5;
    __u32 rx_dir11;
    __u32 rx_nondir5;
    __u32 rx_nondir11;
    __u16 ap_table;
    __u16 cq;
} __attribute__((packed)) GROUP_ORD, *PGROUP_ORD;

/*
 * Used for SIOCDEVPRIVATE + 0x2 (get driver info)
 */
typedef struct _S24_DRIVER_INFO
{
#   define                      S24_FW_VERSION_LEN 14
    char                        FwVersion[S24_FW_VERSION_LEN+1];
    char                        FwDate[S24_FW_VERSION_LEN+1];
    __u32                       ApDistance; /* miles */
    __u32                       ApDensity;
    __u32                       Diversity;
    __u32                       ConfiguredTxRateMask; /* 1=1Mbit, 2=2Mb, 4=5.5Mb, 8=11Mb */
    __u32                       ActualTxRate; /* 1..100 */
    __u32                       RcvAllMcast;
    __u32                       ConfiguredChannel;
    __u32                       UptimeSecs;
    __u32                       Associated;
    __u32                       CommsCq;
    __u32                       CommsAsl;
    __u32                       CommsAnl;
} S24_DRIVER_INFO, *PS24_DRIVER_INFO;

/*
 * CW10 register structure.
 */
#define MAX_CW10_REGISTER   64
typedef struct {
    __u8    RegNum;
    __u8    RegVal;
} S24_CW10_REG;

/*
 * Used for SIOCDEVPRIVATE + 0x3 (set driver info) for features that are unique to
 * Symbol Trilogy.
 */
typedef enum {
    S24InfoSetApDistance,
    S24InfoSetApDensity,
    S24InfoSetAntennaDiversity,
    S24InfoSetTxRateMask,
    S24InfoSetRcvAllMcast,
    S24InfoSetHostScan,
    S24InfoSetCw10Reg
} S24_DRIVER_INFO_SET_ENUM;

typedef struct _S24_DRIVER_INFO_SET
{
    S24_DRIVER_INFO_SET_ENUM    InfoSet;
    union {
        __u32           ApDistance; /* in miles */
        __u32           ApDensity; /* 0-3 */
        __u32           Diversity;
        __u32           TxRateMask;
        __u32           RcvAllMcast;
        __u32           HostScan;
        S24_CW10_REG    Reg;
    } u;
} S24_DRIVER_INFO_SET, *PS24_DRIVER_INFO_SET;

/*
 * Used for SIOCDEVPRIVATE + 0x6 (read CW10 registers)
 */
typedef struct {
    __u8 Regs[MAX_CW10_REGISTER];
} S24_CW10_REGS;
#endif

