/*      MIPv6 authen        
 *	
 *      Authors: 
 *      Henrik Petander         <lpetande@tml.hut.fi>
 * 
 *      $Id: auth_subopt.c,v 1.4 2002/05/16 15:36:25 jamey Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *      Changes: 
 *
 */

#include <linux/autoconf.h>
#include <linux/icmpv6.h>

#include <net/mipv6.h>
#include "debug.h"
#include "ah_algo.h"
#include "auth_subopt.h"
#include "sadb.h"

/*
 *  Authentication Data Sub-Option (alignment requirement: 8n+6)
 *
 *   0                   1                   2                   3
 *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 
 *                                  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *                                  |       4       |    Length     |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |                 Security Parameters Index (SPI)               |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |                                                               |
 *  =             Authentication Data (variable length)             =
 *  |                                                               |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */

/* Data BU:
  dest. addr. (home address),
  source address (care-of oaddress),
  home address from hoa option, 
  BU-opt: t,l,flags, reserved, seq.nr., ltime, 
  all other BU subopts,
  auth. subopt: type, length, SPI
*/

/* Data BA:
   dest. addr., (home address of the receiver)
   source address, (home address  of the sender)
   BA: option type, length, seq. nr., lifetime field, refresh field,
   all other BA suboptions,
   auth suboption: type, length, SPI
*/

int mipv6_auth_build(struct in6_addr *daddr, struct in6_addr *coa, 
		     struct in6_addr *hoa, __u8 *opt, __u8 *opt_end)
{
/* First look up the peer from sadb based on his address */ 
      
	struct ah_processing ahp;
	struct sec_as *sa;
	__u8 buf[MAX_HASH_LENGTH];  
	struct mipv6_subopt_auth_data *aud = (struct mipv6_subopt_auth_data *)opt_end;
	if ((sa = mipv6_sa_get(daddr, OUTBOUND, 0))  == NULL) {
		DEBUG((DBG_WARNING, " Authentication failed due to a missing security association"));
		return -1;
	}
	if (!aud) {
		DEBUG((DBG_ERROR, "Auth subopt missing"));
		goto error;
	}

	aud->type = MIPV6_SUBOPT_AUTH_DATA;
	aud->length = 4 + sa->auth_data_length; /* Type and length not included */
	aud->spi = htonl(sa->spi);
	opt[1] += (aud->length +2);
	if (sa->alg_auth.init(&ahp, sa) < 0) { 
                DEBUG((DBG_ERROR, "Auth subopt: internal error"));
		mipv6_sa_put(&sa); 
                return -1; 
        } 
	/* First the common part */
	if (daddr)
		sa->alg_auth.loop(&ahp, daddr, sizeof(struct in6_addr));
	else {
		DEBUG((DBG_ERROR, "hoa missing from auth subopt calculation"));
		goto error;
	}
	if (coa)
		sa->alg_auth.loop(&ahp, coa, sizeof(struct in6_addr));
	if (hoa) /* can also be regular source address for BA */
		sa->alg_auth.loop(&ahp, hoa, sizeof(struct in6_addr));
	else {
		DEBUG((DBG_ERROR, "hoa missing from auth subopt calculation"));
		goto error;
	}
	sa->alg_auth.loop(&ahp, opt, opt_end - opt);

	/* Include the type, length and SPI fields and from suboption */
	sa->alg_auth.loop(&ahp, (__u8 *)aud, 2*sizeof(__u8) + sizeof(__u32));
	sa->alg_auth.result(&ahp, buf);
	memcpy(aud->data, buf, sa->auth_data_length);
	mipv6_sa_put(&sa); 

	return (aud->length + 2);

error:	
	mipv6_sa_put(&sa); 
	DEBUG((DBG_ERROR, "Calculation of hash failed in authentication suboption"));
	return -1;
}

int mipv6_auth_check(struct in6_addr *daddr, struct in6_addr *coa, 
		     struct in6_addr *hoa, __u8 *opt, __u8 optlen, 
		     struct mipv6_subopt_auth_data *aud)
{
	int ret = 0, spi;
	struct ah_processing ahp;
	struct sec_as *sa;
	__u8 htarget[MAX_HASH_LENGTH];
	spi = ntohl(aud->spi);
	/* Look up peer by home address */ 
	if ((sa = mipv6_sa_get(hoa, INBOUND, spi))  == NULL) {
		DEBUG((DBG_ERROR, " Authentication failed due to a missing security association"));
		return -1;
	}
	if (!aud)
		goto out;

	if (aud->length != (4 + sa->auth_data_length)) {
		DEBUG((DBG_ERROR, "Incorrect authentication suboption length %d", aud->length)); 
		ret = -1;
		goto out; 
	}
	if (ntohl(aud->spi) != sa->spi) {
		DEBUG((DBG_ERROR, "Incorrect spi in authentication suboption  %d", aud->length)); 
		ret = -1;
		goto out;
	}
	if (sa->alg_auth.init(&ahp, sa) < 0) { 
                DEBUG((DBG_ERROR, "Auth subopt receive: internal error in initialization of authentication algorithm"));

                ret = -1;
		goto out;
        } 

	if (daddr)
		sa->alg_auth.loop(&ahp, daddr, sizeof(struct in6_addr));
	else {
		DEBUG((DBG_ERROR,"Auth subopt check: destination addr. missing"));
		ret = -1;
		goto out;
	}
	if (coa)
		sa->alg_auth.loop(&ahp, coa, sizeof(struct in6_addr));
	if (hoa) /* can also be regular source address for BA */
		sa->alg_auth.loop(&ahp, hoa, sizeof(struct in6_addr));
	else {
		DEBUG((DBG_ERROR,"Auth subopt check: source addr. missing"));
		ret = -1;
		goto out;
	}
	sa->alg_auth.loop(&ahp, opt, (opt[1] + 2 - sa->auth_data_length));

	/* Include the type, length and SPI fields and from suboption */
	sa->alg_auth.loop(&ahp, (__u8*)aud, 2*sizeof(__u8) + sizeof(__u32));

	sa->alg_auth.result(&ahp, htarget);
	if (memcmp(htarget, aud->data, sa->auth_data_length))
		ret = 0;
	else
		ret = -1;

out:	
	mipv6_sa_put(&sa); 
	return ret;
}

