/*
 *	Header file for sorted list which can be accessed via hashkey
 *
 *	Authors:
 *	Sami Kivisaari		<skivisaa@cc.hut.fi>	
 *
 *	$Id: hashlist.h,v 1.7 2002/01/15 15:19:45 antti Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 */

#ifndef _HASHLIST_H
#define _HASHLIST_H

#define ITERATOR_ERR -1
#define ITERATOR_CONT 0
#define ITERATOR_STOP 1
#define ITERATOR_DELETE_ENTRY 2

/*
 * hashlist_create - create new hashlist
 * @max_entries: maximum number of entries in the hashlist
 * @hashsize: number of different hash values in the hashlist
 * @compare: function used for comparing entries to hashkey
 * @hash_function: function that provides hash value for the given hashkey
 * @get_hashkey: function used for getting hashkey for given data
 *
 * compare function is used to determine if two entries sharing the
 * same hash value (mod hashsize) are equal. compare must return 0 when
 * the entries equal. hash_function is supposed to return __u32 hash for
 * the given hashkey. hashlist does not store hashkeys, but needs to know
 * it in certain cases. get_hashkey must kmalloc space for the key and return
 * pointer to it. NOTE: hashlist will kfree the entry after it is no more
 * needed.
 */
struct hashlist * hashlist_create(
	int max_entries, int hashsize,
	int (*compare)(void *data, void *hashkey),
	__u32 (*hash_function)(void *hashkey),
	void *(*get_hashkey)(void *data));

/**
 * hashlist_destroy - destroy a hashlist object
 * @hashlist: hashlist to be destroyed
 **/
void hashlist_destroy(struct hashlist *hashlist);

/**
 * hashlist_get - get element from hashlist
 * @hashlist: hashlist
 * @hashkey: hashkey of the desired entry
 **/
void * hashlist_get(struct hashlist *hashlist, void *hashkey);

/**
 * hashlist_exists - check if element exists in hashlist
 * @hashlist: hashlist
 * @hashkey: hashkey of the entry to be checked
 *
 * uses hashlist_get -> not any faster than it
 **/
int hashlist_exists(struct hashlist *hashlist, void *hashkey);

/**
 * hashlist_delete - delete entry from hashlist
 * @hashlist: hashlist
 * @hashkey: hashkey of the doomed entry
 **/
int hashlist_delete(struct hashlist *hashlist, void *hashkey);

/**
 * hashlist_add - add a new entry to hashlist
 * @hashlist: hashlist
 * @hashkey: hashkey of the new entry
 * @sortkey: sortkey of the new entry
 * @data: the actual entry
 *
 * data must have been allocated beforehand
 *
 * WARNING! do NOT use this to update entries
 **/
int hashlist_add(struct hashlist *hashlist, void *hashkey,
		 unsigned long sortkey, void *data);

/* iterator function
 * 
 * args may contain any additional data for the iterator as well as
 * pointers to data returned by the iterator 
 */
typedef int (*hashlist_iterator_t)(void *, void *, unsigned long *);

/**
 * hashlist_iterate - apply an iterator function for all items in hashlist
 * @hashlist: hashlist
 * @args: arguments passed to func
 * @func: apply this function to all entries
 *
 * iterator will apply func to all entries in the hashlist as long as there
 * are entries and func returns ITERATOR_CONT. If func returns
 * ITERATOR_DELETE_ENTRY, the corresponding entry is deleted from the hashlist.
 * Iteration can be stopped prematurely by returning ITERATOR_STOP.
 **/
int hashlist_iterate(struct hashlist *hashlist, void *args,
		     hashlist_iterator_t func);

/**
 * hashlist_count - return number of elements in hashlist
 * @hashlist: hashlist
 **/
int hashlist_count(struct hashlist *hashlist);

/**
 * hashlist_is_full - return true if hashlist is full, false otherwise
 * @hashlist: hashlist
 **/
int hashlist_is_full(struct hashlist *hashlist);

/**
 * hashlist_get_first - return pointer to first element in structure
 * @hashlist: hashlist
 **/
void * hashlist_get_first(struct hashlist *hashlist);

/**
 * hashlist_get_sortkey_first - get sortkey of first element
 * @hashlist: hashlist
 **/
unsigned long hashlist_get_sortkey_first(struct hashlist *hashlist);

/**
 * hashlist_reschedule - set entry to new position in the list
 * @hashlist: hashlist
 * @hashkey: hashkey of the entry
 * @sortkey: new sortkey of the entry
 **/
int hashlist_reschedule(struct hashlist *hashlist, void *hashkey,
			unsigned long sortkey);

#endif

