/**
 * Generic icmp routines
 *
 * Authors:
 * Jaakko Laine <jola@niksula.hut.fi>
 *
 * $Id: mipv6_icmp.c,v 1.3 2002/05/16 15:36:25 jamey Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <linux/icmpv6.h>
#include <net/checksum.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>

#include "debug.h"

struct icmpv6_msg {
	struct icmp6hdr icmph;
	__u8 *data;
};

#define MIPV6_ICMP_HOP_LIMIT 64

static struct socket *icmpv6_socket = NULL;
extern struct net_proto_family inet6_family_ops;

static __u16 identifier = 0;

/**
 * mipv6_icmpv6_send - generic icmpv6 message send
 * @daddr: destination address
 * @saddr: source address
 * @type: icmp type
 * @code: icmp code
 * @id: packet identifier. If null, uses internal counter to get new id
 * @data: packet data
 * @datalen: length of data in bytes
 */
void mipv6_icmpv6_send(struct in6_addr *daddr, struct in6_addr *saddr, int type,
		       int code, __u16 *id, void *data, int datalen)
{
	int len, err;
	struct sock *sk = icmpv6_socket->sk;
	struct sk_buff *skb;
	struct ipv6hdr *hdr;
	struct icmpv6_msg *msg;
	struct flowi fl;
	struct dst_entry *dst;

	DEBUG_FUNC();

	if (!daddr)
		return;

	fl.proto = IPPROTO_ICMPV6;
	fl.fl6_dst = daddr;
	fl.fl6_src = saddr;
	fl.fl6_flowlabel = 0;
	fl.uli_u.icmpt.type = type;
	fl.uli_u.icmpt.code = code;

	dst = ip6_route_output(NULL, &fl);

	if (dst->error || !dst->dev)
		return;

	len = sizeof(struct icmp6hdr) + datalen;
	skb = sock_alloc_send_skb(sk, MAX_HEADER + len +
				  dst->dev->hard_header_len + 15, 0, &err);

	if (!skb) {
		DEBUG((DBG_WARNING, "Alloc skb failed"));
		return;
	}

	skb->dst = dst;
	skb->dev = dst->dev;

	/* ll header */
	if (!skb->dev->hard_header)
		goto fail;

	skb_reserve(skb, (skb->dev->hard_header_len + 15) & ~15);

	if (skb->dev->hard_header(skb, skb->dev, ETH_P_IPV6,
				  skb->dst->neighbour->ha,
				  NULL, len) < 0)
		goto fail;

	/* IP */
	skb->protocol = __constant_htons(ETH_P_IPV6);

	hdr = (struct ipv6hdr *) skb_put(skb, sizeof(struct ipv6hdr));
	skb->nh.ipv6h = hdr;
	*(u32 *)hdr = htonl(0x60000000);

	hdr->payload_len = htons(len);
	hdr->nexthdr = IPPROTO_ICMPV6;
	hdr->hop_limit = sk->net_pinfo.af_inet6.hop_limit;

	ipv6_addr_copy(&hdr->saddr, saddr);
	ipv6_addr_copy(&hdr->daddr, daddr);

	/* icmp */
	msg = (struct icmpv6_msg *) skb_put(skb, sizeof(struct icmp6hdr));
	msg->icmph.icmp6_type = type;
	msg->icmph.icmp6_code = code;
	msg->icmph.icmp6_cksum = 0;

	if (id)
		msg->icmph.icmp6_identifier = htons(*id);
	else
		msg->icmph.icmp6_identifier = htons(identifier++);

	/* data */
	if (datalen > 0) {
		msg->data = skb_put(skb, datalen);
		memcpy(msg->data, data, datalen);
	}

	msg->icmph.icmp6_cksum = csum_ipv6_magic(&skb->nh.ipv6h->saddr,
						 daddr, len, 
						 IPPROTO_ICMPV6,
						 csum_partial((__u8 *) msg, 
							      len, 0));

	/* pojehali! */
	dev_queue_xmit(skb);

	ICMP6_INC_STATS_BH(Icmp6OutMsgs);

	return;

 fail:
	kfree_skb(skb);
}

int mipv6_initialize_icmpv6(void)
{
	struct net_proto_family *ops = &inet6_family_ops;
	struct sock *sk;
	int err;

	if ((icmpv6_socket = sock_alloc()) == NULL) {
		DEBUG((DBG_ERROR, "Cannot allocate icmpv6_socket"));
		return -1;
	}

	icmpv6_socket->inode->i_uid = 0;
	icmpv6_socket->inode->i_gid = 0;
	icmpv6_socket->type = SOCK_RAW;

	if ((err = ops->create(icmpv6_socket, NEXTHDR_NONE)) < 0) {
		DEBUG((DBG_ERROR, "Cannot initialize icmpv6_socket"));
		sock_release(icmpv6_socket);
		icmpv6_socket = NULL; /* For safety */
		return err;
	}

	sk = icmpv6_socket->sk;
	sk->allocation = GFP_ATOMIC;
	sk->net_pinfo.af_inet6.hop_limit = MIPV6_ICMP_HOP_LIMIT;
	sk->prot->unhash(sk);

	return 0;
}

void mipv6_shutdown_icmpv6(void)
{
	if (icmpv6_socket)
		sock_release(icmpv6_socket);
	icmpv6_socket = NULL; /* For safety */
}
