/* 
   xstroke - X based gesture recognition program based on libstroke.

   Copyright (C) 2000 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>

#include "stroke.h"
#include "rec_node.h"
#include "rec_file.h"
#include "xmalloc.h"

char *progname;

/* This parameter only affects the smoothness of the displayed stroke,
 * not the quality of the recognition. */
#define MAX_STROKE_POINTS 100
/* When MAX_STOKE_POINTS is exceeded, we throw out every Nth
 * point. This is N. */
#define STROKE_DECIMATION 4
/* How wide should the line be that is used to draw the stroke. */
#define DEFAULT_STROKE_WIDTH 3
#define DEFAULT_STROKE_FILE ".xstrokerc"
#define DEFAULT_STROKE_DIR  "."
#define DEFAULT_FONT        "fixed"
#define DEFAULT_NAME        "xstroke"
#define DEFAULT_CLASS       "Xstroke"
#define DEFAULT_GEOMETRY    "100x100"
#define DEFAULT_FG_COLOR    "black"
#define DEFAULT_BG_COLOR    "white"

struct stroke_lines {
  XPoint pts[MAX_STROKE_POINTS];
  int num_pts;
  int last_decimated;
};

static Display *display;
static Window window;
static GC gc;
static Colormap colormap;
static XFontStruct *font;
static XColor fg_color;
static XColor bg_color;
static XrmDatabase db;
static struct stroke_lines lines;
static char sequence[MAX_SEQUENCE+1];
#define MAX_SUCCESS 2*MAX_SEQUENCE
static XTextItem success_text;
static struct rec_node *rec_tree_root;
static KeyCode shift_keycode;
static XrmOptionDescRec op_table[] = {
  {"-foreground", "*foreground",  XrmoptionSepArg, (XPointer) NULL},
  {"-background", "*background",  XrmoptionSepArg, (XPointer) NULL},
  {"-fg",         "*foreground",  XrmoptionSepArg, (XPointer) NULL},
  {"-bg",         "*background",  XrmoptionSepArg, (XPointer) NULL},
  {"-display",    ".display",     XrmoptionSepArg, (XPointer) NULL},
  {"-width",      ".strokeWidth", XrmoptionSepArg, (XPointer) NULL},
  {"-strokeWidth",".strokeWidth", XrmoptionSepArg, (XPointer) NULL},
  {"-fn",         "*font",        XrmoptionSepArg, (XPointer) NULL},
  {"-font",       "*font",        XrmoptionSepArg, (XPointer) NULL},
  {"-geometry",   "*geometry",    XrmoptionSepArg, (XPointer) NULL},  
  {"-f",          ".strokeFile",  XrmoptionSepArg, (XPointer) NULL},
  {"-strokeFile", ".strokeFile",  XrmoptionSepArg, (XPointer) NULL}
};
XSizeHints wm_size_hints;

static void initialize(int *argc, char *argv[]);
static void initializeX(int *argc, char *argv[]);
static void initialize_keymap(void);
static void initialize_stroke_rec(int *argc, char *argv[]);
static void main_event_loop(void);
static void cleanup_and_exit(int exit_code);
static void decimate_stroke_lines(struct stroke_lines *lines, int mod);
static int  find_key(Display *display, KeySym keysym, KeyCode *code_ret, int *col_ret);
static void send_key_press_release(char *string);

static void client_message_handler(XEvent *xev);
static void expose_event_handler(XEvent *xev);
static void button_press_handler(XEvent *xev);
static void button_release_handler(XEvent *xev);
static void motion_notify_handler(XEvent *xev);
static void key_press_handler(XEvent *xev);

int main(int argc, char *argv[])
{
  progname = argv[0];
  initialize(&argc, argv);

  /*
  print_rec_tree(rec_tree_root);
  */
  /*
  print_node_stats();
  */

  main_event_loop();

  /* This never happens... */
  return 1;
}

static void main_event_loop() {
  XSelectInput(display, window, 
	       ExposureMask
	       | ButtonPressMask
	       | ButtonReleaseMask
	       | ButtonMotionMask
	       | KeyPressMask
	       );
  while (1)  {
    XEvent xev;

    XNextEvent(display, &xev);
    switch  (xev.type) {
    case ClientMessage:
      client_message_handler(&xev);
      break;
    case Expose:   
      expose_event_handler(&xev);
      break;
    case ButtonPress:
      button_press_handler(&xev);
      break;
    case ButtonRelease:
      button_release_handler(&xev);
      break;
    case MotionNotify:
      motion_notify_handler(&xev);
      break;
    case KeyPress:
      key_press_handler(&xev);
      break;
    }
  }
}

static void initialize(int *argc, char *argv[]) {
  initializeX(argc, argv);
  initialize_stroke_rec(argc, argv);
  initialize_keymap();

  lines.num_pts = 0;
  lines.last_decimated = 0;
}

static void initializeX(int *argc, char *argv[]) {
  char *displayname = NULL;
  char *fontname = DEFAULT_FONT;
  char *fg_name = DEFAULT_FG_COLOR;
  char *bg_name = DEFAULT_BG_COLOR;
  int stroke_width = DEFAULT_STROKE_WIDTH;
  int screen;
  XrmDatabase cmd_db = NULL;
  char *res_type;
  XrmValue res_value;
  int x, y, width, height, gravity;

  XrmInitialize();
  XrmParseCommand(&cmd_db,
		  op_table, sizeof(op_table) / sizeof(op_table[0]),
		  DEFAULT_NAME,
		  argc,
		  argv);
  if (XrmGetResource(cmd_db, "xstroke.display", "Xstroke.Display",
		     &res_type, &res_value) && (strcmp(res_type,"String") ==0)) {
    displayname = res_value.addr;
  }

  display = XOpenDisplay(displayname);
  if (display) {
    Atom wm_protocols[] = { XInternAtom(display, "WM_DELETE_WINDOW", False)};
    XClassHint class_hint;
    XWMHints wm_hints;

    screen = DefaultScreen(display);

    /* TODO: Figure out the right way to merge default resources with
       those on the command line. I've looked over the Xlib
       documentation and I'm guessing that this code is correct, but
       XrmGetDatabase is always returning NULL for some reason. */
    db = XrmGetDatabase(display);
    XrmMergeDatabases(cmd_db, &db);
    XrmGetResource(db, "xstroke.geometry", "Xstroke.Geometry",
		   &res_type, &res_value);
    XWMGeometry(display, screen, (char *) res_value.addr, DEFAULT_GEOMETRY, 0, &wm_size_hints, &x, &y, &width, &height, &gravity);
    wm_size_hints.flags |= PWinGravity;
    wm_size_hints.win_gravity = gravity;

    colormap = DefaultColormap(display, screen);
    if (XrmGetResource(db, "xstroke.foreground", "Xstroke.Foreground",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      fg_name = res_value.addr;
    }
    if (! XParseColor(display, colormap, fg_name, &fg_color)) {
      fprintf(stderr,"%s: Failed to parse foreground color %s. Falling back to %s.\n", progname, fg_name, DEFAULT_FG_COLOR);
      if (! XParseColor(display, colormap, DEFAULT_FG_COLOR, &fg_color)) {
	fprintf(stderr,"%s: Failed to parse foreground color %s. Falling back to BlackPixel.\n", progname, DEFAULT_FG_COLOR);
	fg_color.pixel = BlackPixel(display, screen);
      }
    }
    XAllocColor(display, colormap, &fg_color);

    if (XrmGetResource(db, "xstroke.background", "Xstroke.Background",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      bg_name = res_value.addr;
    }
    if (! XParseColor(display, colormap, bg_name, &bg_color)) {
      fprintf(stderr,"%s: Failed to parse background color %s. Falling back to %s.\n", progname, bg_name, DEFAULT_BG_COLOR);
      if (! XParseColor(display, colormap, DEFAULT_BG_COLOR, &bg_color)) {
	fprintf(stderr,"%s: Failed to parse background color %s. Falling back to WhitePixel.\n", progname, DEFAULT_BG_COLOR);
	bg_color.pixel = WhitePixel(display, screen);
      }
    }
    XAllocColor(display, colormap, &bg_color);

    if (XrmGetResource(db, "xstroke.strokeWidth", "Xstroke.StrokeWidth",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      stroke_width = atoi(res_value.addr);
    }
    if (stroke_width <= 0) {
      fprintf(stderr, "%s: Invalid stroke_width %s. Using %d\n", progname, res_value.addr, DEFAULT_STROKE_WIDTH);
      stroke_width = DEFAULT_STROKE_WIDTH;
    }

    window = XCreateSimpleWindow(display, DefaultRootWindow(display),
				 x, y, width, height,
				 1,
				 fg_color.pixel, bg_color.pixel);
    gc = XCreateGC(display, window, 0, 0);
    XSetLineAttributes(display, gc, stroke_width, LineSolid, CapRound, JoinRound);
    XSetForeground(display, gc, fg_color.pixel);
    XSetBackground(display, gc, bg_color.pixel);

    if (XrmGetResource(db, "xstroke.font", "Xstroke.Font",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      fontname = res_value.addr;
    }
    font = XLoadQueryFont(display, fontname);
    if (font == NULL) {
      fprintf(stderr, "%s: Failed to load font %s. Falling back to %s.\n", progname, fontname, DEFAULT_FONT);
      font = XLoadQueryFont(display, DEFAULT_FONT);
    }
    if (font == NULL) {
      fprintf(stderr, "%s: Failed to load font %s. Continuing without a font!\n", progname, DEFAULT_FONT);
    } else {
      XSetFont(display, gc, font->fid);
    }
    success_text.chars = xmalloc(MAX_SUCCESS);
    success_text.delta = 0;
    success_text.font = None;

    wm_hints.flags = InputHint;
    wm_hints.input = False;
    class_hint.res_name = DEFAULT_NAME;
    class_hint.res_class = DEFAULT_CLASS;
    XmbSetWMProperties(display, window, DEFAULT_NAME, DEFAULT_NAME, argv, *argc,
		     &wm_size_hints,
		     &wm_hints, &class_hint);

    XSetWMProtocols(display, window, wm_protocols, sizeof(wm_protocols) / sizeof(Atom));
    XStoreName(display, window, DEFAULT_NAME);
    XMapWindow(display, window);
  } else {
    fprintf(stderr, "%s: Could not open display %s\n", progname, displayname ? displayname : getenv("DISPLAY"));
    exit(1);
  }
}

static void initialize_stroke_rec(int *argc, char *argv[]) {
  char *stroke_file = DEFAULT_STROKE_FILE;
  char *stroke_dir = getenv("HOME");
  char *filename;
  char *res_type;
  XrmValue res_value;

  stroke_init();

  if (stroke_dir == NULL)
    stroke_dir = DEFAULT_STROKE_DIR;

  if (XrmGetResource(db, "xstroke.strokeFile", "Xstroke.StrokeFile",
		     &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
    filename = res_value.addr;
    rec_tree_root = create_rec_tree_from_file(filename);
  } else {
    filename = xmalloc(strlen(stroke_dir) + 1 + strlen(stroke_file) + 1);
    sprintf(filename, "%s/%s", stroke_dir, stroke_file);
    rec_tree_root = create_rec_tree_from_file(filename);
    free(filename);
  }

  if (rec_tree_root == NULL && strcmp(stroke_dir, DEFAULT_STROKE_DIR)) {
    /* Try again with DEFAULT_STROKE_DIR */
    filename = xmalloc(strlen(DEFAULT_STROKE_DIR) + 1 + strlen(stroke_file) + 1);
    sprintf(filename, "%s/%s", DEFAULT_STROKE_DIR, stroke_file);
    rec_tree_root = create_rec_tree_from_file(filename);
    free(filename);
  }

  if (rec_tree_root == NULL) {
    fprintf(stderr, "%s: Could not load stroke definition file.\n", progname);
    fprintf(stderr, "%s: Tried %s/%s", progname, stroke_dir, stroke_file);
    if (strcmp(stroke_dir,DEFAULT_STROKE_DIR))
      fprintf(stderr, " and %s/%s", DEFAULT_STROKE_DIR, stroke_file);
    fprintf(stderr, "\n%s: Exiting.\n\n", progname);
    exit(1);
  }
}

static void initialize_keymap(void) {
  find_key(display, XK_Shift_L, &shift_keycode, 0);
}

static void expose_event_handler(XEvent *xev) {
  if (font)
    XDrawText(display, window, gc, 1, 1 + font->ascent, &success_text, 1);
  XDrawLines(display, window, gc, lines.pts, lines.num_pts, CoordModeOrigin);
}

static void client_message_handler(XEvent *xev) {
  XClientMessageEvent *cmev = &xev->xclient;

  if (cmev->data.l[0] == XInternAtom(display, "WM_DELETE_WINDOW", False)) {
    cleanup_and_exit(0);
  }
}

static void button_press_handler(XEvent *xev) {
  /* Simply throw away any incomplete stroke sequence. (This is for
   * the case when the user left the window with the button pressed
   * and didn't reenter before the button release) */
  stroke_trans(sequence);

  XClearWindow(display, window);
  lines.num_pts = 0;
  success_text.nchars = 0;
}

static int find_key(Display *display, KeySym keysym, KeyCode *code_ret, int *col_ret)
{
  int col;
  int keycode;
  KeySym k;
  int min_keycode, max_keycode;

  XDisplayKeycodes (display, &min_keycode, &max_keycode);

  for (keycode = min_keycode; keycode <= max_keycode; keycode++) {
    for (col = 0; (k = XKeycodeToKeysym (display, keycode, col)) != NoSymbol; col++)
      if (k == keysym) {
	*code_ret = keycode;
	if (col_ret)
	  *col_ret = col;
	return 1;
      }
  }
  return 0;
}

static void send_key_press_release(char *string) {
  KeySym keysym;
  KeyCode keycode;
  int col;

  keysym = XStringToKeysym(string);
  if (keysym != NoSymbol) {
    if (find_key(display, keysym, &keycode, &col)) {
      if (col & 1)
	XTestFakeKeyEvent (display, shift_keycode, True, 0);
      XTestFakeKeyEvent (display, keycode, True, 0);
      XTestFakeKeyEvent (display, keycode, False, 0);
      if (col & 1)
	XTestFakeKeyEvent (display, shift_keycode, False, 0);
    } else {
      fprintf(stderr, "No key code found for keysym %ld, (from string %s)\n", keysym, string);
    }
  } else {
    fprintf(stderr, "No Keysym found for %s\n", string);
  }
}

static void button_release_handler(XEvent *xev) {
  int len;
  /* We'll have to deal with these in one way or another: Button1Mask,
     Button2Mask, Button3Mask, Button4Mask, Button5Mask, ShiftMask,
     LockMask, ControlMask, Mod1Mask, Mod2Mask, Mod3Mask, Mod4Mask,
     Mod5Mask */
  if (stroke_trans(sequence)) {
    char *string = lookup_sequence(rec_tree_root, sequence);
    if (string) {
      send_key_press_release(string);
      len = snprintf(success_text.chars, MAX_SUCCESS, "%s (%s)", string, sequence);
      success_text.nchars = (len > MAX_SUCCESS) ? MAX_SUCCESS : len;
      if (font)
	XDrawText(display, window, gc, 1, 1 + font->ascent, &success_text, 1);
      /*
      fprintf(stderr, "%s => %s\n",sequence, string);
      */
    } else {
      len = snprintf(success_text.chars, MAX_SUCCESS, "%s (%s)", "unknown", sequence);
      success_text.nchars = (len > MAX_SUCCESS) ? MAX_SUCCESS : len;
      if (font)
	XDrawText(display, window, gc, 1, 1 + font->ascent, &success_text, 1);
      fprintf(stdout, "unknown=%s*\n",sequence);
    }
  } else {
    /*
    fprintf(stderr, "No stroke (%s)\n", sequence);
    */
  }
}

/* Throw out several points to make some more room.  Originally, this
 * function threw out every modth point, but when it was called
 * repeatedly, this stripped a lot more points from the fron of the
 * stroke than the end. I've modified it to remember where it last
 * left off compressing to start from there again and restart at the
 * beginning when the starting point gets close to the end again. This
 * helps, but it's still not perfect for some reason. Oh well...  */
static void decimate_stroke_lines(struct stroke_lines *lines, int mod) {
  int dest, src;

  if ((MAX_STROKE_POINTS - lines->last_decimated) < (2 * mod))
    lines->last_decimated = 0;

  for (dest = lines->last_decimated, src=dest; src < lines->num_pts; dest++, src++) {
    if ((dest % (mod - 1)) == 0) {
      src++;
      if (src >= lines->num_pts)
	break;
    }
    lines->pts[dest].x = lines->pts[src].x;
    lines->pts[dest].y = lines->pts[src].y;
  }
  lines->num_pts = dest;
  lines->last_decimated = dest - 1;
}

static void motion_notify_handler(XEvent *xev) {
  XMotionEvent *mev = &xev->xmotion;

  stroke_record (mev->x, mev->y);

  if (lines.num_pts >= MAX_STROKE_POINTS) {
    decimate_stroke_lines(&lines, STROKE_DECIMATION);
  }
  lines.pts[lines.num_pts].x = mev->x;
  lines.pts[lines.num_pts].y = mev->y;
  if (lines.num_pts > 0)
    XDrawLine(display, window, gc,
	      lines.pts[lines.num_pts-1].x, lines.pts[lines.num_pts-1].y,
	      lines.pts[lines.num_pts].x, lines.pts[lines.num_pts].y);

  lines.num_pts++;
}

static void key_press_handler(XEvent *xev) {
  XKeyEvent *kev = &xev->xkey;

  if (XLookupKeysym(kev, 0) == XK_q) {
    cleanup_and_exit(0);
  }
}

static void cleanup_and_exit(int exit_code) {
  if (display) {
    if (font)
      XFreeFont(display, font);
    if (gc)
      XFreeGC(display, gc);
    if (window)
      XDestroyWindow(display, window);
    XCloseDisplay(display);
  }
  free_rec_tree(rec_tree_root);
  /*
  print_node_stats();
  */
  exit(exit_code);
}

