/* 
   xstroke - X based gesture recognition program based on libstroke.

   Copyright (C) 2000 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>

#include "stroke_rec.h"
#include "stroke.h"
#include "rec_node.h"
#include "rec_file.h"
#include "xmalloc.h"
#include "rec-interface.h"

char *progname;

#define DEFAULT_STROKE_FILE ".xstrokerc"
#define DEFAULT_STROKE_DIR  "."

RecInterface rec_alloc() {
  RecInterface rec = xmalloc(sizeof(RecInterfacePrivate));

  rec->rec_tree_root = NULL;
  return rec;
}

/* This functions purpose is to take the location of a configuration
 * File, and with it load any configurations needed for character
 * recognition. 
 * file_name -- location of primary config file
 * returns 0 on success, non 0 on failure;
 */
int rec_load_configuration(RecInterface rec, char *file_name, 
			   KeySym* table, KeyCode min_kc, KeyCode max_kc,
			   int width) {
  char *def_file_name;

  stroke_init();
  rec->kc2ks_table = table;
  rec->kc2ks_start = min_kc;
  rec->kc2ks_end   = max_kc;
  rec->kc2ks_width = width;
  /* The keymap stuff must be initialized before create_rec_tree is called */
  rec->rec_tree_root = create_rec_tree_from_file(rec, file_name);

  /* Try again with DEFAULT_STROKE_DIR */
  def_file_name = xmalloc(strlen(DEFAULT_STROKE_DIR) + 1
			  + strlen(DEFAULT_STROKE_FILE) + 1);
  sprintf(def_file_name, "%s/%s", DEFAULT_STROKE_DIR, DEFAULT_STROKE_FILE);

  if (rec->rec_tree_root == NULL && strcmp(file_name, def_file_name)) {
    rec->rec_tree_root = create_rec_tree_from_file(rec, def_file_name);
  }

  if (rec->rec_tree_root == NULL) {
    fprintf(stderr, "Failed to load stroke configuration.\n");
    fprintf(stderr, "Tried %s", file_name);
    if (strcmp(file_name, def_file_name))
      fprintf(stderr, " and %s", def_file_name);
    return 1;
  }

  free(def_file_name);

  return 0;
}

action_list *rec_recognize_gesture(RecInterface rec,
				   XPoint* points, int num_points) {
  int i;

  stroke_trans(rec->sequence);
  for (i=0; i < num_points; i++) {
    stroke_record(points[i].x, points[i].y);
  }

  if (stroke_trans(rec->sequence)) {
    return lookup_sequence(rec->rec_tree_root, rec->sequence);
  }

  return NULL;
}

void rec_free(RecInterface rec) {
  free_rec_tree(rec->rec_tree_root);
  free(rec);
}



