/* rec_file.c -- functions to deal with loading a recognition file

   Copyright (C) 2000 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <X11/keysym.h>

#include "rec_node.h"
#include "rec_file.h"
#include "xmalloc.h"
#include "rec-interface.h"

#define MAX_LINE_LENGTH 512

static KeyCode shift_keycode;

/* Prototypes for private functions */
static char *strip_whitespace(char *str);
static int find_modifier_keycode(Display *display, int modmap_index, KeyCode *code_ret);
static int find_key(Display *display, KeySym keysym, KeyCode *code_ret, int *col_ret);

static int find_modifier_keycode(Display *display, int modmap_index, KeyCode *code_ret) {
  XModifierKeymap *map;
  int key;
  KeyCode keycode;

  map = XGetModifierMapping(display);
  for (key=0; key < map->max_keypermod; key++) {
      keycode = map->modifiermap[modmap_index * map->max_keypermod + key]; 
      if (keycode != 0) {
	*code_ret = keycode;
	XFreeModifiermap(map);
	return 0;
      }
  }
  XFreeModifiermap(map);
  return 1;
}


static int find_key(Display *display, KeySym keysym, KeyCode *code_ret, int *col_ret)
{
  int col;
  int keycode;
  KeySym k;
  int min_kc, max_kc;

  XDisplayKeycodes(display, &min_kc, &max_kc);

  for (keycode = min_kc; keycode <= max_kc; keycode++) {
    for (col = 0; (k = XKeycodeToKeysym (display, keycode, col)) != NoSymbol; col++)
      if (k == keysym) {
	*code_ret = keycode;
	if (col_ret)
	  *col_ret = col;
	return 1;
      }
  }
  return 0;
}
  /*
  int i, j;
  int min_kc, max_kc;
  int ks_per_kc;
  KeySym *current_row, *current_elem;
  int size = rec->kc2ks_end - rec->kc2ks_start;

  XDisplayKeycodes(display, &min_kc, &max_kc);
  key_map = XGetKeyboardMapping(display, min_kc,
				(max_kc - min_kc) + 1,
				&ks_per_kc);


  current_row = key_map;
  for (i=0; i <= size; i++) {
    current_elem = current_row;
    current_row += rec->kc2ks_width;
    for(j=0; j < rec->kc2ks_width; j++) {
      if(*current_elem == keysym) 
	{
	  if (col_ret) 
	    *col_ret = j;
	  if (code_ret)
	    *code_ret = i + rec->kc2ks_start;
	  return 1;
	}
      if(*current_elem == NoSymbol) 
	break;
      current_elem++;
    }
  }
  return 0;
}
*/


/* Return a pointer the the given string with initial and trailing
 * whitespace removed. Warning: This function modifies the original
 * string. */
static char *strip_whitespace(char *str) {
  if (str && *str) {
    int n;
    while (*str && isspace (*str)) str++;

    n = strlen (str) - 1;
    while (n > 0 && isspace (str [n])) n--;
    str [n+1] = '\0';
  }
  return str;
}

struct rec_node *create_rec_tree_from_file(Display *display, RecInterface rec, char *filename) {
  FILE *file;
  int line_num = 0;
  int err;
  char buf[MAX_LINE_LENGTH];
  char *key, *seq;
  struct rec_node *root;
  action_list *alist;
  KeySym keysym;
  KeyCode keycode;
  int col;

  err = find_modifier_keycode(display, ShiftMapIndex, &shift_keycode);
  if (err) {
    fprintf(stderr, "ERROR: No keycode is mapped to the Shift modifier.\n");
    return NULL;
  }

  file = fopen(filename, "r");
  if (!file) {
    /*
    fprintf(stderr, "%s:%d: create_rec_tree_from_file: failed to open file %s\n", __FILE__, __LINE__, filename);
    perror("create_rec_tree_from_file");
    */
    return NULL;
  }

  root = new_rec_tree();
  while (fgets(buf, MAX_LINE_LENGTH, file)) {
    char *p;
    line_num++;

    p = buf;
    while (isspace(p[0]))
      p++;
    if (p[0] == '\0' || p[0] == '#')
      goto NEXT_LINE;

    key = strtok(buf, "=");
    seq = strtok(NULL, "\n");

    key = strip_whitespace(key);
    seq = strip_whitespace(seq);

    if (strspn(seq,"123456789?[]*") != strlen(seq)) {
      fprintf(stderr,"%s:%d: Unrecognized character %c in digit sequence %s\n", filename, line_num, seq[strspn(seq,"123456789?")], seq);
      goto NEXT_LINE;
    }

    keysym = XStringToKeysym(key);
    if (keysym != NoSymbol) {
      if (find_key(display, keysym, &keycode, &col)) {
	int shifted = (col & 1);
	int i;
	keycode_action_data *data;

	alist = xmalloc(sizeof(action_list));
	alist->label = xmalloc(strlen(key)+1);
	strcpy(alist->label, key);
	
	i = 0;
	if (shifted) {
	  alist->num_actions = 4;
	  alist->actions = xmalloc(4 * sizeof(action));
	  alist->actions[i].type = KEYCODE_ACTION;
	  alist->actions[i].data = xmalloc(sizeof(keycode_action_data));
	  data = (keycode_action_data *) alist->actions[i].data;
	  data->keycode = shift_keycode;
	  data->press = 1;
	  i++;
	} else {
	  alist->num_actions = 2;
	  alist->actions = xmalloc(2 * sizeof(action));
	}
	
	alist->actions[i].type = KEYCODE_ACTION;
	alist->actions[i].data = xmalloc(sizeof(keycode_action_data));
	data = (keycode_action_data *) alist->actions[i].data;
	data->keycode = keycode;
	data->press = 1;
	i++;
	alist->actions[i].type = KEYCODE_ACTION;
	alist->actions[i].data = xmalloc(sizeof(keycode_action_data));
	data = (keycode_action_data *) alist->actions[i].data;
	data->keycode = keycode;
	data->press = 0;
	i++;
	
	if (shifted) {
	  alist->actions[i].type = KEYCODE_ACTION;
	  alist->actions[i].data = xmalloc(sizeof(keycode_action_data));
	  data = (keycode_action_data *) alist->actions[i].data;
	  data->keycode = shift_keycode;
	  data->press = 0;
	}
      
	err = add_sequence(root, seq, alist);
	if (err) {
	  fprintf(stderr, "%s:%d: Error adding sequence %s to recognition tree\n", filename, line_num, seq);
	}
      } else {
	fprintf(stderr, "%s:%d: Error adding sequence %s to recognition tree:\n\tNo key code found for keysym %ld, (from string %s)\n", filename, line_num, seq, keysym, key);
	goto NEXT_LINE;
      }
    } else {
      fprintf(stderr, "%s:%d: Error adding sequence %s to recognition tree:\n\tNo Keysym found for %s\n", filename, line_num, seq, key);
      goto NEXT_LINE;
    }
    
  NEXT_LINE:
  }

  return root;
}

