/* board-htcleo-microp.c
 * Copyright (C) 2009 Google.
 * Copyright (C) 2009 HTC Corporation.
 *
 * The Microp on htcleo is an i2c device that supports
 * the following functions
 *   - Lightsensor
 *   - Headset remotekeys
 *   - G-sensor
 *   - Proximity (capella cm3602)
 *   - Interrupts
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
*/

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/init.h>
#include <linux/leds.h>
#include <linux/workqueue.h>
#include <linux/i2c.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/miscdevice.h>
#include <linux/input.h>
#include <asm/uaccess.h>
#include <linux/wakelock.h>
#include <asm/mach-types.h>
#include <mach/htc_pwrsink.h>
#include <linux/earlysuspend.h>
#include <linux/bma150.h>
#include <linux/lightsensor.h>
#include <linux/capella_cm3602.h>
#include <asm/mach/mmc.h>
#include <mach/htc_35mm_jack.h>
#include <asm/setup.h>
#include <linux/debugfs.h>
#include <linux/seq_file.h>
#include <linux/mutex.h>
#include <linux/jiffies.h>

#include "board-htcleo.h"


#define MICROP_I2C_NAME "htcleo-microp"

#define MICROP_LSENSOR_ADC_CHAN		6
#define MICROP_REMOTE_KEY_ADC_CHAN	7

#define MICROP_I2C_WCMD_MISC				0x20
#define MICROP_I2C_WCMD_SPI_EN				0x21
#define MICROP_I2C_WCMD_BKL_CTRL			0x22
#define MICROP_I2C_WCMD_AUTO_BL_CTL			0x23
#define MICROP_I2C_RCMD_SPI_BL_STATUS			0x24
#define MICROP_I2C_RCMD_VERSION				0x30
#define MICROP_I2C_WCMD_ADC_TABLE			0x42
#define MICROP_I2C_WCMD_LED_MODE			0x51
#define MICROP_I2C_WCMD_READ_ADC_VALUE_REQ		0x60
#define MICROP_I2C_RCMD_ADC_VALUE			0x62
#define MICROP_I2C_WCMD_REMOTEKEY_TABLE			0x63
#define MICROP_I2C_WCMD_ADC_REQ				0x64
#define MICROP_I2C_WCMD_LCM_REGISTER			0x70
#define MICROP_I2C_WCMD_GSENSOR_REG			0x73
#define MICROP_I2C_WCMD_GSENSOR_REG_DATA_REQ		0x74
#define MICROP_I2C_RCMD_GSENSOR_REG_DATA		0x75
#define MICROP_I2C_WCMD_GSENSOR_DATA_REQ		0x76
#define MICROP_I2C_RCMD_GSENSOR_X_DATA			0x77
#define MICROP_I2C_RCMD_GSENSOR_Y_DATA			0x78
#define MICROP_I2C_RCMD_GSENSOR_Z_DATA			0x79
#define MICROP_I2C_RCMD_GSENSOR_DATA			0x7A
#define MICROP_I2C_WCMD_GPI_INT_CTL_EN			0x80
#define MICROP_I2C_WCMD_GPI_INT_CTL_DIS			0x81
#define MICROP_I2C_RCMD_GPI_INT_STATUS			0x82
#define MICROP_I2C_RCMD_GPI_STATUS			0x83
#define MICROP_I2C_WCMD_GPI_INT_STATUS_CLR		0x84
#define MICROP_I2C_RCMD_GPI_INT_SETTING			0x85
#define MICROP_I2C_RCMD_REMOTE_KEYCODE			0x87
#define MICROP_I2C_WCMD_REMOTE_KEY_DEBN_TIME		0x88
#define MICROP_I2C_WCMD_REMOTE_PLUG_DEBN_TIME		0x89
#define MICROP_I2C_WCMD_SIMCARD_DEBN_TIME		0x8A
#define MICROP_I2C_WCMD_ENABLE_GPO			0x90
#define MICROP_I2C_WCMD_DISABLE_GPO			0x91
#define MICROP_I2C_RCMD_GPO_STATUS			0x92

#define IRQ_GSENSOR	    (1<<10)
#define IRQ_LSENSOR  	(1<<9)
#define IRQ_REMOTEKEY	(1<<7)
#define IRQ_HEADSETIN	(1<<2)
#define IRQ_PROXIMITY   (1<<1)
#define IRQ_SDCARD	    (1<<0)

#define READ_GPI_STATE_HPIN	(1<<2)
#define READ_GPI_STATE_SDCARD	(1<<0)

#define GPO_PROXIMITY   0x3

#define ALS_CALIBRATE_MODE  147

/* Check pattern, to check if ALS has been calibrated */
#define ALS_CALIBRATED	0x6DA5

/* delay for deferred light sensor read */
#define LS_READ_DELAY   (HZ/2)


static uint16_t lsensor_adc_table[10] = {
	0x000, 0x001, 0x00F, 0x01E, 0x03C, 0x121, 0x190, 0x2BA, 0x26E, 0x3FF
};

static uint16_t remote_key_adc_table[6] = {
	0, 33, 43, 110, 129, 220
};

static uint32_t golden_adc = 0xC0;
static uint32_t als_kadc;

static struct wake_lock microp_i2c_wakelock;

static struct i2c_client *private_microp_client;

struct microp_int_pin {
	uint16_t int_gsensor;
	uint16_t int_lsensor;
	uint16_t int_reset;
	uint16_t int_simcard;
	uint16_t int_hpin;
	uint16_t int_remotekey;
};

struct microp_led_data {
	int type;
	struct led_classdev ldev;
	struct mutex led_data_mutex;
	struct work_struct brightness_work;
	spinlock_t brightness_lock;
	enum led_brightness brightness;
	uint8_t mode;
	uint8_t blink;
};

struct microp_i2c_work {
	struct work_struct work;
	struct i2c_client *client;
	int (*intr_debounce)(uint8_t *pin_status);
	void (*intr_function)(uint8_t *pin_status);
};

enum led_type {
	GREEN_LED,
	AMBER_LED,
	NUM_LEDS,
};

struct microp_i2c_client_data {
	struct microp_led_data leds[NUM_LEDS];
	uint16_t version;
	struct microp_i2c_work work;
	struct delayed_work hpin_debounce_work;
	struct delayed_work ls_read_work;
	struct early_suspend early_suspend;
	uint8_t enable_early_suspend;
	uint8_t enable_reset_button;
	int microp_is_suspend;
	int auto_backlight_enabled;
	uint8_t light_sensor_enabled;
    uint8_t proximity_sensor_enabled;
	uint8_t force_light_sensor_read;
	uint8_t button_led_value;
	int headset_is_in;
	int is_hpin_pin_stable;
    struct input_dev *pr_input_dev;
    struct input_dev *ls_input_dev;
	uint32_t als_kadc;
	uint32_t als_gadc;
	uint8_t als_calibrating;
};


static char *hex2string(uint8_t *data, int len)
{
	static char buf[101];
	int i;

	i = (sizeof(buf) - 1) / 4;
	if (len > i)
		len = i;

	for (i = 0; i < len; i++)
		sprintf(buf + i * 4, "[%02X]", data[i]);

	return buf;
}

#define I2C_READ_RETRY_TIMES  10/*
 * SD slot card-detect support
 */
#define I2C_WRITE_RETRY_TIMES 10

static int i2c_read_block(struct i2c_client *client, uint8_t addr,
	uint8_t *data, int length)
{
	int retry;
	int ret;
	struct i2c_msg msgs[] = {
	{
		.addr = client->addr,
		.flags = 0,
		.len = 1,
		.buf = &addr,
	},
	{
		.addr = client->addr,
		.flags = I2C_M_RD,
		.len = length,
		.buf = data,
	}
	};

	for (retry = 0; retry <= I2C_READ_RETRY_TIMES; retry++)
	{
		ret = i2c_transfer(client->adapter, msgs, 2);
		if (ret == 2)
		{
			dev_dbg(&client->dev, "R [%02X] = %s\n", addr, hex2string(data, length));
			msleep(3);
            return 0;
		}
		msleep(10);
	}

	dev_err(&client->dev, "i2c_read_block retry over %d\n", I2C_READ_RETRY_TIMES);
	return -EIO;
}

#define MICROP_I2C_WRITE_BLOCK_SIZE 21
static int i2c_write_block(struct i2c_client *client, uint8_t addr,
	uint8_t *data, int length)
{
	int retry;
	uint8_t buf[MICROP_I2C_WRITE_BLOCK_SIZE];
	int ret;

	struct i2c_msg msg[] = {
		{
			.addr = client->addr,
			.flags = 0,
			.len = length + 1,
			.buf = buf,
		}
	};

    dev_dbg(&client->dev, "W [%02X] = %s\n", addr,
         hex2string(data, length));

	if (length + 1 > MICROP_I2C_WRITE_BLOCK_SIZE) {
		dev_err(&client->dev, "i2c_write_block length too long\n");
		return -E2BIG;
	}

	buf[0] = addr;
	memcpy((void *)&buf[1], (void *)data, length);

//	mdelay(1);
// Cotulla: extra delay
//	msleep(10);
	for (retry = 0; retry <= I2C_WRITE_RETRY_TIMES; retry++)
	{
		ret = i2c_transfer(client->adapter, msg, 1);
		if (ret == 1)
		{
			msleep(3);
			return 0;
		}
		msleep(10);
	}
	dev_err(&client->dev, "i2c_write_block retry over %d\n",
			I2C_WRITE_RETRY_TIMES);
	return -EIO;
}

int microp_disable_lights()
{
	struct i2c_client *client;
	int ret;
	uint8_t data[4];

	client = private_microp_client;

	data[0] = 0;
	data[1] = 0;
	data[2] = 0;
	data[3] = 0;
	ret = i2c_write_block(client, 0x51, data, 1);
	if (ret != 0)
	{
		pr_err("%s: set failed\n", __func__);
	}
	return 0;
}

int micorp_onoff_bkl(int enable)
{
	struct i2c_client *client;
	int ret;
	uint8_t data[1];

	client = private_microp_client;

	data[0] = enable ? 1 : 0;
	ret = i2c_write_block(client, 0x26, data, 1);
	if (ret != 0)
		pr_err("%s: set failed\n", __func__);
	return 0;
}

int microp_set_bkl(uint8_t value)
{
	struct i2c_client *client;
	int ret;
	uint8_t cmd[2], data[2];

	printk("microp_set_bkl(%d)\n", value);

	if (value > 9)
	{
		value = 9;
	}
	client = private_microp_client;

	// disable autobrigtness
	data[0] = 0;
	data[1] = 0;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_AUTO_BL_CTL, data, 2); // 23
	if (ret != 0)
		pr_err("%s: set auto light sensor fail\n", __func__);

	// setvalue
	cmd[0] = value << 4;
//	printk("22LEVEL %02X\n", cmd[0]);
	ret = i2c_write_block(client, MICROP_I2C_WCMD_BKL_CTRL, cmd, 1); // 22
	if (ret < 0)
	{
		dev_err(&client->dev, "%s: request adc fail\n", __func__);
		return -EIO;
	}

	return 0;
}


// LEO proc
int microp_set_adc_req(uint8_t value, int enable)
{
	struct i2c_client *client;
	int ret;
	uint8_t cmd[2], data[2];

	client = private_microp_client;	
	cmd[0] = 1; //value; TODO finish code... now only keys ADC
	ret = i2c_write_block(client, MICROP_I2C_WCMD_ADC_REQ, cmd, 1);
	if (ret < 0) 
	{
		dev_err(&client->dev, "%s: request adc fail\n", __func__);
		return -EIO;
	}

	return 0;
}

// LEO proc
int microp_get_remote_adc(uint32_t *val)
{
	struct i2c_client *client;
	int ret;
	uint8_t data[4];

	if (!val)
		return -EIO; 

	client = private_microp_client;	
	ret = i2c_read_block(client, MICROP_I2C_RCMD_ADC_VALUE, data, 2);
	if (ret < 0) 
	{
		dev_err(&client->dev, "%s: request adc fail\n", __func__);
		return -EIO;
	}

//	printk("%x %x\n", data[0], data[1]);
	*val = data[1] | (data[0] << 8);
	printk("remote adc %d\n", *val);
	return 0;
}


static ssize_t microp_i2c_remotekey_adc_show(struct device *dev,
				  struct device_attribute *attr, char *buf)
{
	uint32_t val;
	int ret;

	microp_get_remote_adc(&val);

	ret = sprintf(buf, "Remote Key[%d]\n", val);

	return ret;
}

static DEVICE_ATTR(key_adc, 0644, microp_i2c_remotekey_adc_show, NULL);

static int microp_read_adc(uint8_t channel, uint16_t *value)
{
	struct i2c_client *client;
	int ret;
	uint8_t cmd[2], data[2];

	client = private_microp_client;
	cmd[0] = 0;
	cmd[1] = 1; //channel;
//	ret = i2c_write_block(client, MICROP_I2C_WCMD_READ_ADC_REQ, cmd, 2);
	ret = i2c_write_block(client, MICROP_I2C_WCMD_READ_ADC_VALUE_REQ, cmd, 2);
	if (ret < 0) {
		dev_err(&client->dev, "%s: request adc fail\n", __func__);
		return -EIO;
	}

	ret = i2c_read_block(client, MICROP_I2C_RCMD_ADC_VALUE, data, 2);
	if (ret < 0) {
		dev_err(&client->dev, "%s: read adc fail\n", __func__);
		return -EIO;
	}
	*value = data[0] << 8 | data[1];
	return 0;
}

/**
 * GPI functions
 **/

static int microp_read_gpi_status(struct i2c_client *client, uint16_t *status)
{
	uint8_t data[2];
	int ret;

	ret = i2c_read_block(client, MICROP_I2C_RCMD_GPI_STATUS, data, 2);
	if (ret < 0) {
		dev_err(&client->dev, "%s: read failed\n", __func__);
		return -EIO;
	}
	*status = (data[0] << 8) | data[1];
	return 0;
}

static int microp_interrupt_enable(struct i2c_client *client, uint16_t interrupt_mask)
{
	uint8_t data[2];
	int ret = -1;

	data[0] = interrupt_mask >> 8;
	data[1] = interrupt_mask & 0xFF;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_GPI_INT_CTL_EN, data, 2);

	if (ret < 0)
		dev_err(&client->dev, "%s: enable 0x%x interrupt failed\n",
			__func__, interrupt_mask);
	return ret;
}

static int microp_interrupt_disable(struct i2c_client *client, uint16_t interrupt_mask)
{
	uint8_t data[2];
	int ret = -1;

	data[0] = interrupt_mask >> 8;
	data[1] = interrupt_mask & 0xFF;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_GPI_INT_CTL_DIS, data, 2);

	if (ret < 0)
		dev_err(&client->dev, "%s: disable 0x%x interrupt failed\n",
			__func__, interrupt_mask);
	return ret;
}

/**
 * GPO functions TODO
 **/

static int microp_read_gpo_status(struct i2c_client *client, uint16_t *status)
{
	uint8_t data[2];
	int ret;

	ret = i2c_read_block(client, MICROP_I2C_RCMD_GPO_STATUS, data, 2);
	if (ret < 0) 
	{
		dev_err(&client->dev, "%s: read failed\n", __func__);
		return -EIO;
	}
	*status = (data[0] << 8) | data[1];
	return 0;
}

static int microp_gpo_enable(struct i2c_client *client, uint16_t interrupt_mask)
{
	uint8_t data[2];
	int ret = -1;

	data[0] = interrupt_mask >> 8;
	data[1] = interrupt_mask & 0xFF;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_ENABLE_GPO, data, 2);

	if (ret < 0)
		dev_err(&client->dev, "%s: enable 0x%x interrupt failed\n", __func__, interrupt_mask);
	return ret;
}

static int microp_gpo_disable(struct i2c_client *client, uint16_t interrupt_mask)
{
	uint8_t data[2];
	int ret = -1;

	data[0] = interrupt_mask >> 8;
	data[1] = interrupt_mask & 0xFF;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_DISABLE_GPO, data, 2);

	if (ret < 0)
		dev_err(&client->dev, "%s: disable 0x%x interrupt failed\n", __func__, interrupt_mask);
	return ret;
}


/*
 * SD slot card-detect support
 */
static unsigned int sdslot_cd = 0;
static void (*sdslot_status_cb)(int card_present, void *dev_id);
static void *sdslot_mmc_dev;

int htcleo_microp_sdslot_status_register(
		void (*cb)(int card_present, void *dev_id),
		void *dev_id)
{
	if (sdslot_status_cb)
		return -EBUSY;
	sdslot_status_cb = cb;
	sdslot_mmc_dev = dev_id;
	return 0;
}

unsigned int htcleo_microp_sdslot_status(struct device *dev)
{
	return sdslot_cd;
}

static void htcleo_microp_sdslot_update_status(int status)
{
	sdslot_cd = !(status & READ_GPI_STATE_SDCARD);
	if (sdslot_status_cb)
		sdslot_status_cb(sdslot_cd, sdslot_mmc_dev);
}



/*
 * Proximity
 */
DEFINE_MUTEX(proximity_api_lock);
static int proximity_opened;
struct wake_lock proximity_wake_lock;

static int report_psensor_data(void)
{
	pr_info("%s\n", __func__);

	int ret, ps_data = 0;
	uint8_t data[2] = {0, 0};
    struct i2c_client *client;
	struct microp_i2c_client_data *cdata;

    client = private_microp_client;
    cdata = i2c_get_clientdata(client);

	ret = i2c_read_block(client, MICROP_I2C_RCMD_GPI_STATUS, data, 2);
	if (ret < 0)
		pr_err("%s: read data failed\n", __func__);
	else {
		ps_data = (data[1] & 0x1) ? 1 : 0;
		pr_info("proximity %s\n", ps_data ? "FAR" : "NEAR");

		/* 0 is close, 1 is far */
		input_report_abs(cdata->pr_input_dev, ABS_DISTANCE, ps_data);
		input_sync(cdata->pr_input_dev);

		wake_lock_timeout(&proximity_wake_lock, 2*HZ);
	}

	return ret;
}

static int capella_cm3602_enable(void)
{
	pr_info("%s\n", __func__);

	int ret;
    struct i2c_client *client;
	struct microp_i2c_client_data *cdata;

    client = private_microp_client;
    cdata = i2c_get_clientdata(client);
    
	if (cdata->proximity_sensor_enabled) {
		pr_info("%s: already enabled\n", __func__);
		return 0;
	}

	/* dummy report */
	input_report_abs(cdata->pr_input_dev, ABS_DISTANCE, -1);
	input_sync(cdata->pr_input_dev);

    ret = microp_gpo_enable(client, GPO_PROXIMITY);
	if (ret < 0)
		return -EIO;

	cdata->proximity_sensor_enabled = 1;
	report_psensor_data();

	return ret;
}

static int capella_cm3602_disable(void)
{
	pr_info("%s\n", __func__);

	int ret;
    struct i2c_client *client;
	struct microp_i2c_client_data *cdata;

    client = private_microp_client;
    cdata = i2c_get_clientdata(client);

	if (!cdata->proximity_sensor_enabled) {
		pr_info("%s: already disabled\n", __func__);
		return 0;
	}

    ret = microp_gpo_disable(client, GPO_PROXIMITY);
	if (ret < 0)
		return -EIO;

	cdata->proximity_sensor_enabled = 0;
	return ret;
}

static int capella_cm3602_open(struct inode *inode, struct file *file)
{
	pr_info("%s\n", __func__);
	int ret = 0;
	mutex_lock(&proximity_api_lock);
	if (proximity_opened) {
		pr_err("%s: already opened\n", __func__);
		ret = -EBUSY;
	}
	proximity_opened = 1;
	mutex_unlock(&proximity_api_lock);
	return ret;
}

static int capella_cm3602_release(struct inode *inode, struct file *file)
{
	pr_info("%s\n", __func__);
	mutex_lock(&proximity_api_lock);
	proximity_opened = 0;
	mutex_unlock(&proximity_api_lock);
    return capella_cm3602_disable();
}

static long capella_cm3602_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	pr_info("%s\n", __func__);

	int val, ret;
    struct i2c_client *client;
	struct microp_i2c_client_data *cdata;

    client = private_microp_client;
    cdata = i2c_get_clientdata(client);

	mutex_lock(&proximity_api_lock);

	switch (cmd) {
    	case CAPELLA_CM3602_IOCTL_ENABLE:
    		if (get_user(val, (unsigned long __user *)arg)) {
    			ret = -EFAULT;
                break;
    		}
            if (val)
                ret = capella_cm3602_enable();
            else
                ret = capella_cm3602_disable();
    		break;
    	case CAPELLA_CM3602_IOCTL_GET_ENABLED:
            ret = put_user(cdata->proximity_sensor_enabled, (unsigned long __user *)arg);
    		break;
    	default:
    		pr_err("%s: invalid cmd %d\n", __func__, _IOC_NR(cmd));
    		ret = -EINVAL;
	}
	
	mutex_unlock(&proximity_api_lock);
    return ret;
}

static struct file_operations capella_cm3602_fops = {
	.owner = THIS_MODULE,
	.open = capella_cm3602_open,
	.release = capella_cm3602_release,
	.unlocked_ioctl = capella_cm3602_ioctl
};

struct miscdevice capella_cm3602_misc = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "cm3602",
	.fops = &capella_cm3602_fops
};

/*
* LED Support
*/

static int microp_i2c_write_led_mode(struct i2c_client *client,
				struct led_classdev *led_cdev,
				uint8_t mode)
{
/*	There are 5 different Led Modi;
*	0x0, 0x0: Disabled
*	0x0, 0x1: LED Green
*	0x0, 0x2: LED Amber 
*	0x0, 0x3: LED Green flashing slow ( ca. 6 sek ) 
*	0x0, 0x4: LED Green flashing fast ( ca. 2 sek )
*	0x0, 0x5: LED Amber flashing fast ( ca. 2 sek ) 
*	0x10,0xX: LED Amber and Green flashing alternately
*/
	struct microp_i2c_client_data *cdata;
	struct microp_led_data *ldata;
	uint8_t data[2] = { 0, 0 };
	int ret;
	static uint8_t oldvalgr=0, oldvalam=0, alternately=0;

	cdata = i2c_get_clientdata(client);
	ldata = container_of(led_cdev, struct microp_led_data, ldev);

	data[0] = 0x00;
	if (ldata->type == GREEN_LED) {
		switch(mode) {
		  case 0x0:
			if(alternately) {
				data[1]=oldvalam; 
				alternately=0;
			} else
				data[1] = 0x0;  // Disable Light
			break;
		  case 0x1:
			data[1] = 0x1;  // Enable Light
			break;
		  case 0x2:
			if(oldvalam==0x5) { // alternately blinking
				data[0] = 0x10;	
				alternately=1;
			} else
				alternately=0;
			data[1] = 0x3;  // Slow blinking
			break;
		  case 0x3:
			if(oldvalam==0x5) { // alternately blinking
				data[0] = 0x10;	
				alternately=1;
			} else
				alternately=0;
			data[1] = 0x4;  // Fast blinking
			break;
		}
		oldvalgr=data[1];
	} else if (ldata->type == AMBER_LED) {
		switch(mode) {
		  case 0x0:
			if(alternately) {
				data[1]=oldvalgr; 
				alternately=0;
			} else
				data[1] = 0x0;  // Disable Light
			break;
		  case 0x1:
			data[1] = 0x2;  // Enable Light
			break;
		  case 0x2:
		  case 0x3:
			if(oldvalgr==0x3 || oldvalgr==0x4) { // alternately blinking
				data[0] = 0x10;	
				alternately=1;
			} else
				alternately=0;
			data[1] = 0x5;  // Fast blinking
			break;
		}
		oldvalam=data[1];
	}

	ret = i2c_write_block(client, MICROP_I2C_WCMD_LED_MODE, data, 2);
	if (ret == 0) {
		mutex_lock(&ldata->led_data_mutex);
		if (mode > 1)
			ldata->blink = mode;
		else
			ldata->mode = mode;
		mutex_unlock(&ldata->led_data_mutex);
	}
	return ret;
}

static ssize_t microp_i2c_led_blink_show(struct device *dev,
				  struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev;
	struct microp_led_data *ldata;
	int ret;

	led_cdev = (struct led_classdev *)dev_get_drvdata(dev);
	ldata = container_of(led_cdev, struct microp_led_data, ldev);

	mutex_lock(&ldata->led_data_mutex);
	ret = sprintf(buf, "%d\n", ldata->blink ? ldata->blink - 1 : 0);
	mutex_unlock(&ldata->led_data_mutex);

	return ret;
}

static ssize_t microp_i2c_led_blink_store(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t count)
{
	struct led_classdev *led_cdev;
	struct microp_led_data *ldata;
	struct i2c_client *client;
	int val, ret;
	uint8_t mode;

	val = -1;
	sscanf(buf, "%u", &val);

	led_cdev = (struct led_classdev *)dev_get_drvdata(dev);
	ldata = container_of(led_cdev, struct microp_led_data, ldev);
	client = to_i2c_client(dev->parent);

	mutex_lock(&ldata->led_data_mutex);
	switch (val) {
	case 0: /* stop flashing */
		mode = ldata->mode;
		ldata->blink = 0;
		break;
	case 1:
	case 2:
		mode = val + 1;
		break;

	default:
		mutex_unlock(&ldata->led_data_mutex);
		return -EINVAL;
	}
	mutex_unlock(&ldata->led_data_mutex);

	ret = microp_i2c_write_led_mode(client, led_cdev, mode);
	if (ret)
		dev_err(&client->dev, "%s set blink failed\n", led_cdev->name);

	return count;
}

static DEVICE_ATTR(blink, 0644, microp_i2c_led_blink_show,
				microp_i2c_led_blink_store);

				
static void microp_brightness_set(struct led_classdev *led_cdev,
			       enum led_brightness brightness)
{
	unsigned long flags;
	struct i2c_client *client = to_i2c_client(led_cdev->dev->parent);
	struct microp_led_data *ldata =
		container_of(led_cdev, struct microp_led_data, ldev);

	dev_dbg(&client->dev, "Setting %s brightness current %d new %d\n",
			led_cdev->name, led_cdev->brightness, brightness);

	if (brightness > 255)
		brightness = 255;
	led_cdev->brightness = brightness;

	spin_lock_irqsave(&ldata->brightness_lock, flags);
	ldata->brightness = brightness;
	spin_unlock_irqrestore(&ldata->brightness_lock, flags);

	schedule_work(&ldata->brightness_work);
}

static void microp_led_brightness_set_work(struct work_struct *work)
{
	unsigned long flags;
	struct microp_led_data *ldata =
		container_of(work, struct microp_led_data, brightness_work);
	struct led_classdev *led_cdev = &ldata->ldev;

	struct i2c_client *client = to_i2c_client(led_cdev->dev->parent);

	enum led_brightness brightness;
	int ret;
	uint8_t mode;

	spin_lock_irqsave(&ldata->brightness_lock, flags);
	brightness = ldata->brightness;
	spin_unlock_irqrestore(&ldata->brightness_lock, flags);

	if (brightness)
		mode = 1;
	else
		mode = 0;

	ret = microp_i2c_write_led_mode(client, led_cdev, mode);
	if (ret) {
		dev_err(&client->dev,
			 "led_brightness_set failed to set mode\n");
	}
}

struct device_attribute *green_amber_attrs[] = {
	&dev_attr_blink,
};


/*
 * Light Sensor Support
 */
static int microp_i2c_auto_backlight_mode(struct i2c_client *client,
					    uint8_t enabled)
{
#if 0
	uint8_t data[2];
	int ret = 0;

	data[0] = 0;
	if (enabled)
		data[1] = 0; //1;
	else
		data[1] = 0;

	ret = i2c_write_block(client, MICROP_I2C_WCMD_AUTO_BL_CTL, data, 2);
	if (ret != 0)
		pr_err("%s: set auto light sensor fail\n", __func__);

	return ret;
#else
	return 0;
#endif
}

static int lightsensor_enable(void)
{
#if 0
	struct i2c_client *client;
	struct microp_i2c_client_data *cdata;
	int ret;

	client = private_microp_client;
	cdata = i2c_get_clientdata(client);

	if (cdata->microp_is_suspend) {
		pr_err("%s: abort, uP is going to suspend after #\n",
		       __func__);
		return -EIO;
	}

	disable_irq(client->irq);
	ret = microp_i2c_auto_backlight_mode(client, 1);
	if (ret < 0) {
		pr_err("%s: set auto light sensor fail\n", __func__);
		enable_irq(client->irq);
		return ret;
	}

	cdata->auto_backlight_enabled = 1;
	/* TEMPORARY HACK: schedule a deferred light sensor read
	 * to work around sensor manager race condition
	 */
	schedule_delayed_work(&cdata->ls_read_work, LS_READ_DELAY);
	schedule_work(&cdata->work.work);

	return 0;
#else
	return 0;
#endif
}

static int lightsensor_disable(void)
{
#if 0
	/* update trigger data when done */
	struct i2c_client *client;
	struct microp_i2c_client_data *cdata;
	int ret;

	client = private_microp_client;
	cdata = i2c_get_clientdata(client);

	if (cdata->microp_is_suspend) {
		pr_err("%s: abort, uP is going to suspend after #\n",
		       __func__);
		return -EIO;
	}

	cancel_delayed_work(&cdata->ls_read_work);

	ret = microp_i2c_auto_backlight_mode(client, 0);
	if (ret < 0)
		pr_err("%s: disable auto light sensor fail\n",
		       __func__);
	else
		cdata->auto_backlight_enabled = 0;
	return 0;
#else
	return 0;
#endif
}

static int microp_lightsensor_read(uint16_t *adc_value,
					  uint8_t *adc_level)
{
#if 0
	struct i2c_client *client;
	struct microp_i2c_client_data *cdata;
	uint8_t i;
	int ret;

	client = private_microp_client;
	cdata = i2c_get_clientdata(client);

	ret = microp_read_adc(MICROP_LSENSOR_ADC_CHAN, adc_value);
	if (ret != 0)
		return -1;

	if (*adc_value > 0x3FF) {
		pr_warning("%s: get wrong value: 0x%X\n",
			__func__, *adc_value);
		return -1;
	} else {
		if (!cdata->als_calibrating) {
			*adc_value = *adc_value
				* cdata->als_gadc / cdata->als_kadc;
			if (*adc_value > 0x3FF)
				*adc_value = 0x3FF;
		}

		*adc_level = ARRAY_SIZE(lsensor_adc_table) - 1;
		for (i = 0; i < ARRAY_SIZE(lsensor_adc_table); i++) {
			if (*adc_value <= lsensor_adc_table[i]) {
				*adc_level = i;
				break;
			}
		}
		pr_debug("%s: ADC value: 0x%X, level: %d #\n",
				__func__, *adc_value, *adc_level);
	}

	return 0;
#else
	return 0;
#endif
}

static ssize_t microp_i2c_lightsensor_adc_show(struct device *dev,
				  struct device_attribute *attr, char *buf)
{
#if 0
	uint8_t adc_level = 0;
	uint16_t adc_value = 0;
	int ret;

	ret = microp_lightsensor_read(&adc_value, &adc_level);

	ret = sprintf(buf, "ADC[0x%03X] => level %d\n", adc_value, adc_level);

	return ret;
#else
	return 0;
#endif
}

static DEVICE_ATTR(ls_adc, 0644, microp_i2c_lightsensor_adc_show, NULL);

static ssize_t microp_i2c_ls_auto_show(struct device *dev,
				  struct device_attribute *attr, char *buf)
{
#if 0
	struct i2c_client *client;
	uint8_t data[2] = {0, 0};
	int ret;

	client = to_i2c_client(dev);

	i2c_read_block(client, MICROP_I2C_RCMD_SPI_BL_STATUS, data, 2);
	ret = sprintf(buf, "Light sensor Auto = %d, SPI enable = %d\n",
			data[0], data[1]);

	return ret;
#else
	return 0;
#endif
}

static ssize_t microp_i2c_ls_auto_store(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t count)
{
#if 0
	struct i2c_client *client;
	struct microp_i2c_client_data *cdata;
	uint8_t enable = 0;
	int ls_auto;

	ls_auto = -1;
	sscanf(buf, "%d", &ls_auto);

	if (ls_auto != 0 && ls_auto != 1 && ls_auto != ALS_CALIBRATE_MODE)
		return -EINVAL;

	client = to_i2c_client(dev);
	cdata = i2c_get_clientdata(client);

	if (ls_auto) {
		enable = 1;
		cdata->als_calibrating = (ls_auto == ALS_CALIBRATE_MODE) ? 1 : 0;
		cdata->auto_backlight_enabled = 1;
	} else {
		enable = 0;
		cdata->als_calibrating = 0;
		cdata->auto_backlight_enabled = 0;
	}

	microp_i2c_auto_backlight_mode(client, enable);

	return count;
#else
	return 0;
#endif
}

static DEVICE_ATTR(ls_auto, 0644,  microp_i2c_ls_auto_show,
			microp_i2c_ls_auto_store);

DEFINE_MUTEX(api_lock);
static int lightsensor_opened;

static int lightsensor_open(struct inode *inode, struct file *file)
{
#if 0
	int rc = 0;
	pr_debug("%s\n", __func__);
	mutex_lock(&api_lock);
	if (lightsensor_opened) {
		pr_err("%s: already opened\n", __func__);
		rc = -EBUSY;
	}
	lightsensor_opened = 1;
	mutex_unlock(&api_lock);
	return rc;
#else
	return 0;
#endif
}

static int lightsensor_release(struct inode *inode, struct file *file)
{
#if 0
	pr_debug("%s\n", __func__);
	mutex_lock(&api_lock);
	lightsensor_opened = 0;
	mutex_unlock(&api_lock);
#endif
	return 0;
}

static long lightsensor_ioctl(struct file *file, unsigned int cmd,
		unsigned long arg)
{
#if 0
	int rc, val;
	struct i2c_client *client;
	struct microp_i2c_client_data *cdata;

	mutex_lock(&api_lock);

	client = private_microp_client;
	cdata = i2c_get_clientdata(client);

	pr_debug("%s cmd %d\n", __func__, _IOC_NR(cmd));

	switch (cmd) {
	case LIGHTSENSOR_IOCTL_ENABLE:
		if (get_user(val, (unsigned long __user *)arg)) {
			rc = -EFAULT;
			break;
		}
		rc = val ? lightsensor_enable() : lightsensor_disable();
		break;
	case LIGHTSENSOR_IOCTL_GET_ENABLED:
		val = cdata->auto_backlight_enabled;
		pr_debug("%s enabled %d\n", __func__, val);
		rc = put_user(val, (unsigned long __user *)arg);
		break;
	default:
		pr_err("%s: invalid cmd %d\n", __func__, _IOC_NR(cmd));
		rc = -EINVAL;
	}

	mutex_unlock(&api_lock);
	return rc;
#else
	return 0;
#endif
}

static struct file_operations lightsensor_fops = {
	.owner = THIS_MODULE,
	.open = lightsensor_open,
	.release = lightsensor_release,
	.unlocked_ioctl = lightsensor_ioctl
};

struct miscdevice lightsensor_misc = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "lightsensor",
	.fops = &lightsensor_fops
};

/*
 * G-sensor
 */
static int microp_spi_enable(uint8_t on)
{
	struct i2c_client *client;
	int ret;

	client = private_microp_client;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_SPI_EN, &on, 1);
	if (ret < 0) {
		dev_err(&client->dev,"%s: i2c_write_block fail\n", __func__);
		return ret;
	}
	msleep(10);
	return ret;
}

static int gsensor_read_reg(uint8_t reg, uint8_t *data)
{
	struct i2c_client *client;
	int ret;
	uint8_t tmp[2];

	client = private_microp_client;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_GSENSOR_REG_DATA_REQ,
			      &reg, 1);
	if (ret < 0) {
		dev_err(&client->dev,"%s: i2c_write_block fail\n", __func__);
		return ret;
	}
	msleep(10);

	ret = i2c_read_block(client, MICROP_I2C_RCMD_GSENSOR_REG_DATA, tmp, 2);
	if (ret < 0) {
		dev_err(&client->dev,"%s: i2c_read_block fail\n", __func__);
		return ret;
	}
	*data = tmp[1];
	return ret;
}

static int gsensor_write_reg(uint8_t reg, uint8_t data)
{
	struct i2c_client *client;
	int ret;
	uint8_t tmp[2];

	client = private_microp_client;

	tmp[0] = reg;
	tmp[1] = data;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_GSENSOR_REG, tmp, 2);
	if (ret < 0) {
		dev_err(&client->dev,"%s: i2c_write_block fail\n", __func__);
		return ret;
	}

	return ret;
}

static int gsensor_read_acceleration(short *buf)
{
	struct i2c_client *client;
	int ret;
	uint8_t tmp[6];
	struct microp_i2c_client_data *cdata;

	client = private_microp_client;

	cdata = i2c_get_clientdata(client);

	tmp[0] = 1;
	ret = i2c_write_block(client, MICROP_I2C_WCMD_GSENSOR_DATA_REQ,
			      tmp, 1);
	if (ret < 0) {
		dev_err(&client->dev,"%s: i2c_write_block fail\n", __func__);
		return ret;
	}

	msleep(10);

	if (cdata->version <= 0x615) {
		/*
		 * Note the data is a 10bit signed value from the chip.
		*/
		ret = i2c_read_block(client, MICROP_I2C_RCMD_GSENSOR_X_DATA,
				     tmp, 2);
		if (ret < 0) {
			dev_err(&client->dev, "%s: i2c_read_block fail\n",
				__func__);
			return ret;
		}
		buf[0] = (short)(tmp[0] << 8 | tmp[1]);
		buf[0] >>= 6;

		ret = i2c_read_block(client, MICROP_I2C_RCMD_GSENSOR_Y_DATA,
				     tmp, 2);
		if (ret < 0) {
			dev_err(&client->dev, "%s: i2c_read_block fail\n",
				__func__);
			return ret;
		}
		buf[1] = (short)(tmp[0] << 8 | tmp[1]);
		buf[1] >>= 6;

		ret = i2c_read_block(client, MICROP_I2C_RCMD_GSENSOR_Z_DATA,
				     tmp, 2);
		if (ret < 0) {
			dev_err(&client->dev, "%s: i2c_read_block fail\n",
				__func__);
			return ret;
		}
		buf[2] = (short)(tmp[0] << 8 | tmp[1]);
		buf[2] >>= 6;
	} else {
		ret = i2c_read_block(client, MICROP_I2C_RCMD_GSENSOR_DATA,
				     tmp, 6);
		if (ret < 0) {
			dev_err(&client->dev, "%s: i2c_read_block fail\n",
				__func__);
			return ret;
		}
		buf[0] = (short)(tmp[0] << 8 | tmp[1]);
		buf[0] >>= 6;
		buf[1] = (short)(tmp[2] << 8 | tmp[3]);
		buf[1] >>= 6;
		buf[2] = (short)(tmp[4] << 8 | tmp[5]);
		buf[2] >>= 6;
	}

	return 1;
}

static int gsensor_init_hw(void)
{
	uint8_t reg;
	int ret;

	pr_debug("%s\n", __func__);

	microp_spi_enable(1);

	ret = gsensor_read_reg(RANGE_BWIDTH_REG, &reg);
	if (ret < 0 )
		return -EIO;
	reg &= 0xe0;
	ret = gsensor_write_reg(RANGE_BWIDTH_REG, reg);
	if (ret < 0 )
		return -EIO;

	ret = gsensor_read_reg(SMB150_CONF2_REG, &reg);
	if (ret < 0 )
		return -EIO;
	reg |= (1 << 3);
	ret = gsensor_write_reg(SMB150_CONF2_REG, reg);
        
	return ret;
}

static int bma150_set_mode(char mode)
{
	uint8_t reg;
	int ret;

	pr_debug("%s mode = %d\n", __func__, mode);
	if (mode == BMA_MODE_NORMAL)
		microp_spi_enable(1);


	ret = gsensor_read_reg(SMB150_CTRL_REG, &reg);
	if (ret < 0 )
		return -EIO;
	reg = (reg & 0xfe) | mode;
	ret = gsensor_write_reg(SMB150_CTRL_REG, reg);

	if (mode == BMA_MODE_SLEEP)
		microp_spi_enable(0);

	return ret;
}
static int gsensor_read(uint8_t *data)
{
	int ret;
	uint8_t reg = data[0];

	ret = gsensor_read_reg(reg, &data[1]);
	pr_debug("%s reg = %x data = %x\n", __func__, reg, data[1]);
	return ret;
}

static int gsensor_write(uint8_t *data)
{
	int ret;
	uint8_t reg = data[0];

	pr_debug("%s reg = %x data = %x\n", __func__, reg, data[1]);
	ret = gsensor_write_reg(reg, data[1]);
	return ret;
}

static int bma150_open(struct inode *inode, struct file *file)
{
	pr_debug("%s\n", __func__);
	return nonseekable_open(inode, file);
}

static int bma150_release(struct inode *inode, struct file *file)
{
	return 0;
}

static int bma150_ioctl(struct inode *inode, struct file *file,
			unsigned int cmd, unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	char rwbuf[8];
	int ret = -1;
	short buf[8], temp;

	switch (cmd) {
	case BMA_IOCTL_READ:
	case BMA_IOCTL_WRITE:
	case BMA_IOCTL_SET_MODE:
		if (copy_from_user(&rwbuf, argp, sizeof(rwbuf)))
			return -EFAULT;
		break;
	case BMA_IOCTL_READ_ACCELERATION:
		if (copy_from_user(&buf, argp, sizeof(buf)))
			return -EFAULT;
		break;
	default:
		break;
	}

	switch (cmd) {
	case BMA_IOCTL_INIT:
		ret = gsensor_init_hw();
		if (ret < 0)
			return ret;
		break;

	case BMA_IOCTL_READ:
		if (rwbuf[0] < 1)
			return -EINVAL;
		ret = gsensor_read(rwbuf);
		if (ret < 0)
			return ret;
		break;
	case BMA_IOCTL_WRITE:
		if (rwbuf[0] < 2)
			return -EINVAL;
		ret = gsensor_write(rwbuf);
		if (ret < 0)
			return ret;
		break;
	case BMA_IOCTL_READ_ACCELERATION:
		ret = gsensor_read_acceleration(&buf[0]);
		if (ret < 0)
			return ret;
		break;
	case BMA_IOCTL_SET_MODE:
		bma150_set_mode(rwbuf[0]);
		break;
	case BMA_IOCTL_GET_INT:
		temp = 0;
		break;
	default:
		return -ENOTTY;
	}

	switch (cmd) {
	case BMA_IOCTL_READ:
		if (copy_to_user(argp, &rwbuf, sizeof(rwbuf)))
			return -EFAULT;
		break;
	case BMA_IOCTL_READ_ACCELERATION:
		if (copy_to_user(argp, &buf, sizeof(buf)))
			return -EFAULT;
		break;
	case BMA_IOCTL_GET_INT:
		if (copy_to_user(argp, &temp, sizeof(temp)))
			return -EFAULT;
		break;
	default:
		break;
	}

	return 0;
}

static struct file_operations bma_fops = {
	.owner = THIS_MODULE,
	.open = bma150_open,
	.release = bma150_release,
	.ioctl = bma150_ioctl,
};

static struct miscdevice spi_bma_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = BMA150_G_SENSOR_NAME,
	.fops = &bma_fops,
};

/*
 * Interrupt
 */
static irqreturn_t microp_i2c_intr_irq_handler(int irq, void *dev_id)
{
	struct i2c_client *client;
	struct microp_i2c_client_data *cdata;

	client = to_i2c_client(dev_id);
	cdata = i2c_get_clientdata(client);

	dev_dbg(&client->dev, "intr_irq_handler\n");

	disable_irq_nosync(client->irq);
	schedule_work(&cdata->work.work);
	return IRQ_HANDLED;
}

static void microp_i2c_intr_work_func(struct work_struct *work)
{
	struct microp_i2c_work *up_work;
	struct i2c_client *client;
	struct microp_i2c_client_data *cdata;
	uint8_t data[3], adc_level;
	uint16_t intr_status = 0, adc_value, gpi_status = 0;
	int keycode = 0, ret = 0;

	up_work = container_of(work, struct microp_i2c_work, work);
	client = up_work->client;
	cdata = i2c_get_clientdata(client);

	ret = i2c_read_block(client, MICROP_I2C_RCMD_GPI_INT_STATUS, data, 2);
	if (ret < 0) {
		dev_err(&client->dev, "%s: read interrupt status fail\n",
			 __func__);
	}

	intr_status = data[0]<<8 | data[1];
	ret = i2c_write_block(client, MICROP_I2C_WCMD_GPI_INT_STATUS_CLR, data, 2);
	if (ret < 0) {
		dev_err(&client->dev, "%s: clear interrupt status fail\n",
			 __func__);
	}
	pr_debug("intr_status=0x%02x\n", intr_status);

#if 0
	if ((intr_status & IRQ_LSENSOR) || cdata->force_light_sensor_read) {
		ret = microp_lightsensor_read(&adc_value, &adc_level);
		if (cdata->force_light_sensor_read) {
			/* report an invalid value first to ensure we trigger an event
			 * when adc_level is zero.
			 */
			input_report_abs(cdata->ls_input_dev, ABS_MISC, -1);
			input_sync(cdata->ls_input_dev);
			cdata->force_light_sensor_read = 0;
		}
		input_report_abs(cdata->ls_input_dev, ABS_MISC, (int)adc_level);
		input_sync(cdata->ls_input_dev);
	}
#endif

	if (intr_status & IRQ_SDCARD) {
		microp_read_gpi_status(client, &gpi_status);
		htcleo_microp_sdslot_update_status(gpi_status);
	}

/*
	if (intr_status & IRQ_HEADSETIN) {
		cdata->is_hpin_pin_stable = 0;
		wake_lock_timeout(&microp_i2c_wakelock, 3*HZ);
		if (!cdata->headset_is_in)
			schedule_delayed_work(&cdata->hpin_debounce_work,
					msecs_to_jiffies(500));
		else
			schedule_delayed_work(&cdata->hpin_debounce_work,
					msecs_to_jiffies(300));
	}
*/
	/* if (intr_status & IRQ_REMOTEKEY) {
		if ((get_remote_keycode(&keycode) == 0) &&
			(cdata->is_hpin_pin_stable)) {
			htc_35mm_key_event(keycode, &cdata->is_hpin_pin_stable);
		}
	} */
	
	if (intr_status & IRQ_PROXIMITY) {
        report_psensor_data();
	}

	enable_irq(client->irq);
}

static void ls_read_do_work(struct work_struct *work)
{
	struct i2c_client *client = private_microp_client;
	struct microp_i2c_client_data *cdata = i2c_get_clientdata(client);

	/* force a light sensor reading */
	disable_irq(client->irq);
	cdata->force_light_sensor_read = 1;
	schedule_work(&cdata->work.work);
}

static int microp_function_initialize(struct i2c_client *client)
{    
	struct microp_i2c_client_data *cdata;
	uint8_t data[20];
	uint16_t stat, interrupts = 0;
	int i;
	int ret;

	cdata = i2c_get_clientdata(client);

#if 0
	/* Light Sensor */
	if (als_kadc >> 16 == ALS_CALIBRATED)
		cdata->als_kadc = als_kadc & 0xFFFF;
	else {
		cdata->als_kadc = 0;
		pr_info("%s: no ALS calibrated\n", __func__);
	}

	if (cdata->als_kadc && golden_adc) {
		cdata->als_kadc =
			(cdata->als_kadc > 0 && cdata->als_kadc < 0x400)
			? cdata->als_kadc : golden_adc;
		cdata->als_gadc =
			(golden_adc > 0)
			? golden_adc : cdata->als_kadc;
	} else {
		cdata->als_kadc = 1;
		cdata->als_gadc = 1;
	}
	pr_info("%s: als_kadc=0x%x, als_gadc=0x%x\n",
		__func__, cdata->als_kadc, cdata->als_gadc);

	for (i = 0; i < 10; i++) {
		data[i] = (uint8_t)(lsensor_adc_table[i]
			* cdata->als_kadc / cdata->als_gadc >> 8);
		data[i + 10] = (uint8_t)(lsensor_adc_table[i]
			* cdata->als_kadc / cdata->als_gadc);
	}
	ret = i2c_write_block(client, MICROP_I2C_WCMD_ADC_TABLE, data, 20);
	if (ret)
		goto exit;
/*
	ret = gpio_request(HTCLEO_GPIO_LS_EN_N, "microp_i2c");
	if (ret < 0) {
		dev_err(&client->dev, "failed on request gpio ls_on\n");
		goto exit;
	}
	ret = gpio_direction_output(HTCLEO_GPIO_LS_EN_N, 0);
	if (ret < 0) {
		dev_err(&client->dev, "failed on gpio_direction_output"
				"ls_on\n");
		goto err_gpio_ls;
	}
*/
	cdata->light_sensor_enabled = 1;
#endif

#if 0
	/* Headset */
	for (i = 0; i < 6; i++) {
		data[i] = (uint8_t)(remote_key_adc_table[i] >> 8);
		data[i + 6] = (uint8_t)(remote_key_adc_table[i]);
	}
	ret = i2c_write_block(client,
		MICROP_I2C_WCMD_REMOTEKEY_TABLE, data, 12);
	if (ret)
		goto exit;
#endif

//	INIT_DELAYED_WORK(&cdata->hpin_debounce_work, hpin_debounce_do_work);
	INIT_DELAYED_WORK(&cdata->ls_read_work, ls_read_do_work);

	/* SD Card */
	interrupts |= IRQ_SDCARD;
	interrupts |= IRQ_PROXIMITY;

	/* enable the interrupts */
	ret = microp_interrupt_enable(client, interrupts);
	if (ret < 0) {
		dev_err(&client->dev, "%s: failed to enable gpi irqs\n",
			__func__);
		goto err_irq_en;
	}

	microp_read_gpi_status(client, &stat);
	htcleo_microp_sdslot_update_status(stat);

	microp_disable_lights();

	return 0;

err_irq_en:
err_gpio_ls:
//	gpio_free(HTCLEO_GPIO_LS_EN_N);
exit:
	return ret;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
void microp_early_suspend(struct early_suspend *h)
{
	struct microp_i2c_client_data *cdata;
	struct i2c_client *client = private_microp_client;
	int ret;

	if (!client) {
		pr_err("%s: dataset: client is empty\n", __func__);
		return;
	}
	cdata = i2c_get_clientdata(client);

	cdata->microp_is_suspend = 1;

	disable_irq(client->irq);
	ret = cancel_work_sync(&cdata->work.work);
	if (ret != 0) {
		enable_irq(client->irq);
	}

#if 0
	if (cdata->auto_backlight_enabled)
		microp_i2c_auto_backlight_mode(client, 0);
	if (cdata->light_sensor_enabled == 1) {
	//	gpio_set_value(HTCLEO_GPIO_LS_EN_N, 1);
		cdata->light_sensor_enabled = 0;
	}
#endif
}

void microp_early_resume(struct early_suspend *h)
{
	struct i2c_client *client = private_microp_client;
	struct microp_i2c_client_data *cdata;

	if (!client) {
		pr_err("%s: dataset: client is empty\n", __func__);
		return;
	}
	cdata = i2c_get_clientdata(client);

#if 0
//	gpio_set_value(HTCLEO_GPIO_LS_EN_N, 0);
	cdata->light_sensor_enabled = 1;
	[RED_LED] = {
		.name		= "red",
		.led_set_work   = microp_led_brightness_set_work,
		.attrs		= green_amber_attrs,
		.attr_cnt	= ARRAY_SIZE(green_amber_attrs)
	},

	if (cdata->auto_backlight_enabled)
		microp_i2c_auto_backlight_mode(client, 1);
#endif

	cdata->microp_is_suspend = 0;
	enable_irq(client->irq);
}
#endif

static int microp_i2c_suspend(struct i2c_client *client,
	pm_message_t mesg)
{
	return 0;
}

static int microp_i2c_resume(struct i2c_client *client)
{
	return 0;
}

static struct {
	const char *name;
	void (*led_set_work)(struct work_struct *);
	struct device_attribute **attrs;
	int attr_cnt;
} microp_leds[] = {
	[GREEN_LED] = {
		.name		= "green",
		.led_set_work   = microp_led_brightness_set_work,
		.attrs		= green_amber_attrs,
		.attr_cnt	= ARRAY_SIZE(green_amber_attrs)
	},
	[AMBER_LED] = {
		.name		= "amber",
		.led_set_work   = microp_led_brightness_set_work,
		.attrs		= green_amber_attrs,
		.attr_cnt	= ARRAY_SIZE(green_amber_attrs)
	},
};

static int microp_i2c_probe(struct i2c_client *client,
			    const struct i2c_device_id *id)
{
	struct microp_i2c_client_data *cdata;
	uint8_t data[6];
	int ret;
	int i, j;

	private_microp_client = client;
	ret = i2c_read_block(client, MICROP_I2C_RCMD_VERSION, data, 2);
	if (ret || !(data[0] && data[1])) {
		ret = -ENODEV;
		dev_err(&client->dev, "failed on get microp version\n");
		goto err_exit;
	}
	dev_info(&client->dev, "microp version [%02X][%02X]\n",
		  data[0], data[1]);

	ret = gpio_request(HTCLEO_GPIO_UP_RESET_N, "microp_i2c_wm");
	if (ret < 0) {
		dev_err(&client->dev, "failed on request gpio reset\n");
		goto err_exit;
	}
	ret = gpio_direction_output(HTCLEO_GPIO_UP_RESET_N, 1);
	if (ret < 0) {
		dev_err(&client->dev,
			 "failed on gpio_direction_output reset\n");
		goto err_gpio_reset;
	}

	cdata = kzalloc(sizeof(struct microp_i2c_client_data), GFP_KERNEL);
	if (!cdata) {
		ret = -ENOMEM;
		dev_err(&client->dev, "failed on allocat cdata\n");
		goto err_cdata;
	}

	i2c_set_clientdata(client, cdata);
	cdata->version = data[0] << 8 | data[1];
	cdata->microp_is_suspend = 0;
	cdata->auto_backlight_enabled = 0;
	cdata->light_sensor_enabled = 0;

	wake_lock_init(&microp_i2c_wakelock, WAKE_LOCK_SUSPEND,
			 "microp_i2c_present");
	
	/* Light Sensor */
	ret = device_create_file(&client->dev, &dev_attr_ls_adc);
	ret = device_create_file(&client->dev, &dev_attr_ls_auto);
 	cdata->ls_input_dev = input_allocate_device();
 	if (!cdata->ls_input_dev) {
 		pr_err("%s: could not allocate input device\n", __func__);
 		ret = -ENOMEM;
 		goto err_request_input_dev;
 	}
 	cdata->ls_input_dev->name = "lightsensor-level";
 	set_bit(EV_ABS, cdata->ls_input_dev->evbit);
 	input_set_abs_params(cdata->ls_input_dev, ABS_MISC, 0, 9, 0, 0);

 	ret = input_register_device(cdata->ls_input_dev);
 	if (ret < 0) {
 		dev_err(&client->dev, "%s: can not register input device\n", __func__);
 		goto err_register_input_dev;
 	}

	ret = misc_register(&lightsensor_misc);
	if (ret < 0) {
		dev_err(&client->dev, "%s: can not register misc device\n",
				__func__);
		goto err_register_misc_register;
	}

	/* LEDs */
	ret = 0;
	for (i = 0; i < ARRAY_SIZE(microp_leds) && !ret; ++i) {
		struct microp_led_data *ldata = &cdata->leds[i];

		ldata->type = i;
		ldata->ldev.name = microp_leds[i].name;
		ldata->ldev.brightness_set = microp_brightness_set;
		mutex_init(&ldata->led_data_mutex);
		INIT_WORK(&ldata->brightness_work, microp_leds[i].led_set_work);
		spin_lock_init(&ldata->brightness_lock);
		ret = led_classdev_register(&client->dev, &ldata->ldev);
		if (ret) {
			ldata->ldev.name = NULL;
			break;
		}

		for (j = 0; j < microp_leds[i].attr_cnt && !ret; ++j)
			ret = device_create_file(ldata->ldev.dev,
						 microp_leds[i].attrs[j]);
	}
	if (ret) {
		dev_err(&client->dev, "failed to add leds\n");
		goto err_add_leds;
	}


	/* Headset */
	cdata->headset_is_in = 0;
	cdata->is_hpin_pin_stable = 1;
//	platform_device_register(&htcleo_h35mm);

	ret = device_create_file(&client->dev, &dev_attr_key_adc);
	
	/* Proximity sensor */
	cdata->pr_input_dev = input_allocate_device();
	if (!cdata->pr_input_dev) {
		pr_err("%s: could not allocate input device (proximity)\n", __func__);
		ret = -ENOMEM;
		goto err_request_input_dev;
	}
	cdata->pr_input_dev->name = "proximity";

	set_bit(EV_ABS, cdata->pr_input_dev->evbit);
	input_set_abs_params(cdata->pr_input_dev, ABS_DISTANCE, 0, 1, 0, 0);

	ret = input_register_device(cdata->pr_input_dev);
	if (ret < 0) {
		dev_err(&client->dev, "%s: could not register input device (proximity)\n",
				__func__);
		goto err_register_input_dev;
	}
	
	ret = misc_register(&capella_cm3602_misc);
	if (ret < 0) {
		pr_err("%s: could not register misc device (proximity)\n", __func__);
		goto err_register_misc_register;
	}

	wake_lock_init(&proximity_wake_lock, WAKE_LOCK_SUSPEND, "proximity");

	/* G-sensor */
	ret = misc_register(&spi_bma_device);
	if (ret < 0) {
		pr_err("%s: init bma150 misc_register fail\n",
				__func__);
		goto err_register_bma150;
	}

	/* Setup IRQ handler */
	INIT_WORK(&cdata->work.work, microp_i2c_intr_work_func);
	cdata->work.client = client;

	ret = request_irq(client->irq,
			microp_i2c_intr_irq_handler,
			IRQF_TRIGGER_LOW,
			"microp_interrupt",
			&client->dev);
	if (ret) {
		dev_err(&client->dev, "request_irq failed\n");
		goto err_intr;
	}
	ret = set_irq_wake(client->irq, 1);
	if (ret) {
		dev_err(&client->dev, "set_irq_wake failed\n");
		goto err_intr;
	}

#ifdef CONFIG_HAS_EARLYSUSPEND
	if (cdata->enable_early_suspend) {
		cdata->early_suspend.level =
				EARLY_SUSPEND_LEVEL_BLANK_SCREEN + 1;
		cdata->early_suspend.suspend = microp_early_suspend;
		cdata->early_suspend.resume = microp_early_resume;
		register_early_suspend(&cdata->early_suspend);
	}
#endif

	ret = microp_function_initialize(client);
	if (ret) {
		dev_err(&client->dev, "failed on microp function initialize\n");
		goto err_fun_init;
	}
	
	return 0;

err_fun_init:
err_intr:
	misc_deregister(&spi_bma_device);

err_register_bma150:
//	platform_device_unregister(&htcleo_h35mm);
	device_remove_file(&client->dev, &dev_attr_key_adc);

err_add_leds:
	for (i = 0; i < ARRAY_SIZE(microp_leds); ++i) {
		if (!cdata->leds[i].ldev.name)
			continue;
		led_classdev_unregister(&cdata->leds[i].ldev);
		for (j = 0; j < microp_leds[i].attr_cnt; ++j)
			device_remove_file(cdata->leds[i].ldev.dev,
					   microp_leds[i].attrs[j]);
	}

err_register_misc_register:
	input_unregister_device(cdata->ls_input_dev);

err_register_input_dev:
	input_free_device(cdata->ls_input_dev);

err_request_input_dev:
	wake_lock_destroy(&microp_i2c_wakelock);
    wake_lock_destroy(&proximity_wake_lock);
	device_remove_file(&client->dev, &dev_attr_ls_adc);
	device_remove_file(&client->dev, &dev_attr_ls_auto);
	kfree(cdata);
	i2c_set_clientdata(client, NULL);

err_cdata:
err_gpio_reset:
	gpio_free(HTCLEO_GPIO_UP_RESET_N);
err_exit:
	return ret;
}

static int __devexit microp_i2c_remove(struct i2c_client *client)
{
	struct microp_i2c_client_data *cdata;

	cdata = i2c_get_clientdata(client);

#ifdef CONFIG_HAS_EARLYSUSPEND
	if (cdata->enable_early_suspend) {
		unregister_early_suspend(&cdata->early_suspend);
	}
#endif

	free_irq(client->irq, &client->dev);

	gpio_free(HTCLEO_GPIO_UP_RESET_N);

	misc_deregister(&lightsensor_misc);
	input_unregister_device(cdata->ls_input_dev);
	input_free_device(cdata->ls_input_dev);
	device_remove_file(&client->dev, &dev_attr_ls_adc);
	device_remove_file(&client->dev, &dev_attr_key_adc);
	device_remove_file(&client->dev, &dev_attr_ls_auto);

//	platform_device_unregister(&htcleo_h35mm);

	/* G-sensor */
	misc_deregister(&spi_bma_device);

	kfree(cdata);

	return 0;
}

#define ATAG_ALS	0x5441001b
static int __init parse_tag_als_kadc(const struct tag *tags)
{
	int found = 0;
	struct tag *t = (struct tag *)tags;

	for (; t->hdr.size; t = tag_next(t)) {
		if (t->hdr.tag == ATAG_ALS) {
			found = 1;
			break;
		}
	}

	if (found)
		als_kadc = t->u.revision.rev;
	pr_debug("%s: als_kadc = 0x%x\n", __func__, als_kadc);
	return 0;
}
__tagtable(ATAG_ALS, parse_tag_als_kadc);

static const struct i2c_device_id microp_i2c_id[] =
{
	{ MICROP_I2C_NAME, 0 },
	{ }
};

static struct i2c_driver microp_i2c_driver =
{
	.driver = {
		   .name = MICROP_I2C_NAME,
	},
	.id_table = microp_i2c_id,
	.probe = microp_i2c_probe,
	.suspend = microp_i2c_suspend,
	.resume = microp_i2c_resume,
	.remove = __devexit_p(microp_i2c_remove),
};

static int __init microp_i2c_init(void)
{
	return i2c_add_driver(&microp_i2c_driver);
}

static void __exit microp_i2c_exit(void)
{
	i2c_del_driver(&microp_i2c_driver);
}

module_init(microp_i2c_init);
module_exit(microp_i2c_exit);

MODULE_AUTHOR("Eric Olsen <eolsen@android.com>");
MODULE_DESCRIPTION("MicroP I2C driver");
MODULE_LICENSE("GPL");

static int micropklt_dbg_leds_set(void *dat, u64 val)
{
	struct i2c_client *client;
	char buffer[3] = { 0, 0, 0 };
	int r;

	client = private_microp_client;

	buffer[0] = 0xff & (val >> 8);
	buffer[1] = 0xff & (val >> 16);
	buffer[2] = 0xff & (val >> 24);
	r =i2c_write_block(client, 0xff & val, buffer, 3);
	return r;
}

static int micropklt_dbg_leds_get(void *data, u64 *val) {
	return 0;
}

DEFINE_SIMPLE_ATTRIBUTE(micropklt_dbg_leds_fops,
		micropklt_dbg_leds_get,
		micropklt_dbg_leds_set, "%llu\n");

static int __init micropklt_dbg_init(void)
{
	struct dentry *dent;

	dent = debugfs_create_dir("micropklt", 0);
	if (IS_ERR(dent))
		return PTR_ERR(dent);

	debugfs_create_file("raw", 0444, dent, NULL,
			&micropklt_dbg_leds_fops);
	return 0;
}

device_initcall(micropklt_dbg_init);

