/*
 * Driver for the DS2745 monitor chip
 *
 * Copyright Â© 2010 Markinus
 *
 * Some parts are used from the ds2746 kaiser driver
 *
 * Author:   Markinus  
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */



#include <linux/module.h>
#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>
#include <linux/pm.h>
#include <linux/power_supply.h>
#include <linux/i2c.h>
#include <linux/ds2745.h>
#include <asm/gpio.h>

#ifdef CONFIG_ANDROID_POWER
#include <linux/android_power.h>
static android_suspend_lock_t vbus_suspend_lock;
#endif

#define MODULE_NAME "ds2745"

/* definitions for registers we care about. */
#define DS2745_DATA_SIZE		0x12

#define DS2745_STATUS_REG		0x01

#define DS2745_TEMPERATURE_MSB		0x0a
#define DS2745_TEMPERATURE_LSB		0x0b
#define DS2745_CURRENT_MSB		0x0e
#define DS2745_CURRENT_LSB		0x0f
#define DS2745_VOLTAGE_MSB		0x0c
#define DS2745_VOLTAGE_LSB		0x0d
#define DS2745_CURRENT_MSB		0x0e
#define DS2745_CURRENT_LSB		0x0f
#define DS2745_CURRENT_ACCUM_MSB	0x10
#define DS2745_CURRENT_ACCUM_LSB	0x11

/* oprations */
#define DS2745_CTL_ENABLE_SUSPEND	1		

struct ds2745_data {
	struct i2c_client *client;
	struct mutex update_lock;

	uint8_t 	rnsn_id;			/*  rsns ID */
	uint32_t	bat_capa;			/*  Battery capacity in mAh */
	uint32_t	bat_voltage;			/*  Battery nominal voltage mV */
	uint32_t	bat_max_voltage;		/*  Battery maximal charging voltage mV */
	uint32_t	bat_min_voltage;		/*  Battery minimal voltage (off) mV */
	uint32_t	bat_vol_up_thresholt;		/*  Battery voltage upper thresholt in percent from bat_max_voltage for 100% e.G. 95 */
	uint32_t	bat_vol_lo_thresholt;		/*  Battery voltage lower thresholt in percent from bat_min_voltage for approximate formula e.g. 10 */
	uint32_t	bat_cap_lo_thresholt;		/*  Battery capacity lower thresholt in percent from bat_capa for approximate formula e.g. 5 */
	uint8_t		power_on :1;			/*  Is the chip always on or allow to sleep */
	
};

// Voltage resolution in nV
const uint32_t volt_res = 4880;	// nV	

// Voltage resolution for current measurement
const uint32_t cur_volt_res = 1562500;	// pV	

// Voltage resolution for accumulated current measurement
const uint32_t acc_cur_volt_res = 6250;	// nVh	

// Temperature resolution in m°C
const uint32_t temp_res = 125; 	// m°C	

// The resistances in mOHM
const uint8_t rsns[] = {
	25,			// 25 mOHM
	20,			// 20 mOHM
	15,			// 15 mOHM
	10,			// 10 mOHM
	5,			// 5 mOHM
};

struct ds2745_data drv_data;

static int battery_capacity = -1;
module_param(battery_capacity, int, 0644);
MODULE_PARM_DESC(battery_capacity, "Estimated battery capacity in mAh");

static int i2c_read(int r) {
	unsigned char i2c_msg[1];
	unsigned char i2c_data[2];
	i2c_msg[0]=r;
	i2c_master_send(drv_data.client, i2c_msg, 1);
	i2c_master_recv(drv_data.client, i2c_data, 2);
	printk("ds2745_i2c_read(%x)=%x %x\n",r,i2c_data[0],i2c_data[1]);
	return i2c_data[0];
}

static void i2c_write(int r, int v) {
	unsigned char i2c_msg[3];
	i2c_msg[0]=r;
	i2c_msg[1]=v>>8;
	i2c_msg[2]=v&0xFF;
	i2c_master_send(drv_data.client, i2c_msg, 3);
}

void ds2745_control(int oper,int param)
{
	unsigned char dat;
	switch (oper)
	{
		case DS2745_CTL_ENABLE_SUSPEND:
			if(param!= 0 && param!= 1) return; 
			dat = i2c_read(DS2745_STATUS_REG);
			dat = param ? dat | 1<<5 : dat & ~(1<<5);
			i2c_write(DS2745_STATUS_REG, dat);
			break;
	}
}


int ds2745_battery_read_status(struct battery_info_reply *b)
{
	short s;
	uint32_t full, step, capa;
	//	printk("read battery status\n");
	if(!drv_data.client) {
		printk("client is null\n");
		b->level=50;
		return 0;
	}

	s=i2c_read(DS2745_VOLTAGE_LSB) | i2c_read(DS2745_VOLTAGE_MSB)<<8;
	b->batt_vol=((s>>5)*volt_res)/1000;	// the lowest 5 bits are reserved, mV
	
	s=i2c_read(DS2745_CURRENT_LSB) | i2c_read(DS2745_CURRENT_MSB)<<8;
	b->batt_current=((uint32_t)s*cur_volt_res) / (rsns[drv_data.rnsn_id]*1000000);
	
	/* Calculate the max for 100% battery */
	full = (drv_data.bat_capa * rsns[drv_data.rnsn_id] * 1000) /acc_cur_volt_res;
	
	/* if battery voltage is < bat_vol_lo_thresholt and depleting, we assume it's almost empty! */
	if (b->batt_vol < (drv_data.bat_vol_lo_thresholt*drv_data.bat_min_voltage/100+drv_data.bat_min_voltage) && b->batt_current < 0) {
	    /* use approximate formula: e.g. 3.3V=5%, 3.0V=0% */
	    step = (full * drv_data.bat_cap_lo_thresholt / 100) / ( drv_data.bat_min_voltage * drv_data.bat_vol_lo_thresholt/100);
	    i2c_write(DS2745_CURRENT_ACCUM_MSB, (b->batt_vol-drv_data.bat_min_voltage) * step);
	}
	
	// if battery voltage is high (>bat_vol_up_thresholt) and it's not discharging but also not charging much then it should be full. 
	if (b->batt_vol > (drv_data.bat_vol_up_thresholt*drv_data.bat_max_voltage/100) && b->batt_current >= 0 && b->batt_current<200) {
	    i2c_write(DS2745_CURRENT_ACCUM_MSB, full);
	}
	
	s=i2c_read(DS2745_CURRENT_ACCUM_LSB) | i2c_read(DS2745_CURRENT_ACCUM_MSB)<<8;
	pr_info("ds2745: s= %d res=%d capa=%d\n", s, rsns[drv_data.rnsn_id], drv_data.bat_capa);
	capa = (s * acc_cur_volt_res) / (drv_data.bat_capa * rsns[drv_data.rnsn_id]);
	b->level=capa;  //( 0 - 1000)
	/* if we read 0%, */
	if(b->level<1) {
		/* only report 0% if we're really down, < %3 over min voltage */
		b->level=((b->batt_vol <= drv_data.bat_min_voltage+(drv_data.bat_min_voltage*3/100)) ? 0 : 1);
	}
	
	if(b->level>1000) b->level=1000;
	b->full_bat = 1000;
	

	s=i2c_read(DS2745_TEMPERATURE_LSB) | i2c_read(DS2745_TEMPERATURE_MSB)<<8;
	b->batt_temp = (s >> 5) * temp_res;	// 5 lower bits are reserved m°C
	s=i2c_read(0x09) | i2c_read(0x08)<<8;
	
	
	pr_info("ds2745: %dmV %dmA charge: %d/%d  temp: %d m°C\n",b->batt_vol,b->batt_current,b->level, b->full_bat, b->batt_temp);
	return 0;
}

static int ds2745_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	printk(KERN_ERR MODULE_NAME ": Initializing DS2745 over i2c driver "
					"at addr: 0x%02x\n", client->addr);
	int err = 0;
	struct i2c_adapter *adapter = to_i2c_adapter(client->dev.parent);
 	struct ds2745_platform_data *pdata;

	if (!i2c_check_functionality(adapter, I2C_FUNC_SMBUS_BYTE)) {
		err = -EIO;
		goto exit;
	}

	drv_data.client = client;
	pdata = client->dev.platform_data;
	if (likely(pdata != NULL)) {
		drv_data.bat_capa = pdata->bat_capa;
		drv_data.bat_voltage = pdata->bat_voltage;	
		drv_data.bat_max_voltage = pdata->bat_max_voltage;
		drv_data.bat_min_voltage = pdata->bat_min_voltage;
		drv_data.bat_vol_up_thresholt = pdata->bat_vol_up_thresholt;
		drv_data.bat_vol_lo_thresholt = pdata->bat_vol_lo_thresholt;
		drv_data.bat_cap_lo_thresholt = pdata->bat_cap_lo_thresholt;
		drv_data.rnsn_id = pdata->rsns_id;
	} else {
		dev_err(&client->dev, " cannot work without platform data\n");
		err = -ENODEV;
		goto exit;
	}

	if(battery_capacity>0) drv_data.bat_capa = battery_capacity;
	mutex_init(&drv_data.update_lock);
	
	struct battery_info_reply b;
        ds2745_battery_read_status(&b);
	pr_info("DS2745 Init done!\n");
	return 0;

exit:
	return err;
}

static int ds2745_remove(struct i2c_client *client)
{
	i2c_unregister_device(drv_data.client);

	return 0;
}

#if CONFIG_PM
static int ds2745_suspend(struct i2c_client * client, pm_message_t mesg)
{
	struct ds2745_data *data = i2c_get_clientdata(client);
	if (data->power_on) {
		ds2745_control(DS2745_CTL_ENABLE_SUSPEND, 1);
		data->power_on = 0;
	}
	return 0;
}

static int ds2745_resume(struct i2c_client * client)
{
	struct ds2745_data *data = i2c_get_clientdata(client);
	if (!data->power_on) {
		ds2745_control(DS2745_CTL_ENABLE_SUSPEND, 0);
		data->power_on = 1;
	}
	return 0;
}
#else
#define ds2745_suspend NULL
#define ds2745_resume NULL
#endif

static const struct i2c_device_id ds2745_battery_id[] = {
	{ DS2745_I2C_NAME, 0 },
	{ }
};

static struct i2c_driver ds2745_battery_driver = {
	.probe	  = ds2745_probe,
	.remove   = ds2745_remove,
#if CONFIG_PM
	.suspend = ds2745_suspend,
	.resume = ds2745_resume,
#endif
	.id_table= ds2745_battery_id,
	.driver = {
		.name = DS2745_I2C_NAME,
	},
};

static int __init ds2745_battery_init(void)
{
	pr_info("DS-2745 Init\n");
#ifdef CONFIG_ANDROID_POWER
	vbus_suspend_lock.name = "vbus_present";
	android_init_suspend_lock(&vbus_suspend_lock);
#endif
	return i2c_add_driver(&ds2745_battery_driver);
}

static void __exit ds2745_battery_exit(void)
{
	i2c_del_driver(&ds2745_battery_driver);
}

module_init(ds2745_battery_init);
module_exit(ds2745_battery_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Markinus" );
MODULE_DESCRIPTION("Driver for the ds2745 battery monitor");
