/* drivers/rtc/rtc-msm7x00.c
 *
 * Author: Martin Johnson <M.J.Johnson@massey.ac.nz>
 * 	   Markinus <markinus@gmailcom>
 *	   Changed the calls to new dex comm function
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/module.h>
#include <linux/version.h>

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/rtc.h>

#include "../../arch/arm/mach-msm/dex_comm.h"

#define PDEV_NAME "msm_rtc"

#define SECSFROM_1970_TO_1980 315532800

static struct rtc_device *rtc;

static unsigned long rtcalarm_time;

static int
msmrtc_pmlib_set_time(struct device *dev, struct rtc_time *tm)
{
	unsigned long unix_time;

	if (rtc_valid_tm(tm))
		return -EINVAL;

	rtc_tm_to_time(tm, &unix_time);
	unix_time=unix_time-SECSFROM_1970_TO_1980; // MSM RTC starts 10 years after unix time

	dex_comm(DEX_WRITE_RTC, (uint32_t*)&unix_time, 0);

	return 0;
}

static int
msmrtc_pmlib_read_time(struct device *dev, struct rtc_time *tm)
{
	unsigned secs=0;

	dex_comm(DEX_READ_RTC, &secs, 0);
	secs = secs + SECSFROM_1970_TO_1980;	// MSM RTC starts 10 years after unix time
	rtc_time_to_tm(secs, tm);
	return 0;
}

static int
msmrtc_virtual_alarm_set(struct device *dev, struct rtc_wkalrm *a)
{
	unsigned long now = get_seconds();

	if (!a->enabled) {
		rtcalarm_time = 0;
		return 0;
	} else
		rtc_tm_to_time(&a->time, &rtcalarm_time);

	if (now > rtcalarm_time) {
		printk(KERN_ERR "%s: Attempt to set alarm in the past\n",
		       __func__);
		rtcalarm_time = 0;
		return -EINVAL;
	}

	return 0;
}

static struct rtc_class_ops msm_rtc_ops = {
	.read_time	= msmrtc_pmlib_read_time,
	.set_time	= msmrtc_pmlib_set_time,
	.set_alarm	= msmrtc_virtual_alarm_set,
};

static void
msmrtc_alarmtimer_expired(unsigned long _data)
{
	printk(KERN_DEBUG "%s: Generating alarm event (src %lu)\n",
	       rtc->name, _data);

	rtc_update_irq(rtc, 1, RTC_IRQF | RTC_AF);
	rtcalarm_time = 0;
}

static int
msmrtc_probe(struct platform_device *pdev)
{
	pr_info("Probe 7x01a RTC driver\n");
	rtc = rtc_device_register("msm_rtc",
				  &pdev->dev,
				  &msm_rtc_ops,
				  THIS_MODULE);
	if (IS_ERR(rtc)) {
		printk(KERN_ERR "%s: Can't register RTC device (%ld)\n",
		       pdev->name, PTR_ERR(rtc));
		return PTR_ERR(rtc);
	}
	printk("MSM RTC started\n");
	return 0;
}

static unsigned long msmrtc_get_seconds(void)
{
	struct rtc_time tm;
	unsigned long now;

	msmrtc_pmlib_read_time(NULL, &tm);
	rtc_tm_to_time(&tm, &now);
	return now;
}

static int
msmrtc_suspend(struct platform_device *dev, pm_message_t state)
{
	unsigned secs;
	if (rtcalarm_time) {
		unsigned long now = msmrtc_get_seconds();
		int diff = rtcalarm_time - now;
		if (diff <= 0) {
			msmrtc_alarmtimer_expired(1);
			return 0;
		}

		dex_comm(DEX_READ_RTC, &secs, 0);

		printk("Wake up in %d seconds\n",diff);

		secs = secs + diff;
		dex_comm(DEX_SET_ALARM_RTC, &secs, 0);
	}

	return 0;
}

static int
msmrtc_resume(struct platform_device *dev)
{
	unsigned long now;
	int diff;
	if (rtcalarm_time) {
		now = msmrtc_get_seconds();
		diff = rtcalarm_time - now;
		if (diff <= 0)
			msmrtc_alarmtimer_expired(2);
	}
	return 0;
}

static struct platform_driver msmrtc_driver = {
	.probe	= msmrtc_probe,
	.suspend	= msmrtc_suspend,
	.resume		= msmrtc_resume,
	.driver	= {
		.name	= "msm_rtc",
		.owner	= THIS_MODULE,
	},
};

static int __init msmrtc_init(void)
{
	pr_info("Init 7x01a RTC driver\n");
	return platform_driver_register(&msmrtc_driver);
}

module_init(msmrtc_init);

MODULE_DESCRIPTION("RTC driver for Qualcomm MSM7x00 chipsets");
MODULE_AUTHOR("Martin Johnson <M.J.Johnson@massey.ac.nz>"
	       "Markinus <markinus@gmail.com>");
MODULE_LICENSE("GPL");
