/*
================================ COPYRIGHT ================================
The contents of this file are subject to the VTech Informations Ltd. License
of VT-OS Ver. 1.1 operating system (the "License"); you may not use this 
file except in compliance with the License.  
 
Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the License 
for the specific language governing rights and limitations under the License.
 
The Original Code is VT-OS Ver. 1.1 Operating System, released 
on October 1st, 1999

The Initial Developer of the Original Code is VTech Informations Ltd.  All 
codes are Copyright (C) VTech Informations Ltd. 1999.  All Rights Reserved.
===========================================================================
*/

/*
===========================================================================
File        :   ui.h
Author(s)   :   Henry Fok, David Lo
Company     :   VTech Informations Ltd.
Project     :   Helio 
Date:	    :   October 1st, 1999
Purpose:	:   header file for ui related structures
Revision    :   1.1
Note        :   None
===========================================================================
*/              

#ifndef _UI_H_
#define _UI_H_

#include "datatype.h"

typedef USHORT ObjectID;
/* ==================================================================== */
/* ========================= Global Structures ======================== */
/* ==================================================================== */
/* Structure: Save_Behind_Pixels used by all object that will cover another object */
struct Bitmap_Template
{
	SHORT xcoord;			/* the x co-ordinate of the covered image */
	SHORT ycoord;			/* the y co-ordinate of the covered image */
	SHORT width;			/* the width of the covered image */
	SHORT height;			/* the height of the covered image */
	BYTE compressed;		/* indicate whether the bitmap is compressed or not */
	BYTE quantisation;
	USHORT size;			/* size (no. of bytes) of the bitmap */
	UWORD *bitmap_data;		/* Pointer to locations of storing the pixels of
					   the covered image */
};
typedef struct Bitmap_Template BitmapTemplate;

/* Structure: Object_Bounds used by all objects	*/
struct Object_Bounds
{
	SHORT xcoord;			/* window relative x co-ordinate of the object */
	SHORT ycoord;			/* window relative y co-ordinate of the object */
	SHORT width;			/* the width of the object */
	SHORT height;			/* the height of the object */
};
typedef struct Object_Bounds ObjectBounds;

/* Structure: Identification of Object */
struct _Identification
{
	ObjectID ui_object_id;      	/* the DBID of the object */
	BYTE ui_object_type;            /* the type of this object
										0 = form object, 1 = control object
										2 = field object, 3 = list object, 
										4 = menu object, 5 = scroll bar object,
										6 = table object 7 = scheduler line object, 
										8 = string object, 9 = bitmap object, 
										10 = line object, 11 = clipboard object	*/
	ObjectID table_related;			/* the DBID of the table that this object belongs to
									   if no table, then $FFFF is set, otherwise, correct DBID
									   is set */
};
typedef struct _Identification Identification;

/* ==================================================================== */
/* ========================= Form Structures ========================== */
/* ==================================================================== */
/* Structure: Form_Objects_List used by FORM OBJECT */
struct Form_Objects_List
{
	ObjectID	object_id;      /* DBID of the objects within the form */
	BYTE	object_type;		/* the type of the object within the form */
								/* 	1 = control object
									2 = field object
									3 = list object
									4 = menu object
									5 = scroll bar object
									6 = table object
									7 = scheduler line object
									8 = string object
									9 = bitmap object
								   10 = line object		*/
};
typedef struct Form_Objects_List FormObjectsList;

/* Structure: Form_Attr used by FORM OBJECT */
struct Form_Attr
{
	BOOLEAN form_drawn;     	/* indicate whether the form is drawn on the display or not */
	BOOLEAN form_active;		/* indicate whether the form is set as active or not */
	BOOLEAN form_updated;
	BOOLEAN form_save_behind;
};
typedef struct Form_Attr FormAttr;

/* Structure: Form_ */
struct _Form
{
	Identification identification;	/* Identification structure */
	ObjectBounds bounds;            /* bounds */
	ObjectID form_focus; 			/* DBID of object that has the focus in the form */
	BYTE form_style;				/* the type of the form object
										0 = Normal form
										1 = Information Dialog form
										2 = Find Dialog form
										3 = Confirmation Dialog form */
	BYTE * form_dialog_title;		/* Pointer to a BYTE array that stores
									   the title of the dialog title  */
	BitmapTemplate form_bitmap;
	USHORT form_num_objects;
	FormObjectsList * form_objects_list;	/* Pointer to FormObjectsList structure */
	BitmapTemplate save_behind;
	FormAttr form_attr;				/* Pointer to the Form_Attr structure */
};
typedef struct _Form Form;

/* ==================================================================== */
/* ========================= Control Structures ======================= */
/* ==================================================================== */
/* Structure: Control_Attr */
struct Control_Attr
{
	BOOLEAN control_enable;		/* indicate whether the control object is
								   enabled or not */
	BOOLEAN control_drawn;		/* indicate whether the control object is
								   drawn on the display or not */
	BOOLEAN control_save_behind;/* indicate whether the control object covered
								   another image or not */
	BOOLEAN control_active;		/* indicate whether this control has the focus or not */
	BOOLEAN control_enter;		/* indicate whether the pen entered this control object */
	BOOLEAN control_enter1;
	BOOLEAN control_enter2;
	BOOLEAN control_visible;
};
typedef struct Control_Attr ControlAttr;

/* Structure: Control_Template_Button */
struct Control_Template_Button
{
	USHORT button_radius;		/* the radius of the button edge */
	BYTE button_color_on;		/* the color of the button when it is on*/
	BYTE button_color_off;		/* the color of the button when it is off*/
	BOOLEAN control_value;		/* control value */
								/* TRUE = on, FALSE = off */
};
typedef struct Control_Template_Button ControlTemplateButton;

/* Structure: Control_Template_PushButton */
struct Control_Template_PushButton
{
	BYTE push_button_color_on;	/* the color of the push button when it is on*/
	BYTE push_button_color_off;	/* the color of the push button when is is off */
	USHORT push_button_group_id;/* the group id of the group of push buttons */
	BOOLEAN control_value;		/* an on/off value to show whether the push
								   button is pressed or not */
	USHORT push_button_radius;	/* this is for push button with round corner */

};
typedef struct Control_Template_PushButton ControlTemplatePushButton;

/* Structure: Control_Template_RepeatButton */
struct Control_Template_RepeatButton
{
   USHORT repeat_count;			/*a number to show when to send EvtControlReapeatEvent */
   BitmapTemplate repeat_bitmap;/* structure of bitmap template that holds the bitmap */

};
typedef struct Control_Template_RepeatButton ControlTemplateRepeatButton;

/* Structure: Control_Template_CheckBox */
struct Control_Template_CheckBox
{
	BitmapTemplate checkbox_bitmap1;/* Pointer to bitmap of the checkbox (while uncheck) */
	BitmapTemplate checkbox_bitmap2;/* Pointer to bitmap of the checkbox (while check) */
	USHORT checkbox_group_id;		/* the group id of the group of check boxes */
	BOOLEAN control_value;		/* the on/off state */
};
typedef struct Control_Template_CheckBox ControlTemplateCheckBox;

/* Structure: Control_Template_Popup_Trigger */
struct Control_Template_Popup_Trigger
{
	USHORT popup_radius;		/* the radius of the popup trigger object */
	USHORT popup_num_objects;	/* number of items in the list */
	BYTE ** popup_items_list;	/* Pointer to a list of addresses of
								   null-terminated text string of items */
	BitmapTemplate save_behind;	/* structure used to store the image that is being covered */
	ObjectBounds popup_arrow_up_bounds;  /*bounds will be the region with respect to
										 windows */
	ObjectBounds popup_arrow_down_bounds;/*bounds will be the region with respect to
										 windows */
	SHORT popup_selected_item; 	 /* selected item on the list */
	SHORT popup_highlighted_item;
	USHORT popup_max_num_item_display;
	USHORT popup_current_num_item_display;
	USHORT popup_top_item_num;
	ObjectBounds popuped;
	BOOLEAN control_value;
	BOOLEAN popup_arrow_up;
	BOOLEAN popup_arrow_down;
};
typedef struct Control_Template_Popup_Trigger ControlTemplatePopupTrigger;

/* Structure: _Control */
struct _Control
{
	Identification identification;
	BYTE * control_text;		/* Pointer to null-terminated text of the control object
								   If no text, then it is NULL */
	ObjectBounds bounds;		/* structure of bounds */
	ObjectBounds screen_bounds;
	void * control_template;	/* Pointer to the address of the selected control
								   template structure */
	BYTE control_style;			/* indicate the data structure is for which type of control object */
								/* 0 = Button
								   1 = Push button
								   2 = Repeat button
								   3 = CheckBox
								   4 = Popup Trigger */
	BYTE control_subtype;		/* the pattern for each control style */
	BYTE text_alignment;        /* 0 = left aligned, 1 = center aligned
								   2 = right aligned */
	ControlAttr control_attr;
};
typedef struct _Control Control;

/* ==================================================================== */
/* ========================= Field Structures ======================= */
/* ==================================================================== */
/* Structure: Line Info */
struct _LineInfo
{
	WORD start;
	WORD length;
};
typedef struct _LineInfo LineInfo;

/* Structure: Field_Underlined_Section */
struct _FieldUnderlinedSection
{
  WORD field_underlined_start_char;
  WORD field_underlined_length;
};
typedef struct _FieldUnderlinedSection FieldUnderlinedSection;

/* Structure: Field_Attr */
struct Field_Attr
{
	BOOLEAN field_drawn;	/* indicate whether the field object is
							   drawn on the display or not */
	BOOLEAN field_active;	/* indicate whether the field object has the
							   focus or not */
    BOOLEAN field_enable;
    BOOLEAN field_dirty;
	BOOLEAN field_highlight;/* indicate whether the the text in the field object
							   is highlighted or not */
    BOOLEAN field_insert_pt_visible;
    BOOLEAN field_scrollbar;
	BOOLEAN field_full_size;
	BOOLEAN field_visible;
};
typedef struct Field_Attr FieldAttr;

/* Structure: _Field */
struct _Field
{
	Identification identification;
	ObjectBounds bounds;   	/* the bounds for this object */
	ObjectBounds screen_bounds;
	BYTE *field_string;		/* the text string in the field object */
    BYTE field_style;       /* 0 = no frame, 1= framed */
    BYTE field_back_line; /* 0 = no line, 1 = dotdot, 2 = grey line */
    BYTE field_font_id;
    BYTE field_font_color;
    BYTE field_background_color;
    BYTE field_text_alignment; /* 0 = left 1 = centre 2 = right */
	WORD field_max_chars;	/* the max. no. of chars in the field object */
    WORD field_current_num_chars;
    WORD field_total_num_lines;
    WORD field_top_line_num;
    WORD field_num_lines_displayed;
    SHORT field_insert_pt_x;
    SHORT field_insert_pt_y;
    WORD field_insert_pt_char_pos;
    BYTE field_insert_pt_movement;  /* 0 = NO movement
                                       1 = MOVE_UP
                                       2 = MOVE_DOWN
                                       3 = MOVE_LEFT
                                       4 = MOVE_RIGHT */
    WORD field_highlight_start_char;
    WORD field_highlight_end_char;
    WORD field_highlight_length;
    USHORT field_num_underlined_section;
    FieldUnderlinedSection **field_underlined_section;
	LineInfo *field_lineinfo;
    USHORT field_repeat_count;
	FieldAttr field_attr;
};
typedef struct _Field Field;

/* ==================================================================== */
/* ========================= List Structures ======================= */
/* ==================================================================== */
/* Structure: List_Attr */
struct List_Attr
{
	BOOLEAN list_enable;	/* indicate whether the list object
							   is enabled for pen action or not */
	BOOLEAN list_drawn;		/* indicate whether the list object is
							   drawn on the display or not */
	BOOLEAN list_active;	/* indicate whether the list object has the focus
							   or not */
	BOOLEAN list_set_scroll;
	BOOLEAN list_enter1;
	BOOLEAN list_enter2;
	BOOLEAN list_visible;
	BOOLEAN list_synchronous;
};
typedef struct List_Attr ListAttr;

/* Structure: _List */
struct _List
{
	Identification identification;
	ObjectBounds bounds;	/* structure to bounds */
	ObjectBounds screen_bounds;
	USHORT list_num_related_list;
	ObjectID *list_related_list_id;
	USHORT list_total_num_items;		/* the number of items in the list object */
	USHORT list_max_num_items_on_display;
	USHORT list_num_items_on_display;	/* the number of items that are display
										   on the display */
	USHORT list_top_item_num;	/* the item number of the item that is on
								   top of the list on the display */
	BYTE ** list_items;		/* Pointer to array to store the item in the list */
	SHORT list_item_height;
	BYTE list_text_alignment;
	SHORT list_selected_item;		/* the current selected item of the list object */
	SHORT list_highlighted_item;
	BYTE list_style;             /* 0 = no frame */
								 /* 1 = framed */
	ObjectBounds list_arrow_up_bounds;  /*bounds will be the region with respect to
										 windows */
	ObjectBounds list_arrow_down_bounds;/*bounds will be the region with respect to
										 windows */
	BOOLEAN list_arrow_up;
	BOOLEAN list_arrow_down;
	BYTE list_text_color;
	BYTE list_bg_color;
    BYTE list_text_font;                    
    ListAttr list_attr;
};
typedef struct _List List;

/* ==================================================================== */
/* ========================= Menu Structures ======================= */
/* ==================================================================== */
/* Structure: Menu_Attr */
struct Menu_Attr
{
	BOOLEAN menu_drawn;		/* indicate whether menu object is
							drawn on the display or not */
	BOOLEAN menu_visible;
};
typedef struct Menu_Attr MenuAttr;

/* Structure: _Menu */
struct _Menu
{
	Identification identification;
	ObjectBounds bounds;            /* this structure is not required
									   to be set, cos it will be updated later */
	BitmapTemplate save_behind;	/* structure to store covered image */
	USHORT menu_num_items;
	USHORT menu_max_num_items;  /*if normal font, then there can only be 13 items */
	SHORT menu_selected_item;
	SHORT menu_highlighted_item;
	BYTE **menu_items;      /* Pointer to a list of pointer of null-terminated string */
	MenuAttr menu_attr;		/* attributes of menu object */
};
typedef struct _Menu Menu;

/* ==================================================================== */
/* ========================= Scrollbar Structures ======================= */
/* ==================================================================== */
/* Structure: Scrollbar_Attr */
struct Scrollbar_Attr
{
	BOOLEAN scrollbar_drawn;		/* indicate whether the scrollbar is drawn
					   on the display or not */
	BOOLEAN scrollbar_active;		/* indicate whether the scrollbar is active or not */
	BOOLEAN scrollbar_visible;
	BOOLEAN scrollbar_enable;
	BOOLEAN scrollbar_enter;
	BOOLEAN scrollbar_enter1;
	BOOLEAN scrollbar_enter2;
};
typedef struct Scrollbar_Attr ScrollbarAttr;

/* Structure: Scrollbar_Attr */
struct _Scrollbar
{
	Identification identification;
	ObjectBounds bounds;		/* bounds */
	USHORT scrollbar_repeat_count;
	BitmapTemplate scrollbar_arrow1;
	BitmapTemplate scrollbar_arrow2;
	WORD scrollbar_max;		/* Maximum value of the scrollbar */
	WORD scrollbar_min;		/* Minimum value of the scrollbar */
	WORD scrollbar_value;		/* the current scrollbar value */
    WORD scrollbar_old_value;
	WORD scrollbar_pagesize;		/* the number of lines to scroll when user scrolls one page */
	WORD scrollbar_draw_pagesize;	/* Used for draw only */
	WORD scrollbar_total_num_lines;
	BYTE scrollbar_type;            /* 0 = scrollbar
									   1 = not scrollbar with hightlight
									   2 = not scrollbar without highlight*/
	BYTE scrollbar_style;
	SHORT scrollbar_save_pos_x;
	SHORT scrollbar_save_pos_y;
	BYTE scrollbar_clicked_region; /* 0 = NOT_HITTED
									    1 = UPPER_ARROW
										2 = SCROLLCAR
										3 = SCROLLBAR
										4 = LOWER_ARROW */
	BYTE *scrollbar_text;
	ScrollbarAttr scrollbar_attr;	/* scrollbar attribute */
};
typedef struct _Scrollbar Scrollbar;

/* ==================================================================== */
/* ========================= Table Structures ======================= */
/* ==================================================================== */
/* Structure: Table_Attr */
struct Table_Attr
{
	BOOLEAN table_drawn;            /* indicate whether the table is drawn on
					   the display or not */
	BOOLEAN table_scrollbar;	/* whether table has scrollbar or not */
	BOOLEAN table_enable;		/* indicate whether the table is editable or not */
	BOOLEAN table_active;		/* indicate whether one of more items in the table
								  selected or not */
	BOOLEAN table_enter;
	BOOLEAN table_visible;
	BOOLEAN table_highlight_enable;	/*TRUE when highlight after selection
									  FALSE no highlight after selection*/
};
typedef struct Table_Attr TableAttr;

/* Structure: Table_Items */
struct Table_Items
{
	BYTE table_data_type;			/* the data type of the item in the table */
									/* 0 = text, 1 = value/number, 2 = UI Object */
	ObjectID table_item_ui_id;		/* If the item is UI object, then the DBID
								   of the object is placed in here. Otherwise, NULL is placed in here */
	BYTE table_font;
	WORD table_value;				/* the value/number of the item */
	BYTE table_display_alignment;
	BYTE table_text_color;
	BYTE table_text_bg_color;
	BOOLEAN table_cell_highlight;
	BOOLEAN table_cell_has_bitmap;
	BYTE * table_text;				/* Pointer to the text */
};
typedef struct Table_Items TableItems;

/* Structure: _Table */
struct _Table
{
	Identification identification;
	ObjectBounds bounds;		/* bounds */
	USHORT table_num_column;		/* the total number of columns in the
									table object */
	USHORT table_num_row;		/* the total number of rows in the table
								object */
	USHORT table_current_row;		/* the current row number (starting from 0)
									of the item that is selected */
	USHORT table_current_col;		/* the current column number (startign from 0)
									of the item that is selected */
	TableItems ** table_item_ptr;		/* Pointer to a list of addresses of the item data
										structures */
	SHORT * table_column_width;	/* Pointer to a list of column width */
	SHORT * table_row_height;	/* Pointer to a list of row height */
	USHORT table_num_col_display;	/* the number of columns that are on the display */
	USHORT table_num_row_display;	/* the number of rows that are on the display */
	USHORT table_top_row_num;		/* the row number of the top row that is on display */
	USHORT table_left_col_num;		/* the column number of the lefmost column on display */
	BYTE table_style;				/* 0 = no line in the bounds
									1 = line in the bounds */
	BYTE table_bg_color;
	BitmapTemplate table_cell_bitmap;
	TableAttr table_attr;
};
typedef struct _Table Table;

/* ==================================================================== */
/* ========================= Scheduler Line Structures ================
/* ==================================================================== */
/* Structure: Schline_Attr */
struct Schline_Attr
{
	BOOLEAN schline_drawn;
	BOOLEAN schline_visible;
	BOOLEAN schline_active;
	BOOLEAN schline_enable;		
	BOOLEAN	schline_enter;
};
typedef struct Schline_Attr SchlineAttr;

/* Structure: _Schline */
struct _Schline
{
	Identification	identification;
	ObjectBounds	bounds;				/* bounds */
//	BYTE **			schline_line_state;	/* Indicate the state of the particular section */
	ObjectBounds **	schline_string;		/* Location of the specified label */
	ObjectBounds **	schline_line;		/* Location of the particular line */
	ObjectBounds **	schline_bitmap0;	/* Actual location of the diagram 0 */
	ObjectBounds **	schline_bitmap1;	/* Actual location of the diagram 1 */
	ObjectBounds **	schline_bitmap2;	/* Actual location of the diagram 2 */
	BYTE *			schline_bitmap_num;	/* The bitmap number or no bitmap in the specified region */
	BYTE **			schline_text;		/* The text in a particular line */
	BYTE			schline_text_highlight;		/* The current date of the scheduler screen */ 
	SHORT			schline_highlight_region;	/* Indicate the highlight region of the specified line */
	BitmapTemplate ** schline_bitmap_ptr;	/* Pointer to the bitmap structure of the particular diagram */
	BOOLEAN			schline_mode;		/* To indicate the Hours mode of the system (e.g. 0= 12 and 1= 24) */
	SchlineAttr		schline_attr;		/* Attribute */
};
typedef struct _Schline Schline;

/* ==================================================================== */
/* ========================= String Structures ======================= */
/* ==================================================================== */
/* Structure: String_Attr */
struct String_Attr
{
	BOOLEAN string_drawn;		/* indicate whether the string object
					   is drawn on the display or not */
	BOOLEAN string_visible;
};
typedef struct String_Attr StringAttr;

/* Structure: _String */
struct _String
{
	Identification identification;
	ObjectBounds bounds;		/* bounds */
	ObjectBounds screen_bounds;
	BYTE string_color;	    /* color of the font in the string */
							/* 0 = WHITE   1 = BLACK */
	BYTE string_bg_color;
	BYTE string_style;      /* 0 = without line, 1 = line as boundary,
							   2 = line as 3D boundary*/
	BYTE * string_text;		/* Pointer to the text */
	BYTE text_alignment;	/* 0 = Left, 1 = Centre, 2 = Right */
	BYTE text_font;			/* 0 = SMALL FONT, 1 = LARGE FONT */
	StringAttr string_attr;		/* attribute */
};
typedef struct _String String;

/* ==================================================================== */
/* ========================= Bitmap Structures ======================= */
/* ==================================================================== */
/* Structure: Bitmap_Attr */
struct Bitmap_Attr
{
	BOOLEAN bitmap_drawn;           /* indicate whether the object is drawn or not */
	BOOLEAN bitmap_enable;		/* indicate whether the bitmap can respond to
					   pen action or not */
	BOOLEAN bitmap_active;
	BOOLEAN bitmap_enter;
	BOOLEAN bitmap_visible;
};
typedef struct Bitmap_Attr BitmapAttr;

/* Structure: _Bitmap */
struct _Bitmap
{
	Identification identification;
	BitmapTemplate bitmap_bitmap1;  /* the displayed bitmap */
	BitmapTemplate bitmap_bitmap2;
	ObjectBounds bounds;            /* bounds */
	ObjectBounds screen_bounds;
	BYTE bitmap_style;          /* 0 = when the bitmap is clicked, the bitmap is inverted
								   1 = when the bitmap is clicked, another bitmap is pasted on it */
	BitmapAttr bitmap_attr;		/* bitmap_attr */
};
typedef struct  _Bitmap Bitmap;

/* ==================================================================== */
/* ========================= Line Structures ======================= */
/* ==================================================================== */
/* Structure: Line_Attr */
struct Line_Attr
{
	BOOLEAN line_drawn;		/* indicate whether line is drawn or not */
	BOOLEAN line_visible;
};
typedef struct Line_Attr LineAttr;

/* Structure: _Line */
struct _Line
{
	Identification identification;
	ObjectBounds bounds;            /* bounds */
	BYTE line_color;		/* color */
							/* 0 = white
							   1 = black */
	BYTE line_style;		/* 0 = dotted line
							   1 = non-dotted line */
	USHORT line_thick;		/* the thickness of the line
							   i.e. 1 = single line
								2 = double line */
	USHORT line_endpt_xpos;		/* x co-ordinate of the end point of the line */
	USHORT line_endpt_ypos;		/* y co-ordinate of the end point of the line */
	LineAttr line_attr;
};
typedef struct _Line Line;

/* ==================================================================== */
/* ========================= Clipboard Structures ======================= */
/* ==================================================================== */
/* Structure: _Clipboard */
struct _Clipboard
{

	BYTE clipboard_type;		/* 0 = (empty) 1 = text and 2 = bitmap */
	WORD clipboard_size;		/* number of bytes of the object */
	void * clipboard_data;		/* Pointer to the data */
};
typedef struct _Clipboard Clipboard;
/* ==================================================================== */
/* ================= Tree and LookUp Table Structures ================= */
/* ==================================================================== */
/* Structure: Link_List_Form */
struct Link_List_Form
{
	ObjectID form_id;
	Form * current_form;
	struct Link_List_Form * next;
};
typedef struct Link_List_Form LinkListForm;

/* Structure: LookUp_Table */
struct Lookup_Table
{
	ObjectID object_id;
	BYTE object_type;
	void * object_struct_add;
	struct Lookup_Table * next;
};
typedef struct Lookup_Table LookupTable;

/* ==================================================================== */
/* ========================= Keyboard Structures ======================= */
/* ==================================================================== */
/* Structure: Keyboard_Attr */
struct Keyboard_Attr
{
    BOOLEAN keyboard_drawn;
    BOOLEAN keyboard_enable;
    BOOLEAN keyboard_cap;
    BOOLEAN keyboard_shift;
	BOOLEAN keyboard_international;
	BOOLEAN keyboard_enter;
	BOOLEAN keyboard_save_behind;
	BOOLEAN keyboard_visible;
	BOOLEAN	keyboard_customised;
};
typedef struct Keyboard_Attr KeyboardAttr;

/* Structure: _Keyboard */
struct _Keyboard
{
    ObjectBounds   bounds;
    const ObjectBounds   *keyboard_keys_bounds;
    const SHORT *keyboard_bitmap_index;
    BitmapTemplate **keyboard_bitmap;
	SHORT keyboard_current_bitmap_index; /*starting from 0 */
	USHORT keyboard_num_keys;
	USHORT keyboard_num_keyboard_bitmap;
	UBYTE keyboard_highlight_key;
	BitmapTemplate save_behind;
    KeyboardAttr   keyboard_attr;
};
typedef struct _Keyboard Keyboard;

/* ==================================================================== */
/* ========================= Textbox Structures ======================= */
/* ==================================================================== */
/* Structure: Textbox_Attr */
struct Textbox_Attr
{
	BOOLEAN textbox_drawn;	/* indicate whether the textbox object is
							   drawn on the display or not */
	BOOLEAN textbox_active;	/* indicate whether the textbox object has the
							   focus or not */
    BOOLEAN textbox_enable;
    BOOLEAN textbox_dirty;
	BOOLEAN textbox_highlight;/* indicate whether the the text in the textbox object
							   is highlighted or not */
    BOOLEAN textbox_insert_pt_visible;
	BOOLEAN textbox_visible;
};
typedef struct Textbox_Attr TextboxAttr;

/* Structure: _textbox */
struct _Textbox
{
	Identification identification;
	ObjectBounds bounds;   	/* the bounds for this object */
	ObjectBounds screen_bounds;
	BYTE *textbox_string;		/* the text string in the textbox object */
    BYTE textbox_style;       /* 0 = no frame, 1= framed */
    BYTE textbox_back_line; /* 0 = no line, 1 = dotdot, 2 = grey line */
    BYTE textbox_font_id;
    BYTE textbox_font_color;
    BYTE textbox_background_color;
	WORD textbox_max_chars;	/* the max. no. of chars in the textbox object */
    WORD textbox_current_num_chars;
	WORD textbox_left_char_pos;
	WORD textbox_right_char_pos;
	WORD textbox_num_chars_displayed;
    SHORT textbox_insert_pt_x;
    SHORT textbox_insert_pt_y;
    WORD textbox_insert_pt_char_pos;
    BYTE textbox_insert_pt_movement;  /* 0 = NO movement
                                       1 = MOVE_UP
                                       2 = MOVE_DOWN
                                       3 = MOVE_LEFT
                                       4 = MOVE_RIGHT */
    WORD textbox_highlight_start_char;
    WORD textbox_highlight_end_char;
    WORD textbox_highlight_length;
    USHORT textbox_repeat_count;
	TextboxAttr textbox_attr;
};
typedef struct _Textbox Textbox;
#endif
