/*
================================ COPYRIGHT ================================
The contents of this file are subject to the VTech Informations Ltd. License
of VT-OS Ver. 1.1 operating system (the "License"); you may not use this 
file except in compliance with the License.  

Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the License 
for the specific language governing rights and limitations under the License.
  
The Original Code is VT-OS Ver. 1.1 Operating System, released 
on October 1st, 1999
	
The Initial Developer of the Original Code is VTech Informations Ltd.  All 
codes are Copyright (C) VTech Informations Ltd. 1999.  All Rights Reserved.
===========================================================================
*/

/*
===========================================================================
File        :   line.c
Author(s)   :   Henry Fok, David Lo
Company     :   VTech Informations Ltd.
Project     :   Helio 
Date:	    :   October 1st, 1999
Purpose:	:   Line object functions
Revision    :   1.1
Note        :   None
===========================================================================
*/              


#include "stdafx.h"

#include "uidef.h"
#include "uifunc.h"

/********************************************************
* Function:	LineDeleteLine
* Purpose:		to free the memory that is allocated to
the line object
* Scope:		Application
* Input:		line_id		DBID of the line
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     LineEraseLine function should be called before
calling LineDeleteLineFromMemory
*********************************************************/
Err LineDeleteLine(ObjectID line_id)
{
	Line *addr;
	BYTE object_type;
	Err Error;
	/* search the Lookup Table, if the DBID is found
	just free the pointer related to the DBID */
	if (UISearchForAddress(line_id,&object_type, (void**)&addr) == TRUE)
	{
		Error = UIDeleteLookupTableElement(line_id);
		if (Error !=TRUE) return Error;
		qfree(addr);
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}
/********************************************************
* Function:	LineDrawLine
* Purpose:		to draw the line object on the display
* Scope:		Application
* Input:		line_id		DBID of the line object
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err LineDrawLine(ObjectID line_id)
{
	Line *addr;
	BYTE object_type;
	if (UISearchForAddress(line_id,&object_type, (void**)&addr) == TRUE)
	{
		if (addr->line_attr.line_visible == FALSE) return FALSE;
		LcdDrawLine(addr->bounds.xcoord,addr->bounds.ycoord,
			addr->line_endpt_xpos,addr->line_endpt_ypos,
			(BYTE)(addr->line_thick), addr->line_style,
			addr->line_color, color_level[COLOR_WHITE_COLOR]);
		addr->line_attr.line_drawn = TRUE;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
	
}
/********************************************************
* Function:	LineGetAttribute
* Purpose:		to check whether the line is drawn on the
display or not
* Scope:		Application
* Input:		line_id		DBID of the line object
* Output:		line_drawn	Pointer to Boolean value
*				line_active	Pointer to Boolean value that shows 
*							whether it is active or not
*				line_visible Pointer to Boolean value that shows 
*							 whether it is active or not
* Return:  	TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err LineGetAttribute(ObjectID line_id,BOOLEAN *line_drawn, BOOLEAN *line_visible)
{
	Line *addr;
	BYTE object_type;
	/* to find out the pointer for the line object
	for an DBID, then return the line_drawn attribute */
	if (UISearchForAddress(line_id,&object_type,(void**)&addr) == TRUE)
	{
		*line_drawn = addr->line_attr.line_drawn;
		*line_visible = addr->line_attr.line_visible;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}
/*********************************************************************
* Function	: LineSetAttribute
* Purpose	: Used to set the attribute of the Line object
* Scope		: Application
* Input		: Line_id - DBID of the Line object
*			  att_drawn - Set the state of the Line_drawn attribute
*		      att_active - Set the state of the schine_active attribute
*			  attr_visible	- Set the state of the Line_visible attribute
* Output	: None
* Return	: Return True if no error or return ErrResNotFound if
*		  	  Invalid object ID
* Comment	: None
**********************************************************************/
Err LineSetAttribute(ObjectID line_id, BOOLEAN att_drawn, BOOLEAN att_visible)
{
	Line *addr;
	BYTE object_type;
	
	if (UISearchForAddress(line_id,&object_type,(void**)&addr) == TRUE)
	{
		addr->line_attr.line_drawn = att_drawn;
		addr->line_attr.line_visible = att_visible;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}

/********************************************************
* Function:	LineSetCharacteristics
* Purpose:		to set the style, thickness ... of the linecheck
* Scope:		Application
* Input:		line_id		DBID of the line object
line_color  color of the line
line_style  style of the line
line_thick  thickness of the line
* Output:		None
* Return:  	TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err LineSetLineCharacteristics(ObjectID line_id,BYTE line_color,
                               BYTE line_style, USHORT line_thick)
{
	Line *addr;
	BYTE object_type;
	/* to find out the pointer for the line object
	for an DBID, then return the line_drawn attribute */
	if (UISearchForAddress(line_id,&object_type,(void**)&addr) == TRUE)
	{
		addr->line_color = line_color;
		addr->line_style = line_style;
		addr->line_thick = line_thick;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}
/********************************************************
* Function:	LineGetCharacteristics
* Purpose:		to get the style, thickness ... of the linecheck
* Scope:		Application
* Input:		line_id		DBID of the line object
* Output:		line_color  Pointer to color of the line
line_style  Pointer to style of the line
line_thick  Pointer to thickness of the line
* Return:  	TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err LineGetLineCharacteristics(ObjectID line_id,BYTE *line_color,
                               BYTE *line_style, USHORT *line_thick)
{
	Line *addr;
	BYTE object_type;
	/* to find out the pointer for the line object
	for an DBID, then return the line_drawn attribute */
	if (UISearchForAddress(line_id,&object_type,(void**)&addr) == TRUE)
	{
		*line_color = addr->line_color;
		*line_style = addr->line_style;
		*line_thick = addr->line_thick;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}
/********************************************************
* Function:	LineGetPositions
* Purpose:		to get the startign point and the end point of the line
* Scope:		Application
* Input:		line_id		DBID of the line object
* Output:		start_x     Pointer to xcoord of starting point
start_y     Pointer to ycoord of starting point
end_x       Pointer to xcoord of end point
end_y       Pointer to ycoord of end point
* Return:  	TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err LineGetPosition(ObjectID line_id,SHORT *start_x, SHORT *start_y,
					SHORT *end_x, SHORT *end_y)
{
	Line *addr;
	BYTE object_type;
	/* to find out the pointer for the line object
	for an DBID, then return the line_drawn attribute */
	if (UISearchForAddress(line_id,&object_type,(void**)&addr) == TRUE)
	{
		*start_x = addr->bounds.xcoord;
		*start_y = addr->bounds.ycoord;
		*end_x   = addr->line_endpt_xpos;
		*end_y   = addr->line_endpt_ypos;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}
/********************************************************
* Function:	LineSetPositions
* Purpose:		to get the startign point and the end point of the line
* Scope:		Application
* Input:		line_id		DBID of the line object
start_x     xcoord of starting point
start_y     ycoord of starting point
end_x       xcoord of end point
end_y       ycoord of end point
* Output:	    None
* Return:  	TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err LineSetPosition(ObjectID line_id,SHORT start_x, SHORT start_y,
					SHORT end_x, SHORT end_y)
{
	Line *addr;
	BYTE object_type;
	/* to find out the pointer for the line object
	for an DBID, then return the line_drawn attribute */
	if (UISearchForAddress(line_id,&object_type,(void**)&addr) == TRUE)
	{
		addr->bounds.xcoord = start_x;
		addr->bounds.ycoord = start_y;
		addr->line_endpt_xpos = end_x;
		addr->line_endpt_ypos = end_y;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}

/********************************************************
* Function:	LineEraseLine
* Purpose:	to erase the specific line object on the
display
* Scope:	Application
* Input:	DBID of the line object
* Output:	None
* Return:      Error code
* Comment:     None
*********************************************************/
Err LineEraseLine(ObjectID line_id)
{
	Line *addr;
	BYTE object_type;
	if (UISearchForAddress(line_id,&object_type,(void**)&addr) == TRUE)
	{
		if (addr->line_attr.line_drawn)
			LcdDrawLine(addr->bounds.xcoord,addr->bounds.ycoord,
			addr->line_endpt_xpos,addr->line_endpt_ypos,
			(BYTE)(addr->line_thick),addr->line_style,
			0, 0);
		addr->line_attr.line_drawn = FALSE;
		return TRUE;
	}
	return ERR_UI_RES_NOT_FOUND;
}
/********************************************************
* Function:	LineInitLine
* Purpose:	1) to initialise the line object
2) to read the resource file
3) to create a corresponding line structure
in RAM
* Scope:	Application
* Input:	DBID of the line object
* Output:	None
* Return:      Error code
* Comment:     None
*********************************************************/
Err LineInitLine(ObjectID line_id)
{
	BYTE *buffer;
	UWORD byte_return;
	Line *addr;
	Err Error;
	BYTE object_type;
	
	object_type = LINE;
	/*put the pointer to the DBID lookup table */
	Error = UIAddressToLookupTable(line_id,object_type,(void**)&addr); 
	if (Error != TRUE) return Error;
	/*check whether the resource file is opened */
    Error = ResOpen(line_id);
    if (Error != TRUE) return Error;
	
	addr->identification.ui_object_id = line_id;
	/* get field 0 of resource file */
	ResGetField(line_id,0,&buffer,&byte_return);
	addr->identification.ui_object_type = *buffer;
    qfree(buffer);
	/* get field 1 of resource file */
	ResGetField(line_id,1,&buffer,&byte_return);
	addr->identification.table_related = *(ObjectID*)buffer;
    qfree(buffer);
	/* get field 2 from byte 0 for 2 bytes */
	ResReadField(line_id,2,0,2,&buffer,&byte_return);
	addr->bounds.xcoord = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(line_id,2,2,2,&buffer,&byte_return);
	addr->bounds.ycoord = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(line_id,2,4,2,&buffer,&byte_return);
	addr->bounds.width = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(line_id,2,6,2,&buffer,&byte_return);
	addr->bounds.height = *(SHORT*)buffer;
    qfree(buffer);
	/* get field 3 of resource file */
	ResGetField(line_id,3,&buffer,&byte_return);
	addr->line_color = UIColorConversion(*buffer);
    qfree(buffer);
	/* get field 4 of resource file */
	ResGetField(line_id,4,&buffer,&byte_return);
	addr->line_style = *buffer;
    qfree(buffer);
	
	/* get field 5 of resource file */
	ResGetField(line_id,5,&buffer,&byte_return);
	addr->line_thick = *(USHORT*)buffer;
    qfree(buffer);
	/* get field 6 of resource file */
	ResGetField(line_id,6,&buffer,&byte_return);
	addr->line_endpt_xpos = *(SHORT*)buffer;
    qfree(buffer);
	/* get field 7 of resource file */
	ResGetField(line_id,7,&buffer,&byte_return);
	addr->line_endpt_ypos = *(SHORT*)buffer;
    qfree(buffer);
	/* get field 8 from byte 0 for 2 bytes */
	ResGetField(line_id,8,&buffer,&byte_return);
	addr->line_attr.line_visible = *(BOOLEAN*)buffer;
    qfree(buffer);
	addr->line_attr.line_drawn  = FALSE;
	ResClose(line_id);
	return TRUE;
}
