/*
================================ COPYRIGHT ================================
The contents of this file are subject to the VTech Informations Ltd. License
of VT-OS Ver. 1.1 operating system (the "License"); you may not use this 
file except in compliance with the License.  

Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the License 
for the specific language governing rights and limitations under the License.
  
The Original Code is VT-OS Ver. 1.1 Operating System, released 
on October 1st, 1999
	
The Initial Developer of the Original Code is VTech Informations Ltd.  All 
codes are Copyright (C) VTech Informations Ltd. 1999.  All Rights Reserved.
===========================================================================
*/

/*
===========================================================================
File        :   schline.c
Author(s)   :   David Lo
Company     :   VTech Informations Ltd.
Project     :   Helio 
Date:	    :   October 1st, 1999
Purpose:	:   Schline object functions
Revision    :   1.1
Note        :   None
===========================================================================
*/              


#include "stdafx.h"

#include "uidef.h"
#include "uifunc.h"

/********************************************************
* Function:	    SchlineEraseSchline
* Purpose:		to erase the scheduler line on the display
* Scope:		Application
* Input:		schline_id		OBJECTID of the schline object
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err SchlineEraseSchline(ObjectID schline_id)
{
    Schline *addr;
	BYTE object_type;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) 
		return ERR_UI_RES_NOT_FOUND;
    if (addr->schline_attr.schline_drawn)
    {
		LcdEraseRegion(&(addr->bounds));
		addr->schline_attr.schline_drawn = FALSE;
    }
	return TRUE;
}
/********************************************************
* Function:	    SchlineDeleteSchline
* Purpose:		to delete the scheduler line in the memory
* Scope:		Application/internal
* Input:		schline_id		OBJECTID of the schline object
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err SchlineDeleteSchline(ObjectID schline_id)
{
    Schline *addr;
	
	BYTE object_type;
	WORD count;
	Err Error;
	
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) 
		return ERR_UI_RES_NOT_FOUND;
	Error = UIDeleteLookupTableElement(schline_id);
	if (Error !=TRUE) return Error;
	
	for (count =0; count<7; count++)
	{
        qfree(addr->schline_string[count]);
        qfree(addr->schline_line[count]);
        qfree(addr->schline_bitmap0[count]);
        qfree(addr->schline_bitmap1[count]);
        qfree(addr->schline_bitmap2[count]);
        qfree(addr->schline_text[count]);
	}    
    for(count =0; count<10; count++)    
        qfree(addr->schline_bitmap_ptr[count]);
	
    qfree(addr->schline_string);
    qfree(addr->schline_line);    
    qfree(addr->schline_bitmap0);
    qfree(addr->schline_bitmap1);
    qfree(addr->schline_bitmap2);
    qfree(addr->schline_bitmap_ptr);
    qfree(addr->schline_bitmap_num);
    qfree(addr->schline_text);
    qfree(addr);
	
	return TRUE;
}
/********************************************************
* Function:	    SchlineGetAttributes
* Purpose:		to get all the relative attributes in the 
schline structre
* Scope:		Application/internal
* Input:		schline_id			OBJECTID of the schline object
* Output:		drawn
*				visible
*				active
*				enable
*				enter
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err SchlineGetAttributes (ObjectID schline_id, BOOLEAN * drawn, BOOLEAN * visible,
						  BOOLEAN * active, BOOLEAN * enable, BOOLEAN * enter)
{
    Schline *addr;
    BYTE object_type;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) 
		return ERR_UI_RES_NOT_FOUND;
    *drawn = addr->schline_attr.schline_drawn;
    *visible = addr->schline_attr.schline_visible;
    *active = addr->schline_attr.schline_active;
    *enable = addr->schline_attr.schline_enable;
    *enter = addr->schline_attr.schline_enter;
    return TRUE;
}
/********************************************************
* Function:	    SchlineSetAttributes
* Purpose:		to get all the relative attributes in the 
schline structre
* Scope:		Application/internal
* Input:		schline_id		OBJECTID of the schline object
*				schline_drawn
*				schline_visible
*				schline_active
*				schline_enable
*				schline_enter
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err SchlineSetAttributes (ObjectID schline_id, BOOLEAN schline_drawn, BOOLEAN schline_visible,
						  BOOLEAN schline_active, BOOLEAN schline_enable, BOOLEAN schline_enter)
{
    Schline *addr;
    BYTE object_type;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) 
		return ERR_UI_RES_NOT_FOUND;
    addr->schline_attr.schline_drawn   = schline_drawn;
    addr->schline_attr.schline_visible = schline_visible;
    addr->schline_attr.schline_active  = schline_active;
    addr->schline_attr.schline_enable  = schline_enable;
	addr->schline_attr.schline_enter   = schline_enter;
    return TRUE;
}

/*********************************************************************
* Function	: SchlineSetDateText
* Purpose	: Set the text of a particular date
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
*			  date_num			the date_num of the date
*			  text				ptr of the text 	
* Output	: None
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
*			  ERR_UI_ITEM_NUMBER_EXCEED
* Comment	: None
**********************************************************************/
Err SchlineSetDateText (ObjectID schline_id, BYTE date_num, BYTE *date_text)
{
    Schline *addr;
    BYTE object_type;
	USHORT length;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	if (date_num >=7) return ERR_UI_ITEM_NUMBER_EXCEED;
	length = strlen(date_text); /*check the length of the passed-in label */
    addr->schline_text[date_num] = (BYTE *)qmalloc((length+1) * sizeof(BYTE));
	strcpy(addr->schline_text[date_num],date_text);
	return TRUE;
}

/*********************************************************************
* Function	: SchlineGetDateText
* Purpose	: Get the text of a particular date
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
*			  date_num			the date_num of the date
* Output	: date_text			pointer reference to the date text
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
*			  ERR_UI_ITEM_NUMBER_EXCEED	
* Comment	: None
**********************************************************************/
Err SchlineGetDateText (ObjectID schline_id, BYTE date_num, BYTE **date_text)
{
    Schline *addr;
    BYTE object_type;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	if (date_num >=7) return ERR_UI_ITEM_NUMBER_EXCEED;
	*date_text = addr->schline_text[date_num];
	return TRUE;
}

/*********************************************************************
* Function	: SchlineGetDateSchedule
* Purpose	: Get the schedules of a date
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object

  * Output	: date_state		pointer reference to array of BYTES to show the date states
  of all 7 days that are displaying on the schedule line
  * Return	: TRUE If no error
  *			  ERR_UI_RES_NOT_FOUND
  * Comment	: None
**********************************************************************/
/*Err SchlineGetDateSchedule (ObjectID schline_id, BYTE date_num, BYTE **date_state)
{
Schline *addr;
BYTE object_type;

  if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
  if (date_num >=7) return ERR_UI_ITEM_NUMBER_EXCEED;
  *date_state = addr->schline_line_state[date_num];
  return TRUE;
  }
*/
/*********************************************************************
* Function	: SchlineSetDateBitmaps
* Purpose	: Set the bitmaps of a date
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
*			  date_num			the date_num of the date
*			  date_bitmaps		pointer to 3 BYTES array to show what bitmaps should be put
*								for that date	
* Output	: None
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
*			  ERR_UI_ITEM_NUMBER_EXCEED
* Comment	: None
**********************************************************************/
Err SchlineSetDateBitmaps (ObjectID schline_id, BYTE date_num, BYTE *date_bitmaps)
{
    Schline *addr;
    BYTE object_type;
	WORD count;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	if (date_num >=7) return ERR_UI_ITEM_NUMBER_EXCEED;
	/* Set the bitmap number from right to left */
	count = 0;
	while (count < 3)
	{	
		addr->schline_bitmap_num[date_num + count*7] = date_bitmaps[count];
		count ++;
	}
	return TRUE;
}
/*********************************************************************
* Function	: SchlineGetDateBitmaps
* Purpose	: Get the bitmaps setting of a date
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
*			  date_num			the date_num of the date
* Output	: date_bitmaps		pointer reference to the bitmaps for the 7 days
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
*			  ERR_UI_ITEM_NUMBER_EXCEED
* Comment	: None
**********************************************************************/
Err SchlineGetDateBitmaps (ObjectID schline_id, BYTE date_num, BYTE *date_bitmaps)
{    
	Schline *addr;
    BYTE object_type;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	if (date_num >=7) return ERR_UI_ITEM_NUMBER_EXCEED;
	*date_bitmaps = addr->schline_bitmap_num[date_num];
	return TRUE;
}
/*********************************************************************
* Function	: SchlineGetClickedRegion
* Purpose	: 
* Scope		: Internal/Application
* Input		: addr			Pointer to the structure of the scroll bar 
*			  input_x       Input x_coordinate
*			  input_y		Input y_coordinate
* Output	: None
* Return	: Returns the clicked region of the Scheduler Line
*			  -1	  = Not Hitted.
*			  0 ~ 20  = A particular bitmap region.
*			  21 ~ 27 = A particular line. 	
* Comment	:
**********************************************************************/
SHORT SchlineGetClickedRegion(Schline *addr, SHORT input_x, SHORT input_y)
{
	SHORT	count;
	ObjectBounds	*bitmap;
	
	
	/* Check the Scheduler Line Region */
	for (count =1; count <8; count++)
	{
		bitmap = addr->schline_line[count-1];
		
		if ( input_x >= bitmap->xcoord && (input_x < bitmap->xcoord + addr->bounds.width - 1) && 
			(input_y >= bitmap->ycoord - 12) && (input_y < bitmap->ycoord + 1) )
			return count;
	}	
	return SCHLINE_NOT_CLICKED;
}
/*********************************************************************
* Function	: SchlineDrawSchline
* Purpose	: Draw the specified scheduler's screen 
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
* Output	: None
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
* Comment	: None
**********************************************************************/
Err SchlineDrawSchline(ObjectID Schline_id)
{
	Schline		*addr;
	BYTE		object_type, picture_num;
	BitmapTemplate header, picture;
	ObjectBounds line_bounds, text_bounds;
	USHORT		mode;
	WORD		count1, count2;
	
	if (UISearchForAddress(Schline_id,&object_type,(void**)&addr) != TRUE)
		return ERR_UI_RES_NOT_FOUND;
	
	if (addr->schline_attr.schline_visible != TRUE)
		return FALSE;
	
	line_bounds = addr->bounds;
	line_bounds.width = 0;
	
	if (addr->schline_mode == SCHLINE_MODE_24)
		mode = SCHLINE_24_BG_NUM;
	else
		mode = SCHLINE_12_BG_NUM;
	
	/* Draw the Background Line */
	count1 = mode;
	count2 = 0;
	while (count1)
	{
		line_bounds.xcoord += line_bounds.width;
		
		if (addr->schline_mode == SCHLINE_MODE_24)
			line_bounds.width  = SCHLINE_24_BG_WIDTH;
		else
			line_bounds.width  = SCHLINE_12_BG_WIDTH;
		
		if (count2%2 == 0 && LcdGetColorMode() == GREYSCALE_MODE)
			LcdDrawBox(&line_bounds,color_level[COLOR_GREY1_COLOR],color_level[COLOR_GREY2_COLOR],FILL_SOLID);
		else 
			LcdDrawBox(&line_bounds,color_level[COLOR_WHITE_COLOR],color_level[COLOR_WHITE_COLOR],FILL_SOLID);
		
		count1--;
		count2++;
	}
	/* Draw the Top header label */
	if (addr->schline_mode == SCHLINE_MODE_24)
		header.bitmap_data = schline_header2;
	else
		header.bitmap_data = schline_header1;
	
	line_bounds = addr->bounds;	
	header.xcoord = line_bounds.xcoord;
	header.ycoord = line_bounds.ycoord;
	header.width  = line_bounds.width;
	header.height = 12;
	header.size	  = 0;
    header.quantisation = Q_FOUR_BIT;
	header.compressed = FALSE;
	LcdDrawBitmap(&header,NOT_INVERT);
	
	line_bounds = addr->bounds;
	line_bounds.xcoord += 1;
	line_bounds.ycoord += 12; /* Top Line */
	/* Draw normal line */
	LcdDrawLine(line_bounds.xcoord,line_bounds.ycoord,
		line_bounds.xcoord+line_bounds.width-2,line_bounds.ycoord,
		SINGLE_LINE,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
	
	count1 = 7;
	while (count1)
	{
		count1 --;
		line_bounds.ycoord += 14;
		LcdDrawLine(line_bounds.xcoord,line_bounds.ycoord,
			line_bounds.xcoord+line_bounds.width-2,line_bounds.ycoord,
			SINGLE_LINE,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
	}
	
	/* Draw day */
	count1 = 7;
	while (count1)
	{
		count1--;
		text_bounds = *(addr->schline_string[count1]);
		
#ifdef	PC_SIM
		if (count1 != addr->schline_text_highlight)
			LcdDrawFixedString(&text_bounds, addr->schline_text[count1], color_level[COLOR_BLACK_COLOR], color_level[COLOR_WHITE_COLOR], SMALL_FONT, DOTDOT, MARGIN_0);
		else
			LcdDrawFixedString(&text_bounds,addr->schline_text[count1],color_level[COLOR_WHITE_COLOR],color_level[COLOR_BLACK_COLOR],SMALL_FONT,DOTDOT,MARGIN_0);
#else
		if (count1 != addr->schline_text_highlight)
			LcdDrawFixedStringS(&text_bounds, addr->schline_text[count1], color_level[COLOR_BLACK_COLOR], color_level[COLOR_WHITE_COLOR], SMALL_FONT, DOTDOT, MARGIN_0);
		else
			LcdDrawFixedString(&text_bounds,addr->schline_text[count1],color_level[COLOR_WHITE_COLOR],color_level[COLOR_BLACK_COLOR],SMALL_FONT,DOTDOT,MARGIN_0);
#endif
	}
	/* Draw the related bitmap on the scheduler line */
	count1 = 3;
	while (count1)
	{
		count1--;
		count2 = 7;
		while (count2)
		{
			count2--;
			if (addr->schline_bitmap_num[count2 + (7*count1)] != -1)
			{
				if (count1 == 0)
				{
					picture.xcoord = ((ObjectBounds*)(addr->schline_bitmap0[count2]))->xcoord;
					picture.ycoord = ((ObjectBounds*)(addr->schline_bitmap0[count2]))->ycoord;
					picture_num = addr->schline_bitmap_num[count2];
					picture.width = ((BitmapTemplate*)(addr->schline_bitmap_ptr[picture_num]))->width;
					picture.height = addr->schline_bitmap_ptr[picture_num]->height;
					((ObjectBounds*)(addr->schline_bitmap0[count2]))->width = picture.width;
					((ObjectBounds*)(addr->schline_bitmap0[count2]))->height = picture.height;
				}
				else if (count1 == 1)
				{
					picture.xcoord = ((ObjectBounds*)(addr->schline_bitmap1[count2]))->xcoord;
					picture.ycoord = ((ObjectBounds*)(addr->schline_bitmap1[count2]))->ycoord;
					picture_num = addr->schline_bitmap_num[count2+7];
					picture.width = ((BitmapTemplate*)(addr->schline_bitmap_ptr[picture_num]))->width;
					picture.height = addr->schline_bitmap_ptr[picture_num]->height;
					((ObjectBounds*)(addr->schline_bitmap1[count2]))->width = picture.width;
					((ObjectBounds*)(addr->schline_bitmap1[count2]))->height = picture.height;
				}
				else
				{
					picture.xcoord = ((ObjectBounds*)(addr->schline_bitmap2[count2]))->xcoord;
					picture.ycoord = ((ObjectBounds*)(addr->schline_bitmap2[count2]))->ycoord;
					picture_num = addr->schline_bitmap_num[count2+14];
					picture.width = ((BitmapTemplate*)(addr->schline_bitmap_ptr[picture_num]))->width;
					picture.height = addr->schline_bitmap_ptr[picture_num]->height;
					((ObjectBounds*)(addr->schline_bitmap2[count2]))->width = picture.width;
					((ObjectBounds*)(addr->schline_bitmap2[count2]))->width;
					((ObjectBounds*)(addr->schline_bitmap2[count2]))->height = picture.height;	
					((ObjectBounds*)(addr->schline_bitmap2[count2]))->height;
				}
				/* Draw picture */
				picture.size = addr->schline_bitmap_ptr[picture_num]->size;
				picture.bitmap_data = addr->schline_bitmap_ptr[picture_num]->bitmap_data;
				
				if ((count2 + (7*count1) == addr->schline_highlight_region) && addr->schline_attr.schline_enter == TRUE)
					LcdDrawBitmap(&picture,INVERT);
				else
					LcdDrawBitmap(&picture,NOT_INVERT);
			}
		}
	}
	addr->schline_attr.schline_drawn = TRUE;
	return TRUE;
}
/********************************************************
* Function:	SchlineInitSchline
* Purpose:	1)	to initialise the schline object
*			2)	to read the resource file
*			3)	to create a corresponding schline structure
*				in RAM
* Scope:	Application
* Input:	DBID of the schline object
* Output:	None
* Return:   Error code
* Comment:  None
*********************************************************/
Err SchlineInitSchline(ObjectID schline_id)
{
	BYTE *buffer, object_type;
	UWORD byte_return;
	Schline *addr;
	Err Error;
	WORD count;
	BYTE **text_list;
	
	
	object_type = SCHLINE;
	/*put the pointer to the DBID lookup table */
	Error = UIAddressToLookupTable(schline_id,object_type,(void**)&addr); 
	if (Error != TRUE) return Error;
	/*check whether the resource file is opened */
    Error = ResOpen(schline_id);
    if (Error != TRUE) return Error;
	
	addr->identification.ui_object_id = schline_id;
	/* get field 0 of resource file */
	ResGetField(schline_id,0,&buffer,&byte_return);
	addr->identification.ui_object_type = *buffer;
    qfree(buffer);
	/* get field 1 from byte 0 for 2 bytes */
	ResReadField(schline_id,1,0,2,&buffer,&byte_return);
	addr->bounds.xcoord = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(schline_id,1,2,2,&buffer,&byte_return);
	addr->bounds.ycoord = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(schline_id,1,4,2,&buffer,&byte_return);
	addr->bounds.width = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(schline_id,1,6,2,&buffer,&byte_return);
	addr->bounds.height = *(SHORT*)buffer;
    qfree(buffer);
	/* get field 2 of resource file */
	ResReadField(schline_id,2,0,2,&buffer,&byte_return);
	addr->schline_attr.schline_enable = *(BOOLEAN*)buffer;
    qfree(buffer);
	ResReadField(schline_id,2,2,2,&buffer,&byte_return);
	addr->schline_attr.schline_active = *(BOOLEAN*)buffer;
    qfree(buffer);
	ResReadField(schline_id,2,4,2,&buffer,&byte_return);
	addr->schline_attr.schline_visible = *(BOOLEAN*)buffer;
    qfree(buffer);
	addr->schline_attr.schline_drawn = FALSE;
	addr->schline_attr.schline_enter = FALSE;
	/* Set the Hour Settings */
	addr->schline_mode = SCHLINE_MODE_24;
	ResClose(schline_id);
	/* Define the location of the text labels */
    addr->schline_string = (ObjectBounds **)qmalloc(7*sizeof(ObjectBounds *));
    
	count = 7;
	while (count)
	{
		count--;
        addr->schline_string[count] = (ObjectBounds *)qmalloc(sizeof(ObjectBounds));
		((ObjectBounds*)(addr->schline_string[count]))->xcoord = 5;
		((ObjectBounds*)(addr->schline_string[count]))->ycoord = 45+(count * 14);
		((ObjectBounds*)(addr->schline_string[count]))->width  = 34; /* Take 34 as worst case */
		((ObjectBounds*)(addr->schline_string[count]))->height = 8;
	}
	/* Define the location of the lines */
    addr->schline_line = (ObjectBounds **)qmalloc(7*sizeof(ObjectBounds *));
    
	count = 7;
	while (count)
	{
		count--;
        addr->schline_line[count] = (ObjectBounds *)qmalloc(sizeof(ObjectBounds));
		((ObjectBounds*)(addr->schline_line[count]))->xcoord = 3;
		((ObjectBounds*)(addr->schline_line[count]))->ycoord = 56+(count * 14);
		((ObjectBounds*)(addr->schline_line[count]))->width  = 0;
		((ObjectBounds*)(addr->schline_line[count]))->height = 0;
	}
	/* Define the location of the bitmap diagram 0 */
    addr->schline_bitmap0 = (ObjectBounds **)qmalloc(7*sizeof(ObjectBounds *));
    
	count = 7;
	while (count)
	{
		count--;
        addr->schline_bitmap0[count] = (ObjectBounds *)qmalloc(sizeof(ObjectBounds));
		((ObjectBounds*)(addr->schline_bitmap0[count]))->xcoord = 147;
		((ObjectBounds*)(addr->schline_bitmap0[count]))->ycoord = 44+(count * 14);	
	}
	/* Define the location of the bitmap diagram 1 */
    addr->schline_bitmap1 = (ObjectBounds **)qmalloc(7*sizeof(ObjectBounds *));
    
	count = 7;
	while (count)
	{
		count--;
        addr->schline_bitmap1[count] = (ObjectBounds *)qmalloc(sizeof(ObjectBounds));
		((ObjectBounds*)(addr->schline_bitmap1[count]))->xcoord = 138;
		((ObjectBounds*)(addr->schline_bitmap1[count]))->ycoord = 44+(count * 14);
	}
	/* Define the location of the bitmap diagram 2 */
    addr->schline_bitmap2 = (ObjectBounds **)qmalloc(7*sizeof(ObjectBounds *));
    
	count = 7;
	while (count)
	{
		count--;
        addr->schline_bitmap2[count] = (ObjectBounds *)qmalloc(sizeof(ObjectBounds));
		((ObjectBounds*)(addr->schline_bitmap2[count]))->xcoord = 126;
		((ObjectBounds*)(addr->schline_bitmap2[count]))->ycoord = 44+(count * 14);
	}
	/* Define the bitmap pointer*/
    addr->schline_bitmap_ptr = (BitmapTemplate **)qmalloc(10*sizeof(BitmapTemplate *));
    
	count = 10;
	while (count)
	{	
		count--;
        addr->schline_bitmap_ptr[count] = (BitmapTemplate *)qmalloc(sizeof(BitmapTemplate));
        addr->schline_bitmap_ptr[count]->quantisation = Q_FOUR_BIT;
	}
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[0]))->width  = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[0]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[0]))->bitmap_data = sch24_d0;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[1]))->width  = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[1]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[1]))->bitmap_data = sch24_d1;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[2]))->width  = 10;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[2]))->height = 11;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[2]))->bitmap_data = sch24_d2;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[3]))->width  = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[3]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[3]))->bitmap_data = sch24_d3;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[4]))->width  = 11;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[4]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[4]))->bitmap_data = sch24_d4;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[5]))->width  = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[5]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[5]))->bitmap_data = sch12_d0;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[6]))->width  = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[6]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[6]))->bitmap_data = sch12_d1;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[7]))->width  = 10;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[7]))->height = 11;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[7]))->bitmap_data = sch12_d2;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[8]))->width  = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[8]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[8]))->bitmap_data = sch12_d3;
	
	((BitmapTemplate*)(addr->schline_bitmap_ptr[9]))->width  = 11;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[9]))->height = 9;
	((BitmapTemplate*)(addr->schline_bitmap_ptr[9]))->bitmap_data = sch12_d4;
	/* Init the bitmap number of the particular locations */
    addr->schline_bitmap_num = (BYTE *)qmalloc(21 * sizeof(BYTE));
	count = 21;
	while (count)
	{
		count--;
		addr->schline_bitmap_num[count] = SCHLINE_NOT_CLICKED;
	}
	/* Init the text of the particular locations */
    text_list = (BYTE**)qmalloc(7 * sizeof(BYTE*));
	for (count =0; count <7; count++)
	{
        text_list[count] = (BYTE*)qmalloc(sizeof(BYTE)*10);
		strcpy(text_list[count], "");
	}
	addr->schline_text = text_list;
	return TRUE;
}

/*********************************************************************
* Function	: SchlineSetLineState
* Purpose	: Draw the specified scheduler's screen 
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
* Output	: None
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
* Comment	: line_num			1 ~ 7
*			  section_num		1 ~ 16 (24 Hrs mode) / 1 ~ 20 (12 Hrs mode)
**********************************************************************/
Err SchlineSetLineState(ObjectID schline_id, SHORT line_num, SHORT section_num)
{
	Schline			*schline_addr;
	ObjectBounds	line_bounds;
	BYTE			object_type;
	SHORT			count;
	
	if (UISearchForAddress(schline_id, &object_type, (void**)&schline_addr) != TRUE)
		return ERR_UI_RES_NOT_FOUND;
	
	if (((schline_addr->schline_mode == SCHLINE_MODE_24 && section_num > SCHLINE_24_NUM_REGION) ||
		(schline_addr->schline_mode == SCHLINE_MODE_12 && section_num > SCHLINE_12_NUM_REGION)) ||
		line_num > 7)
		return ERR_UI_INV_INPUT_VALUE;
	
	line_bounds = *(schline_addr->schline_line[line_num-1]);
	
	if (schline_addr->schline_mode == SCHLINE_MODE_24)
	{
		for (count=0; count<(section_num-1); count++)
			line_bounds.xcoord += SCHLINE_24_BG_WIDTH;
		
		/* Not equal to First/Last Sector */
		if (count == 0 || count == SCHLINE_24_NUM_REGION-1)
			LcdDrawLine(line_bounds.xcoord+1,line_bounds.ycoord,line_bounds.xcoord+SCHLINE_24_BG_WIDTH*2-2,
			line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
		else
			LcdDrawLine(line_bounds.xcoord+1,line_bounds.ycoord,line_bounds.xcoord+SCHLINE_24_BG_WIDTH-2,
			line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
	}
	else
	{
		for (count =0; count<(section_num-1); count++)
			line_bounds.xcoord += SCHLINE_12_BG_WIDTH;
		
		if (count == 0)
			LcdDrawLine(line_bounds.xcoord+1,line_bounds.ycoord,line_bounds.xcoord+SCHLINE_12_BG_WIDTH*3-2,
			line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
		else if (count == SCHLINE_12_NUM_REGION-1)
			LcdDrawLine(line_bounds.xcoord+1,line_bounds.ycoord,line_bounds.xcoord+SCHLINE_12_BG_WIDTH*2-2,
			line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
		else
			LcdDrawLine(line_bounds.xcoord+1,line_bounds.ycoord,line_bounds.xcoord+SCHLINE_12_BG_WIDTH-2,
			line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
	}
	return TRUE;
}

/********************************************************
* Function:	SchlineSetLineLabel
* Purpose:		to set the schline label of the specified
*				schline object
* Scope:		Application
* Input:		schlnie_id		DBID of schline object
*				schline_label	the label of schline object
* Output:		None
* Return:		Error code
* Comment	: line_num			1 ~ 7
*********************************************************/
Err SchlineSetLineLabel(ObjectID schline_id, SHORT line_num, BYTE *schline_label)
{
	Schline *schline_addr;
	BYTE object_type;
	USHORT length;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&schline_addr) != TRUE) 
		return ERR_UI_RES_NOT_FOUND;
	
	if (line_num > 7)
		return ERR_UI_INV_INPUT_VALUE;
	
	length = strlen(schline_label);
    qfree(schline_addr->schline_text[line_num-1]);
    schline_addr->schline_text[line_num-1] = (BYTE *)qmalloc((length+1) * sizeof(BYTE));
	strcpy(schline_addr->schline_text[line_num-1], schline_label);
	return TRUE;
}

/*********************************************************************
* Function	: SchlineSetLineBitmap
* Purpose	: Draw the specified scheduler's screen 
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
* Output	: None
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
* Comment	: line_num			1 ~ 7
**********************************************************************/
Err SchlineSetLineBitmap(ObjectID schline_id, SHORT line_num, SHORT section_num, BYTE bitmap_num)
{
	Schline *schline_addr;
	BYTE object_type;
	
	if (UISearchForAddress(schline_id, &object_type, (void**)&schline_addr) != TRUE)
		return ERR_UI_RES_NOT_FOUND;
	
	if (((schline_addr->schline_mode == SCHLINE_MODE_24 && section_num > SCHLINE_24_NUM_REGION) ||
		(schline_addr->schline_mode == SCHLINE_MODE_12 && section_num > SCHLINE_12_NUM_REGION)) ||
		line_num > 7)
		return ERR_UI_INV_INPUT_VALUE;
	
	schline_addr->schline_bitmap_num[(line_num-1)+(7*(section_num-1))] = bitmap_num;
	
	return TRUE;
}

/********************************************************
* Function:	SchlineSetHighlightText
* Purpose:	to highlight the schline label of the specified
*			schline object
* Scope:	Application
* Input:	schlnie_id			DBID of schline object
* Output:	None
* Return:	Error code
* Comment:	line_num			1 ~ 7
*********************************************************/
Err SchlineSetHighlightText(ObjectID schline_id, SHORT line_num)
{
	Schline	*schline_addr;
	BYTE	object_type;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&schline_addr) != TRUE) 
		return ERR_UI_RES_NOT_FOUND;
	
	if (line_num > 7)
		return ERR_UI_INV_INPUT_VALUE;
	
	schline_addr->schline_text_highlight = line_num-1;
	return TRUE;
}

/********************************************************
* Function:	SchlineSetHighlightText
* Purpose:	to highlight the schline label of the specified
*			schline object
* Scope:	Application
* Input:	schlnie_id			DBID of schline object
* Output:	None
* Return:	Error code
* Comment:	line_num			1 ~ 7
*********************************************************/
Err SchlineSetHourSettings(ObjectID schline_id, BOOLEAN hour_mode)
{
	Schline	*schline_addr;
	BYTE	object_type;
	
	if (UISearchForAddress(schline_id,&object_type,(void**)&schline_addr) != TRUE) 
		return ERR_UI_RES_NOT_FOUND;
	
	schline_addr->schline_mode = hour_mode;
	return TRUE;
}

/*********************************************************************
* Function	: SchlineDrawContinueSector
* Purpose	: Draw the specified scheduler's screen
* Scope		: Application
* Input		: Schline_id		DBID of the scroll bar object
* Output	: None
* Return	: TRUE If no error
* Comment	: line_num			1 ~ 7
**********************************************************************/
Err SchlineDrawSmallSector(Schline *schline_addr, SHORT line_num, SHORT section_num)
{
	ObjectBounds line_bounds;
	WORD count;
	
	if (((schline_addr->schline_mode == SCHLINE_MODE_24 && section_num > SCHLINE_24_NUM_REGION) ||
		(schline_addr->schline_mode == SCHLINE_MODE_12 && section_num > SCHLINE_12_NUM_REGION)) ||
		line_num > 7)
		return ERR_UI_INV_INPUT_VALUE;
	
	line_bounds = *(schline_addr->schline_line[line_num-1]);
	
	if (schline_addr->schline_mode == SCHLINE_MODE_24)
		for (count =0; count<section_num-1; count++)
			line_bounds.xcoord += SCHLINE_24_BG_WIDTH;
		else
			for (count =0; count<section_num-1; count++)
				line_bounds.xcoord += SCHLINE_12_BG_WIDTH;
			
			if (section_num == 1)
			{
				LcdDrawLine(line_bounds.xcoord-1,line_bounds.ycoord,line_bounds.xcoord,line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
				LcdDrawLine(schline_addr->bounds.xcoord+schline_addr->bounds.width-2,line_bounds.ycoord,schline_addr->bounds.xcoord+schline_addr->bounds.width-1,
					line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
			}	
			else
				LcdDrawLine(line_bounds.xcoord-1,line_bounds.ycoord,line_bounds.xcoord,line_bounds.ycoord,3,NON_DOTTED_LINE,color_level[COLOR_BLACK_COLOR],color_level[COLOR_BLACK_COLOR]);
			
			return TRUE;
}
