/*
================================ COPYRIGHT ================================
The contents of this file are subject to the VTech Informations Ltd. License
of VT-OS Ver. 1.1 operating system (the "License"); you may not use this 
file except in compliance with the License.  

Software distributed under the License is distributed on an "AS IS" basis, 
WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the License 
for the specific language governing rights and limitations under the License.
  
The Original Code is VT-OS Ver. 1.1 Operating System, released 
on October 1st, 1999
	
The Initial Developer of the Original Code is VTech Informations Ltd.  All 
codes are Copyright (C) VTech Informations Ltd. 1999.  All Rights Reserved.
===========================================================================
*/

/*
===========================================================================
File        :   scroll.c
Author(s)   :   Henry Fok, David Lo
Company     :   VTech Informations Ltd.
Project     :   Helio 
Date:	    :   October 1st, 1999
Purpose:	:   Scrollbar object functions
Revision    :   1.1
Note        :   None
===========================================================================
*/              

#include "stdafx.h"

#include "uidef.h"
#include "uifunc.h"

//#define DEBUG
//#define DEBUG_SCROLL

/********************************************************
* Function:	    ScrollbarEraseScrollbar
* Purpose:		to erase the scrollbar on the display
* Scope:		Application
* Input:		scrollbar_id		OBJECTID of the scrollbar object
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err ScrollbarEraseScrollbar(ObjectID scrollbar_id)
{
    Scrollbar *addr;
	BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	//    if (addr->scrollbar_attr.scrollbar_drawn)
	//    {
	LcdEraseRegion(&(addr->bounds));
	addr->scrollbar_attr.scrollbar_drawn = FALSE;
	//    }
	return TRUE;
}
/********************************************************
* Function:	    ScrollbarDeleteScrollbar
* Purpose:		to delete the scrollbar int he memory
* Scope:		Application/internal
* Input:		scrollbar_id		OBJECTID of the scrollbar object
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err ScrollbarDeleteScrollbar(ObjectID scrollbar_id)
{
    Scrollbar *addr;
	BYTE object_type;
	Err Error;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	if (addr->scrollbar_text)
        qfree(addr->scrollbar_text);
	Error = UIDeleteLookupTableElement(scrollbar_id);
	if (Error !=TRUE) return Error;
    qfree(addr);
	return TRUE;
}
/********************************************************
* Function:	    ScrollbarGetScrollbar
* Purpose:		to get all the related values in the scrollbar
structre
* Scope:		Application/internal
* Input:		scrollbar_id		OBJECTID of the scrollbar object
* Output:		value               Pointer to current value of scroll car
max_value           Pointer to value of the max value of
scroll car
min_value           Pointer to value of the min value of
scroll car
pagesize            Pointer to the value of pagesize
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err ScrollbarGetScrollbar(ObjectID scrollbar_id, WORD *value,
                          WORD *max_value, WORD *min_value,
                          WORD *pagesize, WORD *total_num_lines)
{
    Scrollbar *addr;
    BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
    *value = addr->scrollbar_value;
    *max_value = addr->scrollbar_max;
    *min_value = addr->scrollbar_min;
    *pagesize = addr->scrollbar_pagesize;
	*total_num_lines = addr->scrollbar_total_num_lines;
    return TRUE;
}
/********************************************************
* Function:	    ScrollbarSetScrollbar
* Purpose:		to get all the related values in the scrollbar
structre
* Scope:		Application/internal
* Input:		scrollbar_id		OBJECTID of the scrollbar object
scrollbar_value     current value of scroll car
scrollbar_max_value max value of scroll car
scrollbar_min_value min value of scroll car
scrollbar_pagesize  value of pagesize
scrollbar_total_num_lines	the total number fo lines
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err ScrollbarSetScrollbar(ObjectID scrollbar_id, WORD scrollbar_value,
                          WORD scrollbar_max_value, WORD scrollbar_min_value,
                          WORD scrollbar_pagesize,WORD scrollbar_total_num_lines)
{
    Scrollbar *addr;
    BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
    addr->scrollbar_value = scrollbar_value;
    addr->scrollbar_max = scrollbar_max_value;
    addr->scrollbar_min = scrollbar_min_value;
    addr->scrollbar_pagesize = scrollbar_pagesize;
	addr->scrollbar_total_num_lines = scrollbar_total_num_lines;
    return TRUE;
}
/********************************************************
* Function:	    ScrollbarSetScrollbarType
* Purpose:		to set the type of the scrollbar
* Scope:		Application/internal
* Input:		scrollbar_id		OBJECTID of the scrollbar object
scrollbar_type      type of the scrollbar
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:      None
*********************************************************/
Err ScrollbarSetScrollbarType(ObjectID scrollbar_id, BYTE scrollbar_type)
{
    Scrollbar *addr;
    BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
    addr->scrollbar_type = scrollbar_type;
    return TRUE;
}

/********************************************************
* Function:	    ScrollbarSetScrollbarVisible
* Purpose:		to set the attribute visible of scrollbar
* Scope:		Application/internal
* Input:		scrollbar_id		OBJECTID of the scrollbar object
visible
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:      None
*********************************************************/
Err ScrollbarSetScrollbarVisible(ObjectID scrollbar_id, BOOLEAN visible)
{
    Scrollbar *addr;
    BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
    addr->scrollbar_attr.scrollbar_visible = visible;
    return TRUE;
}

/********************************************************
* Function:	    ScrollbarGetScrollbarVisible
* Purpose:		to set the attribute visible of scrollbar
* Scope:		Application/internal
* Input:		scrollbar_id		OBJECTID of the scrollbar object
* Output:		visible
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:      None
*********************************************************/
Err ScrollbarGetScrollbarVisible(ObjectID scrollbar_id, BOOLEAN *visible)
{
    Scrollbar *addr;
    BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
    *visible = addr->scrollbar_attr.scrollbar_visible;
    return TRUE;
}

/********************************************************
* Function:	    ScrollbarSetScrollbarDrawPagesize
* Purpose:		to set draw pagesize of scrollbar
* Scope:		Application/internal
* Input:		scrollbar_id		OBJECTID of the scrollbar object
draw_pagesize
* Output:		None
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:      None
*********************************************************/
Err ScrollbarSetScrollbarDrawPagesize(ObjectID scrollbar_id, SHORT draw_pagesize)
{
    Scrollbar *addr;
    BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
    addr->scrollbar_draw_pagesize = draw_pagesize;
    return TRUE;
}

/*********************************************************************
* Function	: ScrollbarGetClickedRegion
* Purpose	:
* Scope		: Internal/Application
* Input		: addr			Pointer to the structure of the scroll bar
*			  input_x       Input x_coordinate
*			  input_y		Input y_coordinate
* Output	: None
* Return	: Returns the clicked region of the scrollbar or null for not hitted
* Comment	:
**********************************************************************/
BYTE ScrollbarGetClickedRegion(Scrollbar *addr,SHORT input_x,SHORT input_y)
{
	WORD middle_start,middle_end,car_start,car_end,car_size,car_value;
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_0)
	{
		if (addr->scrollbar_style == VERTICAL)
		{
			if ( (addr->scrollbar_clicked_region != SCROLLCAR_REGION) && !(input_x >= addr->bounds.xcoord &&
				input_x < addr->bounds.xcoord+addr->bounds.width && input_y >= addr->bounds.ycoord &&
				input_y < addr->bounds.ycoord+addr->bounds.height) )
				return SCROLL_NOT_HITTED;
			
			if ( (addr->scrollbar_clicked_region == SCROLLCAR_REGION) && !(input_x >= addr->bounds.xcoord-SCROLL_EXT_WIDTH+1 &&
				input_x < addr->bounds.xcoord + addr->bounds.width + SCROLL_EXT_WIDTH - 1 && input_y >= addr->bounds.ycoord &&
				input_y < addr->bounds.ycoord+addr->bounds.height) )
				return SCROLL_NOT_HITTED;
			
			middle_start = addr->bounds.ycoord + addr->scrollbar_arrow1.height + SCROLL_SEPARATOR;
			middle_end = (addr->bounds.ycoord + addr->bounds.height-1) - (addr->scrollbar_arrow2.height + SCROLL_SEPARATOR);
			
			/* Test if the scroll car is at the bottom of the bar */
			if (addr->scrollbar_value > (addr->scrollbar_total_num_lines - addr->scrollbar_draw_pagesize))
				car_value = (addr->scrollbar_total_num_lines - addr->scrollbar_draw_pagesize);
			else
				car_value = addr->scrollbar_value;
			
			car_start = middle_start + (car_value * (addr->bounds.height - (addr->scrollbar_arrow1.height + SCROLL_SEPARATOR)*2)
				/ addr->scrollbar_total_num_lines);
			
            car_size = (addr->scrollbar_draw_pagesize) * (addr->bounds.height - (addr->scrollbar_arrow1.height + SCROLL_SEPARATOR - 1)*2)/
				addr->scrollbar_total_num_lines;
			car_end = car_start + car_size - 1;
			
			if (car_size <= SCROLL_MIN_VALUE)
			{
				car_size = SCROLL_MIN_VALUE;
                car_end = car_start + car_size - 1;
                if (car_end > addr->scrollbar_arrow2.ycoord)
				{
					car_start = addr->scrollbar_arrow2.ycoord - car_size;
                    car_end = car_start + car_size - 1;
                }
			}
			
			if (input_x >= addr->bounds.xcoord && input_x < addr->bounds.xcoord+addr->scrollbar_arrow1.width)
			{
				/* Clicked on the upper arrow */
				if (input_y >= addr->bounds.ycoord && input_y < addr->bounds.ycoord+addr->scrollbar_arrow1.height)
					return SCROLL_UP_ARROW;
				
				/* Clicked at the top of scroll car area */
				if (input_y >= middle_start && input_y < car_start)
					return SCROLLBAR_UP_REGION;
				
				/* Clicked on the scroll car area */
				if (input_y >= car_start && input_y < car_end + 1)
					return SCROLLCAR_REGION;
				
				/* Clicked at the bottom of scroll car area */
				if (input_y > car_end && input_y <= middle_end)
					return SCROLLBAR_DOWN_REGION;
				/* Clicked on the lower arrow */
				if (input_y > middle_end+SCROLL_SEPARATOR && input_y < addr->bounds.ycoord+addr->bounds.height)
					return SCROLL_DOWN_ARROW;
			}
			else
				return SCROLLCAR_REGION;
		}
	}
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_1 || addr->scrollbar_type == SCROLLBAR_STYLE_2)
	{
		if (addr->scrollbar_style == VERTICAL)
		{
			
			if ( ((addr->scrollbar_clicked_region != 0) && !(input_x >= addr->bounds.xcoord && 
				input_x < addr->bounds.xcoord+addr->bounds.width && input_y >= addr->bounds.ycoord && 
				input_y < addr->bounds.ycoord+addr->bounds.height)) ||  ((addr->scrollbar_clicked_region == 0) &&
				!(input_x >= addr->bounds.xcoord-SCROLL_EXT_WIDTH+1 && input_x < addr->bounds.xcoord+addr->bounds.width+SCROLL_EXT_WIDTH-1 &&
				input_y >= addr->bounds.ycoord && input_y < addr->bounds.ycoord+addr->bounds.height)) )
				return SCROLL_NOT_HITTED;
			
			middle_start = addr->bounds.ycoord;
			middle_end = addr->bounds.ycoord+addr->bounds.height-1;
			car_start = middle_start+(addr->scrollbar_value * addr->bounds.height / addr->scrollbar_total_num_lines);
			car_size = (addr->scrollbar_total_num_lines-addr->scrollbar_max) *
				addr->bounds.height / addr->scrollbar_total_num_lines;
			car_end = car_start + car_size - 1;
			
			if (input_x >= addr->bounds.xcoord && input_x < addr->bounds.xcoord+addr->bounds.width)
			{
				/* Clicked on the scroll bar region */
				if (input_y >= middle_start &&	input_y < car_start)				
					return SCROLLBAR_UP_REGION;
				/* Clicked on the scroll car area */
				if (input_y >= car_start && input_y < (car_end + 1))			
					return SCROLLCAR_REGION;					
				/* Clicked at the bottom of scroll car area */
				if (input_y > car_end && input_y <= middle_end)
					return SCROLLBAR_DOWN_REGION;				
			}
			else
				return SCROLLCAR_REGION;
		}
		else
		{
			if ( ((addr->scrollbar_clicked_region != 0) && !(input_x >= addr->bounds.xcoord && 
				input_x < addr->bounds.xcoord+addr->bounds.width && input_y >= addr->bounds.ycoord && 
				input_y < addr->bounds.ycoord+addr->bounds.height)) ||  ((addr->scrollbar_clicked_region == 0) &&
				!(input_x >= addr->bounds.xcoord && input_x < addr->bounds.xcoord+addr->bounds.width &&
				input_y >= addr->bounds.ycoord-SCROLL_EXT_WIDTH+1 && input_y < addr->bounds.ycoord+addr->bounds.height+SCROLL_EXT_WIDTH-1)) )
				return SCROLL_NOT_HITTED;
			
			middle_start = addr->bounds.xcoord;
			middle_end = addr->bounds.xcoord+addr->bounds.width-1;
			car_start = middle_start+(addr->scrollbar_value * addr->bounds.width / addr->scrollbar_total_num_lines);
			car_size = (addr->scrollbar_total_num_lines-addr->scrollbar_max) *
				addr->bounds.width / addr->scrollbar_total_num_lines;
			car_end = car_start + car_size - 1;
			
			if (input_y >= addr->bounds.ycoord && input_y < addr->bounds.ycoord+addr->bounds.height)
			{			
				/* Clicked on the left side of scroll car area */
				if (input_x >= middle_start && input_x < car_start)
					return SCROLLBAR_UP_REGION;
				/* Clicked on the scroll car area */
				if (input_x >= car_start && input_x < (car_end + 1))
					return SCROLLCAR_REGION;					
				/* clicked on the right side of scroll car area */
				if (input_x > car_end && input_x <= middle_end)
					return SCROLLBAR_DOWN_REGION;				
			}
			else
				return SCROLLCAR_REGION;
		}
	}
	return TRUE;
}
/*********************************************************************
* Function	: ScrollbarCalculateCurrentValue
* Purpose	: Calculate the current value of the scroll bar
* Scope		: Internal/Application
* Input		: scrollbar_id	DBID of the scroll bar object
*			  input_x       Input x_coordinate
*			  input_y		Input y_coordinate
*			  last_x		Last x_coordinate
*			  last_y		Last y_coordinate
* Output	: scroll_value	current value of the scroll bar
* Return	: TRUE			If no error
*			  ERR_UI_INV_INPUT_COORD
*							Input pt not within the scroll bar region
* Comment	: Calculate the current value of the scroll bar and update
*			  the scroll_value.
**********************************************************************/
Err ScrollbarCalculateCurrentValue(ObjectID scrollbar_id,SHORT input_x,SHORT input_y,WORD *scroll_value)
{
	Scrollbar *addr;
	BYTE object_type,clicked_region;
	WORD get_value;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void **)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	clicked_region = ScrollbarGetClickedRegion(addr,input_x,input_y);			
	get_value = addr->scrollbar_value;
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_0)
	{		
		switch (clicked_region)	
		{
		case SCROLL_UP_ARROW:		get_value = addr->scrollbar_value-1; 
			break;
			
		case SCROLLBAR_UP_REGION:	get_value = addr->scrollbar_value - addr->scrollbar_pagesize; 
			break;
			
		case SCROLLCAR_REGION:		if (addr->scrollbar_style == VERTICAL)
									{
										if (input_y == addr->scrollbar_save_pos_y)
											get_value = addr->scrollbar_value;
										else
										{												
											if (input_y < addr->scrollbar_save_pos_y)
											{
												if (addr->scrollbar_value == addr->scrollbar_min)
													get_value = addr->scrollbar_value;
												else
													get_value += ((input_y-addr->scrollbar_save_pos_y)*
													addr->scrollbar_total_num_lines /
													(addr->bounds.height-(addr->scrollbar_arrow1.height+SCROLL_SEPARATOR)*2));
											}
											
											if (input_y > addr->scrollbar_save_pos_y)
											{
												if (addr->scrollbar_value == addr->scrollbar_max)
													get_value = addr->scrollbar_value;
												else
													get_value += ((input_y-addr->scrollbar_save_pos_y)*
													addr->scrollbar_total_num_lines /
													(addr->bounds.height-(addr->scrollbar_arrow1.height+SCROLL_SEPARATOR)*2));												
											}
										}
									}
			else
			{	
				if (input_x == addr->scrollbar_save_pos_x)
					get_value = addr->scrollbar_value;
				else
				{	
					if (input_x < addr->scrollbar_save_pos_x)
					{												
						if (addr->scrollbar_value == addr->scrollbar_min)
							get_value = addr->scrollbar_value;
						else
							get_value += ((input_x-addr->scrollbar_save_pos_x)*addr->scrollbar_total_num_lines /
							(addr->bounds.width-((addr->scrollbar_arrow1.width+SCROLL_SEPARATOR)*2)));
					}
					
					if (input_x > addr->scrollbar_save_pos_x)
					{												
						if (addr->scrollbar_value == addr->scrollbar_max)
							get_value = addr->scrollbar_value;
						else
							get_value += ((input_x-addr->scrollbar_save_pos_x)*addr->scrollbar_total_num_lines /
							(addr->bounds.width-((addr->scrollbar_arrow1.width+SCROLL_SEPARATOR)*2)));
					}
				}
			}
			break;
			
		case SCROLLBAR_DOWN_REGION:	get_value = addr->scrollbar_value + addr->scrollbar_pagesize; 
			break;
			
		case SCROLL_DOWN_ARROW:		get_value = addr->scrollbar_value+1;
			break;
			
		default: get_value = addr->scrollbar_value;
			break;
		}
	}
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_1 || addr->scrollbar_type == SCROLLBAR_STYLE_2)
	{
		switch (clicked_region)
		{
		case SCROLLBAR_UP_REGION:	get_value = addr->scrollbar_value - addr->scrollbar_pagesize;
			break;
			
		case SCROLLCAR_REGION:		if (addr->scrollbar_style == VERTICAL)
									{
										if (input_y == addr->scrollbar_save_pos_y)
											get_value = addr->scrollbar_value;
										else
										{	
											if (input_y < addr->scrollbar_save_pos_y)
											{												
												if (addr->scrollbar_value == addr->scrollbar_min)
													get_value = addr->scrollbar_value;
												else
													get_value += ((input_y-addr->scrollbar_save_pos_y) *
													addr->scrollbar_total_num_lines /
													addr->bounds.height);
											}
											
											if (input_y > addr->scrollbar_save_pos_y)
											{												
												if (addr->scrollbar_value == addr->scrollbar_max)
													get_value = addr->scrollbar_value;
												else
													get_value += ((input_y-addr->scrollbar_save_pos_y) *
													addr->scrollbar_total_num_lines /
													addr->bounds.height);
											}
										}
									}
			else
			{											
				if (input_x == addr->scrollbar_save_pos_x)
					get_value = addr->scrollbar_value;
				else
				{																							
					if (input_x < addr->scrollbar_save_pos_x)
					{																							
						if (addr->scrollbar_value == addr->scrollbar_min)
							get_value = addr->scrollbar_value;
						else
							get_value += ((input_x-addr->scrollbar_save_pos_x) *
							addr->scrollbar_total_num_lines /
							addr->bounds.width);
					}
					
					if (input_x > addr->scrollbar_save_pos_x)
					{																							
						if (addr->scrollbar_value == addr->scrollbar_max)
							get_value = addr->scrollbar_value;
						else
							get_value += ((input_x-addr->scrollbar_save_pos_x) *
							addr->scrollbar_total_num_lines /
							addr->bounds.width);
					}
				}
			}
			break;
			
		case SCROLLBAR_DOWN_REGION:	get_value = addr->scrollbar_value + addr->scrollbar_pagesize; 
			break;							
			
		default: get_value=  addr->scrollbar_value;
			break;
		}
	}
	/* At top of the bar */
	if (get_value < addr->scrollbar_min)
		get_value = addr->scrollbar_min;
	/* At bottom of the bar */
	if (get_value > addr->scrollbar_max)
		get_value = addr->scrollbar_max;
	
	*scroll_value = get_value;
	return TRUE;
}
/*********************************************************************
* Function	: ScrollbarDrawScrollbar
* Purpose	: Draw the specified scroll bar
* Scope		: Internal/Application
* Input		: scrollbar_id	DBID of the scroll bar object
* Output	: None
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
* Comment	: None
**********************************************************************/
Err ScrollbarDrawScrollbar(ObjectID scrollbar_id)
{
	Scrollbar *addr;
	BYTE object_type;
	WORD car_size, draw_value;
	ObjectBounds scroll_car,scroll_highlight,scroll_bar,scroll_text;
	BitmapTemplate scroll_button;
	
	
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void **)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	if (addr->scrollbar_attr.scrollbar_visible == FALSE) return FALSE;
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_0)
	{
		addr->scrollbar_arrow1.xcoord = addr->bounds.xcoord;
		addr->scrollbar_arrow1.ycoord = addr->bounds.ycoord;
		addr->scrollbar_arrow2.xcoord = addr->bounds.xcoord;
		addr->scrollbar_arrow2.ycoord = addr->bounds.ycoord + addr->bounds.height -
			addr->scrollbar_arrow2.height;
		
		if (addr->scrollbar_style == VERTICAL)
		{
			if (addr->scrollbar_attr.scrollbar_enter1 == TRUE)
				LcdDrawBitmap(&(addr->scrollbar_arrow1),COLOR_INVERT);
			else
				LcdDrawBitmap(&(addr->scrollbar_arrow1),COLOR_NO_INVERT);
			
			if (addr->scrollbar_attr.scrollbar_enter2 == TRUE)
				LcdDrawBitmap(&(addr->scrollbar_arrow2),COLOR_INVERT);
			else
				LcdDrawBitmap(&(addr->scrollbar_arrow2),COLOR_NO_INVERT);
			/* Draw the middle line */
			scroll_bar.xcoord = addr->bounds.xcoord;
			scroll_bar.ycoord = addr->bounds.ycoord+addr->scrollbar_arrow1.height+SCROLL_SEPARATOR;
			scroll_bar.width  = addr->bounds.width;
			scroll_bar.height = addr->bounds.height-((addr->scrollbar_arrow1.height+SCROLL_SEPARATOR)*2);
			LcdDrawBox(&scroll_bar,color_level[COLOR_GREY1],color_level[COLOR_GREY1],FILL_SOLID);			
			/* Test if the scroll car is at the bottom of the bar */
			if (addr->scrollbar_value > (addr->scrollbar_total_num_lines - addr->scrollbar_draw_pagesize))
				draw_value = (addr->scrollbar_total_num_lines - addr->scrollbar_draw_pagesize);
			else
				draw_value = addr->scrollbar_value;
			
			scroll_car.xcoord = addr->bounds.xcoord;
            scroll_car.ycoord = scroll_bar.ycoord + draw_value * (addr->bounds.height - (addr->scrollbar_arrow1.height + SCROLL_SEPARATOR - 1)*2)/
				addr->scrollbar_total_num_lines;
            car_size = (addr->scrollbar_draw_pagesize) * (addr->bounds.height - (addr->scrollbar_arrow1.height + SCROLL_SEPARATOR - 1)*2)/
				addr->scrollbar_total_num_lines;
			
			if (car_size <= SCROLL_MIN_VALUE)
			{
				car_size = SCROLL_MIN_VALUE;
				if (scroll_car.ycoord+car_size-1 > addr->scrollbar_arrow2.ycoord)
					scroll_car.ycoord = addr->scrollbar_arrow2.ycoord - SCROLL_MIN_VALUE;
			}
			
			scroll_car.width  = addr->bounds.width;
			scroll_car.height = car_size;
			
            if (scroll_car.ycoord+scroll_car.height > addr->scrollbar_arrow2.ycoord)
				scroll_car.ycoord -= 1;
			
			LcdDrawBox(&scroll_car,color_level[COLOR_BLACK_COLOR],color_level[COLOR_GREY1],FILL_EMPTY);
			LcdDrawLine(scroll_car.xcoord+1,scroll_car.ycoord+2,scroll_car.xcoord+1,
				scroll_car.ycoord+scroll_car.height-3,SINGLE_LINE,NON_DOTTED_LINE,color_level[COLOR_WHITE],color_level[COLOR_WHITE]);
            LcdDrawLine(scroll_car.xcoord+scroll_car.width-2,scroll_car.ycoord+2,scroll_car.xcoord+scroll_car.width-2,
				scroll_car.ycoord+scroll_car.height-3,SINGLE_LINE,NON_DOTTED_LINE,color_level[COLOR_GREY2],color_level[COLOR_GREY2]);
			LcdDrawLine(scroll_car.xcoord+1,scroll_car.ycoord+1,scroll_car.xcoord+scroll_car.width-2,
				scroll_car.ycoord+1,SINGLE_LINE,NON_DOTTED_LINE,color_level[COLOR_WHITE],color_level[COLOR_WHITE]);
			LcdDrawLine(scroll_car.xcoord+1,scroll_car.ycoord+scroll_car.height-2,scroll_car.xcoord+scroll_car.width-2,
				scroll_car.ycoord+scroll_car.height-2,SINGLE_LINE,NON_DOTTED_LINE,color_level[COLOR_GREY2],color_level[COLOR_GREY2]);
		}
	}
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_1 || addr->scrollbar_type == SCROLLBAR_STYLE_2)
	{
		if (addr->scrollbar_style == VERTICAL)
		{
			LcdDrawBox(&(addr->bounds),color_level[COLOR_BLACK_COLOR],color_level[COLOR_WHITE],FILL_EMPTY);
			/* Draw the scroll button */
			scroll_button.xcoord = addr->bounds.xcoord;
			scroll_button.ycoord = addr->bounds.ycoord+(addr->scrollbar_value * addr->bounds.height) /
				addr->scrollbar_total_num_lines;
			scroll_button.width  = 7;
			scroll_button.height = 7;
			scroll_button.compressed = FALSE;
			scroll_button.size = FALSE;
            scroll_button.quantisation = Q_FOUR_BIT;
            //if (color_mode == GREYSCALE_MODE)
			scroll_button.bitmap_data = scroll_car_diagram;
			//else scroll_button.bitmap_data = scroll_car_diagram_BW;
			
			if (scroll_button.ycoord > (addr->bounds.ycoord + addr->bounds.width - scroll_button.width))
				scroll_button.ycoord = addr->bounds.ycoord + addr->bounds.width - scroll_button.width;
			
			/* Draw the highlight region */
			if (addr->scrollbar_type == 1 && addr->scrollbar_value != 0)
			{
				scroll_highlight.xcoord = addr->bounds.xcoord+1;
				scroll_highlight.ycoord = addr->bounds.ycoord+1;
				scroll_highlight.width = addr->bounds.width-2;
				scroll_highlight.height = scroll_button.ycoord-addr->bounds.ycoord-1;
				LcdDrawBox(&scroll_highlight,color_level[COLOR_GREY2],color_level[COLOR_GREY1],FILL_EMPTY);
				LcdDrawLine(scroll_highlight.xcoord,scroll_highlight.ycoord,
					scroll_highlight.xcoord,scroll_highlight.ycoord+scroll_highlight.height-1,
					SINGLE_LINE,NON_DOTTED_LINE,color_level[COLOR_WHITE],color_level[COLOR_WHITE]);			
			}
			
			LcdDrawBitmap(&scroll_button,COLOR_NO_INVERT);
		}
		else 		
		{
			LcdEraseRegion(&(addr->bounds));
			LcdDrawBox(&(addr->bounds),color_level[COLOR_BLACK_COLOR],color_level[COLOR_WHITE],FILL_EMPTY);
			/* Draw the scroll button */
			scroll_button.xcoord = addr->bounds.xcoord+(addr->scrollbar_value * addr->bounds.width) /
				addr->scrollbar_total_num_lines;
			scroll_button.ycoord = addr->bounds.ycoord;
			scroll_button.width  = 7;
			scroll_button.height = 7;
			scroll_button.compressed = FALSE;
			scroll_button.size = FALSE;
            scroll_button.quantisation = Q_FOUR_BIT;
            //if (color_mode == GREYSCALE_MODE)
			scroll_button.bitmap_data = scroll_car_diagram;
			//else scroll_button.bitmap_data = scroll_car_diagram_BW;
			
			if (scroll_button.xcoord > (addr->bounds.xcoord + addr->bounds.width - scroll_button.width))
				scroll_button.xcoord = addr->bounds.xcoord + addr->bounds.width - scroll_button.width;
			
			/* Draw the highlight region */
			if (addr->scrollbar_type == 1 && addr->scrollbar_value != 0)
			{
				scroll_highlight.xcoord = addr->bounds.xcoord+1;
				scroll_highlight.ycoord = addr->bounds.ycoord+1;
				scroll_highlight.width = scroll_button.xcoord-addr->bounds.xcoord-1;
				scroll_highlight.height = addr->bounds.height-2;
				LcdDrawBox(&scroll_highlight,color_level[COLOR_GREY2],color_level[COLOR_GREY1],FILL_EMPTY);
				
				LcdDrawLine(scroll_highlight.xcoord+1,scroll_highlight.ycoord + 
					scroll_highlight.height-1,scroll_highlight.xcoord +
					scroll_highlight.width-1,scroll_highlight.ycoord + 
					scroll_highlight.height-1,SINGLE_LINE,NON_DOTTED_LINE,
					color_level[COLOR_WHITE],color_level[COLOR_WHITE]);			
			}
			/* Draw the scroll button */
			LcdDrawBitmap(&scroll_button,COLOR_NO_INVERT);
		}
	}
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_3) 
	{
		if (addr->scrollbar_style == HORIZONTAL)
		{
			LcdDrawBox(&(addr->bounds),color_level[COLOR_BLACK_COLOR],color_level[COLOR_WHITE],FILL_EMPTY);
			/* Draw the scrollbar text */
			scroll_text.height = FONT_HEIGHT[0];
			scroll_text.width  = StrGetWidth(addr->scrollbar_text,SMALL_FONT);
			scroll_text.xcoord = addr->bounds.xcoord + (addr->bounds.width - scroll_text.width)/2 -1;
			scroll_text.ycoord = addr->bounds.ycoord + (addr->bounds.height - FONT_HEIGHT[0])/2 -1;
			LcdDrawFixedString(&scroll_text,addr->scrollbar_text,color_level[COLOR_BLACK_COLOR],color_level[COLOR_WHITE],SMALL_FONT,NO_DOTDOT,0);
			/* Draw the highlight region */
			scroll_highlight.xcoord = addr->bounds.xcoord+1;
			scroll_highlight.ycoord = addr->bounds.ycoord+1;
			scroll_highlight.width = (addr->scrollbar_value * addr->bounds.width / addr->scrollbar_total_num_lines) - 1;
			scroll_highlight.height = addr->bounds.height-2;
			LcdInvertBox(&scroll_highlight);
		}
	}
	addr->scrollbar_attr.scrollbar_drawn = TRUE;
	return TRUE;
}
/********************************************************
* Function:	ScrollbarInitScrollbar
* Purpose:	1) to initialise the Scrollbar object
2) to read the resource file
3) to create a corresponding Scrollbar structure
in RAM
* Scope:	Application
* Input:	DBID of the Scrollbar object
* Output:	None
* Return:      Error code
* Comment:     None
*********************************************************/
Err ScrollbarInitScrollbar(ObjectID scrollbar_id)
{
	BYTE *buffer;
	UWORD byte_return;
	Scrollbar *addr;
	Err Error;
	BYTE object_type;
	UWORD field_size;
	
	object_type = SCROLLBAR;
	/*put the pointer to the DBID lookup table */
	Error = UIAddressToLookupTable(scrollbar_id,object_type,(void**)&addr); 
	if (Error != TRUE) return Error;
	/*check whether the resource file is opened */
    Error = ResOpen(scrollbar_id);
    if (Error != TRUE) return Error;
	
	addr->identification.ui_object_id = scrollbar_id;
	/* get field 0 of resource file */
	ResGetField(scrollbar_id,0,&buffer,&byte_return);
	addr->identification.ui_object_type = *buffer;
    qfree(buffer);
	/* get field 1 of resource file */
	ResGetField(scrollbar_id,1,&buffer,&byte_return);
	addr->identification.table_related = *(ObjectID*)buffer;
    qfree(buffer);
	/* get field 2 from byte 0 for 2 bytes */
	ResReadField(scrollbar_id,2,0,2,&buffer,&byte_return);
	addr->bounds.xcoord = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(scrollbar_id,2,2,2,&buffer,&byte_return);
	addr->bounds.ycoord = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(scrollbar_id,2,4,2,&buffer,&byte_return);
	addr->bounds.width = *(SHORT*)buffer;
    qfree(buffer);
	ResReadField(scrollbar_id,2,6,2,&buffer,&byte_return);
	addr->bounds.height = *(SHORT*)buffer;
    qfree(buffer);
	/* get field 3 of resource file */
	ResGetField(scrollbar_id,3,&buffer,&byte_return);
	addr->scrollbar_max = *(WORD*)buffer;
    qfree(buffer);
	/* get field 4 of resource file */
	ResGetField(scrollbar_id,4,&buffer,&byte_return);
	addr->scrollbar_min = *(WORD*)buffer;
    qfree(buffer);
	/* get field 5 of resource file */
	ResGetField(scrollbar_id,5,&buffer,&byte_return);
	addr->scrollbar_value = *(WORD*)buffer;
    qfree(buffer);
	/* get field 6 of resource file */
	ResGetField(scrollbar_id,6,&buffer,&byte_return);
	addr->scrollbar_pagesize = *(WORD*)buffer;
    qfree(buffer);
	/* get field 7 of resource file */
	ResGetField(scrollbar_id,7,&buffer,&byte_return);
	addr->scrollbar_total_num_lines = *(WORD*)buffer;
    qfree(buffer);
	/* get field 8 of resource file */
	ResGetField(scrollbar_id,8,&buffer,&byte_return);
	addr->scrollbar_style = *buffer;
    qfree(buffer);
	/* get field 9 of resource file */
	ResGetField(scrollbar_id,9,&buffer,&byte_return);
	addr->scrollbar_type = *buffer;
    qfree(buffer);
	/* get field 10 of resource file */
	ResFieldSize(scrollbar_id,10,&field_size);
	if (field_size > 8)
	{
		ResReadField(scrollbar_id,10,0,2,&buffer,&byte_return);
		addr->scrollbar_arrow1.xcoord = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,10,2,2,&buffer,&byte_return);
		addr->scrollbar_arrow1.ycoord = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,10,4,2,&buffer,&byte_return);
		addr->scrollbar_arrow1.width = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,10,6,2,&buffer,&byte_return);
		addr->scrollbar_arrow1.height = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,10,8,1,&buffer,&byte_return);
		addr->scrollbar_arrow1.compressed = *buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,10,9,1,&buffer,&byte_return);
		addr->scrollbar_arrow1.quantisation = *buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,10,10,2,&buffer,&byte_return);
		addr->scrollbar_arrow1.size = *(USHORT*)buffer;
        qfree(buffer);
		ResGetPointer(scrollbar_id,10,12,(void**)&buffer);
		addr->scrollbar_arrow1.bitmap_data = (UWORD*)buffer;
	}
	/* get field 11 of resource file */
	ResFieldSize(scrollbar_id,11,&field_size);
	if (field_size > 8)
	{
		ResReadField(scrollbar_id,11,0,2,&buffer,&byte_return);
		addr->scrollbar_arrow2.xcoord = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,11,2,2,&buffer,&byte_return);
		addr->scrollbar_arrow2.ycoord = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,11,4,2,&buffer,&byte_return);
		addr->scrollbar_arrow2.width = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,11,6,2,&buffer,&byte_return);
		addr->scrollbar_arrow2.height = *(SHORT*)buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,11,8,1,&buffer,&byte_return);
		addr->scrollbar_arrow2.compressed = *buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,11,9,1,&buffer,&byte_return);
		addr->scrollbar_arrow2.quantisation = *buffer;
        qfree(buffer);
		ResReadField(scrollbar_id,11,10,2,&buffer,&byte_return);
		addr->scrollbar_arrow2.size = *(USHORT*)buffer;
        qfree(buffer);
		ResGetPointer(scrollbar_id,11,12,(void**)&buffer);
		addr->scrollbar_arrow2.bitmap_data = (UWORD*)buffer;
	}
	/* Field 12 */
	ResGetField(scrollbar_id,12,&buffer,&byte_return);
    addr->scrollbar_text = (BYTE*)qmalloc((strlen(buffer)+1)*sizeof(BYTE));
	strcpy(addr->scrollbar_text,buffer);
    qfree(buffer);
	/* get field 13 of resource file */
	ResReadField(scrollbar_id,13,0,2,&buffer,&byte_return);
	addr->scrollbar_attr.scrollbar_active = *(BOOLEAN*)buffer;
    qfree(buffer);
	ResReadField(scrollbar_id,13,2,2,&buffer,&byte_return);
	addr->scrollbar_attr.scrollbar_enable = *(BOOLEAN*)buffer;
    qfree(buffer);
	ResReadField(scrollbar_id,13,4,2,&buffer,&byte_return);
	addr->scrollbar_attr.scrollbar_visible = *(BOOLEAN*)buffer;
    qfree(buffer);
	addr->scrollbar_attr.scrollbar_drawn  = FALSE;
	addr->scrollbar_attr.scrollbar_enter  = FALSE;
	addr->scrollbar_attr.scrollbar_enter1 = FALSE;
	addr->scrollbar_attr.scrollbar_enter2 = FALSE;
	ResClose(scrollbar_id);
	return TRUE;
}
/*********************************************************************
* Function	: ScrollbarGetScrollbarText
* Purpose	: Get the text of the specified scrollbar
* Scope		: Internal/Application
* Input		: scrollbar_id		DBID of the scroll bar object
* Output	: scrollbar_text	Pointer to the text of the scrollbar
* Return	: TRUE If no error
*			  ERR_UI_RES_NOT_FOUND
* Comment	: None
**********************************************************************/
Err ScrollbarGetScrollbarText(ObjectID scrollbar_id, BYTE **scrollbar_text)
{
	Scrollbar *addr;
	BYTE object_type;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	*scrollbar_text = addr->scrollbar_text;
	return TRUE;
}
/********************************************************
* Function:	ScrollbarSetScrollbarText
* Purpose:		to set the text of the specified scrollbar 
*				object
* Scope:		Application
* Input:		scrollbar_id		DBID of scrollbar object
*				scrollbar_text		Pointer to the text of scrollbar 
*									object
* Output:		None
* Return:		Error code
* Comment:     None
*********************************************************/
Err ScrollbarSetScrollbarText(ObjectID scrollbar_id,BYTE *scrollbar_text)
{
	Scrollbar *addr;
	BYTE object_type;
	USHORT length;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
    qfree(addr->scrollbar_text);
	length = strlen(scrollbar_text); /*check the length of the passed-in label */
    addr->scrollbar_text = (BYTE *)qmalloc((length+1) * sizeof(BYTE));
	strcpy(addr->scrollbar_text,scrollbar_text);
	return TRUE;
}

/********************************************************
* Function:	    ScrollbarHardButtonSetScrollbar
* Purpose:		This function is called to output a corresponding
EVT_SCROLLBAR_REPEAT event while the up or down
hardward button is pressed
* Scope:		Application
* Input:		scrollbar_id		OBJECTID of the scrollbar object
UP_DOWN				SCROLLBAR_UP_ARROW
SCROLLBAR_DOWN_ARROW
* Output:		evt					ouput event
* Return:		TRUE if no error
ERR_UI_RES_NOT_FOUND
* Comment:     None
*********************************************************/
Err ScrollbarHardButtonSetScrollbar(ObjectID scrollbar_id, BYTE UP_DOWN)
{
	Scrollbar       *addr;
	BYTE		object_type;
	WORD		new_value;
	EvtType         evt;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void**)&addr) != TRUE)
		return ERR_UI_RES_NOT_FOUND;
    if (addr->scrollbar_attr.scrollbar_drawn == FALSE ||
        addr->scrollbar_attr.scrollbar_enable == FALSE)
		return TRUE;
	
#ifdef DEBUG_SCROLL
    printf("\n SCROLL 1");
#endif
	
    evt.eventID    =       scrollbar_id;
    evt.para1      =       (WORD)(addr->scrollbar_value);
	
#ifdef DEBUG_SCROLL
    printf("\n SCROLL 2");
#endif
	
	
    if (addr->scrollbar_value == 0 && UP_DOWN == SCROLLBAR_UP_ARROW)
        return TRUE;
    else if (addr->scrollbar_value == addr->scrollbar_max && UP_DOWN == SCROLLBAR_DOWN_ARROW)
        return TRUE;
	
	
#ifdef DEBUG_SCROLL
    printf("\n SCROLL 3");
#endif
	
	
#ifdef DEBUG_SCROLL
    printf("\n SCROLL 4");
#endif
	
    new_value = addr->scrollbar_value;
	
#ifdef DEBUG_SCROLL
    printf("\n SCROLL 5");
#endif
	
	
    if (UP_DOWN == SCROLLBAR_UP_ARROW)
		new_value --;
    else if (UP_DOWN == SCROLLBAR_DOWN_ARROW)
		new_value ++;
	
	if (new_value < 0)
		new_value = 0;
    else if (new_value > addr->scrollbar_max)
		new_value = addr->scrollbar_max;
    evt.para2              =       new_value;
    evt.evtPBP             =       (void*)addr;
	
#ifdef DEBUG_SCROLL
    printf("\n SCROLL 6");
#endif
	
	
    by_pass_scrollbar_event = TRUE;
    addr->scrollbar_value = evt.para2;
    EvtAppendEvtReplace(EVT_SCROLLBAR_REPEAT, evt.eventID, evt.para1, evt.para2, evt.evtPBP);
	//New added Michelle 11082000
	EvtAppendEvtReplace(EVT_SCROLLBAR_SELECT, evt.eventID, evt.para1, evt.para2, evt.evtPBP);
	return TRUE;
}

/*********************************************************************
* Function  : ScrollbarCalculateCurrentValue1
* Purpose	: Calculate the current value of the scroll bar
* Scope		: Internal/Application
* Input		: scrollbar_id	DBID of the scroll bar object
*			  input_x       Input x_coordinate
*			  input_y		Input y_coordinate
*			  last_x		Last x_coordinate
*			  last_y		Last y_coordinate
* Output	: scroll_value	current value of the scroll bar
* Return	: TRUE			If no error
*			  ERR_UI_INV_INPUT_COORD
*							Input pt not within the scroll bar region
* Comment   : 1)Calculate the current value of the scroll bar and update
*               the scroll_value.
*             2)Asuming that the clicked region is nto important
**********************************************************************/
Err ScrollbarCalculateCurrentValue1(ObjectID scrollbar_id, SHORT input_x, SHORT input_y, WORD *scroll_value)
{
	Scrollbar *addr;
	BYTE object_type,clicked_region;
	WORD get_value;
	
	if (UISearchForAddress(scrollbar_id,&object_type,(void **)&addr) != TRUE) return ERR_UI_RES_NOT_FOUND;
	get_value = addr->scrollbar_value;
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_0)
	{
        if (addr->scrollbar_style == VERTICAL)
        {
            if (input_y == addr->scrollbar_save_pos_y)
                get_value = addr->scrollbar_value;
            else
            {
                if (input_y < addr->scrollbar_save_pos_y)
                {
                    if (addr->scrollbar_value == addr->scrollbar_min)
                        get_value = addr->scrollbar_value;
                    else
                        get_value = ((input_y - (addr->bounds.ycoord + addr->scrollbar_arrow1.height + SCROLL_SEPARATOR))*
						addr->scrollbar_total_num_lines /
						(addr->bounds.height-(addr->scrollbar_arrow1.height+SCROLL_SEPARATOR)*2));
                }
                else if (input_y > addr->scrollbar_save_pos_y)
                {
                    if (addr->scrollbar_value == addr->scrollbar_max)
                        get_value = addr->scrollbar_value;
                    else
                        get_value = ((input_y - (addr->bounds.ycoord + addr->scrollbar_arrow1.height + SCROLL_SEPARATOR))*
						addr->scrollbar_total_num_lines /
						(addr->bounds.height-(addr->scrollbar_arrow1.height+SCROLL_SEPARATOR)*2));
					
                }
            }
        }
        else
        { 
            if (input_x == addr->scrollbar_save_pos_x)
                get_value = addr->scrollbar_value;
            else
            { 
                if (input_x < addr->scrollbar_save_pos_x)
                {                                            
                    if (addr->scrollbar_value == addr->scrollbar_min)
                        get_value = addr->scrollbar_value;
                    else
                        get_value += ((input_x-addr->scrollbar_save_pos_x)*addr->scrollbar_total_num_lines /
						(addr->bounds.width-((addr->scrollbar_arrow1.width+SCROLL_SEPARATOR)*2)));
                }
                else if (input_x > addr->scrollbar_save_pos_x)
                {                                            
					if (addr->scrollbar_value == addr->scrollbar_max)
						get_value = addr->scrollbar_value;
					else
						get_value += ((input_x-addr->scrollbar_save_pos_x)*addr->scrollbar_total_num_lines /
						(addr->bounds.width-((addr->scrollbar_arrow1.width+SCROLL_SEPARATOR)*2)));
                }
            }
        }
    }
    else if (addr->scrollbar_type == SCROLLBAR_STYLE_1 || addr->scrollbar_type == SCROLLBAR_STYLE_2)
	{
        if (addr->scrollbar_style == VERTICAL)
        {
            if (input_y == addr->scrollbar_save_pos_y)
                get_value = addr->scrollbar_value;
            else
            { 
                if (input_y < addr->scrollbar_save_pos_y)
                {                                            
                    if (addr->scrollbar_value == addr->scrollbar_min)
                        get_value = addr->scrollbar_value;
                    else
                        get_value += ((input_y-addr->scrollbar_save_pos_y) *
						addr->scrollbar_total_num_lines /
						addr->bounds.height);
                }
                else if (input_y > addr->scrollbar_save_pos_y)
                {                                            
                    if (addr->scrollbar_value == addr->scrollbar_max)
                        get_value = addr->scrollbar_value;
                    else
                        get_value += ((input_y-addr->scrollbar_save_pos_y) *
						addr->scrollbar_total_num_lines /
						addr->bounds.height);
                }
            }
        }
        else
        {                                        
            if (input_x == addr->scrollbar_save_pos_x)
                get_value = addr->scrollbar_value;
            else
            {                                                                                        
                if (input_x < addr->scrollbar_save_pos_x)
                {                                                                                        
                    if (addr->scrollbar_value == addr->scrollbar_min)
                        get_value = addr->scrollbar_value;
                    else
                        get_value += ((input_x-addr->scrollbar_save_pos_x) *
						addr->scrollbar_total_num_lines /
						addr->bounds.width);
                }
                else if (input_x > addr->scrollbar_save_pos_x)
                {                                                                                        
                    if (addr->scrollbar_value == addr->scrollbar_max)
                        get_value = addr->scrollbar_value;
                    else
                        get_value += ((input_x-addr->scrollbar_save_pos_x) *
						addr->scrollbar_total_num_lines /
						addr->bounds.width);
                }
			}
        }
	}
	
    /* At top of the bar */
	if (get_value < addr->scrollbar_min)
		get_value = addr->scrollbar_min;
	/* At bottom of the bar */
	if (get_value > addr->scrollbar_max)
		get_value = addr->scrollbar_max;
	
	*scroll_value = get_value;
	return TRUE;
}

/********************************************************
* Function:	ScrollbarGetCarStartCoord
* Purpose:	This function find out the starting coordinate of the 
*			scroll car
* Scope:	Internal
* Input:	scroll_value	Scroll value of the scrollbar
* Output:	None
* Return:	car_start		Scroll car starting position in pixel
* Comment:	None
*********************************************************/
SHORT	ScrollbarGetCarStartCoord(Scrollbar *addr, WORD scroll_value)
{
	USHORT	car_start;
	
	if (addr->scrollbar_type == SCROLLBAR_STYLE_0)
	{
		if (addr->scrollbar_style == VERTICAL)
			car_start = addr->bounds.ycoord + scroll_value * (addr->bounds.height - (addr->scrollbar_arrow1.height-1)*2) /
				        addr->scrollbar_total_num_lines;
	}
	else if (addr->scrollbar_type == SCROLLBAR_STYLE_1 || addr->scrollbar_type == SCROLLBAR_STYLE_2)
	{
		if (addr->scrollbar_style == VERTICAL)
			car_start = addr->bounds.ycoord + (scroll_value * addr->bounds.height) / addr->scrollbar_total_num_lines;
		else
			car_start = addr->bounds.xcoord + (scroll_value * addr->bounds.width) / addr->scrollbar_total_num_lines;
	}
	return car_start;
}
