/*
 * @(#)helio.c
 *
 * Copyright 1999-2001, Aaron Ardiri (mailto:aaron@ardiri.com)
 * All rights reserved.
 *
 * The  source code outlines a number of basic Helio Computing Programming
 * principles and you  should be able to take the core structure and write 
 * a large complex program. It is distributed WITHOUT ANY WARRANTY; use it
 * "AS IS" and at your own risk.
 *
 * The code presented is Copyright 2000 by Aaron Ardiri. It should be used
 * for educational purposes only.  You  shall not modify the Cube3D source 
 * code in any way and re-distribute it as your own, however you  are free 
 * to  use the  code as a  guide  for  developing  programs on  the  Helio 
 * Computing Platform.
 */

#include "helio.h"

// global variable structure
typedef struct
{
  WORD    evtTimeOut;
  RTM     evtNullEventTime;
  SHORT   ticksPerFrame;
} Globals;

// globals reference
static Globals *globals;

// function prototypes
BOOLEAN (*FormDispatchEvent)(EvtType *Event);

// interface
static BOOLEAN mainFormEventHandler(EvtType *);
static BOOLEAN infoFormEventHandler(EvtType *);
static BOOLEAN helpFormEventHandler(EvtType *);

/**
 * The Form:mainForm event handling routine.
 *
 * @param event the event to process.
 * @return true if the event was handled, false otherwise.
 */
static BOOLEAN 
mainFormEventHandler(EvtType *event)
{
  BOOLEAN processed = FALSE;

  switch (event->eventType)
  {
    case EVT_FORM_OPEN:
         FormDrawForm(mainForm);

         // initialize the cube environment
         CubeInitialize(35);

         // draw the "seperators"
	 GfxDrawLine(4,142,156,142,gfx_dgray);
	 GfxDrawLine(4,143,156,143,gfx_dgray);

         // configure the display window
         {
	   GfxWindow *backBuffer = GfxCreateWindow(152,120);
	   GfxSetDrawWindow(backBuffer);
	 }
         EvtAppendEvt(EVT_FORM_CONFIGURE,0,0,0,NULL);

         processed = TRUE;
         break;

    case EVT_FORM_CONFIGURE:
         globals->evtTimeOut = 0;                     // get first event ASAP
         processed = TRUE;
         break;

    case EVT_BITMAP_SELECT:

         switch (event->eventID) 
         {
           case mainFormHelpBitmap:

                // display the info form [modal dialog]
                ApplicationDisplayDialog(helpForm);

                break;

           case mainFormAboutBitmap:

                // display the info form [modal dialog]
                ApplicationDisplayDialog(infoForm);

                processed = TRUE;
                break;

           default:
                break;
         }
         break;

    case NULL_EVENT:
         {
           GfxRegion region = {{   0,   0 }, { 152, 120 }};

           // clear the screen buffer
           GfxFillRegion(GfxGetDrawWindow(), &region, gfx_white);

           // animate the cube
           CubeAnimate();

	   // copy backbuffer to screen
	   GfxCopyRegion(GfxGetDrawWindow(), GfxGetDisplayWindow(), 
                         &region, 4, 20, gfxPaint);
         }

         processed = TRUE;
         break;

    case EVT_FORM_CLOSE:

         // terminate the cube environment
         CubeTerminate();

         // re-adjust the display window
         {
	   GfxDisposeWindow(GfxGetDrawWindow());
	   GfxSetDrawWindow(GfxGetDisplayWindow());
	 }

         processed = TRUE;
         break;

    default:
         break;
  }

  return processed;
}

/**
 * The Form:infoForm event handling routine.
 *
 * @param event the event to process.
 * @return true if the event was handled, false otherwise.
 */
static BOOLEAN 
infoFormEventHandler(EvtType *event)
{
  BOOLEAN processed = FALSE;

  switch (event->eventType)
  {
    case EVT_FORM_OPEN:
         FormDrawForm(infoForm);

         // configure
         EvtAppendEvt(EVT_FORM_CONFIGURE,0,0,0,NULL);

         processed = TRUE;
         break;

    case EVT_FORM_CONFIGURE:
         globals->evtTimeOut = evtWaitForever;
         processed = TRUE;
         break;

    case EVT_CONTROL_SELECT:

         switch (event->eventID) 
         {
           case infoFormOkButton:

                // close the current form
                EvtAppendEvt(EVT_FORM_CLOSE,0,0,0,NULL);

                processed = TRUE;
                break;

           default:
                break;
         }
         break;

    case EVT_INLAY_SELECT:

         // which button?
	 switch (event->para1)
	 {
	   case INLAY_OK:
                // relay the event
                EvtAppendEvt(EVT_CONTROL_SELECT,infoFormOkButton,0,0,NULL);

                // play a little snd
		SndPlaySndEffect(SNDRES5_BEEP);

                processed = TRUE;
                break;

           default:
                break;
         }
         break;

    case EVT_FORM_CLOSE:
         processed = TRUE;
         break;

    default:
         break;
  }

  return processed;
}

/**
 * The Form:helpForm event handling routine.
 *
 * @param event the event to process.
 * @return true if the event was handled, false otherwise.
 */
static BOOLEAN 
helpFormEventHandler(EvtType *event)
{
  BOOLEAN processed = FALSE;

  switch (event->eventType)
  {
    case EVT_FORM_OPEN:
         FormDrawForm(helpForm);

         // initialize
	 {
	   WORD      val,max,min,pge,tot;
	   BYTE      objectType;
	   Scrollbar *scrollbar;

           // adjust the scrollbar "draw" pagesize
	   ScrollbarGetScrollbar(helpFormScrollbar,&val,&max,&min,&pge,&tot);
	   FormGetObjectPointer(helpFormScrollbar, 
                                &objectType, (void **)&scrollbar);

           val = HelpInitialize();
	   max = (val > (pge+16)) ? (val-(pge+16)) : 0;
	   tot = max+pge;
	   val = 0;

           // adjust the scroll bar
	   ScrollbarSetScrollbar(helpFormScrollbar,
                                 val,max,min,pge,tot);

           // the scroll bar is visible? adjust "scroll car size"
           if (max > 0) {
             scrollbar->scrollbar_attr.scrollbar_visible = TRUE;
             scrollbar->scrollbar_draw_pagesize          = pge; 
	     ScrollbarDrawScrollbar(helpFormScrollbar);
           }

           // draw the instructions
	   HelpDrawInstructions(val);
	 }

         // configure
         EvtAppendEvt(EVT_FORM_CONFIGURE,0,0,0,NULL);

         processed = TRUE;
         break;

    case EVT_FORM_CONFIGURE:
         globals->evtTimeOut = evtWaitForever;
         processed = TRUE;
         break;

    case EVT_IO_KEY_CTRL:

         switch (event->eventID) 
         {
           case EVT_IO_KEY_PRESS:
           case EVT_IO_KEY_REPEAT:

                switch (event->para2)
		{
		  case IO_UP_ARROW:
                       {
	                 WORD val,max,min,pge,tot;

                         // adjust the scrollbar "draw" pagesize
	                 ScrollbarGetScrollbar(helpFormScrollbar,
                                               &val,&max,&min,&pge,&tot);
                         val = (pge > val) ? 0 : (val-pge);
	                 ScrollbarSetScrollbar(helpFormScrollbar,
                                               val,max,min,pge,tot);
	                 ScrollbarDrawScrollbar(helpFormScrollbar);

                         // draw the instructions
                         HelpDrawInstructions(val);
	               }
		       processed = TRUE;
                       break;

		  case IO_DOWN_ARROW:
                       {
	                 WORD val,max,min,pge,tot;

                         // adjust the scrollbar "draw" pagesize
	                 ScrollbarGetScrollbar(helpFormScrollbar,
                                               &val,&max,&min,&pge,&tot);
                         val = (max < (val+pge)) ? max : (val+pge);
	                 ScrollbarSetScrollbar(helpFormScrollbar,
                                               val,max,min,pge,tot);
	                 ScrollbarDrawScrollbar(helpFormScrollbar);

                         // draw the instructions
                         HelpDrawInstructions(val);
	               }
                       processed = TRUE;
                       break;

                  default:
                       break;
		}
                break;

           default:
                break;
         }
         break;

    case EVT_SCROLLBAR_REPEAT:
    case EVT_SCROLLBAR_SELECT:
         {
	   WORD val,max,min,pge,tot;

           // draw the scroll bar
	   ScrollbarGetScrollbar(helpFormScrollbar,
                                 &val,&max,&min,&pge,&tot);
	   ScrollbarDrawScrollbar(helpFormScrollbar);

           // draw the instructions
           HelpDrawInstructions(val);
	 }
         break;

    case EVT_CONTROL_SELECT:

         switch (event->eventID) 
         {
           case helpFormOkButton:

                // close the current form
                EvtAppendEvt(EVT_FORM_CLOSE,0,0,0,NULL);

                processed = TRUE;
                break;

           default:
                break;
         }
         break;

    case EVT_INLAY_SELECT:

         // which button?
	 switch (event->para1)
	 {
	   case INLAY_OK:
                // relay the event
                EvtAppendEvt(EVT_CONTROL_SELECT,helpFormOkButton,0,0,NULL);

                // play a little snd
		SndPlaySndEffect(SNDRES5_BEEP);

                processed = TRUE;
                break;

           default:
                break;
         }
         break;

    case EVT_FORM_CLOSE:

         // clean up
	 HelpTerminate();

         processed = TRUE;
         break;

    default:
         break;
  }

  return processed;
}

/**
 * The Palm Computing Platform initialization routine.
 */
void  
InitApplication()
{
  // initialize the device
  DeviceInitialize();
  GfxInitialize();

  // create the globals reference
  globals = (Globals *)pmalloc(sizeof(Globals));

  globals->evtTimeOut    = 0;
  globals->ticksPerFrame = 1000 / CUBE_FPS;

  // goto the main form
  FormGotoForm(mainForm);
}

/**
 * The application event handling routine.
 *
 * @param event the event to process.
 * @return true if the event was handled, false otherwise.
 */
BOOLEAN 
ApplicationHandleEvent(EvtType *event)
{
  BOOLEAN processed = FALSE;

  switch (event->eventType)
  {
    case EVT_FORM_LOAD:
         {
           ObjectID formID, currFormID;
           Form     *frm;
           BYTE     objectType;
           Err      error;

           // initialize the form
           formID = (ObjectID)event->eventID;
           error  = UISearchForAddress(formID, &objectType, (void **)&frm);
           if ((event->para1 == 1) || (error != TRUE)) {
             FormInitForm(formID);
           }

           // load it
           if (UISearchForAddress(formID, &objectType, (void **)&frm)) {

             switch (formID) 
             {
               case mainForm:
                    FormSetEventHandler(mainForm, 
                                        (void **)&FormDispatchEvent, 
                                        (void *)mainFormEventHandler);
                    break;

               case infoForm:
                    FormSetEventHandler(infoForm, 
                                        (void **)&FormDispatchEvent, 
                                        (void *)infoFormEventHandler);
                    break;

               case helpForm:
                    FormSetEventHandler(helpForm, 
                                        (void **)&FormDispatchEvent, 
                                        (void *)helpFormEventHandler);
                    break;

               default:
                    break;
             }

             // make it active
             FormSetActiveForm(formID);
             processed = TRUE;
           }
         }
         break;
         
    case EVT_INLAY_SELECT:

         // which button?
	 switch (event->para1) 
	 {
	   case INLAY_OK:

                // does the form wanna handle it?
                processed = FormDispatchEvent(event);

                // no process? relay the event
                if (!processed) {
                  EvtAppendEvt(EVT_INLAY_SELECT,0,INLAY_EXIT,0,NULL);
                  processed = TRUE;
                }
	        break; 

           case INLAY_EXIT:

                // does the form wanna handle it?
                processed = FormDispatchEvent(event);

                // no process? relay the event
                if (!processed) {
                  EvtAppendEvt(EVT_INLAY_SELECT,0,INLAY_MAIN_MENU,0,NULL);
                  processed = TRUE;
                }
                break;

           case INLAY_MAIN_MENU:

                // play a little snd
		SndPlaySndEffect(SNDRES5_BEEP);

                // close the current form
                EvtAppendEvt(EVT_FORM_CLOSE,0,0,0,NULL);

                // send an app-stop event
                EvtAppendEvt(EVT_APP_STOP,0,0,0,NULL);

                // lets return to the launcher
                {
                  AppID appID;
                  SysGetAppID((BYTE *)("Mainmenu"), &appID);
                  EvtAppendEvt(EVT_APP_LAUNCH,appID,0,0,NULL);
                }
                processed = TRUE;
                break;

           default:
                break;
         }
         break;

    default:
         break;
  }

  return processed;
}

/**
 * Display a MODAL dialog to the user.
 *
 * @param formID the ID of the form to display.
 */
void
ApplicationDisplayDialog(ObjectID formID)
{
  ObjectID   prevFormID;
  BOOLEAN    appStop, powerDown;
  EvtType    bufEvent;
  GfxWindow *currWindow;

  // save the active form/window
  FormGetActiveFormID(&prevFormID);
  FormSaveBehindBits(prevFormID);
  currWindow = GfxGetDrawWindow();

  GfxSetDrawWindow(GfxGetDisplayWindow());
  appStop   = FALSE;
  powerDown = FALSE;
  {
    BOOLEAN keepFormOpen;
    EvtType event;

    // send load form and open form events
    EvtAppendEvt(EVT_FORM_LOAD, formID, 0, 0, NULL);
    EvtAppendEvt(EVT_FORM_OPEN, formID, 0, 0, NULL);

    // handle all events here (trap them before the OS does) :)
    keepFormOpen = TRUE;
    while (keepFormOpen) {

      DeviceGetEvent(&event, globals->evtTimeOut);

      // this is our exit condition! :)
      keepFormOpen = (event.eventType != EVT_FORM_CLOSE);

      // we have to process the following events in a special way
      if ((event.eventType == EVT_INLAY_SELECT) &&
	  (event.para1     == INLAY_MAIN_MENU)) {

        // close the dialog
        EvtAppendEvt(EVT_FORM_CLOSE,0,0,0,NULL);

        // relay the main menu event
        EvtAppendEvt(EVT_INLAY_SELECT,0,INLAY_MAIN_MENU,0,NULL);
      }
      else
      if ((event.eventType == EVT_INLAY_SELECT) &&
	  (event.para1     == INLAY_EXIT)) {

        // relay it as an "OK" button press
        EvtAppendEvt(EVT_INLAY_SELECT,0,INLAY_OK,0,NULL);
      }

      // handle the event as required
      else
      if (!ApplicationHandleEvent(&event)) 
        if (!SystemHandleEvent(&event)) 
          if (!MenuHandleEvent(&event)) 
            FormDispatchEvent(&event);

      // stop the app?
      if (event.eventType == EVT_APP_STOP) {

        keepFormOpen = FALSE;
        appStop      = TRUE;

        DeviceGetEvent(&bufEvent, 0);

        // is there a power-down event coming?
	if (bufEvent.eventType == EVT_POWER_DOWN)
	  powerDown = TRUE;
      }
    }
  }

  // restore the active form/window
  FormRestoreBitBehind(prevFormID);
  FormSetActiveForm(prevFormID);

  // if not exiting app, return to the previous app
  if (!appStop) {
    EvtAppendEvt(EVT_FORM_LOAD,prevFormID,0,0,NULL);
    EvtAppendEvt(EVT_FORM_CONFIGURE,0,0,0,NULL);
  }
  // need to add the "APP_STOP" event?
  else {
    EvtAppendEvt(EVT_APP_STOP, 0, 0, 0, NULL);

    // need to add the "POWER_DOWN" event?
    if (powerDown) EvtAppendEvt(EVT_POWER_DOWN, 0, 0, 0, NULL);
    else           EvtAppendEvt(bufEvent.eventType,bufEvent.eventID,
                                bufEvent.para1,bufEvent.para2,bufEvent.evtPBP);
  }

  GfxSetDrawWindow(currWindow);
}

/**
 * The Helio Computing Platform event processing loop.
 */
void  
EventLoop()
{
  ObjectID formID;
  EvtType  event;
  RTM      time;
  WORD     timerDiff;

  // reset the timer (just in case)
  RtcGetTime(&globals->evtNullEventTime);

  do {

    DeviceGetEvent(&event, globals->evtTimeOut);
    FormGetActiveFormID(&formID);

    //
    // ANIMATION HANDLING
    // 

    if ((formID == mainForm) && (event.eventType == NULL_EVENT))
    {
      RtcGetTime(&globals->evtNullEventTime);
    }

    // 
    // EVENT HANDLING
    //

    if (!ApplicationHandleEvent(&event)) 
      if (!SystemHandleEvent(&event)) 
        if (!MenuHandleEvent(&event)) 
          FormDispatchEvent(&event);

    //
    // ANIMATION HANDLING
    // 

    if (formID == mainForm)
    {
      // calculate the delay required
      RtcGetTime(&time);
      timerDiff = RtcDiffTime(&globals->evtNullEventTime, &time).l;
      globals->evtTimeOut =
        (timerDiff > globals->ticksPerFrame)
        ? 0
        : (globals->ticksPerFrame - timerDiff);
    }

  } while (event.eventType != EVT_APP_STOP);
}

/**
 * The Helio Computing Platform termination routine.
 */
void  
EndApplication()
{
  // clean up
  pfree(globals);

  // reset the device to its normal state
  GfxTerminate();
  DeviceTerminate();
}

/**
 * The Helio Computing Platform entry routine (mainline).
 *
 * @param cmd    a word value specifying the launch code.
 * @param cmdPBP pointer to a structure associated with the launch code.
 * @return zero if launch successful, non zero otherwise.
 */
BOOLEAN
HelioMain(WORD cmd, void *cmd_ptr)
{
  BOOLEAN result = FALSE;

  // what type of launch was this?
  switch (cmd) 
  {
    case LAUNCH_CMD_NORMAL_LAUNCH:
         {
           InitApplication();
           EventLoop();
           EndApplication();
         }
         result = TRUE;
         break;

    default:
         break;
  }

  return result;
}
