/*
*	bc module. Bytecode manipulation routines (bytecode segments API)
*
*	2001/02/23, Nicholas Christopoulos
*
*	This program is distributed under the terms of the GPL v2.0 or later
*	Download the GNU Public License (GPL) from www.gnu.org
*/

#if !defined(_bc_h)
#define	_bc_h

#include "sys.h"
#include "pmem.h"
#include "kw.h"

#define	BC_ALLOC_INCR			1024
#if defined(OS_ADDR16)
#define	BC_MAX_STORE_SIZE		0x7FFF
#else
#define	BC_MAX_STORE_SIZE		0x7FFFFFFF
#endif

typedef	struct {
	MemHandle	mem_h;		// handle
	code_t		*ptr;		// pointer
	addr_t		cp;			// current position (used by readers not writers)
	addr_t		size;		// allocation size
	addr_t		count;		// current size		(used by writers as the current position)
	}	bc_t;

void	sc_raise(const char *fmt, ...)			SEC(BCSCAN);

void	bc_create(bc_t *bc)						SEC(BCSCAN);
void	bc_destroy(bc_t *bc)  					SEC(BCSCAN);
void	bc_resize(bc_t *bc, dword newsize)		SEC(BCSCAN);

void	bc_add1 (bc_t *bc, byte code) 			SEC(BCSCAN);
void	bc_store1 (bc_t *bc, addr_t offset, byte code) 			SEC(BCSCAN);

void	bc_add_word(bc_t *bc, word code) 			SEC(BCSCAN);
void	bc_add_dword(bc_t *bc, dword code) 			SEC(BCSCAN);

char*	bc_store_string(bc_t *bc, char *src)	SEC(BCSCAN);
char*	bc_store_macro(bc_t *bc, char *src)	SEC(BCSCAN);
void	bc_eoc(bc_t *bc)						SEC(BCSCAN);

void	bc_append(bc_t *dest, bc_t *src)		SEC(BCSCAN);
void	bc_add_n(bc_t *dest, byte *src, dword n)		SEC(BCSCAN);

// add a code
#define	bc_add_code(d,i)		bc_add1((d),(i))

// add a buildin function
void	bc_add_fcode(bc_t *dest, long idx)		SEC(BCSCAN);

// add a buildin procedure
void	bc_add_pcode(bc_t *dest, long idx)		SEC(BCSCAN);

// add an external function
void	bc_add_extfcode(bc_t *dest, int lib, long idx)		SEC(BCSCAN);

// add an external procedure
void	bc_add_extpcode(bc_t *dest, int lib, long idx)		SEC(BCSCAN);

// add an address
void	bc_add_addr(bc_t *bc, addr_t idx)		SEC(BCSCAN);

// add an control (if,repeat,while,...)
void	bc_add_ctrl(bc_t *bc, code_t code, addr_t true_ip, addr_t false_ip) SEC(BCSCAN);

// add an REAL number
#if defined(OS_PREC64)
void	bc_add_creal(bc_t *bc, long double v)			SEC(BCSCAN);
#else
void	bc_add_creal(bc_t *bc, double v)				SEC(BCSCAN);
#endif

// add an INTEGER number
#if defined(OS_PREC64)
void	bc_add_cint(bc_t *bc, long long v)				SEC(BCSCAN);
#else
void	bc_add_cint(bc_t *bc, long v)					SEC(BCSCAN);
#endif

#endif

