/**
*	SmallBASIC, pseudo-compiler: Converts the source to byte-code.
*
*	2000-05-27, Nicholas Christopoulos
*
*	This program is distributed under the terms of the GPL v2.0 or later
*	Download the GNU Public License (GPL) from www.gnu.org
*/

#define SCAN_MODULE
#include "sys.h"
#include "device.h"
#include "kw.h"
#include "bc.h"
#include "scan.h"
#include "smbas.h"

void	bc_scan_cmd(char *text) 				SEC(BCSCAN);
int		bc_scan_inline_if(char *text) 			SEC(BCSCAN);
int		bc_is_keyword(char *name) 				SEC(BCSCAN);
fcode_t	bc_is_func(char *name) 					SEC(BCSCAN);
pcode_t	bc_is_proc(char *name) 					SEC(BCSCAN);
int		bc_is_spopr(char *name) 				SEC(BCSCAN);
long	bc_is_opr(char *name) 					SEC(BCSCAN);
addr_t	bc_search(addr_t ip, code_t code) 		SEC(BCSCAN);

static char	bc_fileName[OS_PATHNAME_SIZE+1];

extern void	expr_parser(bc_t *bc)			 	SEC(BCSCAN);

/* ----------------------------------------------------------------------------------------------------------------------- */

/*
*	GENERIC KEYWORDS (basic bc-types & oldest code)
*/
struct keyword_s keyword_table[] = {
/* real commands */
{ "LOCAL", 		kwLOCAL	},
{ "SUB", 		kwPROC	},
{ "FUNC", 		kwFUNC	},
{ "DEF", 		kwFUNC	},
{ "BYREF",		kwBYREF	},
{ "DECLARE",	kwDECLARE	},
{ "IMPORT",		kwIMPORT	},

{ "LET", 		kwLET	},
{ "CONST", 		kwCONST	},
{ "DIM", 		kwDIM	},
{ "REDIM", 		kwREDIM	},
{ "STOP",		kwSTOP	},
{ "END",		kwEND	},
{ "PRINT",		kwPRINT	},
{ "SPRINT",		kwSPRINT },
{ "INPUT",		kwINPUT	},
{ "SINPUT",		kwSINPUT	},
{ "REM",		kwREM	},
{ "CHAIN",		kwCHAIN	},
{ "ON",			kwON	},

{ "LABEL",		kwLABEL	},
{ "GOTO",		kwGOTO	},
{ "IF",			kwIF	},
{ "THEN",		kwTHEN	},
{ "ELSE",		kwELSE	},
{ "ELIF",		kwELIF	},
{ "ELSEIF",		kwELIF	},
{ "ENDIF",		kwENDIF	},
{ "FI",			kwENDIF	},
{ "FOR",		kwFOR	},
{ "TO",			kwTO	},
{ "STEP",		kwSTEP	},
{ "NEXT",		kwNEXT	},
{ "WHILE",		kwWHILE	},
{ "WEND",		kwWEND	},
{ "REPEAT",		kwREPEAT	},
{ "UNTIL",		kwUNTIL	},
{ "GOSUB",		kwGOSUB	},
{ "RETURN",		kwRETURN	},
{ "READ",		kwREAD	},
{ "DATA",		kwDATA	},
{ "RESTORE",	kwRESTORE	},
{ "EXIT",		kwEXIT	},

{ "ERASE", 		kwERASE	},
{ "USE",		kwUSE	},
{ "USING",		kwUSING		},

{ "LINE",		kwLINE	},
{ "COLOR",		kwCOLOR	},

{ "RUN",		kwRUN	},

{ "OPEN", 		kwOPEN	},
{ "APPEND",		kwAPPEND	},
{ "AS",			kwAS },		// OPEN's args
{ "CLOSE", 		kwCLOSE  },
{ "LINEINPUT", 	kwLINEINPUT },		// The QB's keyword is "LINE INPUT"
{ "SEEK", 		kwSEEK	},
{ "WRITE", 		kwFILEWRITE	},
//{ "READ", 	kwFILEREAD	},
//{ "INPUT", 		kwFINPUT },	// not needed

{ "INSERT",		kwINSERT },
{ "DELETE",		kwDELETE },

/* DEBUG */
{ "TRON",		kwTRON	},
{ "TROFF",		kwTROFF	},

/* for debug */
/* by using small letters, */
/* the keywords are invisible by compiler */
{ "$i32", 		kwTYPE_INT		},
{ "$r64",		kwTYPE_NUM		},
{ "$str",		kwTYPE_STR		},
{ "$log",		kwTYPE_LOGOPR	},
{ "$cmp",		kwTYPE_CMPOPR	},
{ "$add",		kwTYPE_ADDOPR	},
{ "$mul",		kwTYPE_MULOPR	},
{ "$pow",		kwTYPE_POWOPR	},
{ "$unr",		kwTYPE_UNROPR	},
{ "$var",		kwTYPE_VAR	},
{ "$tln",		kwTYPE_LINE	},
{ "$lpr",		kwTYPE_LEVEL_BEGIN	},
{ "$rpr",		kwTYPE_LEVEL_END	},
{ "$crv",		kwTYPE_CRVAR	},
{ "$sep",		kwTYPE_SEP		},
{ "$biF",		kwTYPE_CALLF	},
{ "$biP",		kwTYPE_CALLP	},
{ "$exF",		kwTYPE_CALLEXTF	},
{ "$exP",		kwTYPE_CALLEXTP	},
{ "$ret",		kwTYPE_RET	},
{ "$udp",		kwTYPE_CALL_UDP	},
{ "$udf",		kwTYPE_CALL_UDF	},

{ "", 0 }
};

/* ----------------------------------------------------------------------------------------------------------------------- */

/*
*	OPERATORS (not the symbols)
*/
struct opr_keyword_s opr_table[] = {
{ "AND",  kwTYPE_LOGOPR, '&' 		},
{ "OR",   kwTYPE_LOGOPR, '|' 		},
{ "BAND",  kwTYPE_LOGOPR, OPLOG_BAND },
{ "BOR",   kwTYPE_LOGOPR, OPLOG_BOR	},
{ "XOR",  kwTYPE_LOGOPR, '~'        },
{ "NOT",  kwTYPE_UNROPR, '!'        },
{ "MOD",  kwTYPE_MULOPR, OPLOG_MOD	},
{ "EQV",  kwTYPE_LOGOPR, OPLOG_EQV	},
{ "IMP",  kwTYPE_LOGOPR, OPLOG_IMP	},
{ "NAND", kwTYPE_LOGOPR, OPLOG_NAND	},
{ "NOR",  kwTYPE_LOGOPR, OPLOG_NOR	},
{ "XNOR",  kwTYPE_LOGOPR, OPLOG_NOR	},
{ "IN",  kwTYPE_LOGOPR, OPLOG_IN	},

{ "", 0, 0 }
};

/* ----------------------------------------------------------------------------------------------------------------------- */

/*
*	SPECIAL SEPERATORS
*/
struct spopr_keyword_s spopr_table[] = {
{ "COLOR",		kwCOLOR 	},
{ "FILLED",		kwFILLED	},
{ "FOR",		kwFORSEP	},
{ "INPUT",		kwINPUTSEP	},
{ "OUTPUT",		kwOUTPUTSEP	},
{ "APPEND",		kwAPPENDSEP	},
{ "ACCESS",		kwACCESS	},
{ "USING",		kwUSING		},
{ "SHARED",		kwSHARED	},
{ "AS",			kwAS		},
{ "TO",			kwTO		},
{ "DO",			kwDO		},
{ "STEP",		kwSTEP		},
{ "THEN",		kwTHEN		},
{ "SUB", 		kwPROCSEP	},
{ "FUNC", 		kwFUNCSEP	},
{ "DEF", 		kwFUNCSEP	},
{ "LOOP", 		kwLOOPSEP	},
{ "ON",			kwON		},
{ "OFF",		kwOFF		},
{ "USE",		kwUSE		},

{ "", 0 }
};

/* ----------------------------------------------------------------------------------------------------------------------- */

/*
*	BUILDIN-FUNCTIONS
*/
struct func_keyword_s func_table[] = {
///1234567890123456
{ "ASC",			kwASC },
{ "VAL",			kwVAL },
{ "CHR",			kwCHR },
{ "STR",			kwSTR },
{ "OCT",			kwOCT },
{ "HEX",			kwHEX },
{ "LCASE",			kwLCASE },
{ "LOWER",			kwLCASE },
{ "UCASE",			kwUCASE },
{ "UPPER",			kwUCASE },
{ "LTRIM",			kwLTRIM },
{ "RTRIM",			kwRTRIM },
{ "SPACE",			kwSPACE }, 
{ "SPC",			kwSPACE }, 
{ "TAB",			kwTAB },
{ "CAT",			kwCAT },
{ "ENVIRON",		kwENVIRONF },
{ "ENV",			kwENVIRONF },
{ "TRIM",			kwTRIM },
{ "STRING",			kwSTRING },
{ "SQUEEZE",		kwSQUEEZE },
{ "LEFT",			kwLEFT },
{ "RIGHT",			kwRIGHT },
{ "LEFTOF",			kwLEFTOF },
{ "RIGHTOF",		kwRIGHTOF },
{ "LEFTOFLAST",		kwLEFTOFLAST },
{ "RIGHTOFLAST",	kwRIGHTOFLAST },
{ "MID",			kwMID },
{ "REPLACE",		kwREPLACE },
{ "RUN",			kwRUNF },
{ "INKEY",			kwINKEY },
{ "TIME",			kwTIME },
{ "DATE",			kwDATE },
{ "INSTR",			kwINSTR },
{ "RINSTR",			kwINSTR },
{ "LBOUND",			kwLBOUND },
{ "UBOUND",			kwUBOUND },
{ "LEN",			kwLEN },
{ "EMPTY",			kwEMPTY },
{ "ISARRAY",		kwISARRAY },
{ "ISNUMBER",		kwISNUMBER },
{ "ISSTRING",		kwISSTRING },
{ "ATAN2",			kwATAN2 },
{ "POW",			kwPOW },
{ "ROUND",			kwROUND },
{ "COS",			kwCOS },
{ "SIN",			kwSIN },
{ "TAN",			kwTAN },
{ "COSH",			kwCOSH },
{ "SINH",			kwSINH },
{ "TANH",			kwTANH },
{ "ACOS",			kwACOS },
{ "ASIN",			kwASIN },
{ "ATAN",			kwATAN },
{ "ATN",			kwATAN },
{ "ACOSH",			kwACOSH },
{ "ASINH",	   		kwASINH },
{ "ATANH",			kwATANH },
{ "SQR",	   		kwSQR },
{ "ABS",			kwABS },
{ "EXP",			kwEXP },
{ "LOG",			kwLOG },
{ "LOG10",			kwLOG10 },
{ "FIX",			kwFIX },
{ "INT",			kwINT },
{ "CDBL",			kwCDBL },
{ "CREAL",			kwCDBL },
{ "DEG",			kwDEG },
{ "RAD",			kwRAD },
{ "PEN",			kwPENF },
{ "FLOOR",			kwFLOOR },
{ "CEIL",			kwCEIL },
{ "FRAC",			kwFRAC },
{ "FRE",			kwFRE },
{ "SGN",			kwSGN },
{ "CINT",			kwCINT },
{ "EOF",			kwEOF },
{ "SEEK",			kwSEEKF },
{ "LOF",			kwLOF },
{ "RND",			kwRND },
{ "MAX",			kwMAX },
{ "MIN",			kwMIN },
{ "ABSMAX",			kwABSMAX },
{ "ABSMIN",			kwABSMIN },
{ "SUM",			kwSUM },
{ "SUMSQ",			kwSUMSV },
{ "STATMEAN",		kwSTATMEAN },
{ "STATMEANDEV",	kwSTATMEANDEV },
{ "STATSPREADS",	kwSTATSPREADS },
{ "STATSPREADP",	kwSTATSPREADP },
{ "SEGCOS",			kwSEGCOS },
{ "SEGSIN",			kwSEGSIN },
{ "SEGLEN",			kwSEGLEN },
{ "POLYAREA",		kwPOLYAREA },
{ "PTDISTSEG",		kwPTDISTSEG },
{ "PTSIGN",	   		kwPTSIGN },
{ "PTDISTLN",		kwPTDISTLN },
{ "POINT",			kwPOINT },
{ "XPOS",			kwXPOS },
{ "YPOS",			kwYPOS },
{ "INPUT",			kwINPUTF },
{ "ARRAY",			kwCODEARRAY }, 
{ "LINEQN",			kwGAUSSJORDAN },
{ "FILES",			kwFILES },
{ "INVERSE",		kwINVERSE },
{ "DETERM",			kwDETERM },
{ "JULIAN",			kwJULIAN },
{ "DATEFMT",		kwDATEFMT },
{ "WEEKDAY",		kwWDAY },
{ "IF",				kwIFF },
{ "IFF",			kwIFF },
{ "FORMAT",			kwFORMAT },
{ "FREEFILE",		kwFREEFILE },
{ "TICKS",			kwTICKS },
{ "TICKSPERSEC",	kwTICKSPERSEC },
{ "TIMER", 			kwTIMER }, 
{ "PROGLINE",		kwPROGLINE },
{ "RUN",			kwRUNF	},
{ "TXTW",			kwTEXTWIDTH },
{ "TXTH",			kwTEXTHEIGHT },
{ "TEXTWIDTH", 		kwTEXTWIDTH },
{ "TEXTHEIGHT",		kwTEXTHEIGHT },
{ "EXIST",			kwEXIST },
{ "ISFILE",			kwISFILE },
{ "ISDIR",			kwISDIR },
{ "ISLINK",			kwISLINK },
{ "ACCESS",			kwACCESSF },
{ "RGB",			kwRGB },
{ "RGBF",			kwRGBF },
{ "BIN",			kwBIN },
{ "ENCLOSE",		kwENCLOSE },
{ "DISCLOSE",		kwDISCLOSE },
{ "TRANSLATE",		kwTRANSLATEF },

{ "", 0 }
};

/* ----------------------------------------------------------------------------------------------------------------------- */

/*
*	BUILD-IN PROCEDURES
*/
struct proc_keyword_s proc_table[] = {
///1234567890123456
{ "CLS",		kwCLS	},
{ "RTE",		kwRTE	},
//kwSHELL,
{ "ENVIRON",	kwENVIRON },
{ "ENV",		kwENVIRON },
{ "LOCATE",		kwLOCATE	},
{ "AT",			kwAT		},
{ "PEN",		kwPEN	},
{ "DATEDMY",	kwDATEDMY },
{ "BEEP",		kwBEEP	},
{ "SOUND",		kwSOUND	},
{ "PSET",		kwPSET	},
{ "RECT",		kwRECT	},
{ "CIRCLE",		kwCIRCLE	},
{ "RANDOMIZE",	kwRANDOMIZE	},
{ "SPLIT",		kwSPLIT },
{ "WSPLIT",		kwWSPLIT },
{ "JOIN",		kwWJOIN	},
{ "PAUSE",		kwPAUSE	},
{ "DELAY",		kwDELAY	},
{ "ARC",		kwARC	},
{ "DRAW",		kwDRAW	},
{ "PAINT",		kwPAINT	},
{ "PLAY",		kwPLAY	},
{ "SORT",		kwSORT	},
{ "SEARCH",		kwSEARCH },
{ "ROOT",		kwROOT },
{ "DIFFEQN",	kwDIFFEQ },
{ "CHART",		kwCHART	},
{ "WINDOW",		kwWINDOW },
{ "VIEW",		kwVIEW },
{ "DRAWPOLY",	kwDRAWPOLY },
{ "M3IDENT",	kwM3IDENT },
{ "M3ROTATE",	kwM3ROTATE },
{ "M3SCALE", 	kwM3SCALE },
{ "M3TRANS",	kwM3TRANSLATE },
{ "M3APPLY",	kwM3APPLY },
{ "INTERSECT",	kwSEGINTERSECT },
{ "POLYEXT",	kwPOLYEXT },
{ "DERIV",		kwDERIV },
{ "KILL", 		kwKILL	 },
{ "RENAME", 	kwRENAME	},
{ "COPY", 		kwCOPY	},
{ "CHDIR", 		kwCHDIR	},
{ "MKDIR", 		kwMKDIR	},
{ "RMDIR", 		kwRMDIR	},
{ "TLOAD",	 	kwLOADLN },
{ "TSAVE",		kwSAVELN },
{ "LOCK",		kwFLOCK },
{ "CHMOD",		kwCHMOD },
{ "PLOT",		kwPLOT },
{ "LOGPRINT",	kwLOGPRINT	},
{ "STKDUMP",	kwSTKDUMP	},
{ "SWAP",		kwSWAP	},

{ "", 0 }
};

/* ----------------------------------------------------------------------------------------------------------------------- */

/*
*	EXTERNAL FUNCTIONS & PROCEDURES
*/
ext_proc_node_t	*ext_proc_table;
int		ext_proc_size;
int		ext_proc_count;

ext_func_node_t	*ext_func_table;
int		ext_func_size;
int		ext_func_count;

/*
*	reset the external proc/func lists
*/
void	bc_reset_extlist(void)		SEC(BCSCAN);
void	bc_reset_extlist(void)
{
	if	( ext_func_table )
		tmp_free(ext_func_table);
	ext_func_table = NULL;
	ext_func_count = ext_func_size = 0;

	if	( ext_proc_table )
		tmp_free(ext_proc_table);
	ext_proc_table = NULL;
	ext_proc_count = ext_proc_size = 0;
}

/*
*	Add an external procedure to the list
*/
int		bc_add_extproc(const char *proc_name, int lib_id)	SEC(BCSCAN);
int		bc_add_extproc(const char *proc_name, int lib_id)
{
	// TODO: scan for conflicts
	if	( ext_proc_table == NULL )	{
		ext_proc_size   = 16;
		ext_proc_table  = (ext_proc_node_t*) tmp_alloc(sizeof(ext_proc_node_t) * ext_proc_size);
		}
	else if ( ext_proc_size <= (ext_proc_count+1) )	{
		ext_proc_size  += 16;
		ext_proc_table  = (ext_proc_node_t*) tmp_realloc(ext_proc_table, sizeof(ext_proc_node_t) * ext_proc_size);
		}

	ext_proc_table[ext_proc_count].lib_id = lib_id;
	strcpy(ext_proc_table[ext_proc_count].name, proc_name);
	strupr(ext_proc_table[ext_proc_count].name);
	ext_proc_count ++;
	return ext_proc_count-1;
}

/*
*	Add an external function to the list
*/
int		bc_add_extfunc(const char *func_name, int lib_id)	SEC(BCSCAN);
int		bc_add_extfunc(const char *func_name, int lib_id)
{
	// TODO: scan for conflicts
	if	( ext_func_table == NULL )	{
		ext_func_size   = 16;
		ext_func_table  = (ext_func_node_t*) tmp_alloc(sizeof(ext_func_node_t) * ext_func_size);
		}
	else if ( ext_func_size <= (ext_func_count+1) )	{
		ext_func_size  += 16;
		ext_func_table  = (ext_func_node_t*) tmp_realloc(ext_func_table, sizeof(ext_func_node_t) * ext_func_size);
		}

	ext_func_table[ext_func_count].lib_id = lib_id;
	strcpy(ext_func_table[ext_func_count].name, func_name);
	strupr(ext_func_table[ext_func_count].name);
	ext_func_count ++;
	return ext_func_count-1;
}

/*
*	returns the external procedure id
*/
int		bc_is_extproc(const char *name)		SEC(BCSCAN);
int		bc_is_extproc(const char *name)
{
	int		i;

	for ( i = 0; i < ext_proc_count; i ++ )	{
		if	( strcmp(ext_proc_table[i].name, name) == 0 )
			return i;
		}
	return -1;
}

/*
*	returns the external function id
*/
int		bc_is_extfunc(const char *name)		SEC(BCSCAN);
int		bc_is_extfunc(const char *name)
{
	int		i;

	for ( i = 0; i < ext_func_count; i ++ )	{
		if	( strcmp(ext_func_table[i].name, name) == 0 )
			return i;
		}
	return -1;
}

/* ----------------------------------------------------------------------------------------------------------------------- */

/*
*	Notes:
*		block_level = the depth of nested block
*		block_id	= unique number of each block (based on stack use)
*
*		Example:
*		? xxx			' level 0, id 0
*		for i=1 to 20	' level 1, id 1
*			? yyy		' level 1, id 1
*			if a=1 		' level 2, id 2 (our IF uses stack)
*				...		' level 2, id 2
*			else		' level 2, id 2	// not 3
*				...		' level 2, id 2
*			fi			' level 2, id 2
*			if a=2		' level 2, id 3
*				...		' level 2, id 3
*			fi			' level 2, id 3
*			? zzz		' level 1, id 1
*		next			' level 1, id 1
*		? ooo			' level 0, id 0
*/

int				scan_error;
int				scan_line;			// source line count

static int		block_level;		// block level (FOR-NEXT,IF-FI,etc)
static int		block_id;			// unique ID for blocks (FOR-NEXT,IF-FI,etc)

static addr_t	first_data_ip = INVALID_ADDR;

// ndc: 2001-03-01 static allocation (non tmp_alloc()) does not use the dynamic RAM and reduces the fragmentation
static char	*bc_name;
static char	*bc_parm;
static char	*bc_temp;
static char *bc_proc;
static char	*bc_sec;	// used by sberr.c
//char		bc_sec[SB_KEYWORD_SIZE+1];	// used by sberr.c
static int  bc_proc_level = 0;
static byte	bc_use_global_vartable = 0;		// flag - uses global variable-table for the next commands

static bc_t	bc_prog;
static bc_t	bc_data;

static char kw_do_close_cmd[64];

#define	GROWSIZE	128

/*
*	variable node
*/
struct var_s	{
	char	*name;
	};
typedef struct var_s var_t;

static var_t	*var_table;
static bid_t	var_count;
static bid_t	var_size;

/*
*	LABELS LIST
*/
struct label_s	{
	char	name[SB_KEYWORD_SIZE+1];
	addr_t	ip;			// instruction pointer
	byte	level;		// block level (used for GOTOs)
	bid_t	block_id;	// block_id (FOR-NEXT,IF-FI,etc) used for GOTOs
	addr_t	dp;			// data pointer
	};
typedef struct label_s label_t;

static dbt_t	lab_table;
static bid_t	lab_count;

/*
*	PROCEDURES/FUNCTIONS LIST
*/
struct proc_s	{
	char	*name;
	addr_t	ip;			// instruction pointer
	bid_t	vid;		// variable index (for functions)
	byte	level;		// block level (used for GOTOs)
	bid_t	block_id;	// block_id (FOR-NEXT,IF-FI,etc) used for GOTOs
	};
typedef struct proc_s udp_t;

static udp_t	*udp_table;
static bid_t	udp_count;
static bid_t	udp_size;

/*
*	PASS2 - STACK
*/
struct pass_node_s	{
	char	sec[SB_KEYWORD_SIZE+1];
	addr_t	pos;		// instruction position
	int		line;		// the source-text line
	byte	level;		// block level
	bid_t	block_id;	// block ID	
	};
typedef struct pass_node_s pass_node_t;

static dbt_t	bc_stack;
static bid_t		stk_count;

extern void	sc_raise2(const char *fmt, int line, const char *buff);	// sberr

/*
*	error messages
*/
void	sc_raise(const char *fmt, ...)
{
	char	*buff;
	va_list ap;

	va_start(ap, fmt);

	scan_error = 1;

	buff = tmp_alloc(SB_SOURCELINE_SIZE+1);
	#if defined(_PalmOS)
	StrVPrintF(buff, fmt, ap);
	#else
	vsprintf(buff, fmt, ap);
	#endif
	va_end(ap);

	sc_raise2(bc_sec, scan_line, buff);	// sberr.h
	tmp_free(buff);
}

/*
*	prepare name (keywords, variables, labels, proc/func names)
*/
char*	bc_prepname(char *dest, const char *source, int size) SEC(BCSCAN);
char*	bc_prepname(char *dest, const char *source, int size)
{
	char	*p = (char *) source;

	while ( *p == ' ' )		p ++;
	strncpy(dest, p, size);
	dest[size] = '\0';
	p = dest;
	while ( *p && (is_alpha(*p) || is_digit(*p) || *p == '$' || *p == '/' || *p == '_') )
		p ++;
	*p = '\0';

	str_alltrim(dest);
	return dest;
}

/*
*	returns the ID of the label. If there is no one, then it creates one
*/
bid_t	bc_get_label_id(const char *label_name) SEC(BCSCAN);
bid_t	bc_get_label_id(const char *label_name)
{
	bid_t	idx = -1, i;
	char	name[SB_KEYWORD_SIZE+1];
	label_t	label;

	bc_prepname(name, label_name, SB_KEYWORD_SIZE);

	for ( i = 0; i < lab_count; i ++ )	{
		dbt_read(lab_table, i, &label, sizeof(label_t));
		if	( strcmp(label.name, name) == 0 )	{
			idx = i;
			break;
			}
		}

	if	( idx == -1 )	{
		strcpy(label.name, name);
		label.ip = INVALID_ADDR;
		label.dp = INVALID_ADDR;
		label.level = block_level;
		label.block_id = block_id;

		dbt_write(lab_table, lab_count, &label, sizeof(label_t));
		idx = lab_count;
		lab_count ++;
		}

	return idx;
}

/*
*	set LABEL's position (IP)
*/
void	bc_set_label_ip(bid_t idx) SEC(BCSCAN);
void	bc_set_label_ip(bid_t idx)
{
	label_t	label;

	dbt_read(lab_table, idx, &label, sizeof(label_t));
	label.ip = bc_prog.count;
	label.dp = bc_data.count;
	label.level = block_level;
	label.block_id = block_id;
	dbt_write(lab_table, idx, &label, sizeof(label_t));
}

/*
*	returns the ID of the UDP/UDF
*/
bid_t	bc_getudp_id(const char *proc_name) SEC(BCSCAN);
bid_t	bc_getudp_id(const char *proc_name)
{
	bid_t	idx = -1;
	bid_t	i;
	char	tmp[SB_KEYWORD_SIZE+1];
	char	*name = bc_temp;

	if	( bc_proc_level )	{
		// search the procedure's table for this LOCAL PROC
		bc_prepname(tmp, proc_name, SB_KEYWORD_SIZE);
		strcpy(name, bc_proc);
		strcat(name, "/");
		strcat(name, tmp);
		for ( i = 0; i < udp_count; i ++ )	{
			if	( strcmp(udp_table[i].name, name) == 0 )	{
				idx = i;
				break;
				}
			}
		}

	if	( idx == -1 )	{
		// search the procedure's table for this GLOBAL PROC
		bc_prepname(name, proc_name, SB_KEYWORD_SIZE);
		for ( i = 0; i < udp_count; i ++ )	{
			if	( strcmp(udp_table[i].name, name) == 0 )	{
				idx = i;
				break;
				}
			}
		}

	return idx;
}

/*
*	creates a new UDP/UDF node
*	and returns the new ID
*/
bid_t	bc_addudp(const char *proc_name) SEC(BCSCAN);
bid_t	bc_addudp(const char *proc_name)
{
	char	tmp[SB_KEYWORD_SIZE+1];
	char	*name = bc_temp;
	bid_t	idx = -1, i;

	//
	bc_prepname(tmp, proc_name, SB_KEYWORD_SIZE);
	if	( bc_proc_level )	{
		strcpy(name, bc_proc);
		strcat(name, "/");
		strcat(name, tmp);
		}
	else
		strcpy(name, tmp);

	for ( i = 0; i < udp_count; i ++ )	{
		if	( strcmp(udp_table[i].name, name) == 0 )	{
			idx = i;
			break;
			}
		}

	if	( idx == -1 )	{
		if ( udp_count >= udp_size ) {
			udp_size += GROWSIZE;
			udp_table = tmp_realloc(udp_table, udp_size * sizeof(udp_t));
			}

		if	( !is_alpha(proc_name[0]) )
			sc_raise("Wrong procedure/function name: %s", proc_name);
		else	{
			udp_table[udp_count].name = tmp_alloc(strlen(name)+1);
			udp_table[udp_count].ip = INVALID_ADDR; //bc_prog.count;
			udp_table[udp_count].level = block_level;
			udp_table[udp_count].block_id = block_id;
			strcpy(udp_table[udp_count].name, name);
			idx = udp_count;
			udp_count ++;
			}
		}

	return idx;
}

/*
*	sets the IP of the user-defined-procedure (or function)
*/
bid_t	bc_udp_setip(const char *proc_name, addr_t ip) SEC(BCSCAN);
bid_t	bc_udp_setip(const char *proc_name, addr_t ip)
{
	bid_t	idx;
	
	idx = bc_getudp_id(proc_name);
	if	( idx != -1 )	{
		udp_table[idx].ip = bc_prog.count;
		udp_table[idx].level = block_level;
		udp_table[idx].block_id = block_id;
		}
	return idx;
}

/*
*	Returns the IP of an UDP/UDF
*/
addr_t	bc_udp_getip(const char *proc_name) SEC(BCSCAN);
addr_t	bc_udp_getip(const char *proc_name)
{
	bid_t	idx;
	
	idx = bc_getudp_id(proc_name);
	if	( idx != -1 )	
		return udp_table[idx].ip;
	return INVALID_ADDR;
}

/*
*/
char	*get_param_sect(char *text, const char *delim, char *dest) SEC(BCSCAN);
char	*get_param_sect(char *text, const char *delim, char *dest)
{
	char	*p = (char *) text;
	char	*d = dest;
	int		quotes = 0, level = 0, skip_ch = 0;

	if	( p == NULL )	{
		*dest = '\0';
		return 0;
		}

	while ( is_space(*p) )	p ++;

	while ( *p )	{
		if	( quotes )	{
			if	( *p == '\"' )
				quotes = 0;
			}
		else	{
			switch ( *p )	{
			case	'\"':
				quotes = 1;
				break;
			case	'(':
				level ++;
				break;
			case	')':
				level --;
				break;
			case	'\n':
			case	'\r':
				skip_ch = 1;
				break;
				};
			}

		// delim check
		if	( delim != NULL && level <= 0 && quotes == 0 )	{
			if	( strchr(delim, *p) != NULL )	
				break;
			}

		// copy
		if	( !skip_ch )	{
			*d = *p;
			d ++;
			}
		else
			skip_ch = 0;

		p ++;
		}

	*d = '\0';
	
	if	( quotes )
		sc_raise("Expression: Missing (\"), string remains open");
	if	( level > 0 )
		sc_raise("Expression: Missing \')\'");
	if	( level < 0 )
		sc_raise("Expression: Missing \'(\'");

	str_alltrim(dest);
	return p;
}

/*
*/
int		bc_get_error(void) SEC(BCSCAN);
int		bc_get_error()
{
	return scan_error;
}

/*
*	checking for missing labels
*/
void	bc_check_labels(void) SEC(BCSCAN);
void	bc_check_labels()
{
	bid_t		i;
	label_t		label;

	for ( i = 0; i < lab_count; i ++ )	{
		dbt_read(lab_table, i, &label, sizeof(label_t));
		if	( label.ip == INVALID_ADDR )	{
			sc_raise("Label [%s] is not defined", label.name);
			break;
			}
		}
}

/*
*	returns the id of the variable 'name'
*/
bid_t	bc_get_var_id(const char *var_name) SEC(BCSCAN);
bid_t	bc_get_var_id(const char *var_name)
{
	bid_t	idx = -1, i;
	char	tmp[SB_KEYWORD_SIZE+1];
	char	*name = bc_temp;

	if	( bc_proc_level && (bc_use_global_vartable == 0) )	{
		// search local name-space
		bc_prepname(tmp, var_name, SB_KEYWORD_SIZE);
		xsprintf(name, "%s/%s", bc_proc, tmp);
		for ( i = 0; i < var_count; i ++ )	{
			if	( strcmp(var_table[i].name, name) == 0 )	{
				idx = i;
				break;
				}
			}
		}

	if	( idx == -1 )	{
		// search global name-space
		bc_prepname(name, var_name, SB_KEYWORD_SIZE);
		for ( i = 0; i < var_count; i ++ )	{
			if	( strcmp(var_table[i].name, name) == 0 )	{
				idx = i;
				break;
				}
			}
		}

	if	( idx == -1 )	{
		// create one
		if ( var_count >= var_size ) {
			var_size += GROWSIZE;
			var_table = tmp_realloc(var_table, var_size * sizeof(var_t));
			}

		if	( !(is_alpha(var_name[0]) || var_name[0] == '_') )
			sc_raise("Wrong variable name: %s", var_name);
		else	{
			var_table[var_count].name = tmp_alloc(strlen(name)+1);
			strcpy(var_table[var_count].name, name);
			idx = var_count;
			var_count ++;
			}
		}

	return idx;
}

/*
*	adds a mark in stack at the current code position
*/
void	bc_push1(addr_t ip) SEC(BCSCAN);
void	bc_push1(addr_t ip)
{
	pass_node_t		node;

	strcpy(node.sec, bc_sec);
	node.pos      = ip;
	node.level    = block_level;
	node.block_id = block_id;
	node.line     = scan_line;
	dbt_write(bc_stack, stk_count, &node, sizeof(pass_node_t));

	stk_count ++;
}

/*
*	adds a mark in stack at the current code position
*/
void	bc_push(void) SEC(BCSCAN);
void	bc_push()
{
	bc_push1(bc_prog.count);
}

/*
*	returns the keyword code 
*/
int		bc_is_keyword(char *name)
{
	int		i, idx;
	byte	dolar_bug = 0;

//	Code to enable the $ but not for keywords (INKEY$=INKEY, PRINT$=PRINT !!!)
//	I don't want to increase the size of keywords table.
	idx = strlen(name) - 1;
	if	( name[idx] == '$' )	{
		name[idx] = '\0';
		dolar_bug ++;
		}

	for ( i = 0; keyword_table[i].name[0] != '\0'; i ++ )	{
		if	( strcmp(keyword_table[i].name, name) == 0 )
			return keyword_table[i].code;
		}

	if	( dolar_bug )
		name[idx] = '$';

	return -1;
}

/*
*	returns the keyword code (buildin functions)
*/
fcode_t	bc_is_func(char *name)
{
	fcode_t		i;
	int			idx;
	byte		dolar_bug = 0;

//	Code to enable the $ but not for keywords (INKEY$=INKEY, PRINT$=PRINT !!!)
//	I don't want to increase the size of keywords table.
	idx = strlen(name) - 1;
	if	( name[idx] == '$' )	{
		name[idx] = '\0';
		dolar_bug ++;
		}

	for ( i = 0; func_table[i].name[0] != '\0'; i ++ )	{
		if	( strcmp(func_table[i].name, name) == 0 )
			return func_table[i].fcode;
		}

	if	( dolar_bug )
		name[idx] = '$';

	return -1;
}

/*
*	returns the keyword code (buildin procedures)
*/
pcode_t	bc_is_proc(char *name)
{
	pcode_t		i;

	for ( i = 0; proc_table[i].name[0] != '\0'; i ++ )	{
		if	( strcmp(proc_table[i].name, name) == 0 )
			return proc_table[i].pcode;
		}

	return -1;
}

/*
*	returns the keyword code (special separators)
*/
int		bc_is_spopr(char *name)
{
	int		i;

	for ( i = 0; spopr_table[i].name[0] != '\0'; i ++ )	{
		if	( strcmp(spopr_table[i].name, name) == 0 )
			return spopr_table[i].code;
		}

	return -1;
}

/*
*	returns the keyword code (operators)
*/
long	bc_is_opr(char *name)
{
	int		i;

	for ( i = 0; opr_table[i].name[0] != '\0'; i ++ )	{
		if	( strcmp(opr_table[i].name, name) == 0 )
			return ((opr_table[i].code << 8) | opr_table[i].opr);
		}

	return -1;
}

/*
*/
char*	bc_next_char(char *source)	SEC(BCSCAN);
char*	bc_next_char(char *source)
{
	char	*p = source;

	while ( *p )	{
		if	( *p != ' ' )
			return p;
		p ++;
		}
	return p;
}

/*
*/
char*	bc_prev_char(const char *root, const char *ptr)	SEC(BCSCAN);
char*	bc_prev_char(const char *root, const char *ptr)
{
	char	*p = (char *) ptr;
	
	if	( p > root )
		p --;
	else
		return (char *) root;

	while ( p > root )	{
		if	( *p != ' ' )
			return p;
		p --;
		}
	return p;
}

/*
*	get keyword name
*/
char	*bc_getkeywordstr(char *text, char *dest) SEC(BCSCAN);
char	*bc_getkeywordstr(char *text, char *dest)
{
	char	*p = (char *) text;
	char	*d = dest;

	if	( p == NULL )	{
		*dest = '\0';
		return 0;
		}

	while ( is_space(*p) )	p ++;

	if	( *p == '?' )	{
		strcpy(dest, "PRINT");
		p ++;
		while ( is_space(*p) )	p ++;
		return p;
		}

	if	( *p == '\'' || *p == '#' )	{
		strcpy(dest, "REM");
		p ++;
		while ( is_space(*p) )	p ++;
		return p;
		}

	while ( is_alnum(*p) || (*p == '_') )	{
		*d = *p;
		d ++; p ++;
		}

//	Code to kill the $
//	if	( *p == '$' )	
//		p ++;

//	Code to enable the $
	if	( *p == '$' )
		*d ++ = *p ++;

	*d = '\0';
	while ( is_space(*p) )	p ++;
	return p;
}

/*
*	scan expression
*/
void	bc_scan_expr(char *expr, byte no_parser) SEC(BCSCAN);
void	bc_scan_expr(char *expr, byte no_parser)
{
	char	*ptr = (char *) expr;
	long	idx;
	int		level = 0, check_udf = 0;
	int		kw_do_found = 0;
	int		tp;
	addr_t	w, stip, cip;
	long	lv = 0;
	double	dv = 0;
	bc_t	bc;

	bc_use_global_vartable = 0;				// check local-variables first
	str_alltrim(expr);

	if	( *ptr == '\0' )	
		return;

	bc_create(&bc);

	while ( *ptr )	{

		if	( is_digit(*ptr) || *ptr == '.' || (*ptr == '&' && strchr("XHOB", *(ptr+1) ) ) )	{
			/*
			*	A CONSTANT NUMBER
			*/
			ptr = get_numexpr(ptr, bc_name, &tp, &lv, &dv);
			switch ( tp )	{
			case	1:
				bc_add_cint(&bc, lv);
				continue;
			case	2:
				bc_add_creal(&bc, dv);
				continue;
			default:
				sc_raise("Error on numeric expression");
				}
			}
		else if ( *ptr == '\'' /* || *ptr == '#' */ )	{ // remarks
			break;		
			}
		else if	( is_alpha(*ptr) || *ptr == '?' || *ptr == '_' )	{
			/*
			*	A NAME 
			*/
			ptr = bc_getkeywordstr(ptr, bc_name);
			idx = bc_is_func(bc_name);

			// special case for INPUT
			if	( idx == kwINPUTF )	{
				if	( *bc_next_char(ptr) != '(' )
					idx = -1;	// INPUT is SPECIAL SEPARATOR (OPEN...FOR INPUT...)
				}

			if	( idx != -1 )	{
				/*
				*	IS A FUNCTION
				*/
				bc_add_fcode(&bc, idx);
				check_udf ++;
				}
			else	{
				/*
				*	CHECK SPECIAL SEPARATORS
				*/
				idx = bc_is_spopr(bc_name);
				if	( idx != -1 )	{
					if	( idx == kwUSE )	{
						bc_add_code(&bc, idx);
						bc_add_addr(&bc, 0);
						bc_add_addr(&bc, 0);
						bc_use_global_vartable = 1;	// all the next variables are global (needed for X)
						check_udf ++;
						}
					else if	( idx == kwDO )	{
						while ( *ptr == ' ' )	ptr ++;
						if	( strlen(ptr) )
							kw_do_found = 1;
						break;
						}
					else
						bc_add_code(&bc, idx);
					}
				else	{
					/*
					*	NOT A COMMAND, CHECK OPERATORS
					*/
					idx = bc_is_opr(bc_name);
					if	( idx != -1 )	{
						bc_add_code(&bc, idx >> 8);
						bc_add_code(&bc, idx & 0xFF);
						}
					else	{
						/*
						*	EXTERNAL FUNCTION
						*/
						idx = bc_is_extfunc(bc_name);
						if	( idx != -1 )	{
							bc_add_extfcode(&bc, ext_func_table[idx].lib_id, idx);
							}
						else	{
							idx = bc_is_keyword(bc_name);
							if	( idx == -1 )
								idx = bc_is_proc(bc_name);

							if	( idx != -1 )	{
								sc_raise("Statement %s must be on the left side (the first keyword of the line)", bc_name);
								}
							else	{
								/*
								*	UDF OR VARIABLE
								*/
								int		udf;

								udf = bc_getudp_id(bc_name);
								if	( udf != -1 )	{
									// UDF
									bc_add_code(&bc, kwTYPE_CALL_UDF);
									bc_add_addr(&bc, udf);
									bc_add_addr(&bc, 0);
									check_udf ++;
									}
								else	{
									// VARIABLE
									while ( *ptr == ' ' ) ptr ++;
									if	( *ptr == '(' )	{
										if	( *(ptr+1) == ')' )	{
											// null array
											ptr += 2;
											}
										}

									w = bc_get_var_id(bc_name);
									bc_add_code(&bc, kwTYPE_VAR);
									bc_add_addr(&bc, w);
									}
								} // kw
							} // extf
						} // opr
					} // sp. sep
				} // check sep
			}	// isalpha
		else if ( *ptr == ',' || *ptr == ';' || *ptr == '#' )	{
			// parameter separator
			bc_add_code(&bc, kwTYPE_SEP);
			bc_add_code(&bc, *ptr);

			ptr ++;
			}
		else if ( *ptr == '\"' )	{
			// string
			ptr = bc_store_string(&bc, ptr);
			}
		else if ( *ptr == '[' )	{
			ptr ++;
			level ++;
			bc_add_fcode(&bc, kwCODEARRAY);
			bc_add_code(&bc, kwTYPE_LEVEL_BEGIN);
			}
		else if ( *ptr == '(' )	{
			// parenthesis
			level ++;
			bc_add_code(&bc, kwTYPE_LEVEL_BEGIN);

			ptr ++;
			}
		else if ( *ptr == ')' || *ptr == ']' )	{
			// parenthesis
			bc_add_code(&bc, kwTYPE_LEVEL_END);

			level --;
			ptr ++;
			}
		else if ( is_space(*ptr) )	
			// null characters
			ptr ++;
		else	{
			// operators
			if	( *ptr == '+' || *ptr == '-' )	{
				bc_add_code(&bc, kwTYPE_ADDOPR);
				bc_add_code(&bc, *ptr);
				}
			else if	( *ptr == '*' || *ptr == '/' || *ptr == '\\' || *ptr == '%' )	{
				bc_add_code(&bc, kwTYPE_MULOPR);
				bc_add_code(&bc, *ptr);
				}
			else if	( *ptr == '^' )	{
				bc_add_code(&bc, kwTYPE_POWOPR);
				bc_add_code(&bc, *ptr);
				}
			else if	( strncmp(ptr, "<=", 2) == 0 || strncmp(ptr, "=<", 2) == 0 )	{
				bc_add_code(&bc, kwTYPE_CMPOPR);
				bc_add_code(&bc, OPLOG_LE);

				ptr ++;
				}
			else if	( strncmp(ptr, ">=", 2) == 0 || strncmp(ptr, "=>", 2) == 0 )	{
				bc_add_code(&bc, kwTYPE_CMPOPR);
				bc_add_code(&bc, OPLOG_GE);

				ptr ++;
				}
			else if	( strncmp(ptr, "<>", 2) == 0 || strncmp(ptr, "!=", 2) == 0 )	{
				bc_add_code(&bc, kwTYPE_CMPOPR);
				bc_add_code(&bc, OPLOG_NE);
				ptr ++;
				}
			else if	( *ptr == '=' || *ptr == '>' || *ptr == '<' )	{
				bc_add_code(&bc, kwTYPE_CMPOPR);
				bc_add_code(&bc, *ptr);
				}
			else if	( strncmp(ptr, "&&", 2) == 0 || strncmp(ptr, "||", 2) == 0 )	{
				bc_add_code(&bc, kwTYPE_LOGOPR);
				bc_add_code(&bc, *ptr);
				ptr ++;
				}
			else if	( *ptr == '&' )	{
				bc_add_code(&bc, kwTYPE_LOGOPR);
				bc_add_code(&bc, OPLOG_BAND);
				}
			else if	( *ptr == '|' )	{
				bc_add_code(&bc, kwTYPE_LOGOPR);
				bc_add_code(&bc, OPLOG_BOR);
				}
			else if	( *ptr == '~' )	{
				bc_add_code(&bc, kwTYPE_UNROPR);
				bc_add_code(&bc, OPLOG_INV);
				}
			else if	( *ptr == '!' )	{
				bc_add_code(&bc, kwTYPE_UNROPR);
				bc_add_code(&bc, *ptr);
				}
			else
				sc_raise("Unknown operator: '%c'", *ptr);

			ptr ++;
			}
		};

	///////////////////////////
	if	( level )
		sc_raise("Missing ')'");

	if	( !scan_error )	{
		if	( no_parser == 0 )	{
			// optimization
			bc_add_code(&bc, kwTYPE_EOC);
			expr_parser(&bc);
			}

		if	( bc.count )	{
			stip = bc_prog.count;
			bc_append(&bc_prog, &bc);

			if	( check_udf )	{
				cip = stip;
				while ( (cip = bc_search(cip, kwUSE)) != INVALID_ADDR )	{
					bc_push1(cip);
					cip += (1+ADDRSZ+ADDRSZ);
					}

				cip = stip;
				while ( (cip = bc_search(cip, kwTYPE_CALL_UDF)) != INVALID_ADDR )	{
					bc_push1(cip);
					cip += (1+ADDRSZ+ADDRSZ);
					}

				}
			}
		bc_eoc(&bc_prog);
		}

	// clean-up
	bc_use_global_vartable = 0;				// check local-variables first
	bc_destroy(&bc);

	if	( kw_do_found )	{
		if	( strlen(kw_do_close_cmd) )	{
			bc_scan_cmd(ptr);
			bc_scan_cmd(kw_do_close_cmd);
			strcpy(kw_do_close_cmd, "");
			}
		else	
			sc_raise("Invalid use of DO keyword");
		}
}

/*
*	Converts DATA commands to bytecode
*/
void	bc_scan_data(char *source) SEC(BCSCAN);
void	bc_scan_data(char *source)
{
	char	*ptr = source;
	char	*commap;
	long	lv = 0;
	double	dv = 0, sign = 1;
	char	*tmp = bc_temp;
	int		quotes;
	int		tp;

	while ( *ptr )	{
		while ( *ptr == ' ' )	ptr ++;

		if	( *ptr == '\0' )	
			break;
		else if	( *ptr == ',' )	{
			bc_add_code(&bc_data, kwTYPE_EOC);
			ptr ++;
			}
		else	{
			// find the end of the element
			commap = ptr;
			quotes = 0;
			while ( *commap )	{
				if	( *commap == '\"' )
					quotes = !quotes;
				else if ( (*commap == ',') && (quotes == 0) )
					break;

				commap ++;
				}
			if	( *commap == '\0' )
				commap = NULL;

			if	( commap != NULL )
				*commap = '\0';

			if  ((*ptr == '-' || *ptr == '+') && strchr("0123456789.", *(ptr+1)))	{ 
				if	( *ptr == '-' )
					sign = -1;
				ptr ++;
				}
			else
				sign = 1;

			if	( is_digit(*ptr) || *ptr == '.' 
					|| (*ptr == '&' && strchr("XHOB", *(ptr+1))) )	{

				//	number - constant
				ptr = get_numexpr(ptr, tmp, &tp, &lv, &dv);
				switch ( tp )	{
				case	1:
					bc_add_cint(&bc_data, lv * sign);
					break;
				case	2:
					bc_add_creal(&bc_data, dv * sign);
					break;
				default:
					sc_raise("Error on numeric expression");
					}
				}
			else	{
				// add it as string
				if	( *ptr != '\"' )	{
					strcpy(tmp, "\"");
					strcat(tmp,  ptr);
					strcat(tmp, "\"");
					bc_store_string(&bc_data, tmp);
					if	( commap )
						ptr = commap;
					else
						ptr = ptr + strlen(ptr);
					}
				else
					ptr = bc_store_string(&bc_data, ptr);
				}

			if	( commap != NULL )	
				*commap = ',';
			}
		}

	bc_add_code(&bc_data, kwTYPE_EOC);	// no bc_eoc
}

/*
*	Scans the 'source' for "names" separated by 'delims' and returns the elements (pointer in source)
*	into args array.
*
*	Returns the number of items
*/
int		bc_getlist(char *source, char_p_t *args, char *delims, int maxarg) SEC(BCSCAN);
int		bc_getlist(char *source, char_p_t *args, char *delims, int maxarg)
{
	char	*p, *ps;
	int		count = 0;

	ps = p = source;
	while ( *p )	{
		if	( strchr(delims, *p) ) {
			*p = '\0';
			while (*ps == ' ')	ps ++;
			args[count] = ps;
			count ++;
			if	( count == maxarg )	{
				if	( *ps )
					sc_raise("Paremeters limit: %d", maxarg);
				return count;
				}
			ps = p+1;
			}

		p ++;
		}

	if	( *ps )	{
		while (*ps == ' ')	ps ++;
		if	( *ps )	{
			*p = '\0';
			args[count] = ps;
			count ++;
			}
		}

	return count;
}

/*
*	returns a list of names
*
*	the list is included between sep[0] and sep[1] characters
*	each element is separated by 'delims' characters
*
*	the 'source' is the raw string (null chars will be placed at the end of each name)
*	the 'args' is the names (pointers on the 'source')
*	maxarg is the maximum number of names (actually the size of args)
*	the count is the number of names which are found by this routine.
*
*	returns the next position in 'source' (after the sep[1])
*/
char	*bc_getlist_insep(char *source, char_p_t *args, char *sep, char *delims, int maxarg, int *count) SEC(BCSCAN);
char	*bc_getlist_insep(char *source, char_p_t *args, char *sep, char *delims, int maxarg, int *count)
{
	char	*p = source;
	char	*ps;
	int		level = 1;
	
	*count = 0;
	p = strchr(source, sep[0]);

	if	( p )	{
		ps = p+1;
		p ++;

		while ( *p )	{
			if	( *p == sep[1] )	{
				level --;
				if ( level == 0 )
					break;
				}
			else if	( *p == sep[0] )
				level ++;

			p ++;
			}

		if	( *p == sep[1] )	{
			*p = '\0';
			if	( strlen(ps) )	{
				while ( *ps == ' ' || *ps == '\t' ) ps ++;
				if	( strlen(ps) )	
					*count = bc_getlist(ps, args, delims, maxarg);
				else
					sc_raise("'Empty' parameters are not allowed");
				}
			}
		else	
			sc_raise("Missing '%c'", sep[1]);
		}
	else
		p = source;

	return p;
}

/*
*	INLINE IFs
*
*	returns true if there is an inline IF
*
*	IF expr THEN ... ---> IF expr THEN (:) .... (:FI)
*	IF expr THEN ... ELSE ... ---> IF expr THEN (:) .... (:ELSE:) ... (:FI)
*/
int		bc_scan_inline_if(char *text)
{
	char	*p = (char *) text;	// *text points to 'expr'
	char	*pthen, *pelse;
	char	buf[SB_SOURCELINE_SIZE+1];

	if	( scan_error )		return 0;

	pthen = p;
	do	{
		pthen = strstr(pthen+1, " THEN ");
		if	( pthen )	{
			// store the expression
			while ( *p == ' ' )	p ++;
			strcpy(buf, p);
			p = strstr(buf, " THEN ");
			*p = '\0';

			// check for ':'
			p = pthen+6;
			while ( *p == ' ' )	p ++;

			if	( *p != ':' && *p != '\0' )	{
				// store the IF
				block_level ++;	block_id ++;
				bc_push();
				bc_add_ctrl(&bc_prog, kwIF, 0, 0);

				bc_scan_expr(buf, 0);
				if	( scan_error )		return 0;
				// store EOC
				bc_add_code(&bc_prog, kwTYPE_EOC);	//bc_eoc();

				// auto-goto 
				p = pthen + 6;
				while ( *p == ' ' )	p ++;
				if	( is_digit(*p) )	{
					// add goto
					strcpy(buf, "GOTO ");
					strcat(buf, p);
					}
				else
					strcpy(buf, p);

				// ELSE command
				// If there are more inline-ifs (nested) the ELSE belongs to the first IF (that's an error)
				pelse = strstr(buf+1, "ELSE");
				if	( pelse )	{
					do	{
						if	( (*(pelse-1) == ' ' || *(pelse-1) == '\t') &&
							  (*(pelse+4) == ' ' || *(pelse+4) == '\t') )	{

							*pelse = '\0';

							// scan the commands before ELSE
							bc_scan_cmd(buf);
							// add EOC
							bc_eoc(&bc_prog);

							// auto-goto
							strcpy(buf, "ELSE:");
							p = pelse + 4;
							while ( *p == ' ' || *p == '\t' )	p ++;
							if	( is_digit(*p) )	{
								// add goto
								strcat(buf, "GOTO ");
								strcat(buf, p);
								}
							else
								strcat(buf, p);
							
							//
							break;
							}
						else
							pelse = strstr(pelse+1, "ELSE");
						} while ( pelse != NULL );
					}

				// scan the rest commands
				bc_scan_cmd(buf);
				// add EOC
				bc_eoc(&bc_prog);

				// add ENDIF
				bc_push();	
				bc_add_ctrl(&bc_prog, kwENDIF, 0, 0);
				block_level --;

				return 1;
				}
			else	// *p == ':'
				return 0;
			}
		else
			break;

		} while ( pthen != NULL );

	return 0;	// false
}

/*
*	array's args 
*/
void	bc_scan_arridx(char *src)	SEC(BCSCAN);
void	bc_scan_arridx(char *src)
{
	char	*p = src;
	char	*ss = NULL, *se = NULL;
	int		level = 0;

	while ( *p )	{
		switch ( *p )	{
		case	'(':
			if	( level == 0 )
				ss = p;
			level ++;
			break;
		case	')':
			level --;

			if	( level == 0 )	{
				se = p;
				// store this index
				if	( !ss )	
					sc_raise("Array: syntax error");
				else	{
					*ss = ' ';	*se = '\0';
	
					bc_add_code(&bc_prog, kwTYPE_LEVEL_BEGIN);
					bc_scan_expr(ss, 0);
					bc_store1(&bc_prog, bc_prog.count-1, kwTYPE_LEVEL_END);

					*ss = '(';	*se = ')';
					ss = se = NULL;
					}
				}	// lev = 0
			break;
			};

		p ++;
		}

	//
	if ( level > 0 )	
		sc_raise("Array: Missing ')', (left side of expression)");
	else if ( level < 0 )
		sc_raise("Array: Missing '(', (left side of expression)");
}

/*
*	PASS1: scan source line
*/
void	bc_scan_cmd(char *text)
{
	char	*p;
	char	*lb_end;
	char	*last_cmd;
	#if defined(OS_ADDR16)
	int		idx;
	#else
	long	idx;
	#endif
	int		sharp, ladd, linc, ldec, decl = 0, vattr;
	int		leqop;
	char	pname[SB_KEYWORD_SIZE+1], vname[SB_KEYWORD_SIZE+1];

	if	( scan_error )		return;

	str_alltrim(text);
	p = text;

	if	( *p == ':' )	{	
		p ++;
		bc_scan_cmd(p);
		return;
		}

	// remark
	if	( *p == '\'' || *p == '#' )	return;

	// empty line
	if	( *p == '\0' )		return;

	lb_end = p = bc_getkeywordstr((char *)text, bc_name);
	last_cmd = p;
	p = get_param_sect(p, ":", bc_parm);

	/*
	*	check old style labels 
	*/
	if	( is_all_digits(bc_name) )	{
		str_alltrim(bc_name);
		idx = bc_get_label_id(bc_name);
		bc_set_label_ip(idx);
		if	( scan_error )		return;

		// continue
		last_cmd = p = bc_getkeywordstr(lb_end, bc_name);
		if	( strlen(bc_name) == 0 )	{
			if	( !p )	return;
			if	( *p == '\0' ) return;
			}
		p = get_param_sect(p, ":", bc_parm);
		}

	/* what's this ? */
	idx = bc_is_keyword(bc_name);
	if	( idx == kwREM )	return;		// remarks... return
	if	( idx == -1 )	{
		idx = bc_is_proc(bc_name);
		if	( idx != -1 )	{
			//	simple buildin procedure
			//
			//	there is no need to check it more...
			//	save it and return (go to the next)
			bc_add_pcode(&bc_prog, idx);
			bc_scan_expr(bc_parm, 0);

			if	( *p == ':' )	{	// command separator
				bc_eoc(&bc_prog);
				p ++;
				bc_scan_cmd(p);
				}

			return;
			}
		}

	if	( idx == kwLET )	{	// old-style keyword LET
		char	*p;
		idx = -1;
		p = bc_getkeywordstr(bc_parm, bc_name);
		strcpy(bc_parm, p);
		}
	else if ( idx == kwDECLARE )	{ // declaration
		char	*p;

		decl = 1;
		p = bc_getkeywordstr(bc_parm, bc_name);
		idx = bc_is_keyword(bc_name);
		if	( idx == -1 )	idx = bc_is_proc(bc_name);
		strcpy(bc_parm, p);
		if	( idx != kwPROC && idx != kwFUNC )	{
			sc_raise("Use DECLARE with SUB or FUNC keyword");
			return;
			}
		}

	//////////////////////////////////////////
	if	( idx == kwREM )	return;
	sharp = (bc_parm[0] == '#');			// if #  -> file commands
	ladd  = (strncmp(bc_parm,"<<",2)==0);	// if << -> array, append
	linc  = (strncmp(bc_parm,"++",2)==0);	// 
	ldec  = (strncmp(bc_parm,"--",2)==0);	// 
	if	( bc_parm[1] == '=' && strchr("-+/\\*^%&|", bc_parm[0]) )	
		leqop = bc_parm[0];
	else
		leqop = 0;

	if	( (bc_parm[0] == '=' || 
			ladd || linc || ldec || leqop
			) && (idx != -1) )	{

		sc_raise("%s: is keyword (left side)", bc_name);
		return;
		}
	else if	( (idx == kwCONST)
			|| (
				(bc_parm[0] == '=' || bc_parm[0] == '(' ||
				ladd || linc || ldec || leqop
				) 
				&& (idx == -1) 
				)
			)	{
		//
		//	LET/CONST commands
		//
		char	*parms = bc_parm;

		if	( idx == kwCONST )	{
			p = bc_getkeywordstr(bc_parm, bc_name);
			p = get_param_sect(p, ":", bc_parm);
			if	( bc_is_keyword(bc_name) != -1 )
				sc_raise("%s: is keyword (left side)", bc_name);
			parms = bc_parm;
			bc_add_code(&bc_prog, kwCONST);
			}
		else if	( ladd )	{
			bc_add_code(&bc_prog, kwAPPEND);
			parms += 2;
			}
		else if ( linc )	{
			bc_add_code(&bc_prog, kwLET);
			strcpy(bc_parm, "=");
			strcat(bc_parm, bc_name);
			strcat(bc_parm, "+1");
			}
		else if ( ldec )	{
			bc_add_code(&bc_prog, kwLET);
			strcpy(bc_parm, "=");
			strcat(bc_parm, bc_name);
			strcat(bc_parm, "-1");
			}
		else if ( leqop )	{
			char	*buf;
			int		l;

			bc_add_code(&bc_prog, kwLET);
			l = strlen(bc_parm)+strlen(bc_name)+1;
			buf = tmp_alloc(l);
			memset(buf, 0, l);
			strcpy(buf, "=");
			strcat(buf, bc_name);
			buf[strlen(buf)] = leqop;
			strcat(buf, bc_parm+2);

			strcpy(bc_parm, buf);
			tmp_free(buf);
			}
		else
			bc_add_code(&bc_prog, kwLET);

		bc_add_code(&bc_prog, kwTYPE_VAR);
		bc_add_addr(&bc_prog, bc_get_var_id(bc_name));

		if	( !scan_error )	{
			if	( parms[0] == '(' )	{
				char	*p = strchr(parms, '=');
			
				if	( !p )	
					sc_raise("LET/CONST/APPEND: Missing '='");
				else	{
					if	( *bc_next_char(parms+1) == ')' )	{
						// its the variable's name only
						bc_scan_expr(p, 0);
						}
					else	{
						// ARRAY (LEFT)
						*p = '\0';
						bc_scan_arridx(parms);

						*p = '=';
						if	( !scan_error )	{
							bc_add_code(&bc_prog, kwTYPE_CMPOPR);
							bc_add_code(&bc_prog, '=');
							bc_scan_expr(p+1, 0);
							}
						}
					}
				}
			else	{
				bc_add_code(&bc_prog, kwTYPE_CMPOPR);
				bc_add_code(&bc_prog, '=');
				bc_scan_expr(parms+1, 0);
				}
			}
		}
	else {
		// add generic command
		if	( idx != -1 )	{
			if	( idx == kwLABEL )	{
				str_alltrim(bc_parm);
				idx = bc_get_label_id(bc_parm);
				bc_set_label_ip(idx);
				}

			else if	( idx == kwEXIT )	{
				bc_add_code(&bc_prog, idx);
				str_alltrim(bc_parm);
				if	( strlen(bc_parm) && bc_parm[0] != '\'' )	{
					idx = bc_is_spopr(bc_parm);
					if	( idx == kwFORSEP || idx == kwLOOPSEP || idx == kwPROCSEP || idx == kwFUNCSEP )	
						bc_add_code(&bc_prog, idx);
					else
						sc_raise("Use EXIT [FOR|LOOP|SUB|FUNC]");
					}
				else
					bc_add_code(&bc_prog, 0);
				}

			else if	( idx == kwDECLARE )	{
				}

			else if	( idx == kwPROC || idx == kwFUNC )	{
				//
				// 	USER-DEFINED PROCEDURES/FUNCTIONS
				//
				#if defined(OS_LIMITED)
				char_p_t pars[32];
				#else
				char_p_t pars[256];
				#endif
				char	*lpar_ptr, *eq_ptr;
				int		i, count, pidx;

				// inline function (DEF FN)
				if	( (eq_ptr   = strchr(bc_parm, '=')) )		*eq_ptr = '\0';

				// parameters start
				if	( (lpar_ptr = strchr(bc_parm, '(')) )		*lpar_ptr = '\0';

				bc_prepname(pname, bc_parm, SB_KEYWORD_SIZE);
				if	( decl )	{
					// its only a declaration (DECLARE)
					if	( bc_udp_getip(pname) == INVALID_ADDR )
						bc_addudp(pname);
					}
				else	{
					// func/sub
					if	( bc_udp_getip(pname) != INVALID_ADDR )
						sc_raise("The SUB/FUNC %s is already defined", pname);
					else {

						// setup routine's address (and get an id)
						if	( (pidx = bc_udp_setip(pname, bc_prog.count)) == -1 )	{
							pidx = bc_addudp(pname);
							bc_udp_setip(pname, bc_prog.count);
							}

						// put JMP to the next command after the END 
						// (now we just keep the rq space, pass2 will update that)
						bc_add_code(&bc_prog, kwGOTO);
						bc_add_addr(&bc_prog, 0);
						bc_add_code(&bc_prog, 0);

						block_level ++; block_id ++;
						bc_push();						// keep it in stack for 'pass2'
						bc_add_code(&bc_prog, idx);		// store (FUNC/PROC) code

						// func/proc name
						if	( bc_proc_level )	{
							strcat(bc_proc, "/");
							strcat(bc_proc, pname);
							}
						else
							strcpy(bc_proc, pname);

						if	( !scan_error )	{
							bc_proc_level ++;

							// if its a function,
							// setup the code for the return-value (vid={F}/{F})
							if	( idx == kwFUNC )	{	
								char	*buf;
								buf = tmp_alloc(strlen(bc_proc)+strlen(pname)+2);
								xsprintf(buf, "%s/%s", bc_proc, pname);
								udp_table[pidx].vid = bc_get_var_id(buf);
								tmp_free(buf);
								}
							else
								// procedure, no return value here
								udp_table[pidx].vid = INVALID_ADDR;	

							// parameters
							if	( lpar_ptr )	{
								char	*buf;

								buf = tmp_alloc(SB_SOURCELINE_SIZE+1);

								*lpar_ptr = '(';
								#if defined(OS_LIMITED)
								bc_getlist_insep(bc_parm, pars, "()", ",", 32, &count);
								#else
								bc_getlist_insep(bc_parm, pars, "()", ",", 256, &count);
								#endif
								bc_add_code(&bc_prog, kwTYPE_PARAM);
								bc_add_code(&bc_prog, count);

								for ( i = 0; i < count; i ++ )	{
									if	( strncmp(pars[i], "BYREF ", 6) == 0 )	{
										bc_prepname(vname, pars[i]+6, SB_KEYWORD_SIZE);
										vattr = 0x80;
										}
									else	{
										bc_prepname(vname, pars[i], SB_KEYWORD_SIZE);
										vattr = 0;
										}
										
									if	( strchr(pars[i], '(') )	{
										vattr |= 1;
										}

									bc_add_code(&bc_prog, vattr);
									xsprintf(buf, "%s/%s", bc_proc, vname);
									bc_add_addr(&bc_prog, bc_get_var_id(buf));
									}

								tmp_free(buf);

								}
							else	{
								// no parameters
								bc_add_code(&bc_prog, kwTYPE_PARAM);	// params
								bc_add_code(&bc_prog, 0);				// pcount = 0
								}

							bc_eoc(&bc_prog); // EOC

							// -----------------------------------------------
							// scan for inline function (inline DEF FN format)
							if	( eq_ptr && idx == kwFUNC )	{
								eq_ptr ++;	// *eq_ptr was '\0'
								while ( *eq_ptr == ' ' || *eq_ptr == '\t' )	eq_ptr ++;

								if	( strlen(eq_ptr) )	{
									char	*macro;

									macro = tmp_alloc(SB_SOURCELINE_SIZE+1);
									xsprintf(macro, "%s=%s:END", pname, eq_ptr);

									//	run bc_scan_cmd again
									bc_scan_cmd(macro);
									tmp_free(macro);
									}
								else
									sc_raise("FUNC/DEF it has an '=' on declaration, but I didn't found the body!");
								}
							}
						}
					}
				}
			else if	( idx == kwLOCAL )	{	// local variables
				#if defined(OS_LIMITED)
				char_p_t pars[32];
				#else
				char_p_t pars[256];
				#endif
				int		i, count;

				#if defined(OS_LIMITED)
				count = bc_getlist(bc_parm, pars, ",", 32);
				#else
				count = bc_getlist(bc_parm, pars, ",", 256);
				#endif
				bc_add_code(&bc_prog, kwTYPE_CRVAR);
				bc_add_code(&bc_prog, count);
				for ( i = 0; i < count; i ++ )	{
					bc_prepname(vname, pars[i], SB_KEYWORD_SIZE);

					if	( bc_proc_level )	{
						char	*buf;
						buf = tmp_alloc(strlen(bc_proc)+strlen(vname)+2);
						#if defined(_PalmOS)
						StrPrintF(buf, "%s/%s", bc_proc, vname);
						#else
						sprintf(buf, "%s/%s", bc_proc, vname);
						#endif
						bc_add_addr(&bc_prog, bc_get_var_id(buf));
						tmp_free(buf);
						}
					else
						bc_add_addr(&bc_prog, bc_get_var_id(vname));
					}
				}
			else if ( idx == kwREM )	
				return;
			else if ( idx == kwGOTO )	{
				str_alltrim(bc_parm);
				bc_push();
				bc_add_code(&bc_prog, idx);
				bc_add_addr(&bc_prog, bc_get_label_id(bc_parm));
				bc_add_code(&bc_prog, block_level);
				}
			else if ( idx == kwGOSUB )	{
				str_alltrim(bc_parm);
				bc_add_code(&bc_prog, idx);
				bc_add_addr(&bc_prog, bc_get_label_id(bc_parm));
				}
			//
			//	IF
			//
			else if ( idx == kwIF )	{
				strcpy(kw_do_close_cmd, "ENDIF");

				// from here, we can scan for inline IF
				if	( bc_scan_inline_if(last_cmd) )	{
					// inline-IFs
					return;
					}
				else	{
					block_level ++; block_id ++;
					bc_push();	
					bc_add_ctrl(&bc_prog, idx, 0, 0);
					bc_scan_expr(bc_parm, 0);
					bc_add_code(&bc_prog, kwTYPE_EOC);	//bc_eoc();
					}
				}
			//
			//	ON x GOTO|GOSUB ...
			//
			else if ( idx == kwON )	{
				char	*p;
				int		keep_ip, count;
				#if defined(OS_LIMITED)
				char_p_t pars[32];
				#else
				char_p_t pars[256];
				#endif
				int		i;
			
				idx = kwONJMP;		// WARNING!

				bc_push();
				bc_add_ctrl(&bc_prog, idx, 0, 0);

				if	( (p = strstr(bc_parm, " GOTO ")) != NULL )	{
					bc_add_code(&bc_prog, kwGOTO);	// the command
					*p = '\0'; p += 6;
					keep_ip = bc_prog.count;
					bc_add_code(&bc_prog, 0);			// the counter

					//count = bc_scan_label_list(p);
					#if defined(OS_LIMITED)
					count = bc_getlist(p, pars, ",", 32);
					#else
					count = bc_getlist(p, pars, ",", 256);
					#endif
					for ( i = 0; i < count; i ++ )	
						bc_add_addr(&bc_prog, bc_get_label_id(pars[i]));	// IDs

					if	( count == 0 )
						sc_raise("ON x GOTO WHERE?");
					else	
						bc_prog.ptr[keep_ip] = count;

					bc_scan_expr(bc_parm, 0);	// the expression
					bc_eoc(&bc_prog);
					}
				else if	( (p = strstr(bc_parm, " GOSUB ")) != NULL )	{
					bc_add_code(&bc_prog, kwGOSUB);	// the command
					*p = '\0';	p += 7;
					keep_ip = bc_prog.count;
					bc_add_code(&bc_prog, 0);			// the counter

					//count = bc_scan_label_list(p);
					#if defined(OS_LIMITED)
					count = bc_getlist(p, pars, ",", 32);
					#else
					count = bc_getlist(p, pars, ",", 256);
					#endif
					for ( i = 0; i < count; i ++ )	
						bc_add_addr(&bc_prog, bc_get_label_id(pars[i]));

					if	( count == 0 )
						sc_raise("ON x GOSUB WHERE?");
					else	
						bc_prog.ptr[keep_ip] = count;

					bc_scan_expr(bc_parm, 0);	// the expression
					bc_eoc(&bc_prog);
					}
				else
					sc_raise("ON WHAT?");
				}
			//
			//	FOR
			//
			else if ( idx == kwFOR )	{
				char	*p = strchr(bc_parm, '=');
				char	*p_lev;
				char	*n;
			
				strcpy(kw_do_close_cmd, "NEXT");

				block_level ++; block_id ++;
				bc_push();	
				bc_add_ctrl(&bc_prog, kwFOR, 0, 0);

				if	( !p )	{
					// FOR [EACH] X IN Y
					if	( (p = strstr(bc_parm, " IN ")) == NULL )
						sc_raise("FOR: Missing '=' OR 'IN'");
					
					*p = '\0';
					n = p;
					strcpy(bc_name, bc_parm);
					str_alltrim(bc_name);
					if	( !is_alpha(*bc_name) )
						sc_raise("FOR: %s is not a variable", bc_name);
					else	{
						p_lev = bc_name;
						while ( is_alnum(*p_lev) || *p_lev == ' ' )	p_lev ++;
						if	( *p_lev == '(' )	
							sc_raise("FOR: %s is an array. Arrays are not allowed", bc_name);
						else	{
							bc_add_code(&bc_prog, kwTYPE_VAR);
							bc_add_addr(&bc_prog, bc_get_var_id(bc_name));

							*n = ' ';
							bc_add_code(&bc_prog, kwIN);
							bc_scan_expr(n+3, 0);
							}
						}
					}
				else	{
					// FOR X=Y TO Z [STEP L]
					*p = '\0';
					n = p;

					strcpy(bc_name, bc_parm);
					str_alltrim(bc_name);
					if	( !is_alpha(*bc_name) )
						sc_raise("FOR: %s is not a variable", bc_name);
					else	{
						p_lev = bc_name;
						while ( is_alnum(*p_lev) || *p_lev == ' ' )	p_lev ++;
						if	( *p_lev == '(' )	
							sc_raise("FOR: %s is an array. Arrays are not allowed", bc_name);
						else	{
							bc_add_code(&bc_prog, kwTYPE_VAR);
							bc_add_addr(&bc_prog, bc_get_var_id(bc_name));

							*n = '=';
							bc_scan_expr(n+1, 0);
							}
						}
					}
				}
			//
			//	WHILE - REPEAT
			//
			else if ( idx == kwWHILE )	{
				strcpy(kw_do_close_cmd, "WEND");

				block_level ++; block_id ++;
				bc_push();	
				bc_add_ctrl(&bc_prog, idx, 0, 0);
				bc_scan_expr(bc_parm, 0);
				}
			else if ( idx == kwREPEAT )	{
				// WHILE & REPEAT DOES NOT USE STACK 
				block_level ++; block_id ++;
				bc_push();	
				bc_add_ctrl(&bc_prog, idx, 0, 0);
				bc_scan_expr(bc_parm, 0);
				}
			else if ( idx == kwELSE || idx == kwELIF )	{
				bc_push();	
				bc_add_ctrl(&bc_prog, idx, 0, 0);
				bc_scan_expr(bc_parm, 0);
				}
			else if ( idx == kwENDIF || idx == kwNEXT )	{
				bc_push();	
				bc_add_ctrl(&bc_prog, idx, 0, 0);
				block_level --;
				}
			else if ( idx == kwWEND || idx == kwUNTIL )	{
				bc_push();	
				bc_add_ctrl(&bc_prog, idx, 0, 0);
				block_level --;
				bc_scan_expr(bc_parm, 0);
				}
			else if (  idx == kwSTEP || idx == kwTO || idx == kwIN || idx == kwTHEN 
					|| idx == kwCOS  || idx == kwSIN || idx == kwLEN || idx == kwLOOP  )		// functions...
				{
				sc_raise("%s: Wrong position", bc_name);
				}
			else if ( idx == kwRESTORE )	{
				bc_push();
				bc_add_code(&bc_prog, idx);
				bc_add_addr(&bc_prog, bc_get_label_id(bc_parm));
				}
			else if ( idx == kwEND )	{
				if	( bc_proc_level )	{
					char	*dol;

					// UDP/F RETURN
					dol = strrchr(bc_proc, '/');
					if	( dol )
						*dol = '\0';
					else
						*bc_proc = '\0';

					bc_push();
					bc_add_code(&bc_prog, kwTYPE_RET);

					bc_proc_level --;
					block_level --;
					block_id ++;
					}
				else	{
					// END OF PROG
					bc_add_code(&bc_prog, idx);
					}
				}
			else if ( idx == kwDATA )	
				bc_scan_data(bc_parm);
			else if ( idx == kwREAD && sharp )	{
				bc_add_code(&bc_prog, kwFILEREAD);
				bc_scan_expr(bc_parm, 0);
				}
			else if ( idx == kwINPUT && sharp )	{
				bc_add_code(&bc_prog, kwFILEINPUT);
				bc_scan_expr(bc_parm, 0);
				}
			else if ( idx == kwPRINT && sharp )	{
				bc_add_code(&bc_prog, kwFILEPRINT);
				bc_scan_expr(bc_parm, 0);
				}
			else if ( idx == kwLINE  && strncmp(bc_parm, "INPUT ", 6) == 0  )	{
				bc_add_code(&bc_prog, kwLINEINPUT);
				bc_scan_expr(bc_parm+6, 0);
				}
			else	{
				// something else
				bc_add_code(&bc_prog, idx);
				bc_scan_expr(bc_parm, 0);
				}
			}
		else	{
			/*
			*	EXTERNAL OR USER-DEFINED PROCEDURE
			*/
			int		udp;

			udp = bc_is_extproc(bc_name);
			if	( udp > -1 )	{
				bc_add_extpcode(&bc_prog, ext_proc_table[udp].lib_id, udp);
				bc_add_code(&bc_prog, kwTYPE_LEVEL_BEGIN);
				bc_scan_expr(bc_parm, 0);
				bc_add_code(&bc_prog, kwTYPE_LEVEL_END);
				}
			else	{
				udp = bc_getudp_id(bc_name);

				if	( udp == -1 )
					sc_raise("Unknown command: %s", bc_name);
				else	{
					bc_push();
					bc_add_ctrl(&bc_prog, kwTYPE_CALL_UDP, udp, 0);
					bc_add_code(&bc_prog, kwTYPE_LEVEL_BEGIN);
					bc_scan_expr(bc_parm, 0);
					bc_add_code(&bc_prog, kwTYPE_LEVEL_END);
					}
				}
			}
		}

	///////////////////////////////////////
	if	( *p == ':' )	{
		// command separator
		bc_eoc(&bc_prog);
		p ++;
		bc_scan_cmd(p);
		}
}

/*
*	skip command bytes
*/
addr_t	bc_nextcmd(addr_t ip) SEC(BCSCAN);
addr_t	bc_nextcmd(addr_t ip)
{
	code_t	code;
	#if	defined(ADDR16)
	word	len;
	#else
	dword	len;
	#endif

	code = bc_prog.ptr[ip];
	ip ++;

	switch ( code )	{
	case	kwTYPE_INT:				// integer
		ip += OS_INTSZ;
		break;
	case	kwTYPE_NUM:				// number
		ip += OS_REALSZ;
		break;
	case	kwTYPE_STR:				// string: [2/4B-len][data]
		memcpy(&len, bc_prog.ptr+ip, OS_STRLEN);
		len += OS_STRLEN;
		ip  += len;
		break;
	case	kwTYPE_CALLF:
	case	kwTYPE_CALLP:			// [fcode_t]
		ip += CODESZ;		
		break;
	case	kwTYPE_CALLEXTF:
	case	kwTYPE_CALLEXTP:		// [lib][index]
		ip += (ADDRSZ * 2);
		break;
	case	kwEXIT:
	case	kwTYPE_SEP:
	case	kwTYPE_LOGOPR:
	case	kwTYPE_CMPOPR:
	case	kwTYPE_ADDOPR:
	case	kwTYPE_MULOPR:
	case	kwTYPE_POWOPR:
	case	kwTYPE_UNROPR:			// [1B data]
		ip ++;				
		break;
	case	kwRESTORE:
	case	kwGOSUB:
	case	kwTYPE_LINE:
	case	kwTYPE_VAR:				// [addr|id]
		ip += ADDRSZ;		
		break;
	case	kwTYPE_CALL_UDP:
	case	kwTYPE_CALL_UDF:		// [true-ip][false-ip]
		ip += BC_CTRLSZ;	
		break;
	case	kwGOTO:					// [addr][pop-count]
		ip += (ADDRSZ+1);	
		break;
	case	kwTYPE_CRVAR:			// [1B count][addr1][addr2]...
		len = bc_prog.ptr[ip];	
		ip += ((len * ADDRSZ) + 1);
		break;
	case	kwTYPE_PARAM:			// [1B count] {[1B-pattr][addr1]} ...
		len = bc_prog.ptr[ip];
		ip += ((len * (ADDRSZ+1)) + 1);
		break;
	case	kwONJMP:				// [true-ip][false-ip] [GOTO|GOSUB] [count] [addr1]...
		ip += (BC_CTRLSZ+1);
		ip += (bc_prog.ptr[ip] * ADDRSZ);
		break;
	case	kwIF:		case	kwFOR: 		case	kwWHILE:	case	kwREPEAT:
	case	kwELSE:		case	kwELIF:
	case	kwENDIF:	case	kwNEXT:		case	kwWEND:		case	kwUNTIL:
	case	kwUSE:
		ip += BC_CTRLSZ;
		break;
		};
	return ip;
}

/*
*	search for command (in byte-code)
*/
addr_t	bc_search(addr_t ip, code_t code)
{
	addr_t	i = ip;

	do	{
		if	( code == bc_prog.ptr[i] )
			return i;

		i = bc_nextcmd(i);
		} while ( i < bc_prog.count );
	return INVALID_ADDR;
}

/*
*	search for End-Of-Command mark
*/
addr_t	bc_search_eoc(addr_t ip) SEC(BCSCAN);
addr_t	bc_search_eoc(addr_t ip)
{
	addr_t	i = ip;
	code_t	code;

	do	{
		code = bc_prog.ptr[i];
		if	( code == kwTYPE_EOC || code == kwTYPE_LINE )
			return i;

		i = bc_nextcmd(i);
		} while ( i < bc_prog.count );
	return bc_prog.count;
}

/*
*	search stack
*/
addr_t	bc_search_stack(addr_t start, code_t code, int level) SEC(BCSCAN);
addr_t	bc_search_stack(addr_t start, code_t code, int level)
{
	addr_t		i;
	pass_node_t	node;

	for ( i = start; i < stk_count; i ++ )	{
		dbt_read(bc_stack, i, &node, sizeof(pass_node_t));

		if	( bc_prog.ptr[node.pos] == code )	{
			if	( node.level == level )	
				return node.pos;
			}
		}
	return INVALID_ADDR;
}

/*
*	search stack backward
*/
addr_t	bc_search_stack_backward(addr_t start, code_t code, int level) SEC(BCSCAN);
addr_t	bc_search_stack_backward(addr_t start, code_t code, int level)
{
	addr_t		i = start;
	pass_node_t	node;

	for ( ; i < stk_count; i -- )	{	// WARNING: ITS UNSIGNED, SO WE'LL SEARCH IN RANGE [0..STK_COUNT]
		dbt_read(bc_stack, i, &node, sizeof(pass_node_t));
		if	( bc_prog.ptr[node.pos] == code )	{
			if	( node.level == level )	
				return node.pos;
			}
		}
	return INVALID_ADDR;
}

/*
*	Advanced error messages:
*	Analyze LOOP-END errors
*/
void	print_pass2_stack(addr_t pos, code_t lcode, int level) SEC(TRASH);
void	print_pass2_stack(addr_t pos, code_t lcode, int level)
{
	#if !defined(OS_LIMITED)
	addr_t	ip;
	#endif
	addr_t	i;
	int		j, cs_idx;
	char	cmd[16], cmd2[16];
	pass_node_t	node;
	code_t	ccode[256], code;
	int		csum[256];
	int		cs_count;
	code_t	start_code[] =
	{ kwWHILE, kwREPEAT, kwIF,    kwFOR,  kwFUNC, 0 };
	code_t	end_code[] =
	{ kwWEND,  kwUNTIL,  kwENDIF, kwNEXT, kwTYPE_RET, 0 };

	code = lcode;

	kw_getcmdname(code, cmd);

	/*
	*	search for closest keyword (forward)
	*/	
#if !defined(OS_LIMITED)
	ip = bc_search_stack(pos+1, code, level - 1);
	if	( ip == INVALID_ADDR )	{
		ip = bc_search_stack(pos+1, code, level + 1);
		if	( ip == INVALID_ADDR )	{
			int		cnt = 0;

			for ( i = pos+1; i < stk_count; i ++ )	{
				dbt_read(bc_stack, i, &node, sizeof(pass_node_t));

				if	( bc_prog.ptr[node.pos] == code )	{
					fprintf(stderr, "\n%s found on level %d (@%d) instead of %d (@%d+)\n", cmd, node.level, node.pos, level, pos);
					cnt ++;
					if	( cnt > 3 )
						break;
					}
				}
			}
		else
			fprintf(stderr, "\n%s found on level %d (@%d) instead of %d (@%d+)\n", cmd, level+1, node.pos, level, pos);
		}
	else
		fprintf(stderr, "\n%s found on level %d (@%d) instead of %d (@%d+)\n", cmd, level-1, node.pos, level, pos);
#endif

	/*
	*	print stack
	*/
	cs_count = 0;
#if !defined(OS_LIMITED)
	fprintf(stderr, "\n");
	fprintf(stderr, "--- Pass 2 - stack ------------------------------------------------------\n");
	fprintf(stderr, "%s%4s  %16s %16s %6s %6s %5s %5s %5s\n", "  ", "   i", "Command", "Section", "Addr", "Line", "Level", "BlkID", "Count");
	fprintf(stderr, "-------------------------------------------------------------------------\n");
#endif
	for ( i = 0; i < stk_count; i ++ )	{
		dbt_read(bc_stack, i, &node, sizeof(pass_node_t));

		code = bc_prog.ptr[node.pos];
		if	( node.pos != INVALID_ADDR )	
			kw_getcmdname(code, cmd);
		else
			strcpy(cmd, "---");

		// sum
		cs_idx = -1;
		for ( j = 0; j < cs_count; j ++ )	{
			if	( ccode[j] == code )	{
				cs_idx = j;
				csum[cs_idx] ++;
				break;
				}
			}
		if	( cs_idx == -1 )	{
			cs_idx = cs_count;
			cs_count ++;
			ccode[cs_idx] = code;
			csum[cs_idx] = 1;
			}

#if !defined(OS_LIMITED)
		// info
		fprintf(stderr, "%s%4d: %16s %16s %6d %6d %5d %5d %5d\n",
			((i==pos)?">>":"  "), i, cmd, node.sec, node.pos, node.line, node.level, node.block_id, csum[cs_idx]);
#endif
		}

	/*
	*	sum
	*/
#if !defined(OS_LIMITED)
	fprintf(stderr, "\n");
	fprintf(stderr, "--- Sum -----------------------------------------------------------------\n");
	for ( i = 0; i < cs_count; i ++ )	{
		code = ccode[i];
		if	( !kw_getcmdname(code, cmd) )	sprintf(cmd, "(%d)", code);
		fprintf(stderr, "%16s - %5d\n", cmd, csum[i]);
		}
#endif

	/*
	*	decide
	*/
#if !defined(OS_LIMITED)
	fprintf(stderr, "\n");
#else
	dev_printf("\n");
#endif
	for ( i = 0; start_code[i] != 0; i ++ )	{
		int		sa, sb;
		code_t	ca, cb;

		ca = start_code[i];
		cb = end_code[i];

		sa = 0;
		for ( j = 0; j < cs_count; j ++ )	{
			if	( ccode[j] == ca )	
				sa = csum[j];
			if	( ca == kwFUNC )	{
				if	( ccode[j] == kwPROC )	
					sa += csum[j];
				}
			}

		sb = 0;
		for ( j = 0; j < cs_count; j ++ )	{
			if	( ccode[j] == cb )	{
				sb = csum[j];
				break;
				}
			}

		if	( sa - sb != 0 )	{
			kw_getcmdname(ca, cmd);
			kw_getcmdname(cb, cmd2);
			if	( sa > sb )	
#if !defined(OS_LIMITED)
				fprintf(stderr, "Hint: Missing %d %s or there is/are %d more %s\n", sa - sb, cmd2, sa - sb, cmd);
#else
				dev_printf("Hint: Missing %d %s or there is/are %d more %s\n", sa - sb, cmd2, sa - sb, cmd);
#endif
			else	
#if !defined(OS_LIMITED)
				fprintf(stderr, "Hint: There is/are %d more %s or missing %d %s\n", sb - sa, cmd2, sb-sa, cmd);
#else
				dev_printf("Hint: There is/are %d more %s or missing %d %s\n", sb - sa, cmd2, sb-sa, cmd);
#endif
			}
		}
	
#if !defined(OS_LIMITED)
	fprintf(stderr, "\n");
#else
	dev_printf("\n\n");
#endif
}

/*
*	PASS 2 (write jumps for IF/FOR/WHILE/REPEAT)
*/
void	bc_pass2_scan(void) SEC(BCSCAN);
void	bc_pass2_scan()
{
	addr_t	i = 0, j, true_ip, false_ip, label_id, w;
	addr_t	a_ip, b_ip, c_ip, count;
	code_t	code;
	byte	level;
	pass_node_t	node;
	label_t	label;

	if	( !opt_quite )
		dev_printf("\rPASS2: Node %d/%d", i, stk_count);

	// for each node in stack
	for ( i = 0; i < stk_count; i ++ )	{

		if	( !opt_quite )	{
			if	( (i % SB_KEYWORD_SIZE) == 0 )
				dev_printf("\rPASS2: Node %d/%d", i, stk_count);
			}

		dbt_read(bc_stack, i, &node, sizeof(pass_node_t));

		scan_line = node.line;
		strcpy(bc_sec, node.sec);
		code = bc_prog.ptr[node.pos];

		if	(  code != kwGOTO && code != kwRESTORE && code != kwONJMP 
		    && code != kwTYPE_CALL_UDP && code != kwTYPE_CALL_UDF 
			&& code != kwPROC && code != kwFUNC && code != kwTYPE_RET )	{

			// default - calculate true-ip
			true_ip = bc_search_eoc(node.pos+(BC_CTRLSZ+1));
			memcpy(bc_prog.ptr+node.pos+1, &true_ip, ADDRSZ);
			}

		switch ( code )	{
		case kwPROC:
		case kwFUNC:
			// update start's GOTO
			true_ip = bc_search_stack(i+1, kwTYPE_RET, node.level) + 1;
			if	( true_ip == INVALID_ADDR )	{
				sc_raise("SUB/FUNC: Missing END on the same level");
				print_pass2_stack(i, kwTYPE_RET, node.level);
				return;
				}
			memcpy(bc_prog.ptr+node.pos-(ADDRSZ+1), &true_ip, ADDRSZ);
			break;

		case kwRESTORE:
			// replace the label ID with the real IP
			memcpy(&label_id, bc_prog.ptr+node.pos+1, ADDRSZ);
			dbt_read(lab_table, label_id, &label, sizeof(label_t));
			count = first_data_ip + label.dp;
			memcpy(bc_prog.ptr+node.pos+1, &count, ADDRSZ);	// change LABEL-ID with DataPointer
			break;

		case kwTYPE_CALL_UDP:
		case kwTYPE_CALL_UDF:
			// update real IP
			memcpy(&label_id, bc_prog.ptr+node.pos+1, ADDRSZ);
			true_ip = udp_table[label_id].ip + (ADDRSZ+3);
			memcpy(bc_prog.ptr+node.pos+1, &true_ip, ADDRSZ);	

			// update return-var ID
			true_ip = udp_table[label_id].vid;
			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &true_ip, ADDRSZ);	
			break;

		case kwONJMP:
			// kwONJMP:1 trueip:2 falseip:2 command:1 count:1 label1:2 label2:2 ...
			count = bc_prog.ptr[node.pos+(ADDRSZ+ADDRSZ+2)];

			true_ip = bc_search_eoc(node.pos+BC_CTRLSZ+(count*ADDRSZ)+3);
			memcpy(bc_prog.ptr+node.pos+1, &true_ip, ADDRSZ);

			// change label IDs with the real IPs
			for ( j = 0; j < count; j ++ )	{
				memcpy(&label_id, bc_prog.ptr+node.pos+(j*ADDRSZ)+(ADDRSZ+ADDRSZ+3), ADDRSZ);
				dbt_read(lab_table, label_id, &label, sizeof(label_t));
				w = label.ip;
				memcpy(bc_prog.ptr+node.pos+(j*ADDRSZ)+(ADDRSZ+ADDRSZ+3), &w, ADDRSZ);
				}
			break;

		case kwGOTO:	// LONG JUMPS
			memcpy(&label_id, bc_prog.ptr+node.pos+1, ADDRSZ);
			dbt_read(lab_table, label_id, &label, sizeof(label_t));
			w = label.ip;
			memcpy(bc_prog.ptr+node.pos+1, &w, ADDRSZ);	// change LABEL-ID with IP
			level = bc_prog.ptr[node.pos+(ADDRSZ+1)];
			bc_prog.ptr[node.pos+(ADDRSZ+1)] = 0; // number of POPs ?????????

			if	( level >= label.level )
				bc_prog.ptr[node.pos+(ADDRSZ+1)] = level - label.level; // number of POPs
			else	
				bc_prog.ptr[node.pos+(ADDRSZ+1)] = 0; // number of POPs ?????????

			break;
		case kwFOR:
			a_ip = bc_search(node.pos+(ADDRSZ+ADDRSZ+1), kwTO);
			b_ip = bc_search(node.pos+(ADDRSZ+ADDRSZ+1), kwIN);
			if	( a_ip < b_ip )
				b_ip = INVALID_ADDR;
			else if	( a_ip > b_ip )	
				a_ip = b_ip;

			false_ip = bc_search_stack(i+1, kwNEXT, node.level);

			if	( false_ip == INVALID_ADDR )	{
				sc_raise("FOR: Missing NEXT on the same level");
				print_pass2_stack(i, kwNEXT, node.level);
				return;
				}
			if	( a_ip > false_ip || a_ip == INVALID_ADDR )	{
				if	( b_ip != INVALID_ADDR )
					sc_raise("FOR: Missing IN");
				else
					sc_raise("FOR: Missing TO");
				return;
				}

			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;
		case kwWHILE:
			false_ip = bc_search_stack(i+1, kwWEND, node.level);

			if	( false_ip == INVALID_ADDR )	{
				sc_raise("WHILE: Missing WEND on the same level");
				print_pass2_stack(i, kwWEND, node.level);
				return;
				}

			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;

		case kwREPEAT:
			false_ip = bc_search_stack(i+1, kwUNTIL, node.level);

			if	( false_ip == INVALID_ADDR )	{
				sc_raise("REPEAT: Missing UNTIL on the same level");
				print_pass2_stack(i, kwUNTIL, node.level);
				return;
				}

			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;

		case kwUSE:
			true_ip = node.pos+(ADDRSZ+ADDRSZ+1);
			false_ip = bc_search_eoc(true_ip);
			memcpy(bc_prog.ptr+node.pos+1, &true_ip, ADDRSZ);
			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;

		case kwIF:
		case kwELIF:
			a_ip = bc_search_stack(i+1, kwENDIF, node.level);
			b_ip = bc_search_stack(i+1, kwELSE,  node.level);
			c_ip = bc_search_stack(i+1, kwELIF,  node.level);

			false_ip = a_ip;
			if	( b_ip != INVALID_ADDR && b_ip < false_ip ) false_ip = b_ip;
			if	( c_ip != INVALID_ADDR && c_ip < false_ip ) false_ip = c_ip;

			if	( false_ip == INVALID_ADDR )	{
				sc_raise("IF: Missing ELIF/ELSE/ENDIF on the same level");
				print_pass2_stack(i, kwENDIF, node.level);
				return;
				}

			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;

		case kwELSE:
			false_ip = bc_search_stack(i+1, kwENDIF, node.level);

			if	( false_ip == INVALID_ADDR )	{
				sc_raise("(ELSE) IF: Missing ENDIF on the same level");
				print_pass2_stack(i, kwENDIF, node.level);
				return;
				}

			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;

/////
/////
/////

		case kwTYPE_RET:
			break;

		case kwWEND:
			false_ip = bc_search_stack_backward(i-1, kwWHILE, node.level);
			if	( false_ip == INVALID_ADDR )	{
				sc_raise("WEND: Missing WHILE on the same level");
				print_pass2_stack(i, kwWHILE, node.level);
				return;
				}
			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;
		case kwUNTIL:
			false_ip = bc_search_stack_backward(i-1, kwREPEAT, node.level);
			if	( false_ip == INVALID_ADDR )	{
				sc_raise("UNTIL: Missing REPEAT on the same level");
				print_pass2_stack(i, kwREPEAT, node.level);
				return;
				}
			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;
		case kwNEXT:
			false_ip = bc_search_stack_backward(i-1, kwFOR, node.level);
			if	( false_ip == INVALID_ADDR )	{
				sc_raise("NEXT: Missing FOR on the same level");
				print_pass2_stack(i, kwFOR, node.level);
				return;
				}
			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;
		case kwENDIF:
			false_ip = bc_search_stack_backward(i-1, kwIF, node.level);
			if	( false_ip == INVALID_ADDR )	{
				sc_raise("ENDIF: Missing IF on the same level");
				print_pass2_stack(i, kwIF, node.level);
				return;
				}
			memcpy(bc_prog.ptr+node.pos+(ADDRSZ+1), &false_ip, ADDRSZ);
			break;
			};
		}

	if	( !opt_quite )
		dev_printf("\rPASS2: Node %d/%d\n", stk_count, stk_count);
}

/*
*	initialize
*/
void	bc_init() SEC(BCSCAN);
void	bc_init()
{
	bc_sec  = tmp_alloc(SB_KEYWORD_SIZE+1);
	memset(bc_sec, 0, SB_KEYWORD_SIZE+1);
	bc_name = tmp_alloc(SB_SOURCELINE_SIZE+1);
	bc_parm = tmp_alloc(SB_SOURCELINE_SIZE+1);
	bc_temp = tmp_alloc(SB_SOURCELINE_SIZE+1);
	bc_proc = tmp_alloc(SB_SOURCELINE_SIZE+1);

	scan_line = 0;
	scan_error = 0;
	lab_count = 0;
	var_count = 0;
	stk_count = 0;
	udp_count = 0;
	block_level = 0;
	block_id = 0;
	first_data_ip = INVALID_ADDR;
	bc_proc_level = 0;
	bc_proc[0] = '\0';

	var_table  = (var_t*)       tmp_alloc(GROWSIZE * sizeof(var_t));
	lab_table  = dbt_create("BCS-LBL.VMT");
	udp_table  = (udp_t*)      tmp_alloc(GROWSIZE * sizeof(udp_t));
	bc_stack   = dbt_create("BCS-STK.VMT");

	var_size   = /* lab_size = stk_size = */ udp_size  = GROWSIZE;
	var_count  = lab_count = stk_count = udp_count = 0;

	bc_create(&bc_prog);
	bc_create(&bc_data);

	if	( !var_table || !lab_table || !udp_table || !bc_stack )	
		panic("bc_init(): OUT OF MEMORY");

	if	( !opt_quite )
		dev_printf("VMT Initialization...\n");

	dbt_prealloc(lab_table, os_cclabs1, sizeof(label_t));
	dbt_prealloc(bc_stack, os_ccpass2, sizeof(pass_node_t));

	/*
	*	create system variables
	*/
	bc_get_var_id("OSVER");
	bc_get_var_id("OSNAME");
	bc_get_var_id("SBVER");
	bc_get_var_id("PI");
	bc_get_var_id("XMAX");
	bc_get_var_id("YMAX");
	bc_get_var_id("BPP");
	bc_get_var_id("TRUE");
	bc_get_var_id("FALSE");
	bc_get_var_id("LINECHART");
	bc_get_var_id("BARCHART");
	bc_get_var_id("CWD");
	bc_get_var_id("HOME");
	bc_get_var_id("COMMAND");
	bc_get_var_id("X");	// USE keyword
	bc_get_var_id("Y");	// USE keyword
	bc_get_var_id("Z");	// USE keyword
}

/*
*	clean up
*/
void	bc_close() SEC(BCSCAN);
void	bc_close()
{
	int		i;
	#if defined(_PalmOS)
//	LocalID		lid;
	#endif

	bc_destroy(&bc_prog);
	bc_destroy(&bc_data);

	for ( i = 0; i < var_count; i ++ )
		tmp_free(var_table[i].name);
	for ( i = 0; i < udp_count; i ++ )
		tmp_free(udp_table[i].name);

	tmp_free(var_table);
	dbt_close(lab_table);
	tmp_free(udp_table);
	dbt_close(bc_stack);
//	#if defined(_PalmOS)
//	if ( (lid = DmFindDatabase(0, "BCS-LBL.VMT")) != 0 )	DmDeleteDatabase(0, lid);
//	if ( (lid = DmFindDatabase(0, "BCS-STK.VMT")) != 0 )	DmDeleteDatabase(0, lid);
//	#endif

	var_count = lab_count = stk_count = udp_count = 0;

	tmp_free(bc_proc);
	tmp_free(bc_temp);
	tmp_free(bc_parm);
	tmp_free(bc_name);
	tmp_free(bc_sec);
	bc_reset_extlist();
}

/*
*	returns true if the 'fileName' exists
*/
int		basFileExist(const char *basfile)
{
	int		check = 0;
	char	*p, *fileName;
	#if defined(_PalmOS)
	LocalID	lid;
	#endif

	fileName = tmp_alloc(strlen(basfile)+5);
	strcpy(fileName, basfile);

	p = strchr(fileName, '.');
	if	( !p )	
		strcat(fileName, ".bas");

	#if defined(_PalmOS)
	lid = DmFindDatabase(0, fileName);
	check = (lid != 0);
	#elif defined(_VTOS)
	{
	FILE *fp;
	check=FALSE;
	fp=fopen(fileName,"rb");
	if (fp) {
		fclose(fp);
		check=TRUE;
		}
	}
	#else
		#if !defined(_UnixOS)
		check = (access(fileName, 0) == 0);
		#else
		check = (access(fileName, R_OK) == 0);
		#endif
	#endif

	tmp_free(fileName);
	return check;
}

/*
*/
void	bc_load(const char *fileName) SEC(BCSCAN);
void	bc_load(const char *fileName)
{
#if defined(_PalmOS)
	DmOpenRef	fp;
	LocalID		lid;
	int			l, i;
	VoidPtr		rec_p = NULL;
	VoidHand	rec_h = NULL;
#elif defined(_VTOS)
	FILE *unxh;
#else
	int		unxh;
	struct stat st;
#endif

	strcpy(bc_fileName, fileName);
#if defined(_PalmOS)
	lid = DmFindDatabase(0, (char *) fileName);
	fp = DmOpenDatabase(0, lid, dmModeReadWrite);
	if	( !fp )	{
		panic("LOAD: CAN'T OPEN FILE %s", fileName);
		return;
		}
	l = DmNumRecords(fp) - 1;
	if	( l <= 0 )	{
		panic("LOAD: BAD FILE STRUCTURE %s", fileName);
		return;
		}

	for ( i = 0; i < l; i ++ )	{
		rec_h = DmGetRecord(fp, i+1);
		if	( !rec_h )
			panic("LOAD: CAN'T GET RECORD %s", fileName);
		rec_p = mem_lock(rec_h);
		if	( !rec_p )
			panic("LOAD: CAN'T LOCK RECORD %s", fileName);

		if	( i == 0 && strlen(rec_p+6) == 0 )
			bc_scan("Main", rec_p+70);	// + sizeof(sec_t);
		else
			bc_scan(rec_p+6, rec_p+70);	// + sizeof(sec_t);

		mem_unlock(rec_h);
		DmReleaseRecord(fp, i+1, 0);

		if	( bc_get_error() )
			break;
		}

	DmCloseDatabase(fp);
#else
	#if !defined(_VTOS)
	stat(fileName, &st);
	#endif

	#if defined(_DOS) || defined(_Win32)
	unxh = open(fileName, O_BINARY);
	#elif defined(_VTOS)
	unxh = fopen(fileName, "rb");
	if (unxh==NULL) unxh=(FILE*)-1;
	#else
	unxh = open(fileName, 0);
	#endif

	#if defined(_VTOS)
	if	( unxh == (FILE*)-1 )	{
	#else
	if	( unxh == -1 )	{
	#endif
		panic("can't open %s\n", fileName);
		return;
		}
	else	{
		char	*buff;

		/* read & compile file */
#if defined(_VTOS)
		buff = (char *) tmp_alloc(fsize(unxh)+1);
		fread(buff, 1, fsize(unxh), unxh);
		buff[fsize(unxh)] = '\0';
#else
		buff = (char *) tmp_alloc(st.st_size+1);
		read(unxh, buff, st.st_size);
		buff[st.st_size] = '\0';
#endif

		bc_scan(NULL, buff);

		tmp_free(buff);
#if defined(_VTOS)
		fclose(unxh);
#else
		close(unxh);
#endif
		}
#endif
}

/**
*	PASS 1
*/
void	bc_scan(const char *section, const char *text)
{
	char	*ps, *p, lc = 0;
	int		quotes, len, join_line;
	char	pname[SB_KEYWORD_SIZE+1];
	char	*code_line;
	char	*new_text;

	code_line = tmp_alloc(SB_SOURCELINE_SIZE+1);
	new_text = tmp_alloc((len=(strlen(text) + 4)));
	memset(new_text, 0, len);

	memset(bc_sec, 0, SB_KEYWORD_SIZE+1);
	if	( section )
		strncpy(bc_sec, section, SB_KEYWORD_SIZE);
	else
		strncpy(bc_sec, "Main", SB_KEYWORD_SIZE);

	/*
	*	first
	*
	*	Make it simple text
	*		* space-chars is only the space
	*		* CR/LF are fixed
	*		* control chars are out
	*
	*	TODO: join-lines character (&)
	*/
	p = (char *) text;
	ps = new_text;
	quotes = 0;
	while ( *p )	{
		if	( !quotes )	{
			if ( *p == '\n' )	{
				*ps ++ = '\n';
				while ( *(p+1) == ' ' || *(p+1) == '\t' )
					p ++;
				}
			else if ( (*p == '\t' || *p == ' ') && lc != ' ' )
				*ps ++ = ' ';
			else if	( (*p > ' ') || (*p & 0x80) )
				*ps ++ = to_upper(*p);

			lc = *ps;
			}
		else
			*ps ++ = *p;

		if	( *p == '\"' )	
			quotes = !quotes;

		p ++;
		}
	*ps = '\0';	
	strcat(new_text, "\n");

	/*
	*	second (we can change it to support preprocessor)
	*
	*	Check for:
	*	include (#inc:)
	*	UDF and UDP declarations
	*/
	p = ps = new_text;
	bc_proc_level = 0;
	*bc_proc = '\0';
	while ( *p )	{
		if	( strncmp("#INC:", p, 5) == 0 )	{
			char	*crp  = NULL;

			p += 5;
			if	( *p == '\"' )	{
				p ++;

				crp = p;
				while ( *crp != '\0' && *crp != '\"' )
					crp ++;

				if	( *crp == '\0' )	{
					sc_raise("#INC: Missing \"");
					break;
					}

				(lc = *crp, *crp = '\0');
				}
			else	{
				crp = strchr(p, '\n');
				*crp = '\0';
				lc = '\n';
				}

			strcpy(code_line, p);
			*crp = lc;
			str_alltrim(code_line);
			if	( !basFileExist(code_line) )	
				sc_raise("File %s: File %s does not exist", bc_fileName, code_line);
			else	{
				#if defined(_PalmOS) 
				char	fileName[65];
				#else
				char	fileName[1024];
				#endif
				char	sec[SB_KEYWORD_SIZE+1];
				
				strcpy(sec, bc_sec);
				strcpy(fileName, bc_fileName);
				if	( strchr(code_line, '.') == NULL )
					strcat(code_line, ".bas");
				bc_load(code_line); 
				strcpy(bc_fileName, fileName);
				strcpy(bc_sec, sec);
				}
			}
		if	( (strncmp("SUB ", p, 4) == 0) || (strncmp("FUNC ", p, 5) == 0) || (strncmp("DEF ", p, 4) == 0) )	{
			char	*lpar = NULL;
			char	*crp  = NULL;
			char	*eqn  = NULL;

			if	( *p == 'S' || *p == 'D' )
				p += 4;
			else
				p += 5;

			// close string
			lpar = strchr(p, '(');
			eqn  = strchr(p, '=');
			crp  = strchr(p, '\n');
			if	( lpar && crp )	{
				if	( lpar < crp )		{
					*lpar = '\0';
					crp = NULL;
					}
				else	{
					*crp = '\0';
					lpar = NULL;
					}
				}
			else if ( lpar )
				*lpar = '\0';
			else if ( crp )
				*crp = '\0';

			// add declaration
			bc_prepname(pname, p, SB_KEYWORD_SIZE);
			bc_addudp(pname);
			if	( bc_proc_level )	{
				strcat(bc_proc, "/");
				strcat(bc_proc, pname);
				}
			else
				strcpy(bc_proc, pname);

			if	( !eqn )
				bc_proc_level ++;
			else	{
				// inline (DEF FN)
				char	*dol;

				dol = strrchr(bc_proc, '/');
				if	( dol )
					*dol = '\0';
				else
					*bc_proc = '\0';
				}

			// restore string
			if	( lpar )
				*lpar = '(';
			else	{
				if	( crp )
					*crp = '\n';
				}
			}
		else if ( bc_proc_level )	{
			if	( strncmp("END ", p, 4) == 0 || strncmp("END\n", p, 4) == 0 )	{
				char	*dol;

				dol = strrchr(bc_proc, '/');
				if	( dol )
					*dol = '\0';
				else
					*bc_proc = '\0';

				bc_proc_level --;
				}
			}

		// skip text line
		while ( *p != '\0' && *p != '\n' )
			p ++;

		if	( *p )
			p ++;
		}

	if	( bc_proc_level )
		sc_raise("File %s: SUB/FUNC: Missing END", bc_fileName);

	bc_proc_level = 0;
	*bc_proc = '\0';

	if	( !opt_quite )	{
		#if defined(_PalmOS)
		dev_printf("File: \033[1m%s\033[0m\n\033[80mSection: \033[1m%s\033[0m\033[80m\n", bc_fileName, bc_sec);
		#else
		dev_printf("File: \033[1m%s\033[0m\nSection: \033[1m%s\033[0m\n", bc_fileName, bc_sec);
		#endif
		}

	/*
	*	Start
	*/
	join_line = 0;
	if	( !scan_error )	{
		scan_line = 0;	// remove this for common line nums
		if	( !opt_quite )	
			dev_printf("PASS1: Line %d", scan_line+1);

		ps = p = new_text;
		while ( *p )	{
			if	( *p == '\n' )	{
				if	( *bc_prev_char(new_text, p) == '&' )	{	// I must move it to first-check (the problem is in line-number counter)
					//
					//	join with next line
					//
					char	*jp;

					jp = bc_prev_char(new_text, p);
					*jp = '\0';

					if	( join_line )
						strcat(code_line, ps);
					else	{
						strcpy(code_line, ps);
						join_line = 1;
						}

					bc_add_code(&bc_prog, kwTYPE_LINE);
					bc_add_addr(&bc_prog, scan_line);

					scan_line ++;
					ps = p+1;
					}
				else	{
					//
					//	proceed
					//
					*p = '\0';

					scan_line ++;
					if	( !opt_quite )	{
						#if defined(_PalmOS)
						if	( (scan_line % 8) == 0 )	{
							if	( (scan_line % 32) == 0 )	
								dev_printf("\rPASS1: Line %d", scan_line);
							if	( dev_events(0) < 0 )	{
								dev_print("\n\n\a*** interrupted ***\n");
								scan_error = -1;
								}
							}
						#else
						if	( (scan_line % 256) == 0 )	
							dev_printf("\rPASS1: Line %d", scan_line);
						#endif
						}

					bc_add_code(&bc_prog, kwTYPE_LINE);
					bc_add_addr(&bc_prog, scan_line);

					if	( join_line )	{
						strcat(code_line, ps);
				 		join_line = 0;
						}
					else
						strcpy(code_line, ps);

					bc_scan_cmd(code_line);
					if	( scan_error )
						break;

					ps = p+1;
					}
				}


			if	( scan_error )
				break;

			p ++;
			}
		}

	tmp_free(code_line);
	tmp_free(new_text);

	if	( join_line && !scan_error )
		sc_raise("'&' on the last line!");

	bc_eoc(&bc_prog);
	bc_resize(&bc_prog, bc_prog.count);
	if	( !scan_error )	{
		if	( !opt_quite )	{
			dev_printf("\rPASS1: Line %d; finished\n", scan_line+1);
			#if !defined(_PalmOS)
			dev_printf("\rSB-MemMgr: Maximum use of memory: %dKB\n", (memmgr_getmaxalloc()+512) / 1024);
			#endif
			dev_printf("\n");
			}
		}

}

/*
*	PASS 2
*/
void	bc_pass2(void) SEC(BCSCAN);
void	bc_pass2()
{
	if	( !opt_quite )	
		dev_printf("PASS2...");

	if	( bc_proc_level )	
		sc_raise("SUB/FUNC: Missing END");
	else	{
		bc_add_code(&bc_prog, kwSTOP);
		first_data_ip = bc_prog.count;

		bc_pass2_scan();
		}

	if	( block_level && (scan_error==0) )	
		sc_raise("%d loop(s) remains open", block_level);
	if	( bc_data.count )	
		bc_append(&bc_prog, &bc_data);
}

/*
*	final, create bytecode
*/
mem_t	bc_createbin(void) SEC(BCSCAN);
mem_t	bc_createbin()
{
	int			i;
	mem_t		buff_h;
	byte		*buff, *cp;
	label_t		label;
	bc_head_t	hdr;

	if	( !opt_quite )	
		dev_printf("Creating byte-code...\n");

	hdr.sign      = 'V';
	hdr.ver       = 1;
	hdr.sbver	  = SB_DWORD_VER;
	#if defined(CPU_BIGENDIAN)
	hdr.flags     = 1;
	#else
	hdr.flags     = 0;
	#endif
	#if defined(OS_ADDR16)
	hdr.flags     |= 2;
	#elif defined(OS_ADDR32)
	hdr.flags     |= 4;
	#endif

	hdr.bc_count  = bc_prog.count;
	hdr.var_count = var_count;
	hdr.lab_count = lab_count;
	hdr.data_ip   = first_data_ip;
	hdr.size      = sizeof(bc_head_t) + (lab_count * ADDRSZ) + bc_prog.count;
	
	buff_h = mem_alloc(hdr.size+4); 	// +4 (checkop12SB_KEYWORD_SIZE+1)
	buff = mem_lock(buff_h);

	memcpy(buff, &hdr, sizeof(bc_head_t));
	cp = buff + sizeof(bc_head_t);

	// label table
	for ( i = 0; i < lab_count; i ++ )	{
		dbt_read(lab_table, i, &label, sizeof(label_t));
		memcpy(cp, &label.ip, ADDRSZ);
		cp += ADDRSZ;
		}

	// the program itself
	memcpy(cp, bc_prog.ptr, bc_prog.count);
	mem_unlock(buff_h);

	// print
	if	( !opt_quite )	{
		dev_printf("Variables  %d\n", var_count);
		dev_printf("Labels     %d\n", lab_count);
		dev_printf("Proc/Func  %d\n", udp_count);
		dev_printf("Code size  %d\n", bc_prog.count);
		dev_printf("\n");
		}

	return buff_h;
}


