/**
*	SmallBASIC: Code & Variable Manager.
*
*	2000-05-27, Nicholas Christopoulos
*
*	This program is distributed under the terms of the GPL v2.0 or later
*	Download the GNU Public License (GPL) from www.gnu.org
*/

#if !defined(_sb_cvm_h)
#define _sb_cvm_h

#include "pmem.h"
#include "sberr.h"

/*
*	Variable - types
*/
#if defined(_WinBCB)
	#undef V_INT
    #undef V_ARRAY
#endif
#define	V_NULL		0	//
#define	V_INT		0	// 32bit integer
#define	V_REAL		2	// 64bit float
#define	V_NUM		2	// 64bit float (same as V_NUM)
#define	V_STR		3	// string
#define	V_ARRAY		4	// array of variables
#define	V_PTR		5	// binary - N/A
#define	V_LIST		6	// dynamic list - N/A

/*
*	predefined system variables - index
*/
#define	SYSVAR_OSVER		0
#define	SYSVAR_OSNAME		1
#define	SYSVAR_SBVER		2
#define	SYSVAR_PI			3
#define	SYSVAR_XMAX			4
#define	SYSVAR_YMAX			5
#define	SYSVAR_BPP			6
#define	SYSVAR_TRUE			7
#define	SYSVAR_FALSE		8
#define	SYSVAR_LINECHART	9
#define	SYSVAR_BARCHART		10
#define	SYSVAR_PWD  		11
#define	SYSVAR_HOME  		12
#define	SYSVAR_COMMAND 		13
#define	SYSVAR_X	  		14
#define	SYSVAR_Y	  		15
#define	SYSVAR_Z	  		16

/*
*	Maxium number of array-dimensions
*/
#if defined(OS_LIMITED)
	#define	MAXDIM		3			// think before increase this, (possible stack overflow)
#else
	#if defined(OS_ADDR16)
		#define	MAXDIM		3		// think before increase this, (possible stack overflow)
	#else
		#define	MAXDIM		6		// that's large enough
	#endif
#endif

#if defined(__cplusplus)
extern "C" {
#endif

#include "scan.h"					// compiler structures

/*
*	VARIANT DATA TYPE
*/
struct var_s	{
    byte	type;
	byte	const_flag;

	// value 
	union {
		#if defined(OS_PREC64)
		long double	n;
		long long	i;
		#else
		double		n;					// numeric value
		long		i;					// integer value
		#endif

		// generic ptr (string)
		struct {
			byte	*ptr;				// data ptr
			#if defined(OS_ADDR16)
			int16	size;				// the size of string
			int16	pos;				// position in string (used by pv_* functions)
			#else
			int32	size;				// the size of string
			int32	pos;				// position in string (used by pv_* functions)
			#endif
			} p;

		// array
		struct {
			byte	*ptr;				// array data ptr (sizeof(var_t) * size)
			#if defined(OS_ADDR16)
			int16	size;				// the number of elements
			int16	lbound[MAXDIM];		// lower bound
			int16	ubound[MAXDIM];		// upper bound 
			#else
			int32	size;				// the number of elements
			int32	lbound[MAXDIM];		// lower bound
			int32	ubound[MAXDIM];		// upper bound 
			#endif

			byte	maxdim;				// number of dimensions
			} a;
		} v;
	};

typedef struct var_s var_t;
typedef var_t* var_p_t;

/*
*	label
*/
struct lab_s	{
	addr_t	ip;
	};
typedef struct lab_s lab_t;

/*
*	EXECUTOR's STACK NODE
*/
struct stknode_s	{
	code_t	type;				// type of node (keyword id, i.e. kwGOSUB, kwFOR, etc)
	addr_t	exit_ip;			// EXIT command IP to go

	union	{

		// CMD: FOR-TO-NEXT
		struct {
			code_t	subtype;		// kwTO | kwIN
			var_t	*var_ptr;		// 'FOR' variable
			var_t	*arr_ptr;		// FOR-IN array-variable
			addr_t	to_expr_ip;		// IP of 'TO' expression
			addr_t	step_expr_ip;	// IP of 'STEP' expression (FOR-IN = current element)
			addr_t	jump_ip;		// code block IP
			byte	flags;			// 
			} vfor;

		// CMD: IF/ELIF
		struct {
			addr_t	lcond;	// result of the last condition
			} vif;			

		// CMD: GOSUB
		struct {
			addr_t	ret_ip;	// return ip
			} vgosub;

		// CMD: CALL UDP/F
		struct {
			addr_t	ret_ip;	// return ip
			word 	pcount; // number of parameters
			bid_t	rvid; 	// return-variable ID
			var_t*	retvar;	// return-variable data
			} vcall;

		// CMD: Create dynamic variable (LOCAL or PARAMETER)
		struct {
			bid_t	vid;	// variable index in tvar
			var_t	*vptr;	// previous variable
			} vdvar;

		// CMD: parameter (CALL UDP/F)
		struct {
			word	vcheck;	// checks (1=BYVAL ONLY, 3=BYVAL|BYREF, 2=BYREF ONLY)
			var_t	*res;	// variable pointer (for BYVAL this is a clone)
			} param;

		} x;
	};
typedef struct stknode_s stknode_t;

/*
*	Basic variable's API
*/
var_t*	v_new(void);								// creates a new variable
int		v_is_nonzero(var_t *v);						// returns true if the value is not NULL (0 for numeric, 
double	v_getval(var_t *v);							// returns the floating-point value of a var. (if v is string it will converted to double)
#define v_getnum(a)	v_getval((a))
long	v_igetval(var_t *v);						// returns the integer value of a var. (if v is string it will converted to integer)
#define v_getint(a)	v_igetval((a))
int		v_compare(var_t *a, var_t *b);				// compare two variables
int		v_addtype(var_t *a, var_t *b);
void	v_add(var_t *result, var_t *a, var_t *b);	//
void	v_init(var_t *v);							// initialize a var.
void	v_free(var_t *v);							// free variable's data; warning: only the data
void	v_set(var_t *dest, const var_t *src);		// assing: dest = src
void	v_inc(var_t *a, var_t *b);					// increase the value of variable a by b
int		v_sign(var_t *x);							// returns the sign of a variable
var_t*	v_getelemptr(var_t *v, word index);			// returns the var_p of an array element
void	v_createstr(var_t *v, const char *src);		// create a string variable (with value str)
void	v_tostr(var_t *arg);						// convert variable to string
var_t*	v_clone(const var_t *source);				// create a clone of variable 'source'
void	v_resize_array(var_t *v, int size);			// resize an 1-dim array

void	v_tomatrix(var_t *v, int r, int c)			SEC(BLIB);	// convert variable v to a RxC matrix
var_t	*v_new_matrix(int r, int c)					SEC(BLIB);	// creates and returns a new matrix (array RxC) variable 
void	v_toarray1(var_t *v, int r)					SEC(BLIB);	// converts the variable v to an array of R elements
																// R can be zero for zero-length arrays

int     v_isempty(var_t *v);						// returns true if the 'v' is empty (see EMPTY())
int     v_length(var_t *v);							// returns the length of the variable (see LEN())

/*
*	New API (Dec 2001)
*
*	Use these routines
*
*	Memory free/alloc is contolled inside these functions
*	The only thing that you must care of, is when you declare local var_t elements
*
*	Auto-type-convertion is controlled inside these functions,
*	So if you want a string value of an integer you just do strcpy(buf,v_getstr(&myvar));
*	or a numeric value of a string R = v_getnum(&myvar);
*
*	Using variables on code:
*
*	void	myfunc()					// using them in stack
*	{
*		var_t	myvar;
*		v_init(&myvar);					// DO NOT FORGET THIS! local variables are had random data
*		...
*		v_setstr(&myvar, "Hello, world");
*		...
*		v_set(&myvar, &another_var);	// copy variables (LET myvar = another_var)
*		...
*		v_setint(&myvar, 0x100);		// Variable will be cleared automatically
*		...
*		v_free(&myvar);
*		}
*
*	void	myfunc()					// using dynamic memory
*	{
*		var_t	*myvar_p;
*
*		myvar_p = v_new();				//	create a new variable
*		...
*		v_setstr(myvar_p, "Hello, world");
*		...
*		v_setint(myvar_p, 0x100);		// Variable will be cleared automatically
*		...
*		v_free(myvar_p);				// clear variable's data
*		tmp_free(myvar_p);				// free the variable
*		}
*
*	Old API routines that is good to use them:
*
*	v_init(), v_free()					--- basic
*	v_new(), v_clone(), v_new_matrix()	--- create vars in heap, dont forget the tmp_free()
*
*	v_resize_array()					--- resize an 1-dim array
*	v_tomatrix(), v_toarray1()			--- convertion to matrix (RxC) or 1D array
*/
void	v_setstr        (var_t *var, const char *string)		SEC(BLIB);	// sets a string value to variable 'var'
void	v_strcat        (var_t *var, const char *string)		SEC(BLIB);	// concate string to variable 'var'
void	v_setreal       (var_t *var, double number)				SEC(BLIB);	// sets a real-number value to variable 'var'
void	v_setint        (var_t *var, int32  integer)			SEC(BLIB);	// sets an integer value to variable 'var'
void	v_setintarray   (var_t *var, int32  *itable, int count)	SEC(BLIB);	// makes 'var' an array of integers and copies the itable elements to it
void	v_setrealarray  (var_t *var, double *ntable, int count)	SEC(BLIB);	// makes 'var' an array of reals and copies the ntable elements to it
void	v_setstrarray   (var_t *var, char  **ctable, int count)	SEC(BLIB);	// makes 'var' an array of strings and copies the ctable elements to it

void	v_zerostr(var_t *r)										SEC(BLIB);	// makes 'var' an empty string variable
#define	v_zeroint(r)			v_init((r))									// makes 'var' an empry integer variable

void	v_input2var(const char *str, var_t *var)				SEC(BLIB);	// assign value 'str' to var
																			// the final type of the var will be decided
																			// on that function (numeric if the str is a
																			// numeric-constant string or string)
																			// Info: used by INPUT to convert the variables

#define	v_elem(x,i)		(var_t *) ( (x)->v.a.ptr + (sizeof(var_t) * (i)) )	// returns the var_t pointer of the element i
																			// on the array x. i is a zero-based, one dim, index.

#define	v_asize(x)		((x)->v.a.size)										// the number of the elements of the array (x)

/*
*	new api (dec 2001) - get value
*/
#define	v_getint(v)		v_igetval((v))										// returns the integer value of variable v
#define	v_getreal(v)	v_getval((v))										// returns the real value of variable v
char*	v_getstr(var_t *);													// returns the string-pointer of variable v

/*
*	low-level byte-code parsing 
*
*	Usually you must not use these functions (except the rt_raise)
*	try the parameters API.
*/
void	rt_raise(const char *fmt, ...);					// run-time error

dword	code_getnext32(void);							//	R(long int) <- Code[IP]; IP+=4
double	code_getnext64f(void);							//	R(double)   <- Code[IP]; IP+=8

#if defined(OS_PREC64)
long long	code_getnext64i(void);						//	R(long long)   <- Code[IP]; IP+=8
long double	code_getnext128f(void);						//	R(long double) <- Code[IP]; IP+=16
#endif

int		code_checkop(byte op);
int		code_checkop_s(byte *str);
void	code_jump_label(word label_id);				//  IP <- LABEL_IP_TABLE[label_id]
#define	code_jump(newip)			prog_ip=(newip)	//	IP <- NewIP

void	code_push(stknode_t *node);					// stores a node to stack
void	code_pop(stknode_t *node);					// restores the topmost node from stack

var_t	*code_getvarptr(void);						// R(var_t*) <- Code[IP]; IP += 2;
int		code_isvar(void);							// returns true if the next code is a single variable

#define	code_peek()			prog_source[prog_ip]	// R(byte) <- Code[IP]
#define	code_getnext()		prog_source[prog_ip++]	// R(byte) <- Code[IP]; IP ++;

#define	code_skipnext()		prog_ip++				// IP ++;
#define	code_skipnext16()	prog_ip+=2				// IP += 2;
#define	code_skipnext32()	prog_ip+=4				// IP += 4;
#define	code_skipnext64()	prog_ip+=8				// IP += 8;

#if !defined(_PalmOS) && !defined(_VTOS)
#define	code_getnext16()		(*((word *) (prog_source+(prog_ip+=2)-2)))
#define	code_peeknext16()		(*((word *) (prog_source+prog_ip)))
#define	code_peek16(o)			(*((word *) (prog_source+(o))))
#define	code_peek32(o)			(*((dword *) (prog_source+(o))))
#else	// !!!
	#if defined(CPU_BIGENDIAN)
		#define	code_getnext16()	(prog_ip+=2, (prog_source[prog_ip-1]<<8)|prog_source[prog_ip-2])        // R(2-byte-word+
		#define	code_peeknext16()	((prog_source[prog_ip+1]<<8)|prog_source[prog_ip])
		#define	code_peek16(o)		((prog_source[(o)+1]<<8)|prog_source[(o)])                      // R(2-byte-word) <- Cod+ 
		#define	code_peek32(o)		(((addr_t) code_peek16((o)+2) << 16) + (addr_t) code_peek16((o)))
	#else
		#define	code_getnext16()	(prog_ip+=2, (prog_source[prog_ip-2]<<8)|prog_source[prog_ip-1])	// R(2-byte-word) <- Code[IP]; IP += 2;
		#define	code_peeknext16()	((prog_source[prog_ip]<<8)|prog_source[prog_ip+1])					// R(2-byte-word) <- Code[IP];
		#define	code_peek16(o) 		((prog_source[(o)]<<8)|prog_source[(o)+1])			// R(2-byte-word) <- Code[IP+o];
		#define	code_peek32(o) 		( ((addr_t) code_peek16((o)) << 16) + (addr_t) code_peek16((o)+2) )
	#endif
#endif

/*
*	New API (Dec 2001, 16 & 32 bit support)
*
*	Use these code_x instead of old ones
*/
#define	code_skipopr()		code_skipnext16()						// skip operator
#define	code_skipsep()		code_skipnext16()						// skip separator
#define	code_getsep()		(prog_ip ++, prog_source[prog_ip++])	// returns the separator and advance (IP) to next command
#define	code_peeksep()		(prog_source[prog_ip+1])

#if defined(OS_ADDR16)
#define	code_getaddr()		code_getnext16()
#define	code_skipaddr()		code_skipnext16()
#define code_getstrlen()	code_getnext16()
#define code_peekaddr(i)	code_peek16((i))
#else
#define	code_getaddr()		code_getnext32()						// get address value and advance
#define	code_skipaddr()		code_skipnext32()						// skip address field
#define code_getstrlen()	code_getnext32()						// get strlen	(kwTYPE_STR) and advance
#define code_peekaddr(i)	code_peek32((i))						// peek address field at offset i
#endif

#if defined(OS_PREC64)
#define	code_getint()		code_getnext64i()					
#define	code_getreal()		code_getnext128f()
#else
#define	code_getint()		code_getnext32()						// get integer	(kwTYPE_INT)
#define	code_getreal()		code_getnext64f()						// get real		(kwTYPE_NUM)
#endif

/*
*	System variables (osname, osver, bpp, xmax, etc)
*
*	The variables must be defined
*	a) here (see in first lines) - (variable's index)
*	b) in scan.c (variable's name)
*	c) in brun.c or device.c (variable's value)
*
*	DO NOT LOSE THE ORDER
*/
void	setsysvar_int(int index, long val)			SEC(BLIB);		// sets the integer value of a system variable (constant)
void	setsysvar_num(int index, double val)		SEC(BLIB);		// the same for double
void	setsysvar_str(int index, const char *value)	SEC(BLIB);		// the same for string

/*
*	Rest...
*/
double	*mat_toc(var_t *v, int *rows, int *cols)	SEC(BMATH);
void	mat_tov(var_t *v, double *m, int rows, int cols, int protect_col1) SEC(BMATH);

#if defined(__cplusplus)
}
#endif

#endif

