#include <stdio.h>

#define PSF1_MAGIC0     0x36
#define PSF1_MAGIC1     0x04

#define PSF1_MODE512    0x01
#define PSF1_MODEHASTAB 0x02
#define PSF1_MODEHASSEQ 0x04
#define PSF1_MAXMODE    0x05

#define PSF1_SEPARATOR  0xFFFF
#define PSF1_STARTSEQ   0xFFFE

struct psf1_header {
        unsigned char magic[2];     /* Magic number */
        unsigned char mode;         /* PSF font mode */
        unsigned char charsize;     /* Character size */
};


#define PSF2_MAGIC0     0x72
#define PSF2_MAGIC1     0xb5
#define PSF2_MAGIC2     0x4a
#define PSF2_MAGIC3     0x86

/* bits used in flags */
#define PSF2_HAS_UNICODE_TABLE 0x01

/* max version recognized so far */
#define PSF2_MAXVERSION 0

/* UTF8 separators */
#define PSF2_SEPARATOR  0xFF
#define PSF2_STARTSEQ   0xFE

struct psf2_header {
        unsigned char magic[4];
        unsigned int version;
        unsigned int headersize;    /* offset of bitmaps in file */
        unsigned int flags;
        unsigned int length;        /* number of glyphs */
        unsigned int charsize;      /* number of bytes for each character */
        unsigned int height, width; /* max dimensions of glyphs */
};

void
dumpPSF1(FILE* fp)
{
	struct psf1_header hdr;
	int i, j;

	// read header information
	if (1 != fread(&hdr, sizeof(hdr), 1, fp)) {
		fprintf(stderr, "Error: cannot read font header\n");
		return;
	}
	if (
	    (PSF1_MAGIC0 != hdr.magic[0]) ||
	    (PSF1_MAGIC1 != hdr.magic[1])
	   )
	{
		fprintf(stderr,
		    "Error: magic sequence for PSF1 font is not matched\n");
		return;
	}
	
	// dump header info
	
	printf(
	       "/**********************************************/\n"
	       "/*                                            */\n"
	       "/*       Font file generated by psf2c         */\n"
	       "/*                                            */\n"
	       "/*  character dimensions: 8x%u                */\n"
	       "/*                                            */\n"
	       "/*                                            */\n"
	       "/*                                            */\n"
	       "/**********************************************/\n",
	       hdr.charsize
	      );

/*
	if (hdr.mode & PSF1_MODE512) {
		fprintf(stderr,
		    "Error: only 256 characters font can be processed\n");
		return;
	}
*/

	printf("static unsigned char font8x%u[%d] = {\n",
	    hdr.charsize,256*hdr.charsize);

	// write characters data
	for (i = 0; i < 256; ++i) {
		printf("\n\t/* char %d(%c) */\n", i, (char)(i));
		for (j = 0; j < hdr.charsize; ++j) {
			unsigned char data;
			if (1 != fread(&data, 1, 1, fp)) {
				fprintf(stderr,
				    "Error: cannot read character data\n");
				return;
			}
			printf("\t0x%-4x", data);
			if (255 == i && hdr.charsize -1 == j) {
				printf(" ");
			} else {
				printf(",");
			}

			{
				// binary representation
				int bit;
				printf(" /* ");
				for (bit = 7; bit >= 0; --bit) {
					if ((data >> bit) & 1) {
						printf(" ");
					} else {
						printf("O");
					}
				}
				printf(" */");
				printf("\n");
			}
		}
	}

	printf("};\n\n");


}

void
dumpPSF2(FILE* fp)
{
	struct psf2_header hdr;
	int i, j;
	unsigned char data, data1, *fontdata;

	// read header information
	if (1 != fread(&hdr, sizeof(hdr), 1, fp)) {
		fprintf(stderr, "Error: cannot read font header\n");
		return;
	}
	if (
	    (PSF2_MAGIC0 != hdr.magic[0]) ||
	    (PSF2_MAGIC1 != hdr.magic[1]) ||
	    (PSF2_MAGIC2 != hdr.magic[2]) ||
	    (PSF2_MAGIC3 != hdr.magic[3])
	   )
	{
		fprintf(stderr,
		    "Error: magic sequence for PSF2 font is not matched\n");
		return;
	}
	
	// dump header info
	
	printf(
	       "/**********************************************/\n"
	       "/*                                            */\n"
	       "/*       Font file generated by psf2c         */\n"
	       "/*                                            */\n"
	       "/*  %u characters                             */\n"
	       "/*  %u bytes per character                    */\n"
	       "/*  character dimensions: %ux%u               */\n"
	       "/*                                            */\n"
	       "/*                                            */\n"
	       "/*                                            */\n"
	       "/**********************************************/\n",
	       hdr.length, hdr.charsize, hdr.width, hdr.height
	      );

	if (256 != hdr.length) {
		fprintf(stderr,
		    "Error: only 256 characters font can be processed - you have %d characters\n", hdr.length);
		return;
	}

	if (16 < hdr.width) {
		fprintf(stderr,
		    "Error: only font with character width <=16 can be processed\n");
		return;
	}

	printf("static unsigned char font%ux%u[%d] = {\n",
	    hdr.width, hdr.height,hdr.length*hdr.charsize);

	fontdata = (unsigned char *) malloc(hdr.length*hdr.charsize);
	if (fontdata == NULL) {
		fprintf(stderr, "Error not enough memory for fontdata\n");
		return;
		}

	if ( hdr.length*hdr.charsize != fread(fontdata, 1, hdr.length * hdr.charsize, fp) ) {
		fprintf(stderr,
		    "Error: cannot read character data\n");
		return;
		}

	// write characters data
	for (i = 0; i < 256; ++i) {
		printf("\n\t/* char %d(%c) */\n", i, (char)(i));
		for (j = 0; j < hdr.height; ++j) {
			data = fontdata[i*hdr.charsize + j*((hdr.width+7)/8)];
			printf("\t0x%-4x", data);
			if (hdr.width > 8) {
				data1 = fontdata[i*hdr.charsize + j*((hdr.width+7)/8)+1];
				printf(",\t0x%-4x", data1);
			} else data1 = 0 ;

			if (255 == i && hdr.charsize -1 == j) {
				printf(" ");
			} else {
				printf(",");
			}

			{
				// binary representation
				printf(" /* ");
				if (hdr.width < 8) {
				switch (hdr.width) {
				case 7:
					if (data & (1<<7)) printf(" ");		else printf("O");
					if (data & (1<<6)) printf(" ");		else printf("O");
					if (data & (1<<5)) printf(" ");		else printf("O");
					if (data & (1<<4)) printf(" ");		else printf("O");
					if (data & (1<<3)) printf(" ");		else printf("O");
					if (data & (1<<2)) printf(" ");		else printf("O");
					if (data & (1<<1)) printf(" ");		else printf("O");
					break;
				case 6:
					if (data & (1<<7)) printf(" ");		else printf("O");
					if (data & (1<<6)) printf(" ");		else printf("O");
					if (data & (1<<5)) printf(" ");		else printf("O");
					if (data & (1<<4)) printf(" ");		else printf("O");
					if (data & (1<<3)) printf(" ");		else printf("O");
					if (data & (1<<2)) printf(" ");		else printf("O");
					break;
				case 5:
					if (data & (1<<7)) printf(" ");		else printf("O");
					if (data & (1<<6)) printf(" ");		else printf("O");
					if (data & (1<<5)) printf(" ");		else printf("O");
					if (data & (1<<4)) printf(" ");		else printf("O");
					if (data & (1<<3)) printf(" ");		else printf("O");
					break;
				default:		// character thinner than 4 is not realistic!
					if (data & (1<<7)) printf(" ");		else printf("O");
					if (data & (1<<6)) printf(" ");		else printf("O");
					if (data & (1<<5)) printf(" ");		else printf("O");
					if (data & (1<<4)) printf(" ");		else printf("O");
					break;
				} } else {
					if (data & (1<<7)) printf(" ");		else printf("O");
					if (data & (1<<6)) printf(" ");		else printf("O");
					if (data & (1<<5)) printf(" ");		else printf("O");
					if (data & (1<<4)) printf(" ");		else printf("O");
					if (data & (1<<3)) printf(" ");		else printf("O");
					if (data & (1<<2)) printf(" ");		else printf("O");
					if (data & (1<<1)) printf(" ");		else printf("O");
					if (data & (1<<0)) printf(" ");		else printf("O");
				}
				switch (hdr.width) {
				case 16:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					if (data1 & (1<<6)) printf(" ");		else printf("O");
					if (data1 & (1<<5)) printf(" ");		else printf("O");
					if (data1 & (1<<4)) printf(" ");		else printf("O");
					if (data1 & (1<<3)) printf(" ");		else printf("O");
					if (data1 & (1<<2)) printf(" ");		else printf("O");
					if (data1 & (1<<1)) printf(" ");		else printf("O");
					if (data1 & (1<<0)) printf(" ");		else printf("O");
					break;
				case 15:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					if (data1 & (1<<6)) printf(" ");		else printf("O");
					if (data1 & (1<<5)) printf(" ");		else printf("O");
					if (data1 & (1<<4)) printf(" ");		else printf("O");
					if (data1 & (1<<3)) printf(" ");		else printf("O");
					if (data1 & (1<<2)) printf(" ");		else printf("O");
					if (data1 & (1<<1)) printf(" ");		else printf("O");
					break;
				case 14:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					if (data1 & (1<<6)) printf(" ");		else printf("O");
					if (data1 & (1<<5)) printf(" ");		else printf("O");
					if (data1 & (1<<4)) printf(" ");		else printf("O");
					if (data1 & (1<<3)) printf(" ");		else printf("O");
					if (data1 & (1<<2)) printf(" ");		else printf("O");
					break;
				case 13:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					if (data1 & (1<<6)) printf(" ");		else printf("O");
					if (data1 & (1<<5)) printf(" ");		else printf("O");
					if (data1 & (1<<4)) printf(" ");		else printf("O");
					if (data1 & (1<<3)) printf(" ");		else printf("O");
					break;
				case 12:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					if (data1 & (1<<6)) printf(" ");		else printf("O");
					if (data1 & (1<<5)) printf(" ");		else printf("O");
					if (data1 & (1<<4)) printf(" ");		else printf("O");
					break;
				case 11:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					if (data1 & (1<<6)) printf(" ");		else printf("O");
					if (data1 & (1<<5)) printf(" ");		else printf("O");
					break;
				case 10:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					if (data1 & (1<<6)) printf(" ");		else printf("O");
					break;
				case 9:
					if (data1 & (1<<7)) printf(" ");		else printf("O");
					break;
				default:
					break;
				}
				printf(" */");
				printf("\n");
			}
		}
	}

	printf("};\n\n");

}


int
main(int argc, char* argv[])
{
	FILE* fp;
	unsigned char firstbyte;

	if (2 != argc) {
		fprintf(stderr, "Usage:  psf2c fontfile\n");
		return -1;
	}

	// open file
	fp = fopen(argv[1], "rb");
	if (!fp) {
		fprintf(stderr, "Error: cannot open file %s\n", argv[1]);
		return -1;
	}

	if (1 != fread(&firstbyte, 1, 1, fp)) {
		fprintf(stderr,
		    "Error: cannot read first byte from  %s\n", argv[1]);
		return -1;
	}

	fseek(fp, 0L, SEEK_SET);

	switch(firstbyte) {
		case PSF1_MAGIC0:
			dumpPSF1(fp);
			break;
		case PSF2_MAGIC0:
			dumpPSF2(fp);
			break;
		default:
			fprintf(stderr,
			    "Error: is %s PSF1/PSF2 font file?\n", argv[1]);
			return -1;
	}

	fclose(fp);

	return 0;
}

