//  Copyright (C) 2004 PocketDOS
//
//    PocketDOS
//    http://www.pocketdos.com/
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA


#ifndef _PDOS_CPU_EXT_H
#define _PDOS_CPU_EXT_H

// Specify structure packing
#pragma pack(4)   

// PocketDOS x86 register definition.
typedef struct 
{
	union 
	{
		DWORD dword;
		struct 
		{
			union 
			{
				WORD word;
				struct 
				{
					BYTE lo;
					BYTE hi;
				};
			};
			WORD nReserved;
		};
	};
} tPDOS_CPU_REG;

// Registers passed to and from PocketDOS BIOS functions.
typedef struct
{
	// 80x86 registers
	tPDOS_CPU_REG 
		nAX,
		nCX,
		nDX,
		nBX,
		nSP,
		nBP,
		nSI,
		nDI,
		nIP;
	
	// Pointers to CPU segment registers
	WORD 
		*pES,
		*pCS,
		*pSS,
		*pDS;

	// Pointers to the start of the CPU segment register segments
	BYTE 
		*pCSSegment,
		*pDSSegment,
		*pESSegment,
		*pSSSegment,
		*pStackSegment;

	// Condition flags
	unsigned
		nCF,
		nPF,
		nZF,
		nTF,
		nIF,
		nDF, 
		nAF, 
		nOF, 
		nSF;
} tPDOS_CPU_REGS;

// Memory read/write callbacks
typedef struct
{
	BYTE
		// Read a byte from memory (may be NULL)
		(*pfnReadMemory)( BYTE *pAddress );
	void
		// Write a byte to memory
		(*pfnWriteMemory)( BYTE *pAddress, BYTE nValue );
} tMemoryHandler;

typedef void (*tInterruptNotify)( BOOL );

// Interrupt handler callbacks
typedef struct
{
	void
		// Register interrupt pending callback
		(*pfnRegisterIntNotification)( tInterruptNotify pfnInterruptNotify );
	BOOL
		// Get interrupt vector
		(*pfnGetInterrupt)( BYTE *pInterrupt );
	void
		// Register interrupt pending callback
		(*pfnRegisterHardwareIntNotification)( tInterruptNotify pfnInterruptNotify );
	void
		// Process hardware interrupt
		(*pfnHardwareInterrupt)();
} tIntHandler;

// Port handler callbacks
typedef struct 
{
	BYTE
		// Read from a port
		(*pfnReadPort)( WORD nPort );
	void
		// Write to a port
		(*pfnWritePort)( WORD nPort, BYTE chValue );

} tIOHandler;

// Refresh handler callbacks
typedef struct
{
	DWORD
		// Get system tick count (in ms)
		(*pfnGetSystemTicks)();
	void
		// Refresh hardware emulation
		(*pfnRefreshHardware)( DWORD dwTicks );
} tRefreshHandler;

// BIOS interrupt handler callbacks
typedef struct
{
	void
		// Handle BIOS interrupt
		(*pfnBIOSInterrupt)( tPDOS_CPU_REGS *pREG, BYTE nInterrupt );
} tBIOSHandler;

// Revert to default structure packing
#pragma pack()

// PocketDOS CPU plug in
class PDOS_CPU_EXT
{
public:
	// Constructor/destructor
	PDOS_CPU_EXT(){};
	~PDOS_CPU_EXT(){};

	// Initialise CPU plug in
	virtual BOOL Initialise( BYTE *pMemory, DWORD dwMemorySize, tMemoryHandler *pHighMemoryHandler, tIntHandler *pIntHandler, tIOHandler *pPortHandler, tRefreshHandler *pRefreshHandler, tBIOSHandler *pBIOSHandler )=0;

	// Get/set CPU registers
	virtual void GetRegisters( tPDOS_CPU_REGS *pREG )=0;
	virtual void SetRegisters( tPDOS_CPU_REGS *pREG )=0;

	// Get set hardware refresh rate (in ms)
	inline virtual DWORD GetRefreshRate()=0;
	inline virtual void SetRefreshRate( DWORD dwTicks )=0;

	// Run CPU emulation
	virtual void Run()=0;

	// Stop CPU emulation
	virtual void Shutdown()=0;

	// Execute FPU instruction
	virtual BOOL ExecuteFPU( tPDOS_CPU_REGS *pREG, BYTE nInstruction, BYTE nModRM, WORD nDataSegment, WORD nDataOffset )
	{
		return FALSE;
	}
};

#endif