//  Copyright (C) 2004 PocketDOS
//
//    PocketDOS
//    http://www.pocketdos.com/
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA


//
// This is the glue logic needed to connect the DOSBox
// CPU emulator written by The DOSBox Team to PocketDOS.
//

#include "stdafx.h"
#include "tchar.h"

#ifndef _WIN32_WCE
#include <stdio.h>
#endif

#include "PDOS_DOSBOX_CPU.h"

#include "cpu.h"
#include "cpu/lazyflags.h"

extern CPU_Regs cpu_regs;

void PAGING_Alloc();
bool PAGING_Init();
void PAGING_ShutDown();
bool MEM_Init( Bit8u *pMemory, Bit32u dwMemorySize );
void MEM_ShutDown();
void CPU_Init( int );

// DOSBOX CPU emulator interface
tDOSBOX_Interface theCPU;

// Adapter memory page handler
AdapterMemHandler thePageHandler;

// DOSBOX interrupt pending flag
unsigned int PIC_IRQCheck = 0;

// Dll entry point
BOOL APIENTRY DllMain( HANDLE hModule, DWORD  ul_reason_for_call, LPVOID lpReserved )
{
BOOL bOK = TRUE;

	switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
		{
		HKEY hKey;

			// Initialise CPU interface
			memset( &theCPU, 0, sizeof( theCPU ) );

			// Open registry key
			if (RegOpenKeyEx( HKEY_CURRENT_USER, 
				TEXT( "Software\\DOSBox\\PDOS_DOSBOX_CPU" ),
				0, 0, &hKey ) == ERROR_SUCCESS)
			{
			TCHAR szLevel[256];
			DWORD dwType, dwBytes = sizeof( szLevel ) - sizeof( TCHAR );

				// Get core level
				if (::RegQueryValueEx( hKey, TEXT( "Core" ), NULL, &dwType, 
					(BYTE *) szLevel, &dwBytes ) == ERROR_SUCCESS && dwType == REG_SZ)
				{
					if (_tcsicmp( szLevel, TEXT( "NORMAL" ) ) == 0) 
						theCPU.nCoreLevel = 1;
					else if (_tcsicmp( szLevel, TEXT( "FULL" ) ) == 0) 
						theCPU.nCoreLevel = 2;
					else theCPU.nCoreLevel = 0;
				}

				RegCloseKey( hKey );
			}
		}

		// Allocate paging memory
		PAGING_Alloc();
		bOK = (paging != NULL);
		break;
	case DLL_PROCESS_DETACH:
		// Shut down memory
		MEM_ShutDown();
		
		// Shut down paging
		PAGING_ShutDown();
		break;
	}
	return bOK;
}

// Initialise CPU
bool CPU86_Init( tCPU86_Interface *pInterface )
{
	// Initialise CPU interface
	theCPU.pInterface = pInterface;

	// Initialise paging & memory objects
	if (PAGING_Init() && MEM_Init( (Bit8u *) pInterface->sReg.pMemory, 
		pInterface->sReg.dwMemorySize ))
	{
		// Set up page handlers for entire range of adapter memory
		MEM_SetPageHandler( 0xa0, (0xff-0xa0), &thePageHandler );

		// Set up A20 address mask
		theCPU.bAddressMaskEnabled = false;

		// Initialise CPU object
		PIC_IRQCheck = 0;
		CPU_Init( theCPU.nCoreLevel );
		theCPU.bAsyncEvent = false;

		return true;
	}

	return false;
}

// Flag that interrupt is pending
void DOSBOX_SetIntPending( bool bIntPending )
{
	if (bIntPending)
	{
		// Set CPU interrupt request flag
		PIC_IRQCheck = 1;
		theCPU.bAsyncEvent = true;
	}
	else PIC_IRQCheck = 0;
}

// Get interrupt vector
Bit8u DOSBOX_GetIntVector() 
{
	// Acknowledge CPU interrupt request
	PIC_IRQCheck = 0;

	// Get interrupt vector
	return (*theCPU.pInterface->sInt.pfnGetIntVector)();
}

// Flag that system request is pending
void DOSBOX_SetSysReqPending( bool bSysReqPending )
{
	if ((theCPU.bSysReqPending = bSysReqPending))
		theCPU.bAsyncEvent = true;
}

// Get A20 line status
bool DOSBOX_GetA20Status()
{
	return theCPU.bAddressMaskEnabled;
}

// Set A20 line
void DOSBOX_SetA20Line( bool bEnable )
{
bool bA20Enabled = theCPU.bAddressMaskEnabled;

	// Set up A20 address mask
	theCPU.bAddressMaskEnabled = bEnable;

	// A20 address line changed?
	if (bA20Enabled != theCPU.bAddressMaskEnabled)
	{
		// Notify CPU that A20 address line has changed
		MEM_A20_Enable( bEnable );
	}
}

// Get/set CPU registers
void DOSBOX_GetRegisters( tCPU86_RegisterFile *pReg )
{
	// Get general registers
	pReg->nAX.dword = reg_eax;
	pReg->nCX.dword = reg_ecx;
	pReg->nDX.dword = reg_edx;
	pReg->nBX.dword = reg_ebx;
	pReg->nSP.dword = reg_esp;
	pReg->nBP.dword = reg_ebp;
	pReg->nSI.dword = reg_esi;
	pReg->nDI.dword = reg_edi;

	// Get instruction pointer
	pReg->nIP.dword = reg_eip;
	
	// Get segment registers
	pReg->nES = Segs.val[es];
	pReg->nCS = Segs.val[cs];
	pReg->nSS = Segs.val[ss];
	pReg->nDS = Segs.val[ds];
	pReg->nFS = Segs.val[fs];
	pReg->nGS = Segs.val[gs];

	// Get control word
	pReg->nCR0 = cpu.cr0;

	// Get status flags
	pReg->nEFlags = reg_flags;
}

void DOSBOX_SetRegisters( tCPU86_RegisterFile *pReg )
{
	// Set general registers
	reg_eax = pReg->nAX.dword;
	reg_ecx = pReg->nCX.dword;
	reg_edx = pReg->nDX.dword;
	reg_ebx = pReg->nBX.dword;
	reg_esp = pReg->nSP.dword;
	reg_ebp = pReg->nBP.dword;
	reg_esi = pReg->nSI.dword;
	reg_edi = pReg->nDI.dword;

	// Set instruction pointer
	reg_eip = pReg->nIP.dword;
	
	// Set segment registers
	if (Segs.val[es] != pReg->nES)
		CPU_SetSegGeneral( es, pReg->nES );
	if (Segs.val[cs] != pReg->nCS)
		CPU_SetSegGeneral( cs, pReg->nCS );
	if (Segs.val[ss] != pReg->nSS)
		CPU_SetSegGeneral( ss, pReg->nSS );
	if (Segs.val[ds] != pReg->nDS)
		CPU_SetSegGeneral( ds, pReg->nDS );
	if (Segs.val[fs] != pReg->nFS)
		CPU_SetSegGeneral( fs, pReg->nFS );
	if (Segs.val[gs] != pReg->nGS)
		CPU_SetSegGeneral( gs, pReg->nGS );

	// Set status flags
	CPU_SetFlags( pReg->nEFlags, 0x7FFF );
}

void CPU86_Run( tCPU86_RegisterFile *pReg )
{
	theCPU.bActive = true;

	// Set notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)
			( DOSBOX_SetIntPending );
	if (theCPU.pInterface->sSys.pfnSetSysReqNotify)
		(*theCPU.pInterface->sSys.pfnSetSysReqNotify)
			( DOSBOX_SetSysReqPending );

	// A20 handler required?
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
	{
	tCPU86_A20Handler sA20;

		// Set A20 handler
		sA20.pfnGetA20 = DOSBOX_GetA20Status;
		sA20.pfnSetA20 = DOSBOX_SetA20Line;
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( &sA20 );
	}

	// Initialise CPU registers
	DOSBOX_SetRegisters( pReg );

	// Execute CPU loop
	DOSBOX_RunMachine();

	// Clear notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)( NULL );
	if (theCPU.pInterface->sSys.pfnSetSysReqNotify)
		(*theCPU.pInterface->sSys.pfnSetSysReqNotify)( NULL );

	// Clear A20 handler
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( NULL );

	// Retrieve CPU registers
	DOSBOX_GetRegisters( pReg );

	theCPU.bActive = false;
}

void CPU86_Execute( tCPU86_RegisterFile *pReg )
{
	theCPU.bActive = true;

	// Set notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)
			( DOSBOX_SetIntPending );

	// A20 handler required?
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
	{
	tCPU86_A20Handler sA20;

		// Set A20 handler
		sA20.pfnGetA20 = DOSBOX_GetA20Status;
		sA20.pfnSetA20 = DOSBOX_SetA20Line;
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( &sA20 );
	}

	// Initialise CPU registers
	DOSBOX_SetRegisters( pReg );

	// Execute CPU instruction
	CPU_Cycles = 1;
	(*cpudecoder)();

	// Clear notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)( NULL );

	// Clear A20 handler
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( NULL );

	// Retrieve CPU registers
	DOSBOX_GetRegisters( pReg );

	theCPU.bActive = false;
}

void CPU86_Shutdown()
{
	// Quit CPU loop
	theCPU.bActive = false;
}

// Call system (illegal opcode 0xf1)
void DOSBOX_CallSystem()
{
	// Retrieve CPU registers
	DOSBOX_GetRegisters( &theCPU.pInterface->sReg );

	// Call system
	(*theCPU.pInterface->sSys.pfnCallSystem)( &theCPU.pInterface->sReg );

	// Update CPU registers
	DOSBOX_SetRegisters( &theCPU.pInterface->sReg );
}

Bitu Normal_Loop()
{
DWORD dwCurrentTicks, dwRefreshTicks, dwLastRefreshTicks, dwSystemTicks, dwDelta;

	// Initialise refresh counters
	dwCurrentTicks = 0;
	dwRefreshTicks = 10000;
	dwLastRefreshTicks = 0;

#if !defined( _WIN32_WCE ) || _MSC_VER >= 1400
	try
#endif
	{
		// Run CPU
		while (theCPU.bActive) 
		{
			// Async event pending?
			if (theCPU.bAsyncEvent)
			{
				theCPU.bAsyncEvent = false;

				// Interrupt pending?
				if (PIC_IRQCheck)
				{
				Bit8u nInterrupt = DOSBOX_GetIntVector();

					// Process interrupt
					if (nInterrupt != 0xff) CPU_HW_Interrupt( nInterrupt );
				}

				// System request pending?
				if (theCPU.bSysReqPending)
				{
					// Call hardware interrupt handler
					(*theCPU.pInterface->sSys.pfnSysReq)();
				}
			}

			// Set number of CPU cycles to run
			CPU_Cycles = 0x0f;

			// Run CPU
			if ((*cpudecoder)() < 0) return 1;

			// Refresh?
			if ((dwCurrentTicks += (0x0f - CPU_Cycles)) >= dwRefreshTicks)
			{
				// Get millisecond count
				dwSystemTicks = (*theCPU.pInterface->sSys.pfnGetMilliseconds)();

				// Calculate milliseconds since last refresh
				dwDelta = dwSystemTicks - dwLastRefreshTicks;
				dwLastRefreshTicks = dwSystemTicks;

				// Increase or decrease refresh rate
				if (dwDelta <= theCPU.pInterface->sSys.dwRefreshRate)
					dwRefreshTicks += 250;
				else if (dwRefreshTicks > 250) 
					dwRefreshTicks -= 250;

				// Call refresh handler
				(*theCPU.pInterface->sSys.pfnRefresh)( dwSystemTicks );

				dwCurrentTicks = 0;
			}
		}

		return 1;
	}
#if !defined( _WIN32_WCE ) || _MSC_VER >= 1400
	catch (char*)
	{
	}
#endif

	return -1;
}

void DOSBOX_RunMachine(void)
{
	Bitu ret;
	do {
		// Run CPU loop
		ret=Normal_Loop();
	} while (!ret);
}

// I/O port handlers
Bitu IO_ReadD(Bitu nPort)
{
	// No handler?
	if (!theCPU.pInterface->sIO.pfnReadIODWord)
	{
		// Perform read using byte handler
		return (*theCPU.pInterface->sIO.pfnReadIOByte)( nPort )
			| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+1 ) << 8)
			| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+2 ) << 16)
			| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+3 ) << 24);
	}
	else return (*theCPU.pInterface->sIO.pfnReadIODWord)( nPort ); 
}

Bitu IO_ReadW(Bitu nPort)
{
	// No handler?
	if (!theCPU.pInterface->sIO.pfnReadIOWord)
	{
		// Perform read using byte handler
		return (*theCPU.pInterface->sIO.pfnReadIOByte)( nPort )
			| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+1 ) << 8);
	}
	else return (*theCPU.pInterface->sIO.pfnReadIOWord)( nPort ); 
}

Bitu IO_ReadB(Bitu nPort)
{
	return (Bit8u) (*theCPU.pInterface->sIO.pfnReadIOByte)( nPort );
}

void IO_WriteD(Bitu nPort,Bitu nValue)
{
	// No handler?
	if (!theCPU.pInterface->sIO.pfnWriteIODWord)
	{
		// Perform write using byte handler
		(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort, (BYTE) nValue ); 
		(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+1, (BYTE) (nValue >> 8) ); 
		(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+2, (BYTE) (nValue >> 16) ); 
		(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+3, (BYTE) (nValue >> 24) ); 
	}
	else (*theCPU.pInterface->sIO.pfnWriteIODWord)( nPort, (DWORD) nValue ); 
}

void IO_WriteW(Bitu nPort,Bitu nValue)
{
	// No handler?
	if (!theCPU.pInterface->sIO.pfnWriteIOWord) 
	{
		// Perform write using byte handler
		(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort, (BYTE) nValue ); 
		(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+1, (BYTE) (nValue >> 8) ); 
	}
	else (*theCPU.pInterface->sIO.pfnWriteIOWord)( nPort, (WORD) nValue );
}

void IO_WriteB(Bitu nPort,Bitu nValue)
{
	(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort, nValue );
}

static char buf[1024];
void E_Exit(char const * format,...) 
{
#if !defined( _WIN32_WCE ) || _MSC_VER >= 1400
	va_list msg;
	va_start(msg,format);
	vsprintf(buf,format,msg);
	va_end(msg);
	strcat(buf,"\n");

#if !defined( _WIN32_WCE ) && defined( _DEBUG )
	::MessageBoxA( NULL, buf, "DOSBox", MB_ICONEXCLAMATION );
#endif

	throw(buf);
#else
	CPU86_Shutdown();
#endif
}

