#ifndef _CPU86_INTERFACE_H
#define _CPU86_INTERFACE_H

typedef unsigned __int64 QWORD;

// Specify structure packing
#pragma pack(4)   

// x86 CPU general register (little endian)
typedef struct 
{
	union 
	{
		DWORD dword;
		struct 
		{
			union 
			{
				WORD word;
				struct 
				{
					BYTE lo;
					BYTE hi;
				};
			};
			WORD nReserved;
		};
	};
} tCPU86_GeneralRegister;

// x86 CPU global segment register
typedef struct
{
	DWORD nBase;
	WORD nLimit;	
} tCPU86_GlobalSegRegister;

// x86 CPU register set
typedef struct
{
	// Pointer to start of memory (read-only)
	BYTE *pMemory;

	// Size of memory (in bytes) (read-only)
	DWORD dwMemorySize;

	// General registers
	tCPU86_GeneralRegister
		nAX,
		nCX,
		nDX,
		nBX,
		nSP,
		nBP,
		nSI,
		nDI;

	// Instruction pointer
	tCPU86_GeneralRegister nIP;

	// User segment registers
	WORD
		nES,
		nCS,
		nSS,
		nDS,
		nFS,
		nGS;

	// System segment registers (read-only)
	tCPU86_GlobalSegRegister
		sGDTR,	// Global Descriptor Table register
		sIDTR;	// Interrupt Descriptor Table register

	WORD
		nLDTR,	// Local Descriptor Table register
		nTR;	// Task register

	// Control registers (read-only)
	DWORD
		nCR0,
		nCR1,
		nCR2,
		nCR3,
		nCR4;

	// Status and control flags (upper word read-only)
	DWORD nEFlags;
} tCPU86_RegisterFile;

// x86 CPU A20 line access handler
typedef struct
{
	// Get status of A20 line
	bool (*pfnGetA20)();
	// Set A20 line
	void (*pfnSetA20)( bool );
} tCPU86_A20Handler;

// x86 CPU memory access handler 
// NOTE: used to access adapter RAM/ROM from A0000-FFFFF and A20 address line
typedef struct 
{
	// Read/write byte from/to memory
	BYTE (*pfnReadByte)( DWORD dwAddress );
	void (*pfnWriteByte)( DWORD dwAddress, BYTE nValue );

	// Read/write word from/to memory
	WORD (*pfnReadWord)( DWORD dwAddress );
	void (*pfnWriteWord)( DWORD dwAddress, WORD nValue );

	// Read/write long from/to memory
	DWORD (*pfnReadDWord)( DWORD dwAddress );
	void (*pfnWriteDWord)( DWORD dwAddress, DWORD nValue );

	// Fill memory with byte/word/long
	void (*pfnByteFill)( DWORD dwAddress, DWORD nBytes, BYTE nValue );
	void (*pfnWordFill)( DWORD dwAddress, DWORD nWords, WORD nValue );
	void (*pfnDWordFill)( DWORD dwAddress, DWORD nDWords, DWORD nValue );

	//  Copy bytes to memory
	void (*pfnCopyBytes)( DWORD dwAddress, DWORD dwSource, int nBytes );

	// Set A20 handler functions
	void (*pfnSetA20Handler)( tCPU86_A20Handler *pHandler );
} tCPU86_MemHandler;

// x86 CPU I/O port access handler
typedef struct
{
	// Read/write byte from/to port
	BYTE (*pfnReadIOByte)( WORD nPort );
	void (*pfnWriteIOByte)( WORD nPort, BYTE nValue );

	// Read/write word from/to port
	WORD (*pfnReadIOWord)( WORD nPort );
	void (*pfnWriteIOWord)( WORD nPort, WORD nValue );

	// Read/write long from/to port
	DWORD (*pfnReadIODWord)( WORD nPort );
	void (*pfnWriteIODWord)( WORD nPort, DWORD nValue );
} tCPU86_IOHandler;

// x86 CPU interrupt handler
typedef struct
{
	// Set interrupt pending notification function
	void (*pfnSetIntNotify)( void (*)( bool ) );

	// Get interrupt vector for pending interrupt 
	// NOTE: call from CPU emulator after IntNotify( true )
	//       will return -1 if no interrupt pending
	BYTE (*pfnGetIntVector)();
} tCPU86_IntHandler;

// x86 CPU system interface handler
typedef struct
{
	// Refresh rate (in milliseconds)
	DWORD dwRefreshRate;

	// Get timestamp in milliseconds
	DWORD (*pfnGetMilliseconds)();

	// Get timestamp in microseconds
	QWORD (*pfnGetMicroseconds)();

	// Refresh system (call from CPU emulator only)
	void (*pfnRefresh)( DWORD nMilliseconds );

	// Set system request pending notification function
	void (*pfnSetSysReqNotify)( void (*)( bool ) );

	// Process pending system request 
	// NOTE: call from CPU emulator after SysReqNotify( true )
	void (*pfnSysReq)();

	// Call system function
	// NOTE: call from CPU emulator when it hits illegal 0xf1 instruction
	void (*pfnCallSystem)( tCPU86_RegisterFile *pReg );
} tCPU86_SysHandler;

// x86 CPU emulator interface
typedef struct
{
	// x86 CPU registers
	tCPU86_RegisterFile sReg;

	// Memory handler
	tCPU86_MemHandler sMem;

	// I/O port handler
	tCPU86_IOHandler sIO;

	// Interrupt handler
	tCPU86_IntHandler sInt;

	// System interface handler
	tCPU86_SysHandler sSys;
} tCPU86_Interface;

// x87 FPU emulator interface
typedef struct
{
	// x86 CPU registers
	tCPU86_RegisterFile sReg;

	// Memory handler
	tCPU86_MemHandler sMem;

	// Request hardware interrupt
	void (*pfnRequestInterrupt)( BYTE nInterrupt );

	// FPU instruction and ModR/M byte
	BYTE
		nInstruction,
		nModRM;

	// Data segment and offset
	WORD nDataSegment;
	DWORD nDataOffset;

	// Operand/address size and overrides
	bool
		bIs32bit,
		bIsASOverride,
		bIsOSOverride;
} tFPU87_Interface;

// Revert to default structure packing
#pragma pack()

extern "C" bool CPU86_Init( tCPU86_Interface *pInterface );
extern "C" void CPU86_Run( tCPU86_RegisterFile *pReg );
extern "C" void CPU86_Execute( tCPU86_RegisterFile *pReg );
extern "C" void CPU86_Shutdown();

extern "C" bool FPU87_Init();
extern "C" void FPU87_Execute( tFPU87_Interface *pInterface );
 
#endif // _CPU86_INTERFACE_H