//  Copyright (C) 2004 PocketDOS
//
//    PocketDOS
//    http://www.pocketdos.com/
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA


//
// This is the glue logic needed to connect the bochs
// CPU emulator written by Kevin Lawton and the i387 emulator written 
// by William Metzenthen to PocketDOS.
//

#include "stdafx.h"
#include "PDOS_BOCHS_CPU.h"

// i387 declarations
extern "C" 
{
#include "fpu/fpu_emu.h"
#include "linux/signal.h"

void
math_emulate(fpu_addr_modes addr_modes,
              u_char  FPU_modrm,
              u_char byte1,
              bx_address data_address,
              struct address data_sel_off,
              struct address entry_sel_off);
}

#ifndef PDOS_BOCHS_FPU_ONLY
extern bxInstruction_c *fpu_iptr;
#else
i387_t *current_i387;
#endif

// CPU interface
tBOCHS_Interface theCPU;

#ifdef PDOS_BOCHS_FPU_ONLY
// FPU object
i387_t theFPU;
#endif

// Dll entry point
BOOL APIENTRY DllMain( HANDLE hModule, DWORD  ul_reason_for_call, LPVOID lpReserved )
{
	switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
		// Initialise CPU interface
		memset( &theCPU, 0, sizeof( theCPU ) );
		break;
	}

	return true;
}

#ifndef PDOS_BOCHS_FPU_ONLY

bool CPU86_Init( tCPU86_Interface *pInterface )
{
	// Initialise CPU interface
	theCPU.pInterface = pInterface;

	// Set up A20 address mask
	theCPU.bAddressMaskEnabled = false;
	theCPU.dwAddressMask = 0xffefffff;

	// Initialise memory object
	BX_MEM(0)->init_memory( (Bit8u *) pInterface->sReg.pMemory, 
		pInterface->sReg.dwMemorySize );

	// Initialise CPU object
	BX_CPU(0)->init( BX_MEM(0) );
	BX_CPU(0)->set_cpu_id( 0 );

	// Reset CPU
	BX_CPU(0)->reset( BX_RESET_HARDWARE );

	// Set global FPU pointer
	current_i387 = &(BX_CPU(0)->the_i387);

	// Initialise refresh counters
	theCPU.dwCurrentTicks = 0;
	theCPU.dwRefreshTicks = 10000;
	theCPU.dwLastRefreshTicks = 0;

	return true;
}

// Flag that interrupt is pending
void BOCHS_SetIntPending( bool bIntPending )
{
	// Set CPU interrupt request flag
	BX_CPU(0)->set_INTR( bIntPending );
}

// Get interrupt vector
Bit8u BOCHS_GetIntVector() 
{
	// Acknowledge CPU interrupt request
	BX_CPU(0)->set_INTR( false );

	// Get interrupt vector
	return (*theCPU.pInterface->sInt.pfnGetIntVector)();
}

// Flag that system request is pending
void BOCHS_SetSysReqPending( bool bSysReqPending )
{
	if ((theCPU.bSysReqPending = bSysReqPending))
		BX_CPU(0)->async_event = 1;
}

// Get A20 line status
bool BOCHS_GetA20Status()
{
	return theCPU.bAddressMaskEnabled;
}

// Set A20 line
void BOCHS_SetA20Line( bool bEnable )
{
bool bA20Enabled = theCPU.bAddressMaskEnabled;

	// Enable A20 address line?
	if (bEnable)
	{
		// Set up A20 address mask
		theCPU.bAddressMaskEnabled = true;
		theCPU.dwAddressMask = 0xffffffff;
	}
	else
	{
		// Set up A20 address mask
		theCPU.bAddressMaskEnabled = false;
		theCPU.dwAddressMask = 0xffefffff;
	}

	// A20 address line changed?
	if (bA20Enabled != theCPU.bAddressMaskEnabled)
	{
		// Notify CPU that A20 address line has changed
		BX_CPU(0)->pagingA20Changed();
	}
}

// Read/write memory
Bit8u BOCHS_ReadAdapterByte( Bit32u dwAddress )
{
	// Read memory
	return (*theCPU.pInterface->sMem.pfnReadByte)( dwAddress );
}

void BOCHS_WriteAdapterByte( Bit32u dwAddress, Bit8u nValue )
{
	// Write to memory
	(*theCPU.pInterface->sMem.pfnWriteByte)( dwAddress, nValue );
}

// Read/write IO ports
Bit32u BOCHS_ReadPort( Bit16u nPort, unsigned nLen )
{
	// Check length
	switch (nLen)
	{
	// Read port
	case 1:	
		return (*theCPU.pInterface->sIO.pfnReadIOByte)( nPort ); 
		break;
	case 2:	
		// No handler?
		if (!theCPU.pInterface->sIO.pfnReadIOWord)
		{
			// Perform read using byte handler
			return (*theCPU.pInterface->sIO.pfnReadIOByte)( nPort )
				| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+1 ) << 8);
		}
		else return (*theCPU.pInterface->sIO.pfnReadIOWord)( nPort ); 
		break;
	case 4:	
		// No handler?
		if (!theCPU.pInterface->sIO.pfnReadIODWord)
		{
			// Perform read using byte handler
			return (*theCPU.pInterface->sIO.pfnReadIOByte)( nPort )
				| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+1 ) << 8)
				| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+2 ) << 16)
				| ((*theCPU.pInterface->sIO.pfnReadIOByte)( nPort+3 ) << 24);
		}
		else return (*theCPU.pInterface->sIO.pfnReadIODWord)( nPort ); 
		break;
	default: 
		return -1L; 
		break;
	}
}

void BOCHS_WritePort( Bit16u nPort, Bit32u nValue, unsigned nLen )
{
	// Check length
	switch (nLen)
	{
	// Write to port
	case 1:	
		(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort, (BYTE) nValue ); 
		break;
	case 2:	
		// No handler?
		if (!theCPU.pInterface->sIO.pfnWriteIOWord) 
		{
			// Perform write using byte handler
			(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort, (BYTE) nValue ); 
			(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+1, (BYTE) (nValue >> 8) ); 
		}
		else (*theCPU.pInterface->sIO.pfnWriteIOWord)( nPort, (WORD) nValue );
		break;
	case 4:	
		// No handler?
		if (!theCPU.pInterface->sIO.pfnWriteIODWord)
		{
			// Perform write using byte handler
			(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort, (BYTE) nValue ); 
			(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+1, (BYTE) (nValue >> 8) ); 
			(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+2, (BYTE) (nValue >> 16) ); 
			(*theCPU.pInterface->sIO.pfnWriteIOByte)( nPort+3, (BYTE) (nValue >> 24) ); 
		}
		else (*theCPU.pInterface->sIO.pfnWriteIODWord)( nPort, (DWORD) nValue ); 
		break;
	}
}

// Get/set CPU registers
void BOCHS_GetRegisters( tCPU86_RegisterFile *pReg )
{
	// Get general registers
	pReg->nAX.dword = BX_CPU(0)->get_EAX();
	pReg->nCX.dword = BX_CPU(0)->get_ECX();
	pReg->nDX.dword = BX_CPU(0)->get_EDX();
	pReg->nBX.dword = BX_CPU(0)->get_EBX();
	pReg->nSP.dword = BX_CPU(0)->get_ESP();
	pReg->nBP.dword = BX_CPU(0)->get_EBP();
	pReg->nSI.dword = BX_CPU(0)->get_ESI();
	pReg->nDI.dword = BX_CPU(0)->get_EDI();

	// Get instruction pointer
	pReg->nIP.dword = BX_CPU(0)->get_EIP();
	
	// Get segment Registers
	pReg->nES = BX_CPU(0)->sregs[BX_SEG_REG_ES].selector.value;
	pReg->nCS = BX_CPU(0)->sregs[BX_SEG_REG_CS].selector.value;
	pReg->nSS = BX_CPU(0)->sregs[BX_SEG_REG_SS].selector.value;
	pReg->nDS = BX_CPU(0)->sregs[BX_SEG_REG_DS].selector.value;
	pReg->nFS = BX_CPU(0)->sregs[BX_SEG_REG_FS].selector.value;
	pReg->nGS = BX_CPU(0)->sregs[BX_SEG_REG_GS].selector.value;

	// Get control word
	pReg->nCR0 = BX_CPU(0)->cr0.val32;

	// Get status flags
	pReg->nEFlags = BX_CPU(0)->read_eflags();
}

void BOCHS_SetRegisters( tCPU86_RegisterFile *pReg )
{
	// Set general registers
	BX_CPU(0)->set_EAX( pReg->nAX.dword );
	BX_CPU(0)->set_ECX( pReg->nCX.dword );
	BX_CPU(0)->set_EDX( pReg->nDX.dword );
	BX_CPU(0)->set_EBX( pReg->nBX.dword );
	BX_CPU(0)->set_ESP( pReg->nSP.dword );
	BX_CPU(0)->prev_esp = pReg->nSP.dword;
	BX_CPU(0)->set_EBP( pReg->nBP.dword );
	BX_CPU(0)->set_ESI( pReg->nSI.dword );
	BX_CPU(0)->set_EDI( pReg->nDI.dword );

	// Set instruction pointer
	BX_CPU(0)->dword.eip = pReg->nIP.dword;
	BX_CPU(0)->prev_eip = pReg->nIP.dword;
	
	// Set segment registers
	if (BX_CPU(0)->sregs[BX_SEG_REG_ES].selector.value != pReg->nES)
		BX_CPU(0)->load_seg_reg( &BX_CPU(0)->sregs[BX_SEG_REG_ES], pReg->nES );
	if (BX_CPU(0)->sregs[BX_SEG_REG_CS].selector.value != pReg->nCS)
		BX_CPU(0)->load_seg_reg( &BX_CPU(0)->sregs[BX_SEG_REG_CS], pReg->nCS );
	if (BX_CPU(0)->sregs[BX_SEG_REG_SS].selector.value != pReg->nSS)
		BX_CPU(0)->load_seg_reg( &BX_CPU(0)->sregs[BX_SEG_REG_SS], pReg->nSS );
	if (BX_CPU(0)->sregs[BX_SEG_REG_DS].selector.value != pReg->nDS)
		BX_CPU(0)->load_seg_reg( &BX_CPU(0)->sregs[BX_SEG_REG_DS], pReg->nDS );
	if (BX_CPU(0)->sregs[BX_SEG_REG_FS].selector.value != pReg->nFS)
		BX_CPU(0)->load_seg_reg( &BX_CPU(0)->sregs[BX_SEG_REG_FS], pReg->nFS );
	if (BX_CPU(0)->sregs[BX_SEG_REG_GS].selector.value != pReg->nGS)
		BX_CPU(0)->load_seg_reg( &BX_CPU(0)->sregs[BX_SEG_REG_GS], pReg->nGS );

	// Set status flags
	BX_CPU(0)->writeEFlags( pReg->nEFlags, 0x7FFF );
}

void CPU86_Run( tCPU86_RegisterFile *pReg )
{
	theCPU.bActive = true;

	// Set notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)
			( BOCHS_SetIntPending );
	if (theCPU.pInterface->sSys.pfnSetSysReqNotify)
		(*theCPU.pInterface->sSys.pfnSetSysReqNotify)
			( BOCHS_SetSysReqPending );

	// A20 handler required?
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
	{
	tCPU86_A20Handler sA20;

		// Set A20 handler
		sA20.pfnGetA20 = BOCHS_GetA20Status;
		sA20.pfnSetA20 = BOCHS_SetA20Line;
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( &sA20 );
	}

	// Initialise CPU registers
	BOCHS_SetRegisters( pReg );

	// Execute CPU loop
	BX_CPU(0)->kill_bochs_request = 0;
	BX_CPU(0)->cpu_loop( 1 );

	// Clear notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)( NULL );
	if (theCPU.pInterface->sSys.pfnSetSysReqNotify)
		(*theCPU.pInterface->sSys.pfnSetSysReqNotify)( NULL );

	// Clear A20 handler
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( NULL );

	// Retrieve CPU registers
	BOCHS_GetRegisters( pReg );

	theCPU.bActive = false;
}

void CPU86_Execute( tCPU86_RegisterFile *pReg )
{
	theCPU.bActive = true;

	// Set notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)
			( BOCHS_SetIntPending );

	// A20 handler required?
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
	{
	tCPU86_A20Handler sA20;

		// Set A20 handler
		sA20.pfnGetA20 = BOCHS_GetA20Status;
		sA20.pfnSetA20 = BOCHS_SetA20Line;
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( &sA20 );
	}

	// Initialise CPU registers
	BOCHS_SetRegisters( pReg );

	// Execute CPU instruction
	// TODO

	// Clear notification callbacks
	if (theCPU.pInterface->sInt.pfnSetIntNotify)
		(*theCPU.pInterface->sInt.pfnSetIntNotify)( NULL );

	// Clear A20 handler
	if (theCPU.pInterface->sMem.pfnSetA20Handler)
		(*theCPU.pInterface->sMem.pfnSetA20Handler)( NULL );

	// Retrieve CPU registers
	BOCHS_GetRegisters( pReg );

	theCPU.bActive = false;
}

void CPU86_Shutdown()
{
	// CPU loop running?
	if (theCPU.bActive)
	{
		// Quit CPU loop
		BX_CPU(0)->async_event = 1;
		BX_CPU(0)->kill_bochs_request = 1;
	}
}

// Call system (illegal opcode 0xf1)
void BX_INT1()
{
	// Retrieve CPU registers
	BOCHS_GetRegisters( &theCPU.pInterface->sReg );

	// Call system
	(*theCPU.pInterface->sSys.pfnCallSystem)( &theCPU.pInterface->sReg );

	// Update CPU registers
	BOCHS_SetRegisters( &theCPU.pInterface->sReg );
}

// Refresh
void BOCHS_Tick( Bit32u nTicks )
{
	// Refresh pending?
	if ((theCPU.dwCurrentTicks += nTicks) >= theCPU.dwRefreshTicks)
	{
	DWORD dwSystemTicks, dwDelta;

		// Get millisecond count
		dwSystemTicks = (*theCPU.pInterface->sSys.pfnGetMilliseconds)();

		// Calculate milliseconds since last refresh
		dwDelta = dwSystemTicks - theCPU.dwLastRefreshTicks;
		theCPU.dwLastRefreshTicks = dwSystemTicks;

		// Increase or decrease refresh rate
		if (dwDelta <= theCPU.pInterface->sSys.dwRefreshRate)
			theCPU.dwRefreshTicks += 250;
		else if (theCPU.dwRefreshTicks > 250) 
			theCPU.dwRefreshTicks -= 250;

		// Call refresh handler
		(*theCPU.pInterface->sSys.pfnRefresh)( dwSystemTicks );

		theCPU.dwCurrentTicks = 0;
	}
}

// Get system millisecond count
Bit32u BX_GETTICKS()
{
	return (*theCPU.pInterface->sSys.pfnGetMilliseconds)();
}

#endif

bool FPU87_Init()
{
	// Reset interface pointer
	theCPU.pFPUInterface = NULL;

#ifndef PDOS_BOCHS_FPU_ONLY
	// Set global FPU pointer
	current_i387 = &(BX_CPU(0)->the_i387);
#else
	// Set global FPU pointer
	current_i387 = &theFPU;
#endif

	// Initialise FPU
	finit();

	return true;
}

void FPU87_Execute( tFPU87_Interface *pInterface )
{
fpu_addr_modes sAddrMode;
struct address sInstruction, sData;

	// Initialise FPU interface
	theCPU.pFPUInterface = pInterface;

	// Protected mode?
	if ((pInterface->sReg.nCR0 & 0x01))
	{
		// Not VM86 mode?
		if (!(pInterface->sReg.nEFlags & (1<<17)))
		{
			// Check default operand size
			if (pInterface->bIs32bit) sAddrMode.default_mode = SEG32;
			else sAddrMode.default_mode = PM16;
		}
		// VM86 mode
		else sAddrMode.default_mode = VM86;
	}
	// Real mode, use VM86
	else sAddrMode.default_mode = VM86;

	// Address size override
	if (!pInterface->bIsASOverride) sAddrMode.override.address_size = 0;
	else sAddrMode.override.address_size = ADDR_SIZE_PREFIX;

	// Operand size override
	if (!pInterface->bIsOSOverride) sAddrMode.override.operand_size = 0;
	else sAddrMode.override.operand_size = OP_SIZE_PREFIX;

	// Save CS:IP
	sInstruction.selector = pInterface->sReg.nCS;
	sInstruction.offset = pInterface->sReg.nIP.dword;

	// Save data segment and offset
	sData.selector = pInterface->nDataSegment;
	sData.offset = pInterface->nDataOffset;

	// Emulate FPU instruction
	math_emulate( sAddrMode, pInterface->nModRM, pInterface->nInstruction, 
		pInterface->nDataOffset, sData, sInstruction );

	// Remove FPU interface
	theCPU.pFPUInterface = NULL;
}

unsigned fpu_get_ds()
{
#ifndef PDOS_BOCHS_FPU_ONLY
	if (!theCPU.pFPUInterface) return BX_CPU(0)->sregs[BX_SEG_REG_DS].selector.value;
	else 
#endif
	return theCPU.pFPUInterface->sReg.nDS;
}

void fpu_set_ax( Bit16u val16 )
{
#ifndef PDOS_BOCHS_FPU_ONLY
	if (!theCPU.pFPUInterface) BX_CPU(0)->set_AX( val16 );
	else 
#endif
	theCPU.pFPUInterface->sReg.nAX.word = val16;
}

Bit32u fpu_get_eflags()
{
#ifndef PDOS_BOCHS_FPU_ONLY
	if (!theCPU.pFPUInterface) return BX_CPU(0)->read_eflags();
	else
#endif
	return theCPU.pFPUInterface->sReg.nEFlags;
}

void BX_CPP_AttrRegparmN(3) fpu_verify_area( unsigned what, bx_address ptr, unsigned n )
{
#ifndef PDOS_BOCHS_FPU_ONLY
	if (!theCPU.pFPUInterface) 
	{
	bx_segment_reg_t *seg;

		seg = &BX_CPU(0)->sregs[fpu_iptr->seg()];

		if (what == VERIFY_READ) BX_CPU(0)->read_virtual_checks( seg, ptr, n );
		// VERIFY_WRITE
		else BX_CPU(0)->write_virtual_checks( seg, ptr, n );
	}
#endif
}

Bit32u BX_CPP_AttrRegparmN(2) fpu_get_user( bx_address ptr, unsigned len )
{
tFPU87_Interface *pInterface = theCPU.pFPUInterface;

#ifndef PDOS_BOCHS_FPU_ONLY
	if (!pInterface) 
	{
	Bit32u val32;
	Bit16u val16;
	Bit8u  val8;

		switch (len) 
		{
		case 1:
			BX_CPU(0)->read_virtual_byte( fpu_iptr->seg(), ptr, &val8 );
			val32 = val8;
			break;
		case 2:
			BX_CPU(0)->read_virtual_word( fpu_iptr->seg(), ptr, &val16 );
			val32 = val16;
			break;
		case 4:
			BX_CPU(0)->read_virtual_dword( fpu_iptr->seg(), ptr, &val32 );
			break;
		default:
			BX_PANIC(("fpu_get_user: len=%u", len));
		}

		return val32;
	}
	else
#endif
	{
	bx_address pAddress = (pInterface->nDataSegment << 4) + ptr;
	Bit32u nValue;

		switch (len)
		{
		case 1:
			nValue = (*pInterface->sMem.pfnReadByte)( pAddress );
			break;
		case 2:
			nValue = (*pInterface->sMem.pfnReadWord)( pAddress );
			break;
		case 4:
			nValue = (*pInterface->sMem.pfnReadDWord)( pAddress );
			break;
		}

		return nValue;
	}
}

void BX_CPP_AttrRegparmN(3) fpu_put_user( Bit32u val, bx_address ptr, unsigned len )
{
tFPU87_Interface *pInterface = theCPU.pFPUInterface;

#ifndef PDOS_BOCHS_FPU_ONLY
	if (!pInterface) 
	{
	Bit32u val32;
	Bit16u val16;
	Bit8u  val8;

		switch (len) 
		{
		case 1:
			val8 = val;
			BX_CPU(0)->write_virtual_byte( fpu_iptr->seg(), ptr, &val8 );
			break;
		case 2:
			val16 = val;
			BX_CPU(0)->write_virtual_word( fpu_iptr->seg(), ptr, &val16 );
			break;
		case 4:
			val32 = val;
			BX_CPU(0)->write_virtual_dword( fpu_iptr->seg(), ptr, &val32 );
			break;
		default:
			BX_PANIC(("fpu_put_user: len=%u", len));
		}
	}
	else
#endif
	{
	bx_address pAddress = (pInterface->nDataSegment << 4) + ptr;

		switch (len)
		{
		case 1:
			(*pInterface->sMem.pfnWriteByte)( pAddress, (BYTE) val );
			break;
		case 2:
			(*pInterface->sMem.pfnWriteWord)( pAddress, (WORD) val );
			break;
		case 4:
			(*pInterface->sMem.pfnWriteDWord)( pAddress, (DWORD) val );
			break;
		}
	}
}

void math_abort(void *info, unsigned int signal)
{
	UNUSED(info); // info is always passed NULL
#ifndef PDOS_BOCHS_FPU_ONLY
	if (!theCPU.pFPUInterface) 
	{
#if BX_CPU_LEVEL >= 4
		// values of signal:
		//   SIGILL  : opcodes which are illegal
		//   SIGFPE  : unmasked FP exception before WAIT or non-control instruction
		//   SIGSEGV : access data beyond segment violation
		switch (signal) 
		{
		case SIGFPE:
		  if (BX_CPU(0)->cr0.ne == 0) 
		  {
			  // MSDOS compatibility external interrupt (IRQ13)
			  BX_INFO (("math_abort: MSDOS compatibility FPU exception"));
			  
			  DEV_pic_raise_irq(13);
			  return;
		  }
		  BX_CPU(0)->exception(BX_MF_EXCEPTION, 0, 0);
		  // execution does not reach here
  
		case SIGILL:
		  BX_PANIC (("math_abort: SIGILL not implemented yet."));
		  BX_CPU(0)->UndefinedOpcode(fpu_iptr);
		  break;
  
		case SIGSEGV:
		  BX_PANIC (("math_abort: SIGSEGV not implemented yet."));
		  break;
		}
#else
		UNUSED(signal);
		BX_INFO(("math_abort: CPU<4 not supported yet"));
#endif
	}
	else
	{
		//   SIGFPE  : unmasked FP exception before WAIT or non-control instruction
		switch (signal) 
		{
		case SIGFPE:
			if (!(theCPU.pFPUInterface->sReg.nCR0 & 0x10))
			{
			  // MSDOS compatibility external interrupt (IRQ13)
				if (theCPU.pFPUInterface->pfnRequestInterrupt)
					(*theCPU.pFPUInterface->pfnRequestInterrupt)( 13 );
			}
			break;
		}
	}
#endif
}

extern "C" int printk(const char * fmt, ...)
{
	return 0;
}
