package com.ibm.ive.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import java.io.*;
import java.util.*;
import com.ibm.ive.midp.util.*;
import com.ibm.oti.security.midp.SecurityPolicy;
import com.ibm.oti.vm.VM;
import com.ibm.ive.midp.ams.*;

/**
 *
 * The Device class abstracts the core properties of the implementation's target
 * hardware.
 *
 * <p>The class has three main functions. First, it must answer information about the
 * device hardware - things like display width, height, or ideal icon size. Second,
 * it must trigger operations on the device - flash the backlight, trigger vibration,
 * or dial a phone number. Finally, it must run the lcdui event loop.
 *
 * <p>To implement the event loop, the Device must handle both OS events coming in from
 * the platform and runnables posted by the user or the lcdui code. The implementation
 * is responsible for handling both types events in a synchronous, threadsafe fashion.
 *
 * @author IBM
 *
 */

public class Device {

	public static final int SOUND_ALARM 		= 5;
	public static final int SOUND_WARNING		= 4;
	public static final int SOUND_ERROR			= 3;
	public static final int SOUND_CONFIRMATION	= 2;
	public static final int SOUND_INFO			= 1;

	public static Object gDisplayableLock = new Object();
	public static Object gRepaintLock = new Object();

	static FastVector gEventListeners = new FastVector();
	static FastVector gDisplayIdentifiers = new FastVector();

	public static Runnable[] gSerialRunnables = new Runnable[10];
	public static int gSerialRunnablesSize = 0;
	static int gSerialRunnableLoopCount = 0;
	public static boolean gRunning;
	public static Object gSerialLock = new Object();

	public static Runnable gCanvasFlushRunnable;
	public static boolean gCanvasFlushNeeded;

	static long gLastCanvasFlushTime;
	static long gLastDisplayableUpdateTime;
	static long gLastReadAndDispatchTime;
	public static long gMinTimeBetweenFlushes = 50;
	public static long gMinTimeBetweenUpdates = 130;

	public static Runnable gDisplayableUpdateRunnable;
	public static boolean gDisplayableUpdateNeeded;

	public static Thread gUiThread;
	static int gNumColors;		/* set by JNI */
	static int gNumAlphaLevels;	/* set by JNI */
	static boolean gIsColor;	/* set by JNI */
	static boolean gHasKeyRepeatEvents;	/* set by JNI */
	static boolean gHasPointerEvents;	/* set by JNI */
	static boolean gHasPointerMotionEvents;	/* set by JNI */
	static boolean gRequiresInitialFocus;	/* set by JNI */
	static boolean gAlertsAreFullScreen;	/* set by JNI */
	static boolean gAlertsAreShownInNewWindow;	/* set by JNI */
	static boolean gIsPhone; /* set by JNI */
	static boolean gCanUsePlatformBrowser; /* set by JNI */
	static boolean gCanUsePlatformMailClient; /* set by JNI */
	static boolean gSupportsPointerGrabbing; /* set by JNI */

	public static boolean gObscured;
	public static boolean gCanUpdate = true;

	/*
	 * JSR 248 (MSA) recommends supporting 24x24 size images (6.2.5.5).
	 * Though this recommendation is made for the AMS MIDletList,
	 * we make it applicable for all elements
	 */
	public static final int ICON_HEIGHT = 24;
	public static final int ICON_WIDTH = 24;

	public static int gDisplayClientAreaWidth;	/* set by JNI */
	public static int gDisplayClientAreaHeight;	/* set by JNI */
	public static int gDisplayTotalWidth;		/* set by JNI */
	public static int gDisplayTotalHeight;		/* set by JNI */

	private static boolean gFirstWindowClassNameRequest = true;
	private static String DEFAULT_WINDOW_CLASS_NAME = "IBM-MIDP";

	private static String[] gBootClassPathAndApiInfo;
	private static String gSecurityDomain;

	public static Thread gSerialThread;

	private static Object gUILock = new Object();
	private static Runnable[] gUIRunnables = new Runnable[20];
	private static int gUIRunnablesSize = 0;

	public static int gMessageWindow;

	static final Event WINDOW_ACTIVATE_EVENT = new Event(Event.WINDOW_ACTIVATE);
	static final Event WINDOW_DEACTIVATE_EVENT = new Event(Event.WINDOW_DEACTIVATE);
	static final Event WINDOW_OBSCURED_EVENT = new Event(Event.WINDOW_RESIZE);
	static final Event WINDOW_CLOSED_EVENT = new Event(Event.WINDOW_CLOSED);

	static boolean gFlushRunnableQueue;
	public static boolean gRunningAllRunnables = false;

	public static final long MIN_TIME_BETWEEN_EVENT_PROCESSING = 100;
	static final long WAIT_TIME_FOR_NEXT_RUNNABLE = 50;
	public static final boolean MULTI_THREADED = true;

	public static final boolean DEBUG_SERIAL_QUEUE = false;
	private static final String SEPARATOR = System.getProperty("file.separator"); //$NON-NLS-1$
	public static String gFontPropertiesPath = System.getProperty("java.home") + SEPARATOR + "lib" + SEPARATOR + "jclMidp20" + SEPARATOR + "fontPaths.properties";
//	public static String gFontConfigureMessage = MidpMsg.getString("Font.ConfigureMessage"); //$NON-NLS-1$
//	public static String gFontConfigureWaitTitle = MidpMsg.getString("Font.ConfigureWaitTitle"); //$NON-NLS-1$
	public static String gFontInvalidPathMessage =  MidpMsg.getString("Font.InvalidPath"); //$NON-NLS-1$
	public static String gFontErrorTitle = MidpMsg.getString("Font.ErrorBoxTitle"); //$NON-NLS-1$
	public static String gFontNotAvailable = MidpMsg.getString("Font.NotAvailable"); //$NON-NLS-1$
	/* For network activity indicator. */
	static int gNotificationID = 0;

	/*
	 * Flash and Vibrate LED default configuration override
	 * 0...n - New LED position
	 * -1 - No LED corressponding to this feature is available. Disable the feature.
	 * -2 - Use default. No override.
	 */
	public static int gVibrateLEDPosition = -2;
	public static int gFlashLEDPosition = -2;

	public static boolean gEnablePauseBehavior = false;

 	// Network activity indicator turned off by default due to networking perf. issues.
	static boolean gEnableNetworkActivityIndicator = false;

	// Set this to disable the Back HotKey in smartphones
	public static boolean gDisableBackHotKey = false;
	// Set this to disable the Talk HotKey in smartphones
	public static boolean gDisableTalkHotKey = false;
	static {
		String property = System.getProperty("LcduiEnablePauseBehavior");
		if (property != null && property.equalsIgnoreCase("true")) {
			gEnablePauseBehavior = true;
		}
		property = System.getProperty("LcduiFlushInterval");
		if (property != null) {
			try {
				gMinTimeBetweenFlushes = Integer.parseInt(property);
			} catch (NumberFormatException e) {
			}
		}
		property = System.getProperty("LcduiUpdateInterval");
		if (property != null) {
			try {
				gMinTimeBetweenUpdates = Integer.parseInt(property);
			} catch (NumberFormatException e) {
			}
		}

		property = System.getProperty("LcduiVibrateLEDPosition");
		if (property != null) {
			try {
				gVibrateLEDPosition = Integer.parseInt(property);
			} catch (NumberFormatException e) {
			}
		}

		property = System.getProperty("LcduiFlashLEDPosition");
		if (property != null) {
			try {
				gFlashLEDPosition = Integer.parseInt(property);
			} catch (NumberFormatException e) {
			}
		}

		property = System.getProperty("LcduiDisableBackHotKey");
		if (property != null && property.equalsIgnoreCase("true")) {
			gDisableBackHotKey = true;
		}

		property = System.getProperty("LcduiDisableTalkHotKey");
		if (property != null && property.equalsIgnoreCase("true")) {
			gDisableTalkHotKey = true;
		}

		property = System.getProperty("LcduiEnableNetworkActivityIndicator");
		if (property != null && property.equalsIgnoreCase("true")) {
			gEnableNetworkActivityIndicator = true;
		}
	}

	/*
	 * This method must be called from the UI thread.
	 */
	public static void init() {
		initImpl(getWindowClassName());

		/*
		 * We create this window for two reasons. First, we need to have a window
		 * that we can post events to in order to wake up the UI thread when
		 * appropriate. Secondly, for some reason pocket pc and smartphone platforms
		 * fail to run correctly if you try to start the event loop before any main
		 * windows are created. Note that the windows must also be visible until
		 * some other window is made visible. So - ensure that this is called from
		 * the main thread before the event loop runs.
		 */
		gMessageWindow = createMessageWindow(getWindowClassName());
	}

	/**
	 * Answer the ideal height for list icons on this platform. Each
	 * platform may have a different ideal height, this call allows
	 * the user to pick the best icon if they have multiple icons to
	 * choose from.
	 *
	 * @return Ideal icon height.
	 */
	public static int getIconHeight() {
		return ICON_HEIGHT;
	}

	/**
	 * Answer the ideal width for list icons on this platform. Each
	 * platform may have a different ideal width, this call allows
	 * the user to pick the best icon if they have multiple icons to
	 * choose from.
	 *
	 * @return Ideal icon width.
	 */
	public static int getIconWidth() {
		return ICON_WIDTH;
	}

	/**
	 * Answer the height of the device display in pixels.
	 *
	 * @return Device display client area height in pixels.
	 */
	public static int getDisplayHeight() {
		return gDisplayClientAreaHeight;
	}

	/**
	 * Answer the width of the device display in pixels.
	 *
	 * @return Device display client area width in pixels.
	 */
	public static int getDisplayWidth() {
		return gDisplayClientAreaWidth;
	}

	/**
	 * Answer the number of colors supported by the display.
	 *
	 * @return Number of colors supported by the device display.
	 */
	public static int getNumColors() {
		return gNumColors;
	}

	/**
	 * Answer the number of alpha levels supported by the device display.
	 * There is a required minimum of two alpha levels - the implementation
	 * at a minimum must support fully transparent and fully opaque pixels.
	 *
	 * @return Number of supported alpha levels.
	 */
	public static int numAlphaLevels() {
		return gNumAlphaLevels;
	}

	/**
	 * Answer whether the device has a color screen.
	 *
	 * @return true if the display supports color, false otherwise.
	 */
	public static boolean isColor() {
		return gIsColor;
	}

	private static void readJars(String path, Hashtable fileInfo, Hashtable apiInfo){
		byte[] manifestBytes = VM.manifestFromZip(path);
		Hashtable manifestProperties = null;
		if (manifestBytes != null) {
			manifestProperties = JadParser.parse(new ByteArrayInputStream(manifestBytes));
			if (manifestProperties.containsKey("API-Name") && manifestProperties.containsKey("API-Specification-Version")) {
				String apiName = (String) manifestProperties.get("API-Name");
				String apiVersion = (String) manifestProperties.get("API-Specification-Version");
				if (!apiInfo.containsKey(apiName)){
					apiInfo.put(apiName, apiVersion);
					fileInfo.put(apiName, path);
				}
				else {
					if (new Version((String) apiInfo.get(apiName)).earlierThan(new Version((String) apiVersion))){
						apiInfo.put(apiName, apiVersion);
						fileInfo.put(apiName, path);
					}
				}
			}
		}
	}

	public static String[] getBootClassPathAndApisInfo()  {
		if (gBootClassPathAndApiInfo == null) {
			Hashtable fileInfo = new Hashtable();
			Hashtable apiInfo = new Hashtable();
			String homePath = System.getProperty("java.home").replace('\\', '/');
			String classPath = getBootClassPathAndApisInfoImpl(homePath);
			if (classPath == null) return null;

			gBootClassPathAndApiInfo = new String[2];
			StringBuffer buffer = new StringBuffer();
			int fIndex = -1;
			int lIndex =  classPath.indexOf(",");
			if (lIndex != -1) {
				do {
					readJars(classPath.substring(fIndex + 1, lIndex), fileInfo, apiInfo);
					fIndex = lIndex;
					lIndex = classPath.indexOf(",", fIndex + 1);
				} while (lIndex != -1);
				readJars(classPath.substring(fIndex + 1, classPath.length()),fileInfo,apiInfo);
				Enumeration enums = fileInfo.elements();
				for (int i = 0; i < fileInfo.size() - 1 ; i++) {
					buffer.append((String) enums.nextElement());
					buffer.append(",");
				}
				buffer.append(enums.nextElement());
				gBootClassPathAndApiInfo[0] = buffer.toString();
				gBootClassPathAndApiInfo[1] = classPath;
			} else {
				gBootClassPathAndApiInfo[0] = classPath;
				gBootClassPathAndApiInfo[1] = classPath;
			}
		}
		return gBootClassPathAndApiInfo;
	}

	public static String getSecurityDomain() {
		if (gSecurityDomain == null) {
			SecurityPolicy policy = SecurityPolicy.getInstance();
			int count = policy.getProtectionDomainCount();
			StringBuffer buffer = new StringBuffer();
			for (int i = 0; i < count - 1; i++) {
				buffer.append(policy.getProtectionDomain(i).getId());
				buffer.append(",");
			}
			buffer.append(policy.getProtectionDomain(count - 1).getId());
			gSecurityDomain = buffer.toString();
		}
		return gSecurityDomain;
	}

	/**
	 * Answer whether the device has a pointing device or touchscreen.
	 *
	 * @return true if the device supports pointer events, false otherwise.
	 */
	public static boolean hasPointerEvents() {
		return gHasPointerEvents;
	}

	/**
	 * Answer whether the device supports dragging a pointer device.
	 *
	 * @return true if the device supports pointer drag events, false otherwise.
	 */
	public static boolean hasPointerMotionEvents() {
		return gHasPointerMotionEvents;
	}

	/**
	 * Answer whether the device supports key repeated events. These events
	 * are generated when a key is held down by the user.
	 *
	 * @return true if the device supports pointer drag events, false otherwise.
	 */
	public static boolean hasKeyRepeatEvents() {
		return gHasKeyRepeatEvents;
	}

	/**
	 * Play the specified sound. The argument is a constant which maps to one
	 * of the AlertType values. The implementation should try to play a sound
	 * which matches the given constant as closely as possible. Answer true if
	 * a sound was played or false if the implementation is unable to play a sound.
	 *
	 * @return true if a sound was played, false otherwise.
	 */
	public static boolean playSound(final int sound) {
		asyncExec(new Runnable() {
			public void run() {
				playSoundImpl(sound);
			}
		});
		return true;
	}

	/**
	 * Vibrate the device for the specified number of milliseconds. If the
	 * value passed is zero, then cancel any previous vibrate request. The
	 * call should vibrate the device only when the MIDlet is currently the
	 * foreground application. If this call is made while a previous call
	 * is still running, the time represents the new duration and should not
	 * be added to the current duration. The call must return immediately.
	 *
	 * @return true if the device can be vibrated for the specified duration,
	 * false otherwise.
	 */
	public static boolean vibrate(int millis) {
		boolean result = vibrateStart();
		if (result) {
			final Timer timer = new Timer();
			TimerTask callStop = new TimerTask() {
				public void run() {
					vibrateStop();
					/*
					 * Terminate the TimerTask's execution thread
					 * by calling cancel as termination of this thread automatically,
					 * may take arbitrarily long time
					 */
					timer.cancel();
				}
			};
			timer.schedule(callStop, millis);
		}
		return result;
	}

	/**
	 * Flash the backlight for the specified number of milliseconds. If the
	 * value passed is zero, then cancel any previous request. The
	 * call should flash the backlight only when the MIDlet is currently the
	 * foreground application. If this call is made while a previous call
	 * is still running, the time represents the new duration and should not
	 * be added to the current duration. The call must return immediately.
	 *
	 * @return true if the device can flash the backlight for the specified
	 * duration, false otherwise.
	 */
	public static boolean flashBacklight(int millis) {
		boolean result = flashBacklightStart();
		if (result) {
			final Timer timer = new Timer();
			TimerTask callStop = new TimerTask() {
				public void run() {
					flashBacklightStop();
					/*
					 * Terminate the TimerTask's execution thread
					 * by calling cancel as termination of this thread automatically,
					 * may take arbitrarily long time
					 */
					timer.cancel();
				}
			};
			timer.schedule(callStop, millis);
		}
		return result;
	}

	/*
	 * Answers whether the device is a phone
	 */
	public static boolean isPhone() {
		return gIsPhone;
	}

	/*
	 * Answers whether the device supports opening the
	 * HTTP URL specified through a platformRequest()
	 * call in the platform browser
	 */
	public static boolean canUsePlatformBrowser() {
		return gCanUsePlatformBrowser;
	}

	/*
	 * Answers whether the device supports opening the
	 * platform mail client to handle the mailto: platform
	 * request
	 */
	public static boolean canUsePlatformMailClient() {
		return gCanUsePlatformMailClient;
	}

	/*
	 * Answers whether the device supports grabbing the pointer
	 */
	public static boolean canGrabPointer() {
		return gSupportsPointerGrabbing;
	}

	/**
	 * Set the device's event listener. This method is usually called by DisplayPeer
	 * to register for events. The event listener passed will be notified of all OS
	 * events that are read from the device - this includes things like pointer events,
	 * key events, focus events, etc.
	 */
	static public void addEventListener(IEventListener listener, int id) {
		synchronized (gEventListeners) {
			gEventListeners.addElement(listener);
			gDisplayIdentifiers.addElement(new Integer(id));
		}
	}

	static public void removeEventListener(IEventListener listener, int id) {
		if (null == listener) return;
		synchronized (gEventListeners) {
			gEventListeners.removeElement(listener);
			gDisplayIdentifiers.removeElement(new Integer(id));
		}
	}

	/**
	 * Post a runnable on the queue. This method grabs the queue lock,
	 * posts the runnable to the queue, and then returns. The method
	 * is used to correctly synchronize the order of repaints, input
	 * events, and callSerially() runnables. Runnables posted to the
	 * device in this way should be executed by the event processing
	 * code in the runOsEventLoop() method.
	 *
	 * @see Device#runEventLoop
	 */
	static public void postRunnable(Runnable r) {
		synchronized (gSerialLock) {
			if (gSerialRunnablesSize == gSerialRunnables.length) {
				Runnable[] newArray = new Runnable[gSerialRunnablesSize << 1];
				System.arraycopy(gSerialRunnables, 0, newArray, 0, gSerialRunnablesSize);
				gSerialRunnables = newArray;
			}
			gSerialRunnables[gSerialRunnablesSize++] = r;
			gSerialLock.notifyAll();
		}
	}

	static Runnable getNextSerialRunnable() {
		synchronized (gSerialLock) {
			while (gRunning && gSerialRunnablesSize == 0 && !gCanvasFlushNeeded && !gDisplayableUpdateNeeded) {
				try {
					gSerialLock.wait(WAIT_TIME_FOR_NEXT_RUNNABLE);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			if (!gRunning || gSerialRunnablesSize == 0) return null;
			Runnable result = gSerialRunnables[0];
			gSerialRunnablesSize--;
			if (gSerialRunnablesSize > 0) System.arraycopy(gSerialRunnables, 1, gSerialRunnables, 0, gSerialRunnablesSize);
			return result;
		}
	}

	static Runnable getNextUIRunnable() {
		synchronized (gUILock) {
			if (gUIRunnablesSize == 0) return null;
			Runnable result = gUIRunnables[0];
			System.arraycopy(gUIRunnables, 1, gUIRunnables, 0, gUIRunnablesSize-1);
			gUIRunnables[--gUIRunnablesSize] = null;
			return result;
		}
	}

	static public void postUIRunnable(Runnable r) {
		synchronized (gUILock) {
			if (gUIRunnablesSize == gUIRunnables.length) {
				Runnable[] newArray = new Runnable[gUIRunnablesSize << 1];
				System.arraycopy(gUIRunnables, 0, newArray, 0, gUIRunnablesSize);
				gUIRunnables = newArray;
			}
			gUIRunnables[gUIRunnablesSize++] = r;
			postMessage(gMessageWindow, Event.EVENT_UI_RUNNABLE);
		}
	}

	public static void asyncExec(Runnable r) {
		if (Thread.currentThread() == gUiThread) {
			r.run();
			return;
		}
		postUIRunnable(r);
	}

	public static void syncExec(Runnable r) {
		if (Thread.currentThread() == gUiThread) {
			r.run();
			return;
		}
		synchronized (r) {
			postUIRunnable(r);
			try {
				r.wait();
			} catch (InterruptedException e) {
			}
		}
	}

	static void dispatchUIRunnable() {
		Runnable r = getNextUIRunnable();
		if (r != null) {
			synchronized (r) {
				try {
					r.run();
				} catch (Throwable t) {
					t.printStackTrace();
				}
				r.notify();
			}
		}
	}

	/**
	 * Answer whether the implementation should set an initial focus on Forms or
	 * other displayables. This may be desirable for platforms which have no pointer
	 * events or undesirable for platforms that normally don't assign an initial
	 * widget focus.
	 *
	 * @return true if displayables should assign an initial focus widget, false otherwise.
	 */
	public static boolean getRequiresInitialFocus() {
		return gRequiresInitialFocus;
	}

	/**
	 * Answer a string which represents the given key code. The text
	 * should resemble any characters printed on the key or should clearly
	 * describe which key has generated the keycode.
	 *
	 * @return a String which represents the key pressed.
	 */
	public static String getKeyName(int keyCode){
		//TODO: implement
		return null;
	}

	public static void scheduleDisplayableUpdate() {
		synchronized (gSerialLock) {
			if (!gDisplayableUpdateNeeded) {
				gDisplayableUpdateNeeded = true;
				gSerialLock.notifyAll();
			}
		}
	}

	public static void networkActivityStarted(String operation) {
		if (gEnableNetworkActivityIndicator) addNetworkActivityIndicator(gNotificationID, "Java Notification", "The Java VM is currently accessing the network.");
	}

	public static void networkActivityComplete(String operation) {
		if (gEnableNetworkActivityIndicator) removeNetworkActivityIndicator(gNotificationID++);
	}

	public static boolean isAllowedToRunMIDletSuite() {
		return true;
	}

	static String getMidletWindowName() {
		String midletName = (String) MIDletManager.gProperties.get("MIDlet-Name"); //$NON-NLS-1$
		return "IBM-MIDP-" + midletName; //$NON-NLS-1$
	}

	public static String getWindowClassName() {
		if (gFirstWindowClassNameRequest) {
			gFirstWindowClassNameRequest = false;
			return getMidletWindowName();
		}
		return DEFAULT_WINDOW_CLASS_NAME;
	}

	public static boolean isAlreadyRunningThisSuite() {
		if (Device.bringMidletToFront(getMidletWindowName())) {
			Device.disposeSplashScreen();
			return true;
		}
		return false;
	}

	/**
	 * Free any OS memory allocated by this class. This call is made when the
	 * MIDlet is shutting down.
	 */
	public static void dispose() {
		disposeImpl(gMessageWindow);
		gMessageWindow = 0;
	}

	/**
	 * Stop the OS event loop. This call is made to end event processing when the
	 * MIDlet is shutting down. It should end the event processing loop and
	 * make sure that any OS memory allocated to process OS events has been freed.
	 */
	public static void stopEventLoop() {
		gRunning = false;
		postMessage(gMessageWindow, Event.EVENT_STOP);
	}

	/**
	 * Begin running the event loop. This loop must synchronously handle events
	 * coming in from the device or platform graphics library as well as running
	 * Runnables posted by the user or by the lcdui code base. Events coming from
	 * the OS should be sent to the IEventListener if one has been set on this
	 * device. The method should process events in the current thread rather than
	 * starting a new one. This means that it will block until stopEventLoop()
	 * is called.
	 */
	public static void runEventLoop() {
		synchronized (gDisplayableLock) {
			if (gRunning) return;
			gUiThread = Thread.currentThread();
			gRunning = true;
			gDisplayableLock.notifyAll();
		}

		if (MULTI_THREADED) {
			startSerialRunnableThread();
			while (gRunning) readAndDispatch();
		} else {
			gSerialThread = Thread.currentThread();
			while (gRunning) {
				dispatchSerialRunnable(true, false);
				long currentTime = System.currentTimeMillis();
				if (currentTime - gLastReadAndDispatchTime > MIN_TIME_BETWEEN_EVENT_PROCESSING) {
					gLastReadAndDispatchTime = currentTime;
					readAndDispatch();
				}

				// Copied from the old Qt stream.
				// The delay is introduced because otherwise the event loop can starve user threads.
				// May need to restrict this to every 100 iterations or so to increase jbenchmark scores.
				//
				// sleep for ten millis if there are no runnables waiting to be run, otherwise yield.
				// should sync around serial lock when checking gSerialRunnablesSize, but it doesn't make much of a difference whether or not it's true
				if (gSerialRunnablesSize == 0) {
					// no other runnables are waiting to be executed in the queue, give the processor a rest
					try {
						Thread.sleep(10);
					} catch (InterruptedException e) {
					}
				} else {
					Thread.yield();
				}
			}
		}

		synchronized (gSerialLock) {
			gSerialLock.notify();
		}
		Thread.yield();
	}

	public static void flushDisplay(long currentTime) {
		gCanvasFlushNeeded = false;
		gLastCanvasFlushTime = currentTime;
		try {
			gCanvasFlushRunnable.run();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public static void dispatchSerialRunnable(boolean runAllSerialRunnables, boolean forceFlush) {
		Runnable[] runnables = null;
		int stop = 0;

		if (runAllSerialRunnables) {
			synchronized (gSerialLock) {
				/*
				 * Some runnables will schedule new runnables. Only run the
				 * number that have already been scheduled.
				 */
				runnables = new Runnable[gSerialRunnablesSize];
				System.arraycopy(gSerialRunnables, 0, runnables, 0, gSerialRunnablesSize);
				stop = gSerialRunnablesSize;
				gSerialRunnablesSize = 0;
			}

			for (int i = 0; i < stop; i++) {
				try {
					runnables[i].run();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		} else {
			Runnable r = getNextSerialRunnable();
			if (r != null) {
				try {
					r.run();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		long currentTime = System.currentTimeMillis();
		if ((gCanvasFlushRunnable != null && forceFlush) || (gCanvasFlushNeeded && (currentTime - gLastCanvasFlushTime) > gMinTimeBetweenFlushes)) {
			flushDisplay(currentTime);
		}

		boolean runDisplayableUpdateRunnable = false;
		synchronized (gSerialLock) {
			if (gCanUpdate && gDisplayableUpdateNeeded && (gSerialRunnablesSize == 0 || ((currentTime - gLastDisplayableUpdateTime) > gMinTimeBetweenUpdates))) {
				gLastDisplayableUpdateTime = currentTime;
				gDisplayableUpdateNeeded = false;
				runDisplayableUpdateRunnable = true;
			}
		}
		if (runDisplayableUpdateRunnable && gDisplayableUpdateRunnable != null)  {
			try {
				gDisplayableUpdateRunnable.run();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	static void startSerialRunnableThread() {
		gSerialThread = new Thread() {
			public void run() {
				while (gRunning) {
					dispatchSerialRunnable(false, false);
					if (gSerialRunnableLoopCount++ == 100) {
						Thread.yield();
						gSerialRunnableLoopCount = 0;
					}
				}
			}
		};

		gSerialThread.start();
	}

	/**
	 * Execute all currently queued runnables. This method synchronizes
	 * on the library lock and then executes and removes each Runnable
	 * leaving the queue empty.
	 */
	static public void flushRunnableQueue() {
		dispatchSerialRunnable(true, true);
	}

	public static void enqueueMenuEvent(final int id, final int eventId) {
		postRunnable(new Runnable() {
			public void run() {
				dispatchEvent(new Event(id, eventId, 0, 0));
			}
		});
	}

	static boolean dispatchEvent(Event e) {
		int size = gDisplayIdentifiers.size();
		IEventListener listener = null;

		synchronized (gEventListeners) {
			if (1 == size) { /* For Single MIDlet */
				Integer curID = (Integer) gDisplayIdentifiers.elementAt(0);
				if (e.fHandle == curID.intValue()) {
					listener = (IEventListener) gEventListeners.elementAt(0);
				}
			} else { /* look up the listener */
				int i = 0;
				for (;i < size; i++) {
					Integer curID = (Integer) gDisplayIdentifiers.elementAt(i);
					if (e.fHandle == curID.intValue()) {
						listener = (IEventListener) gEventListeners.elementAt(i);
						break;
					}
				}
			}
		}
		if (listener == null) {
//			System.err.println("Dropping event: " + e.fType + " owner: " + Integer.toHexString(e.fHandle));
			return false;
		}
		switch (e.fType) {
			case Event.POINTER_PRESSED:
			case Event.POINTER_RELEASED:
			case Event.POINTER_MOVED:
				return listener.dispatchPointerEvent(e);
			default:
				return listener.dispatchEvent(e);
		}
	}

	static boolean event(int handle, int msg, int arg1, int arg2) {
		switch (msg) {
			case Event.EVENT_UI_RUNNABLE:
				try {
					dispatchUIRunnable();
				} catch (Exception e) {
					e.printStackTrace();
				}
				return true;
//			case Event.EVENT_DISPLAYABLE_UPDATE_RUNNABLE:
//				try {
//					if (gDisplayableUpdateRunnable != null) gDisplayableUpdateRunnable.run();
//				} catch (Exception e) {
//					e.printStackTrace();
//				}
//				return true;
		}

		final Event event = new Event(handle, msg, arg1, arg2);
		postRunnable(new Runnable() {
			public void run() {
				syncExec(new Runnable() {
					public void run() {
						dispatchEvent(event);
					}
				});
			}
		});

		/*
		 * Special handling for the WINDOW_CLOSED event as this would be
		 * handled by the java side when the queued runnable is executed.
		 * Returning false would affect the freeing of OS resources correctly
		 * as the OS might attempt a cleanup as part of the default handling
		 * (as in the case of Windows).
		 */
		if (msg == Event.WINDOW_CLOSED) return true;
		return false;
	}

	/**
	 * Answer the height of the device display in pixels.
	 *
	 * @return Total Device display height in pixels.
	 */
	public static int getShellHeight() {
		return gDisplayTotalHeight;
	}

	/**
	 * Answer the width of the device display in pixels.
	 *
	 * @return Total Device display width in pixels.
	 */
	public static int getShellWidth() {
		return gDisplayTotalWidth;
	}

	public static boolean alertsAreFullScreen() {
		return gAlertsAreFullScreen;
	}

	public static boolean alertsAreShownInNewWindow() {
		return gAlertsAreShownInNewWindow;
	}

	static native void addNetworkActivityIndicator(int id, String title, String text);
	private static native boolean bringMidletToFront(String windowName);
	public static native boolean bringMidletToForeground(int windowHandle);
	public static native boolean sendMidletToBackground(int windowHandle);
	static native int createMessageWindow(String windowName);
	public static native boolean dialPhone(String number);
	public static native void disposeImpl(int messageWindow);
	public static native void disposeSplashScreen();
	private static native boolean flashBacklightStart();
	private static native void flashBacklightStop();
	static native void initImpl(String windowClassName);
	public static native void midletSuiteClosing();
	public static native boolean platformRequest(String url);
	static native boolean playSoundImpl(int sound);
	static native void postMessage(int handle, int msg);
	public static native int readAndDispatch();
	static native void removeNetworkActivityIndicator(int id);
	private static native boolean vibrateStart();
	private static native void vibrateStop();
	public static native int relaunchMidlet();
	public static native boolean requiresMidletRelaunch(int windowHandle, String message1, String message2, String midletIdentifier);
	public static native boolean closeMidletIdentifier();
	public static native boolean openPlatformBrowser(String URL);
	public static native boolean openPlatformMailClient(String URL);
	static native String getBootClassPathAndApisInfoImpl(String path);
}
