package com.ibm.ive.midp.ams;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import java.io.*;
import java.util.*;
import javax.microedition.io.*;
import com.ibm.ive.midp.*;
import com.ibm.oti.connection.file.*;
import com.ibm.oti.midlet.help.*;
import com.ibm.ive.midp.util.*;
import com.ibm.ive.midp.ams.ui.AmsMidlet;
import com.ibm.oti.vm.VM;

public class InstallProcess implements MidpConstants, Runnable {

	MidletEntry fInstalledMidlet;
	IInstallProcessListener fListener;
	AmsException fFailure;
	int fMode;
	String fURL;
	boolean fPlatformInstallProcess;

	public InstallProcess(String url, int mode, IInstallProcessListener listener, boolean platformInstallProcess) {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
		fListener = listener;
		fMode = mode;
		fURL = getValidatedURL(url);
		fPlatformInstallProcess = platformInstallProcess;
	}

	public void run() {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();

		MidletEntry entry = new MidletEntry();
		try {
			fetchMidlet(entry, fMode, fURL);
			Hashtable properties = new MidletVerifier(fListener).checkMidlet(entry, fListener);
			fInstalledMidlet = entry;

			setSecurityDomain(entry);
			MidletStorage.add(entry);

			if ((fMode == MidpConstants.INSTALL_MIDLET) && fURL != null && fURL.length() > 0) {
				InstallHistory.addToHistory(fURL);
			}
			try {
				MessageDispatch.reportStatus(MidpConstants.STATUS_SUCCESS, properties);
				PendingStatusReporter.sendPendingStatusReports();
			} catch (Exception e1) {
				PendingStatusReporter.addNotification(MidpConstants.STATUS_SUCCESS, (String)properties.get(MidpConstants.KEY_INSTALL_NOTIFY));
				e1.printStackTrace();
			}
			ResourceDisposer.processResourcesAwaitingDeletion();

			fListener.installSucceeded(entry, fMode != MidpConstants.INSTALL_MIDLET, fPlatformInstallProcess);
		} catch (AmsException e) {
			fFailure = e;
			if (!fListener.isCancelled()){
				e.printStackTrace();
				if (e.getNotifyURL() != null) MessageDispatch.reportErrorStatus(e);
				fListener.installFailed(e);
			}

			MidletStorage.throwAwayMidlet(entry);
		}
	}

	void setSecurityDomain(MidletEntry entry) {
		MidletSuiteId suiteId = new MidletSuiteId(entry.getName(), entry.getVendor());
		MidletLoader.Metadata metadata = MidletLoader.getMetadataClass(suiteId, "AMS");				 //$NON-NLS-1$
		metadata.setEntry("domain", entry.getSecurityDomain()); //$NON-NLS-1$
		try {
			metadata.save();
		} catch (IOException e) {
		}
	}

	String getEntry(Hashtable properties, String key, String notifyURL, boolean required) throws AmsException {
		String value = (String) properties.get(key);
		if (required && (value == null || value.length() == 0)) {
			throw new AmsException(MidpConstants.RET_INVALID_DESCRIPTOR_FILE, MidpMsg.getString("InstallProcess.getRequiredEntry.error.missing_required_key", key), notifyURL); //$NON-NLS-1$
		}
		return value;
	}

	void fetchMidlet(MidletEntry newEntry, int installOption, String url) throws AmsException {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();

		long availableStorageSpace = fListener.getAvailableStorageSpace();

		/*
		 * Download the file from the specified URL and assume that
		 * it is a JAD since that will probably be the most common
		 * method of installing a midlet
		 */
		FileDownloader d = new FileDownloader(fListener);
		String filename = FileStorage.getNextAvailableFileName(newEntry.getMidletID(), ".jad");		 //$NON-NLS-1$
		int fileSize = d.downloadFile(url, filename);

		/*
		 * The type of file downloaded needs to be determined and the properties
		 * need to be read so that we can check to see if this is an update.
		 *
		 * To determine the file type, try reading a jar manifest. If this succeeds,
		 * then we've downloaded a jar, otherwise assume we have a jad file.
		 *
		 * TODO - Is it valid to have a jad url or jar url that doesn't end with the
		 * *.jad or *.jar extension? If not, then we can just check the URL to
		 * determine the file type.
		 */

		Hashtable properties;
		byte[] manifest = VM.manifestFromZip(filename);
		boolean isJadFile = false;
		if (manifest != null) {
			newEntry.setJarURL(url);
			newEntry.setJarFile(FileStorage.getNextAvailableFileName(newEntry.getMidletID(), ".jar")); //$NON-NLS-1$
			properties = JadParser.parse(new ByteArrayInputStream(manifest));
			// The file downloaded seems to be a jar file so the file should be renamed
			FileStorage.rename(filename, newEntry.getJarFile()); // returns int to signify success or not
		} else {
			newEntry.setJadURL(url);
			newEntry.setJadFile(filename);
			properties = JadParser.parse(newEntry.getJadFile());
			/*The following line should be un-commented for making the check for Profile, Configuration and
			 *permission a required entry in the JAD file as per section 6.2.5.1 of jsr 248 spec
			 */
			//isJadFile = true;
		}

		// Check for all the required entries
		String installNotifyURL = (String) properties.get(MidpConstants.KEY_INSTALL_NOTIFY);
		newEntry.setName(getEntry(properties, MidpConstants.KEY_MIDLET_NAME, installNotifyURL, true));
		newEntry.setIcon(getEntry(properties, MidpConstants.KEY_MIDLET_ICON, installNotifyURL, false), properties);
		newEntry.setVersion(getEntry(properties, MidpConstants.KEY_MIDLET_VERSION, installNotifyURL, true));
		newEntry.setVendor(getEntry(properties, MidpConstants.KEY_MIDLET_VENDOR, installNotifyURL, true));

		/*JSR 248 mandates that jad SHOULD contain MicroEdition-Profile, MicroEdition-Configuration
		 * & MIDlet-Permissions as well. If the downloaded file is a JAD file, we should check for these
		 * before we download the JAR file.
		 */
		newEntry.setMIDPProfile(getEntry(properties, MidpConstants.KEY_MIDLET_PROFILE, installNotifyURL, isJadFile));
		newEntry.setMIDPConfiguration(getEntry(properties, MidpConstants.KEY_MIDLET_CONFIGURATION, installNotifyURL, isJadFile));
		newEntry.setMidletPermissions(getEntry(properties, MidpConstants.KEY_MIDLET_PERMISSIONS, installNotifyURL, isJadFile));

		if (newEntry.getJadFile() != null) {
			//As per JSR 248, we should check if the profile and configuration are valid before downloading the jar file
			// (section 6.2.5.1 of jsr 248 spec). Commenting this out for now.
			//MidletVerifier.checkProfileAndConfiguration(newEntry, installNotifyURL);

			newEntry.setJarSize(getEntry(properties, KEY_MIDLET_JAR_SIZE, installNotifyURL, true));
			int jarSize = 0;
			try {
				jarSize = Integer.parseInt(newEntry.getJarSize());
				if (jarSize < 0) throw new NumberFormatException();
			} catch (NumberFormatException nfe) {
				throw new AmsException(RET_INVALID_DESCRIPTOR_FILE, MidpMsg.getString("InstallProcess.fetchMidlet.error.invalid_jar_size"), installNotifyURL); //$NON-NLS-1$
			}
			String midletDataSize = getEntry(properties, KEY_MIDLET_DATA_SIZE, installNotifyURL, false);
			long dataSize = 0;
			if (midletDataSize != null) {
				// MIDlet-Data-Size specified. Check it and get it.
				try {
					dataSize = Long.parseLong(midletDataSize);
					if (dataSize < 0) throw new NumberFormatException();
				} catch (NumberFormatException nfe) {
					throw new AmsException(RET_ILLEGAL_ARGUMENT, MidpMsg.getString("InstallProcess.fetchMidlet.error.invalid_midlet_data_size"), installNotifyURL); //$NON-NLS-1$
				}
				newEntry.setDataSize(midletDataSize);
			}
			if (fileSize == -1) fileSize = 0; // Did not get JAD file size. Don't consider it.
			long requiredStorageSpace = fileSize + jarSize + dataSize;
			if (requiredStorageSpace < 0 || // overflow
					requiredStorageSpace > availableStorageSpace) {
				throw new AmsException(MidpConstants.RET_OUT_OF_MEMORY, MidpMsg.getString("FileDownloader.downloadFile.error.noSpace"), installNotifyURL); //$NON-NLS-1$
			} else if (!fListener.isOkToInstallFile(newEntry)) {
				throw new AmsException(RET_OPERATION_CANCELLED, MidpMsg.getString("InstallProcess.promptForDownload.error.user_cancelled")); //$NON-NLS-1$
			}
		}
		// Check to see if the downloaded file would be considered an update
		checkForUpdate(newEntry, installOption, installNotifyURL);

		// If the downloaded file was a jad then the jar still need to be downloaded.
		if (newEntry.getJarFile() == null) {
			newEntry.setJarURL(getValidatedJarURL(properties, installNotifyURL, url));
			newEntry.setJarFile(FileStorage.getNextAvailableFileName(newEntry.getMidletID(), ".jar")); //$NON-NLS-1$
			try {
				int jarSize = Integer.parseInt(getEntry(properties, KEY_MIDLET_JAR_SIZE, installNotifyURL, true));
				fetchJarFile(newEntry.getJarFile(), newEntry.getJarURL(), jarSize, installNotifyURL);
			} catch (NumberFormatException e) {
				throw new AmsException(RET_INVALID_DESCRIPTOR_FILE, MidpMsg.getString("InstallProcess.fetchMidlet.error.invalid_jar_size"), installNotifyURL); //$NON-NLS-1$
			}
		}
	}

	String getValidatedJarURL(Hashtable properties, String installNotifyURL, String jadURL) throws AmsException {
		String url = getEntry(properties,KEY_MIDLET_JAR_URL, installNotifyURL, true);

		// Check to see if the URL is a valid url on its own
		try {
			Connection c = Connector.open(url);
			try {
				c.close();
			} catch (IOException e1) {
			}

			// The JAR URL opened as it was specified in the JAD so it must be ok
			return url;
		} catch (IllegalArgumentException e) {
			// Maybe they tried to specify the JAR as relative to the JAD...
			int lastSlash = jadURL.lastIndexOf('/');
			int lastBackwardSlash = jadURL.lastIndexOf('\\');
			if (lastBackwardSlash > lastSlash) lastSlash = lastBackwardSlash;

			// If there is no last slash then assume the file is in the current directory.
			if (lastSlash == -1) return "file:" + url; //$NON-NLS-1$

			// Remove the last part of the url from the jadURL and append the jarURL to it
			String baseURL = jadURL.substring(0, lastSlash + 1);
			String absoluteJarURL = baseURL + url;

			if (absoluteJarURL.startsWith("file:")) { //$NON-NLS-1$
				try {
					FileConnection file = new FileConnection();
					String fileSpec = absoluteJarURL.substring(5);
					file.setParameters(fileSpec, Connector.READ, false);
					file.openInputStream();
					file.close();
					return absoluteJarURL;
				} catch (IOException e1) {
				}
			}

			// Try to open a connection to the new url and return it as the real url
			// if it opens

			try {
				Connection c = Connector.open(absoluteJarURL);
				try {
					c.close();
				} catch (IOException e2) {
				}

				// The new URL opened so that means the jar url inside the jad must have
				// be relative to the jad.
				return absoluteJarURL;
			} catch (IllegalArgumentException e1) {
				// The url failed to open so the best we can do is return the URL that
				// was specified in the JAD
				return url;
			} catch (IOException e2) {
				return url;
			}
		} catch (IOException e) {
			return url;
		}
	}

	String getValidatedURL(String url) {
		// Check to see if the URL is a valid url on its own
		try {
			Connection c = Connector.open(url);
			try {
				c.close();
			} catch (IOException e1) {
			}

			// The JAR URL opened as it was specified in the JAD so it must be ok
			return url;
		} catch (Exception e) {
			// Try it as a file url
			try {
				FileConnection file = new FileConnection();
				file.setParameters(url, Connector.READ, false);
				file.openInputStream();
				file.close();
				return "file:" + url; //$NON-NLS-1$
			} catch (IOException e1) {
			}
		}
		return url;
	}

	void checkForUpdate(MidletEntry newEntry, int installOption, String installNotifyURL) throws AmsException {
		int prompt = 0;
		boolean promptRMS = false;
		MidletEntry entry = MidletStorage.get(newEntry.getName(), newEntry.getVendor());
		newEntry.updatedMidlet = entry;
		if (entry != null) {
			Version v1 = new Version(newEntry.getVersion());
			Version v2 = new Version(entry.getVersion());

			// Check to see if the new MIDlet is from a different location then the current MIDlet
			if ((installOption & DIFFERENT_LOCATION) != DIFFERENT_LOCATION) {
				String newJad = newEntry.getJadURL();
				String oldJad = entry.getJadURL();

				// Either the new MIDlet or the old MIDlet might have been installed from
				// a jar URL, so be carefull when comparing URLs
				if (newJad != null && oldJad != null && !newJad.equals(oldJad)) {
					prompt = PROMPT_DIFFERENT_LOCATION;
					promptRMS = true;
				}

				String newJar = newEntry.getJarURL();
				String oldJar = entry.getJarURL();
				if (newJar != null && oldJar != null && !newJar.equals(oldJar)) {
					prompt = PROMPT_DIFFERENT_LOCATION;
					promptRMS = true;
				}
				//Since we do not support Application RMS retention on Palm as of now,
				//perform this check only for non Palm platforms
					if (!promptRMS && entry.getKeyPrincipalName() != null && newEntry.getKeyPrincipalName() != null) {
					Hashtable properties = JadParser.parse(entry.getJadFile());
					Hashtable newProperties = JadParser.parse(newEntry.getJadFile());
					String oldSignature = (String)properties.get("MIDlet-Jar-RSA-SHA1");	//$NON-NLS-1$
					String newSignature = (String)newProperties.get("MIDlet-Jar-RSA-SHA1");	//$NON-NLS-1$
					if (newSignature != null && oldSignature != null && !newSignature.equals(oldSignature)) promptRMS = true;
				}
			}

			if (v1.equals(v2)) {
				//* New Version is SAME AS Currently Installed Version
				if ((installOption & UPDATE_IF_SAME_OR_LATER) != UPDATE_IF_SAME_OR_LATER && (installOption & UPDATE_ANY) != UPDATE_ANY) {
					if (!fListener.isOkToUpdateFile(newEntry, prompt | PROMPT_SAME_VERSION_EXISTS)) {
						throw new AmsException(RET_OPERATION_CANCELLED, MidpMsg.getString("InstallProcess.promptForUpdate.error.midlet_already_installed"), installNotifyURL); //$NON-NLS-1$
					}
				}
			} else if (v1.earlierThan(v2)) {
				//* New Version is EARLIER THAN Currently Installed Version
				if ((installOption & UPDATE_ANY) != UPDATE_ANY) {
					if (!fListener.isOkToUpdateFile(newEntry, prompt | PROMPT_LATER_VERSION_EXISTS)) {
						throw new AmsException(RET_OPERATION_CANCELLED, MidpMsg.getString("InstallProcess.promptForUpdate.error.midlet_already_installed"), installNotifyURL); //$NON-NLS-1$
					}
				}
			} else if (v1.laterThan(v2)) {
				//* New Version is LATER THAN Currently Installed Version -
				//* only a problem if INSTALL_MIDLET Option Specifed
				if ((installOption & INSTALL_MIDLET) == INSTALL_MIDLET) {
					if (!fListener.isOkToUpdateFile(newEntry, prompt | PROMPT_EARLIER_VERSION_EXISTS)) {
						throw new AmsException(RET_OPERATION_CANCELLED, MidpMsg.getString("InstallProcess.promptForUpdate.error.midlet_already_installed"), installNotifyURL); //$NON-NLS-1$
					}
				}
			}
			/* Will take care of unknown update requests. */
			fMode = MidpConstants.UPDATE_IF_LATER_VERSION;
			if (promptRMS) {
				String name = entry.getName();
				String vendor = entry.getVendor();
				if (!fListener.isOkToRetainRMS(name)) {
					MidletLoader.init(name, vendor);
					MidletLoader.deleteMidletSuiteRMS(MidletLoader.getMidletSuiteId());
					MidletLoader.init(AmsMidlet.amsMidlet.getAppProperty("MIDlet-Name"), AmsMidlet.amsMidlet.getAppProperty("MIDlet-Vendor"));
				}
			}
		}
	}

	void fetchJarFile(String filename, String jarURL, int size, String notifyURL) throws AmsException {
		try {
			(new FileDownloader(fListener)).downloadFile(jarURL, filename);
		} catch (AmsException e) {
			if (e.getStatus() == RET_OPERATION_CANCELLED) throw e;
			if (e.getStatus() == RET_ILLEGAL_ARGUMENT) {
				throw new AmsException(RET_INVALID_DESCRIPTOR_FILE, MidpMsg.getString("InstallProcess.fetchJarFile.error.invalid_jar_location"), notifyURL); //$NON-NLS-1$
			} else {
				throw new AmsException(RET_IO_ERROR, MidpMsg.getString("InstallProcess.fetchJarFile.error.error_downloading_jar", jarURL), notifyURL); //$NON-NLS-1$
			}
		}

		// Check to make sure that the jar file is the proper size
		try {
			FileInputStream testStream = new FileInputStream(filename);
			try {
				int realJarSize = testStream.available();
				// Double check if InputStream#available() has returned the correct
				// size of the jar file if the sizes do not match
				if (size != realJarSize) {
					int bufferSize = 2048, readBytes = 0;
					byte []buffer = new byte[bufferSize];
					realJarSize = 0;
					while ((readBytes = testStream.read(buffer)) != -1) {
						realJarSize += readBytes;
					}
					//If they still dont match, throw up an Exception
					if (size != realJarSize) throw new AmsException(RET_JAR_SIZE_MISMATCH, MidpMsg.getString("InstallProcess.fetchJarFile.error.jar_size_mistmatch", new Integer(size), new Integer(realJarSize)), notifyURL); //$NON-NLS-1$
				}
			} catch (IOException e2) {
				throw new AmsException(RET_INTERNAL_ERROR, MidpMsg.getString("InstallProcess.fetchJarFile.error.error_reading_jar_file")); //$NON-NLS-1$
			} finally {
				try {
					testStream.close();
				} catch (IOException e3) {
				}
			}
		} catch (ConnectionNotFoundException e1) {
			throw new AmsException(RET_INTERNAL_ERROR, MidpMsg.getString("InstallProcess.fetchJarFile.error.jar_not_found")); //$NON-NLS-1$
		}
	}

	public MidletEntry getInstalledMidlet() {
		return fInstalledMidlet;
	}

	public AmsException getInstallFailure() {
		return fFailure;
	}
}
