package com.ibm.ive.midp.ams;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import java.io.*;
import java.util.*;
import javax.microedition.io.*;
import com.ibm.ive.midp.*;

public class MessageDispatch {

	/**
	 * Reports one of the status codes as defined on page 16 of the MIDP 2.0 spec to a specified
	 * url.  Even reporting the status fails because of communication errors this method will
	 * return safely.
	 *
	 * @param status one of the status constants, MidletConstants.STATUS_*
	 * @param URL the url to which send the status will be sent
	 */
	static void reportStatus(int status, String URL) throws IOException{
		if (URL == null) return;

		Connection connection = null;
		try {
			connection = Connector.open(URL);
		} catch (IllegalArgumentException iae) {
			iae.printStackTrace();
			return;
		} catch (ConnectionNotFoundException cnfe) {
			cnfe.printStackTrace();
			return;
		} catch (SecurityException se) {
			se.printStackTrace();
			return;
		}

		if (connection instanceof HttpConnection) {
			// All HTTP based status reports should be sent through the POST method
			((HttpConnection)connection).setRequestMethod(HttpConnection.POST);
		}

		if (connection instanceof OutputConnection) {
			DataOutputStream outputStream = ((OutputConnection)connection).openDataOutputStream();
			String message = Integer.toString(status) + " "; //$NON-NLS-1$
			// The strings below this point should not be translated since
			// they are required by the spec
			switch (status) {
				case MidpConstants.STATUS_SUCCESS: message += "Success"; break; //$NON-NLS-1$
				case MidpConstants.STATUS_INSUFFICIENT_MEMORY: message += "Insufficient Memory"; break; 	 //$NON-NLS-1$
				case MidpConstants.STATUS_USER_CANCELLED: message += "User Cancelled"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_LOSS_OF_SERVICE: message += "Loss of Service"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_JAR_SIZE_MISMATCH: message += "JAR size mismatch"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_ATTRIBUE_MISMATCH:	message += "Attribue Mismatch"; break; //$NON-NLS-1$
				case MidpConstants.STATUS_INVALID_DESCRIPTOR: message += "Invalid Descriptor"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_INVALID_JAR: message += "Invalid JAR"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_INCOMPATIBLE_PROFILE: message += "Incompatible Configuration or Profile"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_AUTHENTICATION_FAILURE: message += "Application authentication failure"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_AUTHORIZATION_FAILURE: message += "Application authorization failure"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_PUSH_FAILURE: message += "Push registration failure"; break;	 //$NON-NLS-1$
				case MidpConstants.STATUS_DELETE_NOTIFICATION: message += "Deletion Notification"; break;	 //$NON-NLS-1$
			}

			outputStream.write(message.getBytes());
			outputStream.close();
		} else {
			System.err.println(MidpMsg.getString("MidletCatalog.reportStatus.error.output_not_allowed", URL)); //$NON-NLS-1$
		}
	}

	static void reportErrorStatus(AmsException e) {
		int status = -1;
		switch (e.getStatus()) {
			case MidpConstants.RET_ILLEGAL_ARGUMENT:
			case MidpConstants.RET_MIDLET_NOT_FOUND:
			case MidpConstants.RET_CONNECTION_FAILED:
			case MidpConstants.RET_COMM_FAILURE:
			case MidpConstants.RET_GATEWAY_BUSY:
			case MidpConstants.RET_IO_ERROR:
			case MidpConstants.RET_FILE_OPEN_FAILURE:
			case MidpConstants.RET_HOST_NOT_FOUND:
			case MidpConstants.RET_INVALID_SCHEME:
			case MidpConstants.RET_UNKOWN_AUTHENTICATION_MODE:
			case MidpConstants.RET_MALFORMED_URL: break;
			case MidpConstants.RET_SIGNATURE_PROBLEM: status = MidpConstants.STATUS_AUTHENTICATION_FAILURE; break;
			case MidpConstants.RET_CONFIG_PROFILE_PROBLEM: status = MidpConstants.STATUS_INCOMPATIBLE_PROFILE; break;
			case MidpConstants.RET_OUT_OF_MEMORY: status = MidpConstants.STATUS_INSUFFICIENT_MEMORY; break;
			case MidpConstants.RET_INVALID_DESCRIPTOR_FILE: status = MidpConstants.STATUS_INVALID_DESCRIPTOR; break;
			case MidpConstants.RET_OPERATION_CANCELLED: status = MidpConstants.STATUS_USER_CANCELLED; break;
			case MidpConstants.RET_JAR_SIZE_MISMATCH: status = MidpConstants.STATUS_JAR_SIZE_MISMATCH; break;
			case MidpConstants.RET_ATTRIBUTE_MISMATCH: status = MidpConstants.STATUS_ATTRIBUE_MISMATCH; break;
			case MidpConstants.RET_INVALID_JAR_FILE: status = MidpConstants.STATUS_INVALID_JAR; break;
			case MidpConstants.RET_INVALID_PERMISSION: status = MidpConstants.STATUS_AUTHORIZATION_FAILURE; break;
		}

		if (status != -1) {
			try {
				reportStatus(status, e.getNotifyURL());
			} catch (IOException e1) {
				e1.printStackTrace();
			}
		}
	}

	static void reportStatus(int status, Hashtable properties) throws IOException{
		String notifyURL = (String)properties.get(MidpConstants.KEY_INSTALL_NOTIFY);
		reportStatus(status, notifyURL);
	}
}
