package com.ibm.ive.midp.ams;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.util.*;
import java.io.*;

import javax.microedition.io.*;
import javax.microedition.pki.*;

import com.ibm.ive.midp.*;
import com.ibm.oti.connection.file.*;
import com.ibm.oti.security.midp.*;
import com.ibm.oti.security.provider.*;
import com.ibm.ive.midp.util.*;
import com.ibm.oti.vm.*;

class MidletVerifier {

	IInstallProcessListener fListener;

	MidletVerifier(IInstallProcessListener listener) {
		if (listener == null) throw new IllegalArgumentException();
		fListener = listener;
	}

	public Hashtable checkMidlet(MidletEntry entry, IInstallProcessListener installListener) throws AmsException {
		Hashtable jadProperties = null;
		String jadFile = entry.getJadFile();
		String entryName = entry.getName();

		// If a jad file was downloaded with this midlet then it could be signed
		if (jadFile != null) {
			jadProperties = JadParser.parse(jadFile);
			checkSignature(entry, jadProperties);
		} else {
			checkUntrustedMidletSuite(entry, null);
		}

		//Fix for Bug# 113852. Checking X509 certificate's Distinguished Name(DN)
		// of the new and the old entry for the updating of a signed midlet scenario
		MidletEntry oldEntry = MidletStorage.get(entryName, entry.getVendor());
		if (oldEntry != null) {
			String oldPrincipal = oldEntry.getKeyPrincipalName();
			if (oldPrincipal != null) {
				String newPrincipal = entry.getKeyPrincipalName();
				//installed midlet is signed.now check if the current one is signed
				if (newPrincipal == null){
					throw new AmsException(MidpConstants.RET_INVALID_PERMISSION, MidpMsg.getString("InstallProcess.checkForUpdate.error.notallowed"), (String)jadProperties.get(MidpConstants.KEY_INSTALL_NOTIFY)); //$NON-NLS-1$
				} else {
					//now check if the DN of the existing and the current midlets are the same
					if (!newPrincipal.equals(oldPrincipal)) {
						if (!installListener.isOkToUpdateDifferentPrincipal(entryName)) {
							throw new AmsException(MidpConstants.RET_SIGNATURE_PROBLEM, MidpMsg.getString("InstallProcess.checkTrustedMidletSuite.trusted_midlet_not_allowed"), (String)jadProperties.get(MidpConstants.KEY_INSTALL_NOTIFY)); //$NON-NLS-1$
						}
					}
				}
			}
		}

		Hashtable properties = checkJadAndManifest(entry, jadProperties);
		return properties;
	}

	Hashtable checkJadAndManifest(MidletEntry entry, Hashtable jadProperties) throws AmsException {
		String notifyURL = null;
		if (jadProperties != null) notifyURL = (String)jadProperties.get(MidpConstants.KEY_INSTALL_NOTIFY);

		byte[] manifestBytes = VM.manifestFromZip(entry.getJarFile());
		if (manifestBytes == null) throw new AmsException(MidpConstants.RET_INVALID_JAR_FILE, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.missing_manifest")); //$NON-NLS-1$

		if (jadProperties != null) {
			Hashtable manifestProperties = JadParser.parse(new ByteArrayInputStream(manifestBytes));
			if (notifyURL == null) notifyURL = (String)manifestProperties.get(MidpConstants.KEY_INSTALL_NOTIFY);

			// Some properties must be exactly the same in both the manifest
			// and the application descriptor for the midlet to be installed
			checkPresentAndEqual(MidpConstants.KEY_MIDLET_NAME, jadProperties, manifestProperties, notifyURL);
			checkPresentAndEqual(MidpConstants.KEY_MIDLET_VENDOR, jadProperties, manifestProperties, notifyURL);
			checkPresentAndEqual(MidpConstants.KEY_MIDLET_VERSION, jadProperties, manifestProperties, notifyURL);

			// If the midlet suite was signed then even more values have to
			// match exactly between the manifest and JAD.
			String principalName = entry.getKeyPrincipalName();
			if (principalName != null) {
				// If a midlet is signed then the values inside the manifest
				// must not be overridden by the values in the application descriptor
				checkNoOverride(jadProperties, manifestProperties, notifyURL);
			}
		}

		Hashtable properties = entry.getProperties();

		Enumeration keys = properties.keys();
		while (keys.hasMoreElements()) {
			String next = (String) keys.nextElement();
			if (next.startsWith("MIDlet-Push-")) throw new AmsException(MidpConstants.RET_ILLEGAL_ARGUMENT, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.push_tag"), notifyURL); //$NON-NLS-1$ //$NON-NLS-2$
		}

		if (notifyURL == null) notifyURL = (String)properties.get(MidpConstants.KEY_INSTALL_NOTIFY);
		String deleteNotifyURL = (String) properties.get(MidpConstants.KEY_DELETE_NOTIFY);
		if ((notifyURL != null && notifyURL.length() > 256) || (deleteNotifyURL != null && deleteNotifyURL.length() > 256)) {
			throw new AmsException(MidpConstants.RET_INVALID_DESCRIPTOR_FILE, MidpMsg.getString("InstallProcess.fetchJarFile.error.url_length_too_long"), notifyURL); //$NON-NLS-1$
		}
//		if (properties.get("MIDlet-1") == null) {	//$NON-NLS-1$
//			throw new AmsException(MidpConstants.RET_INVALID_DESCRIPTOR_FILE, MidpMsg.getString("InstallProcess.fetchJarFile.error.missing_required_key", "MIDlet-1"), notifyURL);	//$NON-NLS-1$
//		}

		/*
		 * We dont have to check this if the user has downloaded the JAD file as
		 * we have already done this check before downloading the JAR file
		 * Commenting out the check for now, as we aren't yet conforming to 6.2.5.1 of jsr 248 spec
		 */
		//if (entry.getJadURL() == null) {
		checkProfileAndConfiguration(properties, notifyURL);
		//}

		// If the midlet is signed then all permissions in the MIDlet-Permissions
		// key have to be known and allowed for this midlets protection domain
		if (entry.getKeyPrincipalName() != null) {
			String permissions = (String)properties.get(MidpConstants.KEY_MIDLET_PERMISSIONS);
			// Permissions are optional
			if (permissions != null) {
				checkPermissions(permissions, entry.getSecurityDomain(), notifyURL);
			}
		}

		return properties;
	}

	static void checkProfileAndConfiguration(Hashtable properties, String notifyURL) throws AmsException{
		String profile = (String)properties.get(MidpConstants.KEY_MIDLET_PROFILE);
		int profileLength = 0;
		if (profile != null && (profileLength = profile.trim().length()) > 0) {
			int ind = 0;
			boolean isImplMidp10 = System.getProperty("microedition.profiles").equals("MIDP-1.0"); //$NON-NLS-1$
			boolean isMidp10 = false;
			while (ind != -1) {
				int temp = ind;
				ind = profile.indexOf(' ', ++ind);
				String tempProfile = profile.substring(temp, (ind != -1) ? ind : profileLength).trim();
				//Check for invalid entries
				if (!tempProfile.equals("MIDP-1.0") && !tempProfile.equals("MIDP-2.0")) { //$NON-NLS-1$ //$NON-NLS-2$
					throw new AmsException(MidpConstants.RET_CONFIG_PROFILE_PROBLEM, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.invalid_config_profile"), notifyURL); //$NON-NLS-1$
				}
				if(!isMidp10) isMidp10 = tempProfile.equals("MIDP-1.0");
			}
			//Check for compatibility
			if (isImplMidp10 && !isMidp10) {
				throw new AmsException(MidpConstants.RET_CONFIG_PROFILE_PROBLEM, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.unsupported_config_profile"), notifyURL); //$NON-NLS-1$
			}
		} else {
			// The MIDP 2.0a TCK requires that the proper error for a missing
			// Configuration is "907 Invalid Jar"
			throw new AmsException(MidpConstants.RET_INVALID_JAR_FILE, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.missing_profile"), notifyURL); //$NON-NLS-1$
		}

		String configuration = (String)properties.get(MidpConstants.KEY_MIDLET_CONFIGURATION);
		if (configuration != null && configuration.trim().length() > 0) {
			String implConfig = System.getProperty("microedition.configuration");//$NON-NLS-1$
			//Check for invalid entries
			if (!configuration.equals("CLDC-1.0") && !configuration.equals("CLDC-1.1")) { //$NON-NLS-1$ //$NON-NLS-2$
				throw new AmsException(MidpConstants.RET_CONFIG_PROFILE_PROBLEM, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.invalid_config_profile"), notifyURL); //$NON-NLS-1$
			}
			//Check for compatibility
			if (implConfig.equals("CLDC-1.0") && !configuration.equals("CLDC-1.0")) { //$NON-NLS-1$ //$NON-NLS-2$
				throw new AmsException(MidpConstants.RET_CONFIG_PROFILE_PROBLEM, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.unsupported_config_profile"), notifyURL); //$NON-NLS-1$
			}
		} else {
			// The MIDP 2.0a TCK requires that the proper error for a missing
			// Configuration is "907 Invalid Jar"
			throw new AmsException(MidpConstants.RET_INVALID_JAR_FILE, MidpMsg.getString("InstallProcess.checkJadAndManifest.error.missing_configuration"), notifyURL);			 //$NON-NLS-1$
		}
	}

	/**
	 * Checks to make sure that all the permissions are allowed for this domain
	 *
	 * @param permissions the string of permissions to parse and check
	 * @param domainString the domain to verify the permissions against
	 * @param notifyURL the url to notify in case of an error
	 */
	void checkPermissions(String permissions, String domainString, String notifyURL) throws AmsException {
		boolean done = false;
		SecurityPolicy policy = SecurityPolicy.getInstance();
		ProtectionDomain domain = policy.getProtectionDomain(domainString);

		if (domain == null) throw new AmsException(MidpConstants.RET_INVALID_PERMISSION, MidpMsg.getString("InstallProcess.checkPermissions.error.unknown_domain")); //$NON-NLS-1$

		while (!done) {
			String permission;
			int index = permissions.indexOf(","); //$NON-NLS-1$
			if (index != -1) {
				permission = permissions.substring(0, index);
				permissions = permissions.substring(index+1);
			} else {
				done = true;
				permission = permissions;
			}

			permission = permission.trim();
			if (permission.length() > 0) {
				int highest = domain.getHighestPermissionLevel(permission);
				if (highest == ProtectionDomain.PERMISSION_LEVEL_DENY) {
					throw new AmsException(MidpConstants.RET_INVALID_PERMISSION, MidpMsg.getString("InstallProcess.checkPermissions.error.permission_not_granted", permission ), notifyURL); //$NON-NLS-1$
				}
			}
		}
	}

	/**
	 * Checks to make sure that the value for the specified key is present and the same in
	 * both the application descriptor and the mainfiest.
	 *
	 * @param key the key to check the value of
	 * @param jadProperties the application descriptors values
	 * @param manifestProperties the mainfiest's values
	 * @param notifyURL the url to notify in case of an error
	 * @throws AmsException if the value is either not present or the values to not match
	 */
	void checkPresentAndEqual(String key, Hashtable jadProperties, Hashtable manifestProperties, String notifyURL) throws AmsException {
		String jadValue = (String)jadProperties.get(key);
		if (jadValue == null) throw new AmsException(MidpConstants.RET_INVALID_DESCRIPTOR_FILE, MidpMsg.getString("InstallProcess.checkPresentAndEqual.missing_jad_attribute", key), notifyURL); //$NON-NLS-1$

		String manifestValue = (String)manifestProperties.get(key);
		if (manifestValue == null) throw new AmsException(MidpConstants.RET_INVALID_JAR_FILE, MidpMsg.getString("InstallProcess.checkPresentAndEqual.missing_manifest_attribute", key), notifyURL); //$NON-NLS-1$

		if (jadValue.equals(manifestValue) == false) {
			throw new AmsException(MidpConstants.RET_ATTRIBUTE_MISMATCH, MidpMsg.getString("InstallProcess.checkPresentAndEqual.mismatching_attribute", key), notifyURL); //$NON-NLS-1$
		}
	}

	/**
	 * Checks to make sure that no value in the manifiest is overridden by a value
	 * in the application descriptor.
	 *
	 * @param jadProperties the list of application descriptor values
	 * @param manifestProperties the list of manifest values
	 * @param notifyURL the url to notify in case of error
	 * @throws AmsException if one of the values is impropertly overriden
	 */
	void checkNoOverride(Hashtable jadProperties, Hashtable manifestProperties, String notifyURL) throws AmsException {
		Enumeration e = manifestProperties.keys();
		while (e.hasMoreElements()) {
			String key = (String)e.nextElement();
			String manifestValue = (String)manifestProperties.get(key);
			String jadValue = (String)jadProperties.get(key);

			// If the value is not defined in the application descriptor then
			// it is ok
			if (jadValue != null && !jadValue.equals(manifestValue)) {
				// If the values IS defined in the JAD then it is required
				// to be exactly the same as the manifest
				throw new AmsException(MidpConstants.RET_ATTRIBUTE_MISMATCH, MidpMsg.getString("InstallProcess.checkNoOverride.overriding_not_allowed", key), notifyURL); //$NON-NLS-1$
			}
		}
	}

	boolean checkSignature(MidletEntry entry, Hashtable jadProperties) throws AmsException {
		// According to the "Authenticating a MIDlet Suite" section in the
		// "Trusted MIDlet Suites using X.509 PKI" chapter.  If the
		// "MIDlet-Jar-RSA-SHA1" key in the jad file then the midlet MUST be authenticated
		if (jadProperties.get("MIDlet-Jar-RSA-SHA1") != null) { //$NON-NLS-1$
			return checkTrustedMidletSuite(entry, jadProperties);
		} else {
			return checkUntrustedMidletSuite(entry, jadProperties);
		}
	}

	boolean checkTrustedMidletSuite(MidletEntry entry, Hashtable jadProperties) throws AmsException {
		try {
			FileInputStream jarInputStream = new FileInputStream(entry.getJarFile());

			try {
				X500Principal root = AmsMidletSecurityOfficer.verify(jadProperties, jarInputStream);
				String domain = AmsMidletSecurityOfficer.getSecurityDomain(root);
				// Save the x509 principal so that later only midlets
				// from the exact same principal can update this midlet.  Also
				// having a principal is an easy way to tell if a midlet was
				// signed since even signed midlets can end up in the untrusted domain.
				entry.setKeyPrincipalName(root.getName());
				entry.setSecurityDomain(domain);
				return true;
			} catch (SecurityException e) {
				throw new AmsException(MidpConstants.RET_SIGNATURE_PROBLEM, MidpMsg.getString("InstallProcess.checkTrustedMidletSuite.error.SecurityException", e.getMessage())); //$NON-NLS-1$
			} catch (CertificateException e) {
				throw new AmsException(MidpConstants.RET_SIGNATURE_PROBLEM, MidpMsg.getString("InstallProcess.checkTrustedMidletSuite.error.CertificateException", e.getMessage())); //$NON-NLS-1$
			} catch (IOException e) {
				throw new AmsException(MidpConstants.RET_SIGNATURE_PROBLEM, MidpMsg.getString("InstallProcess.checkTrustedMidletSuite.error.IOException")); //$NON-NLS-1$
			} finally {
				// The file stream MUST always be closed or the file will stay around
				// even after the midlet has been deleted.
				try {
					jarInputStream.close();
				} catch (IOException e1) {
				}
			}
		} catch (ConnectionNotFoundException e) {
			throw new AmsException(MidpConstants.RET_SIGNATURE_PROBLEM, MidpMsg.getString("InstallProcess.checkTrustedMidletSuite.error.ConnectionNotFoundException")); //$NON-NLS-1$
		}
	}

	boolean checkUntrustedMidletSuite(MidletEntry entry, Hashtable jadProperties) throws AmsException {
		// The midlet suite is untrusted, and according to page 505 of the MIDP-NG 2.0 spec, section
		// 3.4, the implementation MUST prompt the user to inform them that an untrusted midlet suite
		// has been installed
		String installNotifyURL = null;
		if (jadProperties != null) installNotifyURL = (String) jadProperties.get(MidpConstants.KEY_INSTALL_NOTIFY);
		if (!fListener.isOkToInstallFromUntrustedSource(entry)) {
			throw new AmsException(MidpConstants.RET_OPERATION_CANCELLED, MidpMsg.getString("InstallProcess.checkUntrustedMidletSuite.untrusted_midlet_not_allowed"), installNotifyURL); //$NON-NLS-1$
		}
		return true;
	}

}
