package com.ibm.ive.midp.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import java.io.*;
import java.util.*;
import javax.microedition.io.*;
import com.ibm.ive.midp.*;
import com.ibm.ive.midp.ams.AmsException;
import com.ibm.ive.midp.ams.MidpConstants;
import com.ibm.oti.connection.file.*;

public class FileDownloader {

	// Changed the Buffer Size from 1024.
	static final int BUFFER_SIZE = 16384; //16K

	IFileDownloadListener fListener;

	public FileDownloader(IFileDownloadListener listener) {
		if (listener == null) throw new IllegalArgumentException();
		fListener = listener;
	}

	Connection openConnection(String url) throws AmsException {
		try {
			if (url.startsWith("file:")) { //$NON-NLS-1$
				// Since the JCL team removed the file connector, we will have to create the copied
				// file connector by hand to see if it will work.
				FileConnection fConnection = new FileConnection();
				// Remove the "file:" string from the beginning
				String fileSpec = url.substring(5);
				fConnection.setParameters2(fileSpec, Connector.READ, true);
				return fConnection;
			}
			return Connector.open(url, Connector.READ, true);
		} catch (IOException e) {
			e.printStackTrace();
			throw new AmsException(MidpConstants.RET_CONNECTION_FAILED, MidpMsg.getString("FileDownloader.openConnection.error.IOException", new String[] { url, e.getMessage()})); //$NON-NLS-1$
		}
	}

	public int downloadFile(String sourceURL, String destinationFilePath) throws AmsException {

		InputStream inputStream = null;
		int length = -1;
		try {
			Connection c = openConnection(sourceURL);

			if (c instanceof InputConnection) {
				if (c instanceof FileConnection) {
					//TODO - This code uses available() to get the length. This is most likely incorrect.
					inputStream = ((FileConnection) c).openInputStream();
					length = ((FileInputStream) inputStream).available();
				} else if (c instanceof ContentConnection) {
					if (c instanceof HttpConnection) {
						/*
						 * Loop while handle response code is a new connection because it
						 * could either be following a redirect, or opening a new connection
						 * with a login/password.
						 */
						Connection oldConnection;
						do {
							oldConnection = c;
							c = handleHttpResponseCode((HttpConnection) oldConnection);
						} while (oldConnection != c);
					}

					length = (int) ((ContentConnection) c).getLength();
					inputStream = ((InputConnection) c).openInputStream();
				}

				if (!hasEnoughSpace(length)) throw new AmsException(MidpConstants.RET_OUT_OF_MEMORY, MidpMsg.getString("FileDownloader.downloadFile.error.noSpace"), sourceURL); //$NON-NLS-1$
				copyStreamToFile(inputStream, destinationFilePath, length);
			}
		} catch (IOException e) {
			if (!fListener.isCancelled()) e.printStackTrace();
			throw new AmsException(MidpConstants.RET_COMM_FAILURE, MidpMsg.getString("FileDownloader.downloadFile.error.IOException", new String[] {sourceURL, e.getMessage()})); //$NON-NLS-1$
		} catch (IllegalArgumentException e) {
			if (!fListener.isCancelled()) e.printStackTrace();
			throw new AmsException(MidpConstants.RET_ILLEGAL_ARGUMENT, MidpMsg.getString("FileDownloader.downloadFile.error.IllegalArgumentException", sourceURL)); //$NON-NLS-1$
		} finally {
			if (inputStream != null) {
				try {
					inputStream.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return length;
	}

	void copyStreamToFile(InputStream input, String file, int length) throws AmsException {
		try {
			FileOutputStream fileOutputStream = new FileOutputStream(file, false);
			fListener.notifyDownloadProgress(0, length);

			byte[] buffer = new byte[BUFFER_SIZE];
			int count = 0;
			int totalBytes = 0;

			try {
				while ((count = input.read(buffer, 0, BUFFER_SIZE)) != -1 && !fListener.isCancelled()) {
					totalBytes += count;
					fListener.notifyDownloadProgress(totalBytes, length);
					fileOutputStream.write(buffer, 0, count);
				}
				fListener.notifyDownloadFinished(length);
			} catch (IOException e1) {
				throw new AmsException(MidpConstants.RET_IO_ERROR, MidpMsg.getString("FileDownloader.saveFile.error.IOException", e1.getMessage())); //$NON-NLS-1$
			} finally {
				try {
					fileOutputStream.close();
				} catch (IOException e2) {
					e2.printStackTrace();
				}
			}
		} catch (ConnectionNotFoundException e) {
			throw new AmsException(MidpConstants.RET_FILE_OPEN_FAILURE, MidpMsg.getString("FileDownloader.saveFile.error.ConnectionNotFoundException", e.getMessage())); //$NON-NLS-1$
		}

		if (fListener.isCancelled()) throw new AmsException(MidpConstants.RET_OPERATION_CANCELLED, MidpMsg.getString("FileDownloader.userCancelled")); //$NON-NLS-1$
	}

	Connection handleHttpResponseCode(HttpConnection httpConnection) throws AmsException {
		try {
			int responseCode = httpConnection.getResponseCode();
			switch (responseCode) {
				case HttpConnection.HTTP_OK :
				case HttpConnection.HTTP_CREATED :
				case HttpConnection.HTTP_NOT_AUTHORITATIVE :
				case HttpConnection.HTTP_RESET :
					return httpConnection;

				case HttpConnection.HTTP_UNAVAILABLE :
					throw new AmsException(MidpConstants.RET_GATEWAY_BUSY, MidpMsg.getString("FileDownloader.handleHttpResponseCode.error.busy_server")); //$NON-NLS-1$

				case HttpConnection.HTTP_PROXY_AUTH :
				case HttpConnection.HTTP_UNAUTHORIZED :
					return handleAuthentication(httpConnection);

				case HttpConnection.HTTP_ACCEPTED :
				case HttpConnection.HTTP_NO_CONTENT :
				case HttpConnection.HTTP_BAD_GATEWAY :
				case HttpConnection.HTTP_BAD_METHOD :
				case HttpConnection.HTTP_BAD_REQUEST :
				case HttpConnection.HTTP_CLIENT_TIMEOUT :
				case HttpConnection.HTTP_CONFLICT :
				case HttpConnection.HTTP_ENTITY_TOO_LARGE :
				case HttpConnection.HTTP_EXPECT_FAILED :
				case HttpConnection.HTTP_FORBIDDEN :
				case HttpConnection.HTTP_GATEWAY_TIMEOUT :
				case HttpConnection.HTTP_GONE :
				case HttpConnection.HTTP_INTERNAL_ERROR :
				case HttpConnection.HTTP_LENGTH_REQUIRED :
				case HttpConnection.HTTP_MULT_CHOICE :
				case HttpConnection.HTTP_NOT_ACCEPTABLE :
				case HttpConnection.HTTP_NOT_IMPLEMENTED :
				case HttpConnection.HTTP_NOT_MODIFIED :
				case HttpConnection.HTTP_PARTIAL :
				case HttpConnection.HTTP_PAYMENT_REQUIRED :
				case HttpConnection.HTTP_PRECON_FAILED :
				case HttpConnection.HTTP_REQ_TOO_LONG :
				case HttpConnection.HTTP_SEE_OTHER :
				case HttpConnection.HTTP_UNSUPPORTED_RANGE :
				case HttpConnection.HTTP_UNSUPPORTED_TYPE :
				case HttpConnection.HTTP_VERSION :
					throw new AmsException(MidpConstants.RET_CONNECTION_FAILED, MidpMsg.getString("FileDownloader.handleHttpResponseCode.error.connection_failed",responseCode)); //$NON-NLS-1$

				case HttpConnection.HTTP_NOT_FOUND :
					throw new AmsException(MidpConstants.RET_CONNECTION_FAILED, MidpMsg.getString("FileDownloader.handleHttpResponseCode.error.url_not_found")); //$NON-NLS-1$

				case HttpConnection.HTTP_MOVED_PERM :
				case HttpConnection.HTTP_MOVED_TEMP :
				case HttpConnection.HTTP_USE_PROXY :
				case HttpConnection.HTTP_TEMP_REDIRECT :
					//TODO: the headers for these error codes could be read and followed
					throw new AmsException(MidpConstants.RET_CONNECTION_FAILED, MidpMsg.getString("FileDownloader.handleHttpResponseCode.error.content_has_moved")); //$NON-NLS-1$

				default :
					throw new AmsException(MidpConstants.RET_INTERNAL_ERROR, MidpMsg.getString("FileDownloader.handleHttpResponseCode.error.unknown_HTTP_response_code",responseCode)); //$NON-NLS-1$
			}
		} catch (IOException e) {
			throw new AmsException(MidpConstants.RET_CONNECTION_FAILED, MidpMsg.getString("FileDownloader.handleHttpResponseCode.error.IOException", new String[] { httpConnection.getURL(), e.getMessage() })); //$NON-NLS-1$
		}
	}

	Connection handleAuthentication(HttpConnection connection) throws AmsException {
		try {
			String authHeader = connection.getHeaderField("WWW-Authenticate"); //$NON-NLS-1$
			int index = authHeader.indexOf(' ');
			String authenicationType = authHeader.substring(0, index).trim().toLowerCase();
			Hashtable authParams = parseAuthenticationParameters(authHeader.substring(index).trim());

			if (authenicationType.equals("basic")) { //$NON-NLS-1$
				return handleBasicAuthentication(connection, authParams);
			} else {
				throw new AmsException(MidpConstants.RET_UNKOWN_AUTHENTICATION_MODE, MidpMsg.getString("FileDownloader.handleAuthentication.error.unknown_authentication_mode", authenicationType)); //$NON-NLS-1$
			}
		} catch (IOException e) {
			throw new AmsException(MidpConstants.RET_COMM_FAILURE, MidpMsg.getString("FileDownloader.handleAuthentication.error.IOException", e.getMessage())); //$NON-NLS-1$
		}
	}

	Connection handleBasicAuthentication(HttpConnection connection, Hashtable params) throws AmsException {
		String realm = findAuthenicationRealm(params);
		String[] userNameAndPassword = fListener.getUserNameAndPassword(realm);
		if (userNameAndPassword == null || userNameAndPassword.length != 2 || userNameAndPassword[0] == null || userNameAndPassword[1] == null) {
			throw new AmsException(MidpConstants.RET_OPERATION_CANCELLED, MidpMsg.getString("FileDownloader.handleBasicAuthentication.invalid_username_password")); //$NON-NLS-1$
		}

		Connection newConnection = openConnection(connection.getURL());
		if (newConnection instanceof HttpConnection) {
			HttpConnection newHttpConnection = (HttpConnection) newConnection;
			String userpass = userNameAndPassword[0] + ':' + userNameAndPassword[1];

			try {
				newHttpConnection.setRequestProperty("Authorization", "Basic " + encodeBase64(userpass.getBytes())); //$NON-NLS-1$ //$NON-NLS-2$
				return newHttpConnection;
			} catch (IOException e) {
				try {
					newHttpConnection.close();
				} catch (IOException e1) {
				}
				throw new AmsException(MidpConstants.RET_COMM_FAILURE, MidpMsg.getString("FileDownloader.handleBasicAuthentication.error.failed_to_send_username_password")); //$NON-NLS-1$
			}
		}

		throw new AmsException(MidpConstants.RET_INTERNAL_ERROR, MidpMsg.getString("FileDownloader.handleBasicAuthentication.error.HTTP_connection_expected")); //$NON-NLS-1$
	}

	/**
	 * Encodes a set of bytes into a BASE64 encoded string according to
	 * RFC 1521 ( sec 5.2 ).  The only difference is that the line length
	 * restriction ( 76 characters ) is ignored.
	 *
	 * @param bytes the bytes to encode
	 * @return the encoded string
	 */
	String encodeBase64(byte[] bytes) {
		int triplesCount = bytes.length / 3;
		if (triplesCount * 3 < bytes.length) triplesCount++;
		StringBuffer buffer = new StringBuffer();

		for (int i = 0; i < triplesCount; i++) {
			byte b1 = bytes[i * 3];
			buffer.append(encodeByte((byte) ((b1 >> 2) & 0x3F)));
			byte e2 = (byte) ((b1 << 4) & 0x30);

			if (i * 3 + 1 < bytes.length) {
				byte b2 = bytes[i * 3 + 1];
				e2 |= (b2 >> 4) & 0x0F;
				buffer.append(encodeByte(e2));
				byte e3 = (byte) ((b2 << 2) & 0x3C);

				if (i * 3 + 2 < bytes.length) {
					byte b3 = bytes[i * 3 + 2];
					e3 |= (b3 >> 6) & 0x03;
					buffer.append(encodeByte(e3));
					buffer.append(encodeByte((byte) (b3 & 0x3F)));
				} else {
					buffer.append(encodeByte(e3));
					buffer.append('=');
				}
			} else {
				buffer.append(encodeByte(e2));
				buffer.append('=');
				buffer.append('=');
			}
		}

		return buffer.toString();
	}

	/**
	 * Performs a lookup of the byte into the table for base64
	 * values.  This table can be found in RFC 1521 sec 5.2
	 *
	 * @param b the byte to lookup
	 * @return the correct character value
	 */
	char encodeByte(byte b) {
		if (b <= 25 && b >= 0)
			return (char) (b + 'A');

		if (b <= 51 && b >= 26)
			return (char) (b - 26 + 'a');
		if (b <= 61 && b >= 52)
			return (char) (b - 52 + '0');

		switch (b) {
			case 62 :
				return '+';
			case 63 :
				return '/';
			default :
				throw new IllegalArgumentException();
		}
	}

	/**
	 * Parses a list of authentication parameters and returns the results
	 * in a hashtable.  The values for each of the parameters are stripped
	 * of the quotes.  The specification for parsing this string can be found
	 * in HTTP 1.1 spec.
	 *
	 * @param authParams a string contains all the authorization parameters
	 * @return a table containing key value pairs of all parameters
	 */
	Hashtable parseAuthenticationParameters(String authParams) {
		Hashtable params = new Hashtable();

		int commaIndex;
		while ((commaIndex = authParams.indexOf(',')) != -1) {
			String param = authParams.substring(0, commaIndex);
			addAuthParam(param, params);
			authParams = authParams.substring(commaIndex + 1);
		}

		addAuthParam(authParams, params);
		return params;
	}

	void addAuthParam(String param, Hashtable table) {
		int equalsIndex = param.indexOf('=');
		if (equalsIndex == -1) return;

		String key = param.substring(0, equalsIndex);

		// Strip off the quotes when doing the substring call
		String value = param.substring(equalsIndex + 2, param.length() - 1);

		table.put(key, value);
	}

	/**
	 * Returns the authentication realm from a table of the
	 * authenication parameters.
	 *
	 * @param params the authentication parameters
	 * @return the realm
	 */
	String findAuthenicationRealm(Hashtable params) {
		/*
		 * Since the realm parameter is case in-sensitive
		 * a simple lookup can not be used.  Each key has to
		 * be inspected.
		 */
		Enumeration e = params.keys();
		while (e.hasMoreElements()) {
			String key = (String) e.nextElement();
			if (key.toLowerCase().equals("realm")) return (String) params.get(key); //$NON-NLS-1$
		}
		return null;
	}

	boolean hasEnoughSpace(int bytesToDownload) {
		return bytesToDownload < fListener.getAvailableStorageSpace();
	}
}
