package com.ibm.ive.midp.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import java.io.*;
import java.util.*;
import com.ibm.oti.connection.file.*;
import com.ibm.ive.midp.*;

public class JadParser {

	static byte[] gBuffer;
	static int gOffset;
	static int gBufferLength;

	// If this flag is true then the manifest parse will be more flexible
	// to handle some poorly written jad files.  Hopefully leaving this flag turned on
	// will not affect TCK results.
	private static final boolean FLEXIBLE_PARSING = true;

	// the code that parses the jad and jar manifest REQUIRE
	// utf8, because we assume the lf, cr, and '\' are one byte only.
	private static final String ENCODING = "UTF8"; //$NON-NLS-1$

	static void parse(InputStream inputStream, Hashtable properties) {
		if (inputStream != null) {
			try {
				readProperties(inputStream, properties);
			} catch (IOException e) {
				System.err.println(MidpMsg.getString("JadParser.parse.InvalidJadFile", e.getMessage()));
			} finally {
				try {
					inputStream.close();
				} catch (Exception e) {
				}
			}
		}
	}

	public static Hashtable parse(InputStream inputStream) {
		if (inputStream == null) return null;

		Hashtable properties = new Hashtable(10);
		parse(inputStream, properties);

		return properties;
	}

	public static Hashtable parse(String jadFileName) {
		if (jadFileName == null) return null;

		InputStream jadFileStream = null;
		try {
			jadFileStream = new FileInputStream(jadFileName);
		} catch(IOException e) {
			e.printStackTrace();
			return null;
		}

		return parse(jadFileStream);
	}

	public static Hashtable parse(String jadFileName, InputStream manifestStream) {
		if (jadFileName == null) {
			if (manifestStream == null) return null;
			return parse(manifestStream);
		} else {
			InputStream jadFileStream = null;
			try {
				jadFileStream = new FileInputStream(jadFileName);
			} catch(IOException e) {
				e.printStackTrace();
				return null;
			}
			return parse(jadFileStream, manifestStream);
		}
	}

	public static Hashtable parse(InputStream jadFileStream, InputStream manifestFileStream) {
		if (jadFileStream == null && manifestFileStream == null) return null;
		Hashtable properties = new Hashtable(15);

		// jad takes precedence
		parse(manifestFileStream, properties); // so read in manifest values first
		parse(jadFileStream, properties); // then jad, so that common key values are overwritten by jad

		return properties;
	}

	static void readProperties(InputStream is, Hashtable properties) throws IOException {
		String ln;
		String linein;
		String key;
		String value;

		int pos = 0;
		gOffset = 0;
		gBufferLength = 0;
		gBuffer = new byte[512];

		//* Read/Parse the 'Key: Value' Pairs
		ln = readLine(is);
		for (;;) {
			if (ln == null) break;

			linein = ln;
			ln = readLine(is);
			while (ln != null && (ln.length() > 0) && (ln.charAt(0) == ' ')) {
				linein += ln.substring(1);
				ln = readLine(is);
			}

			key = linein;
			value = ""; //$NON-NLS-1$
			if ((pos = linein.indexOf(": ")) > 0) { //$NON-NLS-1$
				key = linein.substring(0, pos);
				if (key.length() < linein.length())
					value = (linein.substring(pos + 2)).trim();
			} else if (FLEXIBLE_PARSING) {
				// The line did not contain a ": " string so nothing from
				// that line will be parsed.  It is possible that the midlet
				// did an improper job of writing the manifest and forgot the space
				// after the colon.  Many midlets from www.midlet.org have this problem.
				// To be more flexible the first colon will be treated like it has a
				// space following it
				int firstColon = linein.indexOf(':');
				if (firstColon != -1) {
					key = linein.substring(0, firstColon);
					if (key.length() < linein.length())
						value = (linein.substring(firstColon + 1)).trim();
				}
			}

			if (key != null && (value.length() > 0)) {
				properties.put(key, value);
				// System.out.println("(M) "+key+"="+value);
			}
		}

		// Free the buffer so that 500 bytes are not sitting around being wasted
		gBuffer = null;
	}

	/*
	 * Read the data in the input stream and return the string upto the lf (0x0a) or
	 * crlf (0x0d0a). The file is assumed to be utf8 encoded, but may also contain
	 * one or more unicode escape sequences ('\'+'u'+4 digits 0-f).
	 */
	static String readLine(InputStream is) throws IOException {
		int ch;
		boolean done = false;
		StringBuffer sb = new StringBuffer(100);
		final int NONE = 0, SLASH = 1, UNICODE = 2;
		int mode = NONE;
		int unicode = 0;
		int count = 0;
		// where to start encoding the bytes
		int bufferStart = gOffset;

		while (!done) {
			while (gOffset < gBufferLength) {
				ch = gBuffer[gOffset];
				gOffset++;

				if (mode == UNICODE) {
					// read the 4 digits after the /u to make the single unicode char
					int digit = Character.digit((char) ch, 16);
					if (digit >= 0) {
						unicode = (unicode << 4) + digit;
						if (++count == 4) {
							// all 4 unicode chars read
							ch = unicode;
							mode = NONE;
							// now add the encoded char to the encoded string as long
							// as it is not a control char
							if (ch >= 0x20 && ch != 0x7f)
								sb.append((char) ch);
							// next encoding starts after the unicode
							bufferStart = gOffset;
						}
					}
				}
				if (mode == SLASH) {
					if (ch == 'u') {
						// start of unicode sequence
						count = 0;
						unicode = 0;
						mode = UNICODE;
					} else {
						// found some char other than 'u' after '\'
						// append first '\' found to the encoded string
						sb.append('\\');
						// reset the mode so this char is handled
						mode = NONE;
						// don't move the start of the buffer because it is
						// already pointing this char found after the '\' (otherwise
						// this char will be skipped.
						//					bufferStart = fOffset;
					}
				}
				if (mode == NONE) {
					if (ch == '\n' || ch == '\r' || ch == '\\') {
						// we have encountered a char that causes us to encode everything
						// read so far (up to but NOT including this char). By encoding now
						// we ensure that all utf8 bytes are within the buffer.
						int length = gOffset - 1 - bufferStart;
						if (length > 0) sb.append(new String(gBuffer, bufferStart, length, ENCODING));
						// end of the line - just go back
						if (ch == '\n') return sb.toString();
						// possible unicode mode
						if (ch == '\\') mode = SLASH;
						// next encoding starts after this char
						bufferStart = gOffset;
					} else if (
						((ch & 0xff) < 0x20 || (ch & 0xff) == 0x7f)
							&& (ch & 0xff) != 0x09) {
						// these control chars are not allowed
						StringBuffer hex = new StringBuffer(4);
						hex.append("0x"); //$NON-NLS-1$
						int data = ch & 0xff;
						if (data <= 0x0f) hex.append('0');
						hex.append(Integer.toHexString(data));
						throw new IOException(hex.toString());
					}
				}
			}

			int length = gOffset - bufferStart;
			int numRead = 0;
			if (length > 0) {
				// we have scanned through the buffer, but we haven't found the point
				// where we can encode the string yet.  If we encode now we could
				// potentially encode a partially read utf8 char.
				byte[] temp = new byte[gBuffer.length * 2];
				// copy what we have read into a new buffer twice a big as the current
				// one (if we have encoded some of the current buffer then that portion
				// can be thrown away now).
				System.arraycopy(gBuffer, bufferStart, temp, 0, length);
				// read some more storing the data after what has already been scanned
				numRead = is.read(temp, length, temp.length - length);
				// where the end of the bytes are now
				gBufferLength = length + numRead;
				// point our buffer to the newer bigger one
				gBuffer = temp;
				// pick scanning after the data that is already scanned
				gOffset = length;
			} else {
				// the data already read (or this could be the first read), is encoded so
				// it doesn't need to be saved
				numRead = is.read(gBuffer);
				gBufferLength = numRead;
				gOffset = 0;
			}
			// start encoding from the beginning of the buffer
			bufferStart = 0;
			if (numRead == -1) done = true;
		}

		// number of bytes that still need to be encoded
		int length = gOffset - bufferStart;
		if (length > 0) {
			// we never encountered a character to force out the last line so send it out now
			sb.append(new String(gBuffer, bufferStart, length, ENCODING));
			return sb.toString();
		} else {
			return null;
		}

		//	while (ch != -1 && ch != '\n')
		//	  {
		//		if (ch != '\r')
		//	       sb.append((char) ch);
		//	    ch=is.read();
		//	  }
		//	if (ch==-1)
		//	  return null;
		//	String s = sb.toString();
		//	// System.out.println(s);
		//	return s;
	}

	//private String unescapeUnicode(String text) {
	//
	//	final int NONE = 0, SLASH = 1, UNICODE = 2;
	//	int mode = NONE;
	//	int unicode = 0;
	//	int count = 0;
	//	StringBuffer sb = new StringBuffer(text.length());
	//
	//	for (int i=0; i < text.length(); i++) {
	//		char ch = text.charAt(i);
	//
	//		if (mode == UNICODE) {
	//			// read the 4 digits after the /u to make the single unicode char
	//			int digit = Character.digit((char)ch, 16);
	//			if (digit >= 0) {
	//				unicode = (unicode << 4) + digit;
	//				if (++count == 4) {
	//					// all 4 unicode chars read
	//					ch = (char) unicode;
	//					mode = NONE;
	//				}
	//			}
	//		}
	//
	//		if (mode == SLASH) {
	//			if (ch == 'u') {
	//				// start of unicode sequence
	//				count = 0;
	//				unicode = 0;
	//				mode = UNICODE;
	//			} else {
	//				// found some char other than 'u' after '\'
	//				// append first '\' found
	//				sb.append('\\');
	//				// reset the mode so this char is handled
	//				mode = NONE;
	//			}
	//		}
	//		if (mode == NONE) {
	//			switch (ch) {
	//				case '\\': mode = SLASH; break;
	//				default: sb.append((char)ch); break;
	//			}
	//		}
	//	}
	//
	//	return sb.toString();
	//
	//}
}
