package com.ibm.ive.midp.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.*;
import java.util.*;
import javax.microedition.lcdui.*;
import javax.microedition.midlet.*;
import com.ibm.ive.midp.*;
import com.ibm.oti.midlet.help.*;
import com.ibm.oti.security.midp.*;
import javax.microedition.io.*;
import com.ibm.oti.connection.file.*;
import com.ibm.oti.vm.*;

public class MIDletManager {

	public static final int MIDLET_ACTIVE = 0;
	public static final int MIDLET_PAUSED = 1;
	public static final int MIDLET_DESTROYED = 2;

	private static MIDlet[] gMidlets = new MIDlet[10];
	private static Display[] gDisplays = new Display[10];
	private static MIDlet[] gMidletsByIndex = new MIDlet[10];
	private static IMIDletAccessor[] gAccessors = new IMIDletAccessor[10];

	public static IAppManager gAppManager;
	public static Hashtable gProperties;
	private static int gMidpVersion = 0;
	public static String gMidletSuiteName;
	public static boolean gReLaunchMidlet;
	public static String gVendorName;
	private static boolean gUntrusted = false; // set to true if domain is "untrusted"
	private static final String UNTRUSTED_SECURITY_DOMAIN = "untrusted"; //$NON-NLS-1$
	public static final String MIDP_20_STRING = "MIDP-2.0"; //$NON-NLS-1$
	public static Object gLock = new Object();
	private static int gMIDletsToBeAdded = 0;

	public static void initialize(IAppManager appManager, String jad, String jar, String midletName) {
		gAppManager = appManager;

		InputStream manifest = null;
		InputStream jadFileStream = null;

		// get JAD input stream
		if (jad != null) {
			try {
				jadFileStream = new FileInputStream(jad);
			} catch (ConnectionNotFoundException e) {
				// do nothing
			}
			if (jadFileStream == null) jadFileStream = MIDletManager.class.getResourceAsStream("/" + jad);
		}
		if (jadFileStream == null) jadFileStream = MIDletManager.class.getResourceAsStream('/' + getJadFileName());

		// get MANIFEST input stream
		if (jar != null) {
			byte[] fileBytes = VM.manifestFromZip(jar);
			if (fileBytes != null) manifest = new ByteArrayInputStream(fileBytes);
		} else {
			manifest = VM.getManifest();
		}

		gProperties = JadParser.parse(jadFileStream, manifest);
		if(!validateProperties(gProperties)) {
			initializeWithNoProperties(midletName);
			return;
		}

		gMidletSuiteName = (String)gProperties.get("MIDlet-Name"); //$NON-NLS-1$
		gVendorName = (String)gProperties.get("MIDlet-Vendor"); //$NON-NLS-1$
		MidletLoader.init(gMidletSuiteName, gVendorName);
	}

	public static MIDlet getMIDlet(int index) {
		if (index < gMidletsByIndex.length) return gMidletsByIndex[index];
		return null;
	}

	/*
	 * When adding a midlet, we only need to add it to the index-based array.
	 * The reason for this is that it's already been added to the matching
	 * midlet/accessor arrays when the accessor was created.
	 *
	 * This method must be called after the accessor creation because the
	 * accessor does not know it's midlet index at creation time.
	 */
	public static void addMIDlet(MIDlet midlet, int index) {
		synchronized (gMidlets) {
			if (index >= gMidletsByIndex.length) {
				MIDlet[] newArray = new MIDlet[index + 1];
				System.arraycopy(gMidletsByIndex, 0, newArray, 0, gMidletsByIndex.length);
				gMidletsByIndex = newArray;
			}
			gMidletsByIndex[index] = midlet;
			gMIDletsToBeAdded--;
		}
	}

	public static void launchingMIDlet() {
		gMIDletsToBeAdded++;
	}

	public static void cancelLaunchingMIDlet() {
		gMIDletsToBeAdded--;
	}

	/*
	 * Rather than have separate methods for removing a midlet and its accessor,
	 * we provide just one to ensure that the arrays are in sync.
	 */
	public static void deleteMIDlet(MIDlet midlet) {
		for (int i = 0; i < gMidlets.length; i++) {
			if (gMidlets[i] == midlet) {
				int index = gAccessors[i].getMidletIndex();
				gMidlets[i] = null;
				gDisplays[i] = null;
				gAccessors[i] = null;
				gMidletsByIndex[index] = null;
				return;
			}
		}
	}

	public static void addAccessor(MIDlet midlet, IMIDletAccessor accessor) {
		accessor.setAppManager(gAppManager);

		for (int i = 0; i < gMidlets.length; i++) {
			if (gMidlets[i] == null) {
				gMidlets[i] = midlet;
				gAccessors[i] = accessor;
				return;
			}
		}

		/* there was not enough space in the array */
		int index = gMidlets.length;
		MIDlet[] newMIDletArray = new MIDlet[gMidlets.length + 10];
		Display[] newDisplayArray = new Display[newMIDletArray.length];
		IMIDletAccessor[] newAccessorsArray = new IMIDletAccessor[newMIDletArray.length];
		System.arraycopy(gMidlets, 0, newMIDletArray, 0, gMidlets.length);
		System.arraycopy(gDisplays, 0, newDisplayArray, 0, gDisplays.length);
		System.arraycopy(gAccessors, 0, newAccessorsArray, 0, gAccessors.length);
		gMidlets = newMIDletArray;
		gDisplays = newDisplayArray;
		gAccessors = newAccessorsArray;
		gMidlets[index] = midlet;
		gAccessors[index] = accessor;
	}

	public static IMIDletAccessor getAccessor(MIDlet midlet) {
		for (int i = 0; i < gMidlets.length; i++) {
			if (gMidlets[i] == midlet) return gAccessors[i];
		}
		return null;
	}

	public static String[] getMIDletNames() {
		int midletNum = 1;

		FastVector midletNames = new FastVector();
		String midletName = (String) gProperties.get("MIDlet-" + midletNum); //$NON-NLS-1$
		while (midletName != null) {
			midletName = midletName.substring(0, midletName.indexOf(','));
			midletNames.addElement(midletName);
			midletNum++;
			midletName = (String) gProperties.get("MIDlet-" + midletNum); //$NON-NLS-1$
		}

		String[] returnValue = new String[midletNames.size()];
		midletNames.copyInto(returnValue);
		return returnValue;
	}

	public static String[] getMIDletIconNames() {
		int midletNum = 1;

		FastVector midletIconNames = new FastVector();
		String midletEntry = (String) gProperties.get("MIDlet-" + midletNum); //$NON-NLS-1$
		while (midletEntry != null) {
			int index1 = midletEntry.indexOf(',') + 1;
			int index2 = midletEntry.indexOf(',', index1);
			midletEntry = midletEntry.substring(index1, index2);
			midletEntry = midletEntry.trim();
			if (midletEntry.length() == 0) midletEntry = null;
			midletIconNames.addElement(midletEntry);
			midletNum++;
			midletEntry = (String) gProperties.get("MIDlet-" + midletNum); //$NON-NLS-1$
		}

		String[] returnValue = new String[midletIconNames.size()];
		midletIconNames.copyInto(returnValue);
		return returnValue;
	}

	private static void tokenizeProperty(String value, Vector tokens) {
		int pos = 0;
		int length = value.length();
		while (pos < length) {
			while (value.charAt(pos) == ' ' || value.charAt(pos) == '\t') pos++;
			int endPos = value.indexOf(',', pos + 1);
			if (endPos == -1) endPos = length;
			while (value.charAt(endPos - 1) == ' ' || value.charAt(endPos - 1) == '\t') endPos--;
			String token = value.substring(pos, endPos);
			tokens.addElement(token);
			pos = endPos + 1;
		}
	}

	private static String[] parseJadPermissions() {
		Vector permissions = new Vector();
		String value = (String)gProperties.get("MIDlet-Permissions"); //$NON-NLS-1$
		if (value != null && value.length() != 0) tokenizeProperty(value, permissions);

		value = (String)gProperties.get("MIDlet-Permissions-Opt"); //$NON-NLS-1$
		if (value != null && value.length() != 0) tokenizeProperty(value, permissions);

		String[] permStr = new String[permissions.size()];
		permissions.copyInto(permStr);
		return permStr;
	}

	public static String getSecurityDomain() {
		return getSecurityDomain(gMidletSuiteName);
	}

	public static String getSecurityDomain(String midletName) {
		String protectionDomain = null;

		if (midletName == null) return UNTRUSTED_SECURITY_DOMAIN;

		/*
		 * Check for existing metadata about this app/MIDlet
		 */
		MidletLoader.Metadata metadata = MidletLoader.getMetadataClass(null, "AMS"); //$NON-NLS-1$
		if (metadata == null) return UNTRUSTED_SECURITY_DOMAIN;

		protectionDomain = metadata.getEntry("domain"); //$NON-NLS-1$
		if (protectionDomain != null) return protectionDomain;

		/*
		 * If no protection domain metadata exists,
		 * look for a domain in the .prc and then create metadata.
		 */
		protectionDomain = System.getProperty("jad2prc.verify.protectiondomain"); //$NON-NLS-1$
		if (protectionDomain != null) {
			metadata.setEntry("domain", protectionDomain); //$NON-NLS-1$
			try {
				metadata.save();
			} catch(IOException e) {
				System.err.println("Save error: " + e.getMessage()); //$NON-NLS-1$
			}
			return protectionDomain;
		}

		return UNTRUSTED_SECURITY_DOMAIN;
	}

	public static void setSecurity(UserPermissionAgent agent) {
		setSecurity(gMidletSuiteName, agent);
	}

	public static void setSecurity(String midletName, UserPermissionAgent userAgent) {
		// parse the permissions from the JAD's MIDlet-Permission and
		// MIDlet-Permission-Opt attributes
		String[] jadPermissions = parseJadPermissions();

		// the permissionDomain is passed on the command line from the AMS
		if ("AMSMidlet".equals(midletName)) return; // Workaround for JCL bug //$NON-NLS-1$

		String domain = getSecurityDomain(midletName);

		gUntrusted = UNTRUSTED_SECURITY_DOMAIN.equals(domain);

		// set the manager
		PermissionManager.setManager(new PermissionManager(midletName, jadPermissions, domain, userAgent));
	}

	public static boolean isUntrusted() {
		return gUntrusted;
	}

	public static String getJadFileName() {
		String jadFile = "examples.jad"; //$NON-NLS-1$
		try {
			InputStream is = jadFile.getClass().getResourceAsStream("/META-INF/JXE.MF"); //$NON-NLS-1$

			/* Don't do anything if unable to load JXE.MF */
			if (is != null) {
				StringBuffer jxeName = getJxeName(is);
				if (jxeName != null) {
					String newJadFile = jxeName.append(".jad").toString(); //$NON-NLS-1$

					/*
					 * Checks whether the jadFile exists.
					 * If not, will use examples.jad
					 */
					is = newJadFile.getClass().getResourceAsStream("/" + newJadFile); //$NON-NLS-1$
					if (is != null) jadFile = newJadFile;
				}
			}
		} catch(Exception e) {
		}
		return jadFile;
	}

	/**
	 * Looks for the jxeName entry in <code>is</code>.
	 * @return StringBuffer the name of the JXE if found, null otherwise.
	 */
	private static StringBuffer getJxeName(InputStream is)
		throws IOException {
		if (lookFor("jxeName ", is)) { //$NON-NLS-1$
			StringBuffer name = new StringBuffer();
			char c = (char) is.read();

			while (c != -1 && c != '\r' && c != '\n') {
				name.append(c);
				c = (char) is.read();
			}

			/* Delete the trailing space */
			int endPosition = name.length() - 1;
			if (name.charAt(endPosition) <= ' ') {
				name.deleteCharAt(endPosition);
			}
			return name;
		}
		return null;
	}

	/**
	 * Looks for <code>text</code> in <code>is</code>.
	 * @return true if <code>text</code>, false otherwise.
	 */
	private static boolean lookFor(String text, InputStream is) throws IOException {
		boolean found = false;
		while (!found) {
			int i = 0, max = text.length();
			for (i = 0; i < max; i++) {
				char c = (char) is.read();
				if (c == -1) return found;
				if (c != text.charAt(i)) break;
			}
			found = (i == max - 1);
		}
		return found;
	}

	public static boolean isMidp10() {
		return getMidpVersion() == 1;
	}

	/*
	 * The Display needs the MIDP version so that it can
	 * do the right thing for Displayables from 1.0 which
	 * do not respond to sizeChanged().
	 */
	public static int getMidpVersion() {
		if (gMidpVersion == 0) {
			String version = (String) gProperties.get("MicroEdition-Profile"); //$NON-NLS-1$
			/*
			 * gMidpVersion set to 1 unless MicroEdition-Profile
			 * value contains MIDP-2.0
			 */
			gMidpVersion = 1;
			if (version != null) {
				int length = (version = version.trim()).length();
				int index = 0;
				/*
				 * MicroEdition-Profile can have more than one profile specified with
				 * space in between them. So, something like
				 * MicroEdition-Profile: MIDP-1.0 MIDP-2.0  is valid (one of the
				 * Security TCK case does so) and the MIDlet
				 * in this case should be treated as a MIDP 2.0 MIDlet
				 */
				while (index != -1) {
					int prevIndex = index;
					index = version.indexOf(' ', index + 1);
					if (version.substring(prevIndex, index == -1 ? length : index).trim().equals(MIDP_20_STRING)) {
						gMidpVersion = 2;
						break;
					}
				}
			}
		}
		return gMidpVersion;
	}

	public static String getClassName(int num) {
		String s = (String) gProperties.get("MIDlet-" + num); //$NON-NLS-1$
		if (s == null) return null;
		return s.substring(s.lastIndexOf(',') + 1).trim();
	}

	public static Display getDisplay(MIDlet midlet) {
		if (midlet == null) throw new IllegalArgumentException();
		for (int i = 0; i < gMidlets.length; i++) {
			if (gMidlets[i] == midlet) return gDisplays[i];
		}
		return null;
	}

	public static void registerDisplay(MIDlet midlet, Display display) {
		if (midlet == null) throw new IllegalArgumentException();
		for (int i = 0; i < gMidlets.length; i++) {
			if (gMidlets[i] == midlet) gDisplays[i] = display;
		}
	}

	public static int getMIDletCount() {
		int count = 0;

		synchronized (gMidlets) {
			for (int i = 0; i < gMidlets.length; i++) {
				if (gMidlets[i] != null) count++;
			}
			count += gMIDletsToBeAdded;
		}

		return count;
	}

	public static MIDlet getMIDlet(Display display) {
		for (int i = 0; i < gDisplays.length; i++) {
			if (gDisplays[i] == display) return gMidlets[i];
		}
		return null;
	}

	private static boolean validateProperties(Hashtable properties) {
		if (properties == null) return false;
		if (properties.get("MIDlet-Name") == null) return false;
		if (properties.get("MIDlet-Vendor") == null) return false;
		return true;
	}

	private static void initializeWithNoProperties(String midletName) {
		if (midletName == null) throw new NullPointerException();
		gProperties = new Hashtable();
		gProperties.put("MIDlet-Name", midletName);
		gProperties.put("MIDlet-1", midletName);
		gProperties.put("MIDlet-Vendor", "Unknown");  //$NON-NLS-1$
	}

	public static int getIndexFromClassName(String midletClass) {
		int index = 1;
		while (true) {
			String s = (String) gProperties.get("MIDlet-" + index); //$NON-NLS-1$
			if (s == null) return -1;

			if (s.substring(s.lastIndexOf(',') + 1).trim().equals(midletClass)) return index;
			index++;
		}
	}
}
