package com.ibm.microedition.media;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import java.util.Vector;
import javax.microedition.media.control.ToneControl;

/**
 * compiles the tone sequence into midi track events
 *
 */
public class MIDICompiler {

	private byte[]  fSequence;
//	private int     fMidiChannel;
	private int     fTempo;
	private int     fResolution;
	private int     fVolume;
	private int     fMsPerDuration;
	private int[][] fBlockDefs;
//	private int     fProgram;

	public MIDICompiler() {
		super();
	}

	/**
	 * Construct a new instance of this class.
	 */
	private MIDICompiler(byte[] sequence, int midiChannel, int program) {
		super();
		fSequence = sequence;
//		fMidiChannel = midiChannel;
		fTempo = 30;
		fResolution = 64;
		fVolume = 100;
		fBlockDefs = new int[128][];
//		fProgram = program;
	}

	/**
	 * Compile a tone sequence into a set of duration / midievents.
	 */
	static public int[] compileToMidi(byte[] sequence, int midiChannel, int program) {
		if (-1 == program) program = 1;

		MIDICompiler mc = new MIDICompiler(sequence, midiChannel, program);
		return mc.getMidiSequence();
	}

	/**
	 * Add midi events for a single note.
	 */
	private void playNote(Vector v, int note, int duration, int volume) {
		int[] deltaTicks = new int[4];
		int count = 0;

		/* actual duration is given by duration * fMsPerDuration
		   see the MIDI spec for calculating the duration */

		duration *= fMsPerDuration;

		/* write the duration in variable length */
		if (duration > 0x7f) {
			deltaTicks[count] = (duration & 0x7f);
			duration >>= 7;
			while (duration != 0){
				count++;
				deltaTicks[count] = (duration & 0x7f) | 0x80;
				duration >>= 7;
			}
		} else {
			deltaTicks[count] = duration;
		}

		/* for silence events, use a note off instead*/
		if (note == ToneControl.SILENCE) {
			note   = ToneControl.C4;
			volume = 0;
		}

		/*note on */
		v.addElement(new Integer(0));
		v.addElement(new Integer(144));		// 144 <=> 0x90
		v.addElement(new Integer(note));
		v.addElement(new Integer(volume));

		/* filling the variable length duration */
		for (int i = count; i >= 0; i--){
			v.addElement(new Integer(deltaTicks[i] & 0x000000ff));
		}

		/* note off -> volume is 0; remember, the 'tick' value == milliseconds */
		v.addElement(new Integer(128));		// 128 <=> 0x80
		v.addElement(new Integer(note));
		v.addElement(new Integer(volume));
	}

	/**
	 * Play a block.
	 */
	private void playBlock(Vector v, int blockNumber) {
		int[] blockDef = fBlockDefs[blockNumber];
		if (null == blockDef) {
			throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
		}
		playSequence(v,blockDef[0],blockDef[1]);
	}

	/**
	 * Play a sequence from pc up to but not including stop.
	 */
	private void playSequence(Vector v, int pc, int stop) {
		while (pc < stop) {
			int note;
			int duration;
			int volume = (fVolume * 127) / 100;
			switch (fSequence[pc]) {
				/*play a block*/
				case ToneControl.PLAY_BLOCK:
					pc++;
					int blockNumber = fSequence[pc++];
					if (blockNumber < 0 || blockNumber > 127) {
						throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
					}
					playBlock(v,blockNumber);
					break;
				/*set the volume*/
				case ToneControl.SET_VOLUME:
					pc++;
					fVolume = fSequence[pc++];
					volume = (fVolume * 127) / 100;
					if (fVolume < 0 || fVolume > 100) {
						throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg4"));
					}
					break;
				/*play a repeated note*/
				case ToneControl.REPEAT:
					pc++;
					int repeatCount = fSequence[pc++];
					note = fSequence[pc++];
					duration = fSequence[pc++];
					if (repeatCount < 2 || repeatCount > 127) {
						throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg7"));
					}
					if (note < ToneControl.SILENCE || note > 127) {
						throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg6"));
					}
					if (duration < 1 || duration > 127) {
						throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg5"));
					}
					for (int i = 0; i < repeatCount; i++) {
						playNote(v, note, duration, volume);
					}
					break;
				/*play a single note*/
				default:
					note = fSequence[pc++];
					duration = fSequence[pc++];
					if (note < ToneControl.SILENCE || note > 127) {
						throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg6"));
					}
					if (duration < 1 || duration > 127) {
						throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg5"));
					}
					playNote(v, note, duration, volume);
			}
		}
	}

	/**
	 * Compile a sequence into a set of midi events.
	 */
	private void compile(Vector v) {
		int pc = 0;

		/* ensure VERSION tag is there*/
		if (fSequence[pc++] != ToneControl.VERSION) {
			throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg1"));
		}
		/*ensure a VERSION tag == 1*/
		if (fSequence[pc++] != 1) {
			throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg1"));
		}
		/*handle a TEMPO tag*/
		if (fSequence[pc] == ToneControl.TEMPO) {
			pc++;
			fTempo = fSequence[pc++];
			if (fTempo < 5 || fTempo > 127) {
				throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg8"));
			}

			/* converting from bpm to usec per quarter note
			 * which is 1/bpm <=> 60/(tempo * 1usec)
			 * */
			int tttttt = (60 * 1000000)/(fTempo * 4);

			v.addElement(new Integer(255)); 	//midi meta-event
			v.addElement(new Integer(81));		// set tempo
			v.addElement(new Integer(3));		// 3 bytes
			v.addElement(new Integer((tttttt >> 16)  & 0x000000ff));		// T0:DO - translate fTempo to tt tt tt
			v.addElement(new Integer((tttttt >> 8)  & 0x000000ff));
			v.addElement(new Integer(tttttt & 0x000000ff));
			/* cma */
		}

		// handle a RESOLUTION tag
		if (fSequence[pc] == ToneControl.RESOLUTION) {
			pc++;
			fResolution = fSequence[pc++];
			if (fResolution < 1 || fResolution > 127) {
				throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg9"));
			}
		}

		/*calculate milliseconds per duration
		 *need this to set 'ticks' to milliseconds
		 *see the MIDP spec for ToneControl for an explanation,
		 *but realize the tempo value in their doc is the
		 *post-multipied volume,  and ours is the
		 *pre-multiplied version, so we need to multiple
		 *our tempo by 4, where they do not.
		 *
		 */
		fMsPerDuration = (60 * 1000 * 4) / (fResolution * fTempo * 4);

		/*handle block definitions*/
		while (fSequence[pc] == ToneControl.BLOCK_START) {
			pc++;
			int blockNumber = fSequence[pc++];
			if (blockNumber < 0 || blockNumber > 127) {
				throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
			}
			if (null != fBlockDefs[blockNumber]) {
				throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
			}
			/*start saving our block definition*/
			fBlockDefs[blockNumber] = new int[2];
			fBlockDefs[blockNumber][0] = pc;

			while (fSequence[pc] != ToneControl.BLOCK_END) {
				switch (fSequence[pc]) {
					case ToneControl.PLAY_BLOCK:
						pc++;
						int anotherBlockNumber = fSequence[pc++];
						/*special check to make sure any blocks used in this block are
						 *valid and already defined
						 */
						if (anotherBlockNumber < 0 || anotherBlockNumber > 127) {
							throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
						}
						if (null == fBlockDefs[anotherBlockNumber]) {
							throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
						}
						break;

					case ToneControl.SET_VOLUME:
						pc += 2; break;
					case ToneControl.REPEAT:
						pc += 4; break;
					default:
						 pc += 2; break;
				}
			}

			/*save the 'stop' value for the block*/
			fBlockDefs[blockNumber][1] = pc++;

			if (fSequence[pc] != blockNumber) {
				throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
			}
			if (fBlockDefs[blockNumber][1] == fBlockDefs[blockNumber][0]) {
				throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg3"));
			}
			pc++;
		}
		/*the rest is a sequence*/
		playSequence(v, pc, fSequence.length);
	}

	/**
	 * Return the resultant midi events.
	 */
	private int[] getMidiSequence() {
		Vector v = new Vector();
		try {
			compile(v);
		}
		catch (ArrayIndexOutOfBoundsException e) {
			throw new IllegalArgumentException(MediaPlayer.getTranslatedString("MIDICompiler.msg1")); //$NON-NLS-1$
		}
		int[] result = new int[v.size()];
		for (int i = 0; i < result.length; i++) {
			result[i] = ((Integer) v.elementAt(i)).intValue();
		}
		return result;
	}
}
