package com.ibm.microedition.media;

/*
 *
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import java.io.*;
import java.util.*;
import com.ibm.oti.vm.*;

class MediaMessages {

	// Properties holding the system messages.
	static private Hashtable messages = null;

	/**
	 * Retrieves a message which has no arguments.
	 *
	 * @author		OTI
	 * @version		initial
	 *
	 * @param		msg String
	 *					the key to look up.
	 * @return		String
	 *					the message for that key in the system
	 *					message bundle.
	 */
	static public String getString(String msg) {
		if (messages == null) {
			// Attempt to load the messages.
			try {
				messages = loadMessages("com/ibm/microedition/media/ExternalMessages"); //$NON-NLS-1$
			} catch (IOException e) {
				e.printStackTrace();
			}
			if (messages == null) return msg;
		}

		String resource = (String) messages.get(msg);
		if (resource == null) return msg;
		return resource;
	}

	/** Loads properties from the specified resource. The properties are of
	 *  the form <code>key=value</code>, one property per line.
	 */

	static private Hashtable loadMessages(String resourceName) throws IOException {
		String resName;
		InputStream resourceStream;

		String language = System.getProperty("user.language"); //$NON-NLS-1$
		if (language == null) language = "en"; //$NON-NLS-1$
		String region = System.getProperty("user.region"); //$NON-NLS-1$
		if (region == null) region = "US"; //$NON-NLS-1$
		String variant = System.getProperty("user.variant"); //$NON-NLS-1$

		if (variant != null) {
			resName = resourceName + "_" + language + "_" + region + "_" + variant + ".properties"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			resourceStream = VM.getResourceAsStream(resName, false);
			if (resourceStream != null) {
				return load(resourceStream);
			}
		}

		resName = resourceName + "_" + language + "_" + region + ".properties"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		resourceStream = VM.getResourceAsStream(resName, false);
		if (resourceStream != null) {
			return load(resourceStream);
		}

		resName = resourceName + "_" + language + ".properties"; //$NON-NLS-1$ //$NON-NLS-2$
		resourceStream = VM.getResourceAsStream(resName, false);
		if (resourceStream != null) {
			return load(resourceStream);
		}

		resourceStream = VM.getResourceAsStream(resourceName + ".properties", false);
		if (resourceStream != null) {
			return load(resourceStream);
		}

		return null;
	}

	static private Hashtable load(InputStream resourceStream) throws IOException {
		final int NONE = 0, SLASH = 1, UNICODE = 2, CONTINUE = 3, DONE = 4, IGNORE = 5;

		Hashtable messages = new Hashtable();
		int mode = NONE, unicode = 0, count = 0, nextChar;
		char[] buf = new char[40];
		int offset = 0, keyLength = -1;
		boolean firstChar = true;

		while ((nextChar = resourceStream.read()) != -1) {
			if (offset == buf.length) {
				char[] newBuf = new char[buf.length * 2];
				System.arraycopy(buf, 0, newBuf, 0, offset);
				buf = newBuf;
			}
			if (mode == UNICODE) {
				int digit = Character.digit((char) nextChar, 16);
				if (digit >= 0) {
					unicode = (unicode << 4) + digit;
					if (++count < 4) continue;
				}
				mode = NONE;
				buf[offset++] = (char) unicode;
				if (nextChar != '\n')
					continue;
			}
			if (mode == SLASH) {
				mode = NONE;
				switch (nextChar) {
					case '\r' :
						mode = CONTINUE; // Look for a following \n
						continue;
					case '\n' :
						mode = IGNORE; // Ignore whitespace on the next line
						continue;
					case 'b' :
						nextChar = '\b';
						break;
					case 'f' :
						nextChar = '\f';
						break;
					case 'n' :
						nextChar = '\n';
						break;
					case 'r' :
						nextChar = '\r';
						break;
					case 't' :
						nextChar = '\t';
						break;
					case 'u' :
						mode = UNICODE;
						unicode = count = 0;
						continue;
				}
			} else {
				switch (nextChar) {
					case '#' :
					case '!' :
						if (firstChar) {
							while ((nextChar = resourceStream.read()) != -1) {
								if (nextChar == '\r' || nextChar == '\n') break;
							}
							continue;
						}
						break;
					case '\n' :
						if (mode == CONTINUE) { // Part of a \r\n sequence
							mode = IGNORE; // Ignore whitespace on the next line
							continue;
						}
						// fall into the next case
					case '\r' :
						mode = NONE;
						firstChar = true;
						if (keyLength >= 0) {
							String temp = new String(buf, 0, offset);
							messages.put(temp.substring(0, keyLength), temp.substring(keyLength));
						}
						keyLength = -1;
						offset = 0;
						continue;
					case '\\' :
						mode = SLASH;
						continue;
					case ':' :
					case '=' :
						if (keyLength == -1) { // if parsing the key
							keyLength = offset;
							continue;
						}
						break;
				}
				if (isWhitespace((char) nextChar)) {
					if (mode == CONTINUE)
						mode = IGNORE;
					// if key length == 0 or value length == 0
					if (offset == 0 || offset == keyLength || mode == IGNORE)
						continue;
					if (keyLength == -1) { // if parsing the key
						mode = DONE;
						continue;
					}
				}
				if (mode == IGNORE || mode == CONTINUE) mode = NONE;
			}
			firstChar = false;
			if (mode == DONE) {
				keyLength = offset;
				mode = NONE;
			}
			buf[offset++] = (char) nextChar;
		}
		if (keyLength >= 0) {
			String temp = new String(buf, 0, offset);
			messages.put(temp.substring(0, keyLength), temp.substring(keyLength));
		}
		resourceStream.close();
		return messages;
	}

	static boolean isWhitespace(char c) {
		// Optimized case for ASCII
		if ((c >= 0x1c && c <= 0x20) || (c >= 0x9 && c <= 0xd)) return true;
		if (c < 0x2000) return false;
		return c <= 0x200b || c == 0x2028 || c == 0x2029 || c == 0x3000;
	}
}
