package com.ibm.microedition.media;

/*
 *
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

/**
 * This class is the superclass for all the specific media players.  It provides
 * implementation for all the methods specified in Player interface.  The state
 * changing methods delegate the work for their corresponding ..Impl method that
 * needs to be overriden in the subclass by the developer who is porting.
 *
 * The class implements common functionality of:
 * - State validation
 * - State handling
 * - Event handling
 *
 */

import java.util.Vector;
import javax.microedition.media.*;

public abstract class MediaPlayer implements Player {

	//state changing operations
	static public final int OPERATION_DEALLOCATE = 1;
	static public final int OPERATION_PREFETCH = 2;
	static public final int OPERATION_REALIZE = 3;
	static public final int OPERATION_START = 4;
	static public final int OPERATION_STOP = 5;
	static public final int OPERATION_CLOSE = 6;

	//other non-state changing operations
	static public final int OPERATION_SETMEDIATIME = 10;
	static public final int OPERATION_GETMEDIATIME = 11;
	static public final int OPERATION_GETDURATION = 12;
	static public final int OPERATION_GETCONTENTTYPE = 13;
	static public final int OPERATION_SETLOOPCOUNT = 14;
	static public final int OPERATION_ADDLISTENER = 15;
	static public final int OPERATION_REMOVELISTENER = 16;
	static public final int OPERATION_GETCONTROL = 17;
	static public final int OPERATION_GETCONTROLS = 18;
	static public final int OPERATION_SETSEQUENCE = 19;		//set tone sequence- NEW

	//Events for convenience in case it is polled from the player
	static public final int EVT_CLOSED = 1;
	static public final int EVT_DEVICE_AVAILABLE = 2;
	static public final int EVT_DEVICE_UNAVAILABLE = 3;
	static public final int EVT_DURATION_UPDATED = 4;
	static public final int EVT_END_OF_MEDIA = 5;
	static public final int EVT_ERROR = 6;
	static public final int EVT_STARTED = 7;
	static public final int EVT_STOPPED = 8;
	static public final int EVT_VOLUME_CHANGED = 9;

	private int currentState = UNREALIZED;
	private int loopCount = 1;
	protected Vector listeners = new Vector();
	private MediaDescriptor mediaDescriptor;

	//methods that need to be implemented in the concrete implementations of this class
	protected abstract void close(boolean cleanUp);
	public abstract Control getControl(String controlType);
	public abstract Control[] getControls();
	public abstract String getContentType();

	//State changing method implementations
	public synchronized void realize() throws MediaException {
		if (!evaluateStateChangingOperation(OPERATION_REALIZE))
			return;
		realizeImpl();
		setState(REALIZED);
	}

	/**
	* Need to be implemented in the subclasses.
	*/
	protected abstract void realizeImpl() throws MediaException;

	public synchronized void prefetch() throws MediaException {
		if (!evaluateStateChangingOperation(OPERATION_PREFETCH))
			return;
		prefetchImpl();
		setState(PREFETCHED);
	}

	/**
	* Need to be implemented in the subclasses.
	*/
	protected abstract void prefetchImpl() throws MediaException;

	public synchronized void start() throws MediaException {
		if (!evaluateStateChangingOperation(OPERATION_START))
			return;
		setState(STARTED);
		startImpl();
/*
		MediaEventManager.postEventLong(
			this,
			PlayerListener.STARTED,
			TIME_UNKNOWN);
*/
		}

	/**
	* Need to be implemented in the subclasses.
	*/
	protected abstract void startImpl() throws MediaException;

	public synchronized void stop() throws MediaException {
		if (!evaluateStateChangingOperation(OPERATION_STOP))
			return;
		stopImpl();
		setState(PREFETCHED);
/*
		MediaEventManager.postEventLong(
			this,
			PlayerListener.STOPPED,
			TIME_UNKNOWN);
*/
	}

	/**
	* Need to be implemented in the subclasses.
	*/
	protected abstract void stopImpl() throws MediaException;

	public synchronized void deallocate() {
		try {
			if (!evaluateStateChangingOperation(OPERATION_DEALLOCATE))
				return;
		} catch (MediaException me) {
			me.printStackTrace(); //this should never occur
		}
		deallocateImpl();
		setState(REALIZED);
	}

	/**
	* Need to be implemented in the sublcasses.
	*/
	protected abstract void deallocateImpl();

	public synchronized void close() {
		try {
			if (!evaluateStateChangingOperation(OPERATION_CLOSE))
				return;
		} catch (MediaException me) {
			me.printStackTrace(); //this should never occur
		}
		closeImpl();
		setState(CLOSED);
		/* Event posting should be handled from the concrete implementation */
//		MediaEventManager.postEventNull(this, PlayerListener.CLOSED);
		MediaEventManager.unregisterPlayer(this);
	}

	/**
	* Need to be implemented in the subclasses.
	*/
	protected abstract void closeImpl();

	/**
	 * If supported, needs to be overwritten in the concrete implementation.
	 */
	public long setMediaTime(long mediaTime) throws MediaException {
		evaluateOperation(OPERATION_SETMEDIATIME);
		throw new MediaException(getTranslatedString("AbstractPlayer.msg2")); //$NON-NLS-1$
	}

	/**
	* If supported, needs to be overwritten in the concrete implementation.
	*/
	public long getMediaTime() {
		evaluateOperation(OPERATION_GETMEDIATIME);
		return TIME_UNKNOWN;
	}

	public int getState() {
		return currentState;
	}

	/**
	 * Sets the current state of the Player.  Allowed states:
	 * UNREALIZED | REALIZED | PREFETCHED | STARTED | CLOSED
	 * TODO: Error message needs to be externalized.
	 *
	 * @param state
	 * @throws IllegalArgumentException
	 */
	public void setState(int state) throws IllegalArgumentException {
		if (state
			== CLOSED | state
			== PREFETCHED | state
			== REALIZED | state
			== STARTED | state
			== UNREALIZED) {
			currentState = state;
		} else {
			throw new IllegalArgumentException("State should be REALIZED|PREFETCHED|UNREALIZED|CLOSED|STARTED");
		}
	}

	/**
	* If supported, needs to be overwritten in the concrete implementation.
	*/
	public long getDuration() {
		evaluateOperation(OPERATION_GETDURATION);
		return TIME_UNKNOWN;
	}

	public void setLoopCount(int count) {
		evaluateOperation(OPERATION_SETLOOPCOUNT);

		if (count == 0) {
			throw new IllegalArgumentException(getTranslatedString("AbstractPlayer.msg1")); //$NON-NLS-1$
		} else if (count > 0) {
			loopCount = count;
		} else {
			loopCount = -1; //playing indefinetely
		}
	}

	public int getLoopCount() {
		return loopCount;
	}

	public void addPlayerListener(PlayerListener playerListener) {
		evaluateOperation(OPERATION_ADDLISTENER);
		if (null == playerListener)
			return;

		synchronized (listeners) {
			listeners.addElement(playerListener);
		}
	}

	public void removePlayerListener(PlayerListener playerListener) {
		evaluateOperation(OPERATION_REMOVELISTENER);
		if (null == playerListener)
			return;

		synchronized (listeners) {
			listeners.removeElement(playerListener);
		}
	}

	/**
	 * Evaluates the state change operation and controls the state changes.
	 * Common for both Midp20 media package and JSR 135.
	 * Additional operations are in the evaluateOperation(int Operation) method.
	 *
	 * @param operation State change related operatoin.
	 * @return The flag to allow the actual operation and state change.
	 * @throws IllegalStateException
	 */
	public boolean evaluateStateChangingOperation(int operation)
		throws MediaException {
		switch (operation) {
			case OPERATION_REALIZE :
				if (getState() == CLOSED)
					throw new IllegalStateException("IllegalStateException occured");

				return getState() != REALIZED
					&& getState() != PREFETCHED
					&& getState() != STARTED;

			case OPERATION_PREFETCH :
				if (getState() == CLOSED)
					throw new IllegalStateException("IllegalStateException occured");
				if (getState() == STARTED)
					return false;
				if (getState() == PREFETCHED)
					return false; //unclear in spec, assuming doing nothing.
				if (getState() == UNREALIZED)
					realize();
				return true;

			case OPERATION_START :
				if (getState() == CLOSED)
					throw new IllegalStateException("IllegalStateException occured");
				if (getState() == STARTED)
					return false;
				if (getState() == UNREALIZED || getState() == REALIZED)
					prefetch();
				return true;

			case OPERATION_STOP :
				if (getState() == CLOSED)
					throw new IllegalStateException("IllegalStateException occured");
				return getState() == STARTED;

			case OPERATION_CLOSE :
				return getState() != CLOSED;

			case OPERATION_DEALLOCATE :
				if (getState() == CLOSED)
					throw new IllegalStateException("IllegalStateException occured");
				if (getState() == UNREALIZED || getState() == REALIZED)
					return false;
				//if (realizing) - see spec, needs to be taken care of in the concrete implementation
				if (getState() == STARTED)
					stop();
				return true;

			default :
				return false;
		}
	}

	/**
	 * Evaluates the non-state changing operation.  Only for Midp20 media package.
	 * For JSR 135, additional operations need to be added.
	 *
	 * @param operation non-state change related operatoin.
	 * @return The flag to allow the actual operation.
	 * @throws IllegalStateException
	 */
	public boolean evaluateOperation(int operation) {
		switch (operation) {
			case OPERATION_SETMEDIATIME :
				if (getState() == CLOSED || getState() == UNREALIZED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_GETMEDIATIME :
				if (getState() == CLOSED || getState() == UNREALIZED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_GETDURATION :
				if (getState() == CLOSED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_GETCONTENTTYPE :
				if (getState() == CLOSED || getState() == UNREALIZED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_GETCONTROL :
				if (getState() == CLOSED || getState() == UNREALIZED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_GETCONTROLS :
				if (getState() == CLOSED || getState() == UNREALIZED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_SETLOOPCOUNT :
				if (getState() == CLOSED || getState() == STARTED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_ADDLISTENER :
				if (getState() == CLOSED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_REMOVELISTENER :
				if (getState() == CLOSED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			case OPERATION_SETSEQUENCE :
				if (getState() == CLOSED || getState() == PREFETCHED || getState() == STARTED)
					throw new IllegalStateException(
						getTranslatedString("AbstractPlayer.msg3"));
				return true;

			default :
				return false;
		}
	}

	public void fireEventHandlers(String event, Object eventData) {
		synchronized (listeners) {
			for (int i = 0; i < listeners.size(); i++) {
				PlayerListener listener =
					(PlayerListener) listeners.elementAt(i);
				try {
					listener.playerUpdate(this, event, eventData);
				} catch(Throwable t) {
				}
			}
		}
	}

	/**
	* Needs to be overwritten in the concrete player implementation.
	*/
	public abstract int setVolumeLevelImpl(int volumeLevel);

	/**
	* Needs to be overwritten in the concrete player implementation.
	*/
	public abstract int getVolumeLevelImpl();

	/*Used for externalization*/
	public static String getTranslatedString(String id) {
		return MediaMessages.getString(id);
	}

	public MediaDescriptor getMediaDescriptor() {
		return mediaDescriptor;
	}

	public void setMediaDescriptor(MediaDescriptor descriptor) {
		mediaDescriptor = descriptor;
	}

}
