package com.ibm.microedition.media;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import java.io.IOException;
import java.io.InputStream;

import javax.microedition.media.*;
import com.ibm.microedition.media.control.*;

/**
 * ToneSequence Player will extend the MIDI Player
 */
public class ToneSequencePlayer extends MIDIPlayer {

	private int[] fToneData;
	ToneCtrl toneCtrl = new ToneCtrl(this);

	public ToneSequencePlayer() {
		super();
	}

	public Control getControl(String controlType) {
		evaluateOperation(MediaPlayer.OPERATION_GETCONTROL);

		if (null == controlType) throw new IllegalArgumentException();

		if (controlType.equals("VolumeControl") || controlType.equals("javax.microedition.media.control.VolumeControl")) { //$NON-NLS-1$ //$NON-NLS-2$
			return midiVolCtrl;
		}
		if (controlType.equals("ToneControl") || controlType.equals("javax.microedition.media.control.ToneControl")) { //$NON-NLS-1$ //$NON-NLS-2$
			return toneCtrl;
		}
		return null;
	}

	public Control[] getControls() {
		evaluateOperation(MediaPlayer.OPERATION_GETCONTROL);
		return new Control[] {midiVolCtrl, toneCtrl};
	}

	public String getContentType() {
		evaluateOperation(MediaPlayer.OPERATION_GETCONTENTTYPE);
		return MediaConstants.MIME_TYPE_TONE;
	}

	public void realizeImpl() throws MediaException {
		MediaDescriptor md = this.getMediaDescriptor();

		if (md == null) return;
		String protocol = md.getProtocol();
		InputStream stream = md.getMediaIS();

		if ((protocol.equals(MediaConstants.PROTOCOL_FILE)
				&& (stream != null))
			|| ((protocol.equals(MediaConstants.PROTOCOL_HTTP)
				|| protocol.equals(MediaConstants.PROTOCOL_HTTPS))
				&& (stream != null))) {
					try {
						if (readData() == false) {
							throw new MediaException(getTranslatedString("MediaPlayer.msg2") );
						}
					} catch (IOException e) {
						e.printStackTrace();
					}
					toneCtrl.setSequence(fData);
				}
	}

	public void prefetchImpl() throws MediaException {
		if (0 != fHandle) return;

		if (toneCtrl != null) {
			fToneData = toneCtrl.getMidiSequence();
			parseMidiHeader();
		} else {
			throw new MediaException(getTranslatedString("MediaPlayer.msg1"));
		}

		fHandle = midiPlayerCreate(this, fMidiData);
		if (0 == fHandle) {
			throw new MediaException(getTranslatedString("MediaPlayer.native.msg1")); //$NON-NLS-1$
		}
	}

	public long getDuration() {
		evaluateOperation(OPERATION_GETDURATION);
		if (toneCtrl.getMidiSequence() == null) return 0;

		int duration = midiPlayerGetDuration(fHandle);
		if (-1 == duration) return Player.TIME_UNKNOWN;

		return duration * 1000;
	}

	/*  add the Midi header chunk to the track data obtained from sequence */
	private void parseMidiHeader() {
		int length = fToneData != null ? fToneData.length : 0;
		fMidiData = new int[length + 23 + 4];	// track data + header info + end of track data
												// -> we are going to add all these to get total length
		fMidiData[0] = 0x4D; 		//0-3 -> 'MThd'
		fMidiData[1] = 0x54;
		fMidiData[2] = 0x68;
		fMidiData[3] = 0x64;
		fMidiData[4] = 0x00;		// 4-7 -> length = 6
		fMidiData[5] = 0x00;
		fMidiData[6] = 0x00;
		fMidiData[7] = 0x06;
		fMidiData[8] = 0x00;		//8-9 -> format 0
		fMidiData[9] = 0x00;
		fMidiData[10] = 0x00;		//10-11 -> num of tracks = 1
		fMidiData[11] = 0x01;
		fMidiData[12] = 0x02;		//let's set it to the default tpqn
		fMidiData[13] = 0x00;
		fMidiData[14] = 0x4D;		//14-17 -> 'MTrk'
		fMidiData[15] = 0x54;
		fMidiData[16] = 0x72;
		fMidiData[17] = 0x6B;

		fMidiData[18] = ((length >> 24) & 0x000000ff );	//18-21 -> length of the track
		fMidiData[19] = ((length >> 16) & 0x000000ff);
		fMidiData[20] = ((length >> 8) & 0x000000ff);
		fMidiData[21] = (length & 0x000000ff);
		fMidiData[22] = 0x00;		// delta-ticks for first event

		for (int i = 0, j = 23; i < length; i++, j++) {
			fMidiData[j] = fToneData[i] & 0x000000ff;
		}

		fMidiData[23 + length] = 0x00;  // end of track data
		fMidiData[23 + length + 1] = 0xff;
		fMidiData[23 + length + 2] = 0x2f;
		fMidiData[23 + length + 3] = 0x00;
	}
}
