package com.ibm.oti.connection.datagram;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 */

import java.io.*;

import javax.microedition.io.*;
import com.ibm.oti.connection.*;
import com.ibm.oti.connection.socket.Socket;
import com.ibm.oti.util.NetworkActivityHook;
import com.ibm.oti.util.NetworkActivityListener;
import com.ibm.oti.security.midp.PermissionManager;

/**
 * Implements a datagram connection. Accepts the following parameters:
 * <UL>
 * <LI>so_timeout - set the receive timeout in milliseconds, 0 waits forever,
 * the default is zero. If the connection is opened with the timeout argument set
 * to true, and the so_timeout is not set, the default is 3000 milliseconds.</LI>
 * <LI>so_rcvbuf - set the socket receive buffer size in bytes.</LI>
 * <LI>so_sndbuf - set the socket send buffer size in bytes.</LI>
 * </UL>
 *
 * @author		OTI
 * @version		initial
 *
 * @see		javax.microedition.io.DatagramConnection
 */

public class Connection extends Socket implements CreateConnection, UDPDatagramConnection {

	static final int DEFAULT_TIMEOUT = 8000;
	private String server;
	private boolean closed = false;
	private int access, port, timeout = 0;
	private int udpLocalPort;
	private byte[] localAddress = {0,0,0,0};

public Connection() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
}

public void close() throws IOException {
	closed = true;
	socketClose();
}

public int getMaximumLength() throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	return maxDatagramImpl();
}

public int getNominalLength() throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	return nominalDatagramImpl();
}

public Datagram newDatagram(byte[] data, int length) throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if ((length < 0) || isOverMaxLength(length)) throw new IllegalArgumentException();
	Datagram datagram = new DatagramPacket();
	datagram.setData(data, 0, length);
	if (server != null)
		datagram.setAddress("datagram://" + server + ":" + port);
	return datagram;
}

public Datagram newDatagram(byte[] data, int length, String address) throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if ((length < 0) || isOverMaxLength(length)) throw new IllegalArgumentException();
	Datagram datagram = new DatagramPacket();
	datagram.setData(data, 0, length);
	datagram.setAddress(address);
	return datagram;
}

public Datagram newDatagram(int length) throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if ((length < 0) || isOverMaxLength(length)) throw new IllegalArgumentException();
	return newDatagram(new byte[length], length);
}

public Datagram newDatagram(int length, String address) throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if ((length < 0) || isOverMaxLength(length)) throw new IllegalArgumentException();
	Datagram datagram = new DatagramPacket();
	datagram.setData(new byte[length], 0, length);
	datagram.setAddress(address);
	return datagram;
}

private boolean isOverMaxLength(int length){
	return false;
}

public void receive(javax.microedition.io.Datagram datagram) throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if (access != Connector.READ && access != Connector.READ_WRITE)
		// K00a9 = Not open for READ
		throw new IOException(com.ibm.oti.util.Msg.getString("K00a9"));

	synchronized (this) {
		String host;
	 	try {
		 	NetworkActivityHook.getListener().networkActivityStarted(NetworkActivityListener.READ);
			int orgLength = datagram.getLength();
			datagram.setLength(receiveDatagramImpl(datagram.getData(), datagram.getOffset(),
				orgLength, timeout));
			host = getHostByAddr(address);
			if (host == null) {
				host = (address[0] & 0xff) + "." + (address[1] & 0xff) + "." +
					(address[2] & 0xff) + "." + (address[3] & 0xff);
			}
	 	} finally {
		 	NetworkActivityHook.getListener().networkActivityComplete(NetworkActivityListener.READ);
	 	}
		datagram.setAddress("datagram://" + host + ":" + localport);
	}
}

/**
 * Recieve at most <code>count</code> bytes into the buffer <code>data</code>
 * at the <code>offset</code> on the socket.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		data	the receive buffer
 * @param		offset	the offset into the buffer
 * @param		length	the max number of bytes to receive
 * @param		timeout	the max time the read operation should block waiting for data
 * @return		int		the actual number of bytes read
 * @exception	IOException	if an error occurs while reading
 */
native int receiveDatagramImpl(byte[] data, int offset, int length, int timeout) throws IOException;

public void send(Datagram datagram) throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if (access != Connector.WRITE && access != Connector.READ_WRITE)
		// K00aa = Not open for WRITE
		throw new IOException(com.ibm.oti.util.Msg.getString("K00aa"));
	DatagramPacket data;
	if (datagram instanceof DatagramPacket) {
		data = (DatagramPacket)datagram;
	} else {
		data = new DatagramPacket();
		data.setData(datagram.getData(), datagram.getOffset(), datagram.getLength());
		data.setAddress(datagram.getAddress());
	}

	int port;
	String host;
	String serverAddress = data.getAddress();
	if (serverAddress == null) {
		if (server == null) return;
		port = this.port;
		host = server;
	} else {
		port = data.port;
		host = data.getHost();
	}

	byte[] address = resolveHost(host);

 	try {
	 	NetworkActivityHook.getListener().networkActivityStarted(NetworkActivityListener.WRITE);
		sendDatagramImpl(data.getData(), data.getOffset(), data.getLength(),
			address, port);
 	} finally {
	 	NetworkActivityHook.getListener().networkActivityComplete(NetworkActivityListener.WRITE);
 	}
}

private boolean isMulticastAddress(byte[] addr) {
	return ((addr[0] & 255) >>> 4) == 0xE;
}

/**
 * Send <code>count</code> bytes from the buffer <code>data</code>
 * at the <code>offset</code>, on the socket.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		data	the send buffer
 * @param		offset	the offset into the buffer
 * @param		length	the number of bytes to send
 * @param		inetAddr 	remote address to send to
 * @param		remotePort	remote port to send to
 * @exception	IOException	if an error occurs while sending
 */
native void sendDatagramImpl(byte[] data, int offset, int length, byte[] inetAddr, int remotePort) throws IOException;

/**
 * Passes the parameters from the Connector.open() method to this
 * object. Protocol used by MIDP 2.0
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		timeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
public javax.microedition.io.Connection setParameters2(String spec, int access, boolean timeout) throws IOException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();

	setParameters(spec, access, timeout);
	return this;
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object. Protocol used by MIDP 1.0
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		timeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
public void setParameters(String spec, int access, boolean timeout) throws IOException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();

	String[][] equates = ConnectionUtil.NO_PARAMETERS;
	int index = spec.indexOf(';');
	if (index != -1) {
		equates = ConnectionUtil.getParameters(spec.substring(index + 1));
		spec = spec.substring(0, index);
	}
	setParameters(spec, equates, access, timeout);
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		equates String[][]
 *					An 2-dimensional array of strings that
 *					contain the parameters and their keys
 *					passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		throwTimeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
private void setParameters(String spec, String[][] equates, int access, boolean throwTimeout) throws IOException {
	this.access = access;
	int rcvbuf = 0, sndbuf = 0;
	int[] result = new int[1];
	for (int i=0; i<equates.length; i++) {
		String key = equates[i][0];
		equates[i][0] = equates[i][0].toLowerCase();
		if (ConnectionUtil.intParam("so_timeout", equates[i], ConnectionUtil.NEGATIVE, result))
			timeout = result[0];
		else if (ConnectionUtil.intParam("so_rcvbuf", equates[i], ConnectionUtil.NEGATIVE_OR_ZERO, result))
			rcvbuf = result[0];
		else if (ConnectionUtil.intParam("so_sndbuf", equates[i], ConnectionUtil.NEGATIVE_OR_ZERO, result))
			sndbuf = result[0];
		// K00a5 = Invalid parameter - {0}
		else throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a5", key));
	}
	if (throwTimeout && timeout == 0)
		timeout = DEFAULT_TIMEOUT;

	if (!spec.startsWith("//"))
		// K00a1 = Missing //
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a1"));
	String host = Socket.parseURL(spec, result, true, true);
	if (host.length() != 0) {
		server = host;
		if (PermissionManager.getManager() != null)
			PermissionManager.getManager().checkPermission("javax.microedition.io.Connector.datagram");
	} else {
		if (PermissionManager.getManager() != null)
			PermissionManager.getManager().checkPermission("javax.microedition.io.Connector.datagramreceiver");
	}
	port = result[0];

	try {
		bindDatagramImpl(server == null ? port : 0);
		localAddress = address;
		udpLocalPort = localport;
		if (rcvbuf != 0) setSocketOptionImpl(SO_RCVBUF, rcvbuf);
		if (sndbuf != 0) setSocketOptionImpl(SO_SNDBUF, sndbuf);
	} catch (IOException e) {
		socketClose();
		throw e;
	}
	socketOpen();
}

native void bindDatagramImpl(int localport) throws IOException;

native int maxDatagramImpl();
native int nominalDatagramImpl();

private boolean addressZero(byte[] address) {
	boolean addressZero = true;
	for (int i=0; i<address.length; i++) {
		if(address[i] != 0 ) {
			addressZero = false;
			break;
		}
	}
	return addressZero;
}

/**
 * @see javax.microedition.io.UDPDatagramConnection#getLocalAddress()
 */
public String getLocalAddress() throws IOException {
	// K00ac = Connection is closed
	if(closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));

	if ( addressZero(localAddress) ) {
		String hostName = System.getProperty("microedition.hostname");
		if (hostName != null) {
			if (hostName.length() > 0 && hostName.charAt(0) >= '0' &&
				hostName.charAt(0) <= '9') return hostName;
			try {
				localAddress = Socket.getHostByName(hostName);
			} catch (IOException e) {}
		}

		if (addressZero(localAddress)) {
			localAddress = new byte[4];
			localAddress[0]=127; localAddress[1]=0; localAddress[2]=0; localAddress[3]=1;

		}
	}
	return Socket.addressToString(localAddress);
}

/**
 * @see javax.microedition.io.UDPDatagramConnection#getLocalPort()
 */
public int getLocalPort() throws IOException {
	// K00ac = Connection is closed
	if(closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	return udpLocalPort;
}

}
