package com.ibm.oti.connection.file;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

import java.io.*;
import javax.microedition.io.*;

/**
 * FileInputStream is a class for reading bytes from a file.
 *
 * @author		OTI
 * @version		initial
 *
 * @see 		InputStream
 */
public final class FileInputStream extends InputStream
{
	/**
	 * The file descriptor representing this FileInputStream.
	 */
	private long descriptor = -1;

	public static final FileInputStream in = new FileInputStream(0);

FileInputStream(long fid) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	descriptor = fid;
}

/**
 * Constructs a new FileInputStream on the file named <code>fileName</code>.  If the
 * file does not exist, the <code>FileNotFoundException</code> is thrown. The <code>fileName</code>
 * may be absolute or relative to the System property <code>"user.dir"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		fileName	the file on which to stream reads.
 *
 * @exception 	ConnectionNotFoundException If the <code>fileName</code> is not found.
 */
public FileInputStream(String fileName) throws ConnectionNotFoundException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	if ((descriptor = openImpl(com.ibm.oti.util.Util.getBytes(fileName))) == -1)
		throw new ConnectionNotFoundException(fileName);
}

/**
 * Constructs a new FileInputStream on the file named <code>fileName</code>.  If the
 * file does not exist, the <code>FileNotFoundException</code> is thrown. The <code>fileName</code>
 * may be absolute or relative to the System property <code>"user.dir"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		fileName	the file on which to stream reads.
 *
 * @exception 	ConnectionNotFoundException If the <code>fileName</code> is not found.
 */
public FileInputStream(byte[] fileName) throws ConnectionNotFoundException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	if ((descriptor = openImpl(fileName)) == -1)
		throw new ConnectionNotFoundException();
}

/**
 * Answers a int representing then number of bytes that are available
 * before this InputStream will block.  This method always returns the
 * size of the file minus the current position.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		the number of bytes available before blocking.
 *
 * @exception 	java.io.IOException	If an error occurs in this stream.
 */
public int available() throws IOException {
	return availableImpl(descriptor);
};

private native int availableImpl(long descriptor) throws IOException;

/**
 * Close the FileInputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception 	java.io.IOException	If an error occurs attempting to close this FileInputStream.
 */
public void close() throws IOException {
	try {
		closeImpl(descriptor);
	} finally {
		descriptor = -1;
	}
}

private native void closeImpl(long descriptor) throws IOException;

private native long openImpl(byte [] fileName);

/**
 * Reads a single byte from this FileInputStream and returns the result as
 * an int.  The low-order byte is returned or -1 of the end of stream was
 * encountered.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		the byte read or -1 if end of stream.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public int read() throws IOException {
	// K0059 = Stream is closed
	if (descriptor == -1) throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
	return readByteImpl(descriptor);
}

private native int readByteImpl(long descriptor ) throws IOException;

/**
 * Reads at most <code>count</code> bytes from the FileInputStream and stores them in byte
 * array <code>buffer</code> starting at <code>offset</code>. Answer the number of
 * bytes actually read or -1 if no bytes were read and end of stream was encountered.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer	the byte array in which to store the read bytes.
 * @param		offset	the offset in <code>buffer</code> to store the read bytes.
 * @param		count	the maximum number of bytes to store in <code>buffer</code>.
 * @return 		the number of bytes actually read or -1 if end of stream.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public int read(byte[] buffer, int offset, int count) throws IOException {
	// K0059 = Stream is closed
	if (descriptor == -1) throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
	return readImpl(buffer, offset, count, descriptor);
}

private native int readImpl(byte[] buffer, int offset, int count, long descriptor) throws IOException;

/**
 * Skips <code>count</code> number of bytes in this FileInputStream.  Subsequent
 * <code>read()</code>'s will not return these bytes unless <code>reset()</code>
 * is used.  This method may perform multiple reads to read <code>count</code>
 * bytes.  This default implementation reads <code>count</code> bytes into a temporary
 * buffer.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		count		the number of bytes to skip.
 * @return		the number of bytes actually skipped.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public long skip(long count) throws IOException {
	return skipImpl(count, descriptor);
};

private native long skipImpl(long count, long descriptor) throws IOException;

}
