package com.ibm.oti.connection.file;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

import java.io.*;
import javax.microedition.io.*;

/**
 * FileOutputStream is a class whose underlying stream is represented by
 * a file in the operating system.  The bytes that are written to this stream are
 * passed directly to the underlying operating system equivalent function.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			OutputStream
 */
public final class FileOutputStream extends OutputStream {

	/**
	 * The file descriptor representing this FileOutputStream.
	 */
	private long descriptor = -1;

	public static final FileOutputStream out = new FileOutputStream(1);
	public static final FileOutputStream err = new FileOutputStream(2);

FileOutputStream(long fid) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	descriptor = fid;
}

/**
 * Constructs a new FileOutputStream on the file named <code>filename</code>. If
 * the file exists, it is written over.  See the constructor which can append to
 * the file if so desired. The <code>filename</code> may be absolute or relative
 * to the System property <code>"user.dir"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		filename	the file on which to stream writes.
 * @param		append		a boolean indicating whether or not to append to an existing file.
 *
 * @exception 	ConnectionNotFoundException If the <code>filename</code> cannot be opened for writing.
 */
public FileOutputStream(String filename, boolean append) throws ConnectionNotFoundException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	if ((descriptor = openImpl(com.ibm.oti.util.Util.getBytes(filename), append)) == -1)
		throw new ConnectionNotFoundException(filename);
}

/**
 * Close the FileOutputStream.  This implementation closes the underlying OS resources allocated
 * to represent this stream.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception 	java.io.IOException	If an error occurs attempting to close this FileOutputStream.
 */
public void close() throws IOException {
	try {
		closeImpl(descriptor);
	} finally {
		descriptor = -1;
	}
}

private native void closeImpl(long descriptor) throws IOException;

private native long openImpl(byte[] fileName, boolean openAppend);

/**
 * Writes <code>count</code> <code>bytes</code> from the byte array
 * <code>buffer</code> starting at <code>offset</code> to this
 * FileOutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer		the buffer to be written
 * @param		offset		offset in buffer to get bytes
 * @param		count		number of bytes in buffer to write
 *
 * @throws 	java.io.IOException	If an error occurs attempting to write to this FileOutputStream.
 * @throws	java.lang.IndexOutOfBoundsException If offset or count are outside of bounds.
 * @throws	java.lang.NullPointerException If buffer is <code>null</code>.
 */
public void write(byte[] buffer, int offset, int count) throws IOException {
	// K00b7 = File is closed
	if (descriptor == -1) throw new IOException(com.ibm.oti.util.Msg.getString("K00b7"));
	writeImpl(buffer, offset, count, descriptor);
}

private native void writeImpl(byte[] buffer, int offset, int count, long descriptor) throws IOException;

/**
 * Writes the specified byte <code>oneByte</code> to this FileOutputStream.  Only
 * the low order byte of <code>oneByte</code> is written.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		oneByte		the byte to be written
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write to this FileOutputStream.
 */
public void write(int oneByte) throws IOException {
	// K00b7 = File is closed
	if (descriptor == -1) throw new IOException(com.ibm.oti.util.Msg.getString("K00b7"));
	writeByteImpl(oneByte, descriptor);
}

private native void writeByteImpl(int oneByte, long descriptor) throws IOException;
}
