package com.ibm.oti.connection.socket;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 */

import java.io.*;
import javax.microedition.io.*;
import com.ibm.oti.connection.*;

import com.ibm.oti.util.NetworkActivityHook;
import com.ibm.oti.util.NetworkActivityListener;
import com.ibm.oti.security.midp.PermissionManager;

/**
 * Implements a socket connection. Accepts the following parameters:
 * <UL>
 * <LI>so_timeout - set the receive timeout in milliseconds, 0 waits forever,
 * the default is zero. If the connection is opened with the timeout argument set
 * to true, and the so_timeout is not set, the default is 3000 milliseconds.</LI>
 * <LI>so_linger - set the socket linger option in milliseconds.</LI>
 * <LI>tcp_nodelay - true/false, set the socket TCP nodelay option.</LI>
 * <LI>so_keepalive - true/false, set the socket keepalive option.</LI>
 * <LI>so_rcvbuf - set the socket receive buffer size in bytes.</LI>
 * <LI>so_sndbuf - set the socket send buffer size in bytes.</LI>
 * </UL>
 *
 * @author		OTI
 * @version		initial
 *
 * @see		javax.microedition.io.StreamConnection
 */
public class Connection extends Socket implements CreateConnection, SocketConnection {
	static final int DEFAULT_TIMEOUT = 8000;
	private static final int UNOPENED = 0, OPEN = 1, CLOSED = 2;
	private String host;
	private int remoteport;
	private byte[] remoteaddress;
	private int access, port = 0, timeout = 0;
	private int inputStatus = UNOPENED, outputStatus = UNOPENED;
	private boolean checkPermission = true;

	// cached value of TCP_NODELAY socket option
	private int tcpNoDelay;

public Connection() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
}

public Connection(int access) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	host = "";
	this.access = access;
	checkPermission = false;
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object. Protocol used by MIDP 2.0
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		timeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
public javax.microedition.io.Connection setParameters2(String spec, int access, boolean timeout) throws IOException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();

	// if the spec starts with "socket://:" then it is a server socket
	if (spec.startsWith("//") && (spec.length() == 2 || (spec.length() > 2 &&
		(spec.charAt(2) == ':' || spec.charAt(2) == ';'))))
	{
		Class connectionClass = null;
		try {
			connectionClass = Class.forName("com.ibm.oti.connection.serversocket.Connection");
		} catch (ClassNotFoundException e) {
			// K0002 = Scheme not found: {0}
			throw new ConnectionNotFoundException(com.ibm.oti.util.Msg.getString("K0002", "serversocket"));
		}
		boolean exception = false;
		CreateConnection connection = null;
		try {
			connection = (CreateConnection)connectionClass.newInstance();
		} catch (InstantiationException e) {
			exception = true;
		} catch (IllegalAccessException e) {
			exception = true;
		}
		if (exception)
			// K0003 = Cannot instantiate scheme: {0}
			throw new ConnectionNotFoundException(com.ibm.oti.util.Msg.getString("K0003", connectionClass));
		return connection.setParameters2(spec, access, timeout);
	}
	if (checkPermission && PermissionManager.getManager() != null)
		PermissionManager.getManager().checkPermission("javax.microedition.io.Connector.socket");
	setParameters(spec, access, timeout);
	return this;
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object. Protocol used by MIDP 1.0
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		timeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
public void setParameters(String spec, int access, boolean timeout) throws IOException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();

	String[][] equates = ConnectionUtil.NO_PARAMETERS;
	int index = spec.indexOf(';');
	if (index != -1) {
		equates = ConnectionUtil.getParameters(spec.substring(index + 1));
		spec = spec.substring(0, index);
	}
	setParameters(spec, equates, access, timeout);
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		equates String[][]
 *					An 2-dimensional array of strings that
 *					contain the parameters and their keys
 *					passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		throwTimeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
private void setParameters(String spec, String[][] equates, int access, boolean throwTimeout) throws IOException {
	int[] result = new int[1];
	boolean setNoDelay = false, setKeepAlive = false;
	int linger = -1, rcvbuf = 0, sndbuf = 0;
	for (int i=0; i<equates.length; i++) {
		String key = equates[i][0];
		equates[i][0] = equates[i][0].toLowerCase();
		if (ConnectionUtil.intParam("so_timeout", equates[i], ConnectionUtil.NEGATIVE, result))
			timeout = result[0];
		else if (ConnectionUtil.intParam("so_linger", equates[i], ConnectionUtil.NEGATIVE, result)) {
			linger = result[0];
			if (linger > 65535) linger = 65535;
		} else if (equates[i][0].equals("tcp_nodelay")) {
			String value = equates[i][1].toLowerCase();
			if (value.equals("true")) setNoDelay = true;
			else if (!value.equals("false"))
				// K00b5 = Invalid boolean value: {0}
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00b5", equates[i][1]));
		} else if (equates[i][0].equals("so_keepalive")) {
			String value = equates[i][1].toLowerCase();
			if (value.equals("true")) setKeepAlive = true;
			else if (!value.equals("false"))
				// K00b5 = Invalid boolean value: {0}
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00b5", equates[i][1]));
		} else if (ConnectionUtil.intParam("so_rcvbuf", equates[i], ConnectionUtil.NEGATIVE_OR_ZERO, result))
			rcvbuf = result[0];
		else if (ConnectionUtil.intParam("so_sndbuf", equates[i], ConnectionUtil.NEGATIVE_OR_ZERO, result))
			sndbuf = result[0];
		// K00a5 = Invalid parameter - {0}
		else throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a5", key));
	}
	if (throwTimeout && timeout == 0)
		timeout = DEFAULT_TIMEOUT;

	host = spec;
	this.access = access;
	if (!host.startsWith("//"))
		// K01cc = '//' must follow protocol
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01cc"));
	host = Socket.parseURL(spec, result, true, false);

	byte[] addr = {0,0,0,0};
	try {
		addr = resolveHost(host);
	} catch (IOException e) {
		// K01ce = Host lookup failed: {0} - {1}
		throw new ConnectionNotFoundException(com.ibm.oti.util.Msg.getString("K01ce", host, e.getMessage()));
	}

	try {
		try {
 			NetworkActivityHook.getListener().networkActivityStarted(NetworkActivityListener.CONNECT);
			connectStreamImpl2(0, result[0], addr);
		} catch (ConnectionNotFoundException e) {
			// socketClose(); - socket closed by outer catch block
			// K01cf = Could not establish connection: {0} - {1}
			throw new ConnectionNotFoundException(com.ibm.oti.util.Msg.getString("K01cf", "(" + addressToString(addr) + ") " + host + ":" + result[0], e.getMessage()));
		}
			finally {
	 			NetworkActivityHook.getListener().networkActivityComplete(NetworkActivityListener.CONNECT);
			}
		if (rcvbuf != 0) setSocketOptionImpl(SO_RCVBUF, rcvbuf);
		if (sndbuf != 0) setSocketOptionImpl(SO_SNDBUF, sndbuf);
		if (linger != -1) setSocketOptionImpl(SO_LINGER, linger);
		if (setNoDelay) {
			setSocketOptionImpl(TCP_NODELAY, 1);
			tcpNoDelay=1;
		}
		if (setKeepAlive) setSocketOptionImpl(SO_KEEPALIVE, 1);
	} catch (IOException e) {
		socketClose();
		throw e;
	}
	socketOpen();

}

public void close() throws IOException {
	host = null;
	if (inputStatus != OPEN && outputStatus != OPEN) {
		socketClose();
	}
}

public InputStream openInputStream() throws IOException {
	// K00ac = Connection is closed
	if (host == null) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if (access != Connector.READ && access != Connector.READ_WRITE)
		// K00a9 = Not open for READ
		throw new IOException(com.ibm.oti.util.Msg.getString("K00a9"));
	// K0192 = Stream already opened
	if (inputStatus != UNOPENED) throw new IOException(com.ibm.oti.util.Msg.getString("K0192"));
	inputStatus = OPEN;

	return new InputStream() {
		public int available() throws IOException {
			if (inputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
			return availableStreamImpl();
		}
		public void close() throws IOException {
			if(inputStatus != CLOSED) {
				inputStatus = CLOSED;
				if(outputStatus != OPEN && host == null) {
					socketClose();
				}
			}
		}
		public int read(byte[] buffer, int offset, int count) throws IOException {
			if (inputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
			// K0047 = buffer is null
			if (null == buffer) throw new IOException(com.ibm.oti.util.Msg.getString("K0047"));
			if (0 == count) return 0;
			if (0 > offset || 0 > count || offset >= buffer.length || buffer.length - offset < count)
				// K002f = Arguments out of bounds
				throw new ArrayIndexOutOfBoundsException(com.ibm.oti.util.Msg.getString("K002f"));
			NetworkActivityHook.getListener().networkActivityStarted(NetworkActivityListener.READ);
			try {
			int result = receiveStreamImpl(buffer, offset, count, timeout);
			return result;
			} finally {
				NetworkActivityHook.getListener().networkActivityComplete(NetworkActivityListener.READ);
			}
		}
		public int read() throws IOException {
			byte[] buffer = new byte[1];
			int result = read(buffer, 0, 1);
			return (-1 == result) ? result : buffer[0] & 0xFF;
		}
	};
}

public OutputStream openOutputStream() throws IOException {
	// K00ac = Connection is closed
	if (host == null) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	if (access != Connector.WRITE && access != Connector.READ_WRITE)
		// K00aa = Not open for WRITE
		throw new IOException(com.ibm.oti.util.Msg.getString("K00aa"));
	// K0192 = Stream already opened
	if (outputStatus != UNOPENED) throw new IOException(com.ibm.oti.util.Msg.getString("K0192"));
	outputStatus = OPEN;

	return new OutputStream() {
		public void close() throws IOException {
			if(outputStatus != CLOSED) {
				// don't shutdown output on Windows CE with the linger socket option set
				if ((getSocketFlags()&FLAG_BROKEN_SO_LINGER_SHUTDOWN)==0 || getSocketOptionImpl(SO_LINGER)==0) {
					shutdownSideImpl(false);
				}
				outputStatus = CLOSED;
				if(inputStatus != OPEN && host == null) {
					socketClose();
				}
			}
		}
		public void write(int oneByte) throws IOException {
			byte[] buffer = new byte[1];
			buffer[0] = (byte)oneByte;
			write(buffer, 0, 1);
		}
		public void write(byte[] buffer, int offset, int count) throws IOException {
			if (outputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
			if (null == buffer) throw new NullPointerException();
			if (0 == count) return;
			if (0 > offset || 0 > count || offset >= buffer.length || buffer.length - offset < count )
				// K002f = Arguments out of bounds
				throw new ArrayIndexOutOfBoundsException(com.ibm.oti.util.Msg.getString("K002f"));
 			NetworkActivityHook.getListener().networkActivityStarted(NetworkActivityListener.WRITE);
 			try {
			sendStreamImpl(buffer, offset, count);
 			} finally {
	 			NetworkActivityHook.getListener().networkActivityComplete(NetworkActivityListener.WRITE);
 			}
		}
	};
}

public DataInputStream openDataInputStream() throws IOException {
	return new DataInputStream(openInputStream());
}

public DataOutputStream openDataOutputStream() throws IOException {
	return new DataOutputStream(openOutputStream());
}

/**
 * Connect the underlying socket to the nominated remotehost/port.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		localPort	the local machine port to use
 * @param		remotePort	the remote machine port to connect to
 * @param		addr	the address of the remote machine to connect to
 * @exception	IOException	if an error occurs while connecting
 */
native void connectStreamImpl2(int localPort, int remotePort, byte[] addr) throws IOException;

/**
 * Recieve at most <code>count</code> bytes into the buffer <code>data</code>
 * at the <code>offset</code> on the socket.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		data	the receive buffer
 * @param		offset	the offset into the buffer
 * @param		count	the max number of bytes to receive
 * @param		timeout	the max time the read operation should block waiting for data
 * @return		int		the actual number of bytes read
 * @exception	IOException	if an error occurs while reading
 */
native int receiveStreamImpl(byte[] data, int offset, int count, int timeout) throws IOException;

/**
 * Send <code>count</code> bytes from the buffer <code>data</code>
 * at the <code>offset</code>, on the socket.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		data	the send buffer
 * @param		offset	the offset into the buffer
 * @param		count	the number of bytes to send
 * @return		int		the actual number of bytes sent
 * @exception	IOException	if an error occurs while writing
 */
native int sendStreamImpl(byte[] data, int offset, int count) throws IOException;

/**
 * Answer the number of bytes available to read on the socket
 * without blocking.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	IOException	if an error occurs while peeking
 */
native int availableStreamImpl() throws IOException;

/**
 * @see javax.microedition.io.SocketConnection#getLocalAddress()
 */
public String getLocalAddress() throws IOException {
	if(host == null)
		// K00ac = Connection is closed
		throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));

	return addressToString(address);
}

/**
 * @see javax.microedition.io.SocketConnection#getLocalPort()
 */
public int getLocalPort() throws IOException {
	if(host == null)
		// K00ac = Connection is closed
		throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	return localport;
}

/**
 * @see javax.microedition.io.SocketConnection#getAddress()
 */
public String getAddress() throws IOException {

	if(host == null)
		// K00ac = Connection is closed
		throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));

	return addressToString(remoteaddress);
}

/**
 * @see javax.microedition.io.SocketConnection#getPort()
 */
public int getPort() throws IOException {
	if(host == null)
		// K00ac = Connection is closed
		throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	return remoteport;
}

/**
 * @see javax.microedition.io.SocketConnection#getSocketOption(byte)
 */
public int getSocketOption(byte optname)
	throws IllegalArgumentException, IOException {

	if(host == null)
		// K00ac = Connection is closed
		throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));

	try {
		switch (optname) {
			case DELAY :
				int result = getSocketOptionImpl(TCP_NODELAY)==0?1:0;
				if ((getSocketFlags() & FLAG_BROKEN_TCP_NODELAY) != 0) {
					return tcpNoDelay;
				} else {
					return result;
				}
			case KEEPALIVE:
				return getSocketOptionImpl(SO_KEEPALIVE);
			case LINGER:
				return getSocketOptionImpl(SO_LINGER);
			case RCVBUF:
				return getSocketOptionImpl(SO_RCVBUF);
			case SNDBUF:
				return getSocketOptionImpl(SO_SNDBUF);
			default:
				// K01c8 = Invalid socket option
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01c8"));
		}
	} catch (IOException e) {
		return -1;
	}
}

/**
 * @see javax.microedition.io.SocketConnection#setSocketOption(byte, int)
 */
public void setSocketOption(byte optname, int optval)
	throws IllegalArgumentException, IOException {

	if(host == null)
		// K00ac = Connection is closed
		throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));

	if(optval < 0)
		// K01c9 = Socket option value cannot be negative
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01c9"));

	try {
		switch (optname) {
			case DELAY :
				setSocketOptionImpl(TCP_NODELAY, optval == 0?1:0);
				tcpNoDelay = optval==0?0:1;
				break;
			case KEEPALIVE:
				setSocketOptionImpl(SO_KEEPALIVE, optval);
				break;
			case LINGER:
				setSocketOptionImpl(SO_LINGER, optval);
				break;
			case RCVBUF:
				setSocketOptionImpl(SO_RCVBUF, optval);
				break;
			case SNDBUF:
				setSocketOptionImpl(SO_SNDBUF, optval);
				break;
			default:
				// K01c8 = Invalid socket option
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01c8"));
		}
	} catch (IOException e) {
	}
}

private native int shutdownSideImpl(boolean inputSide) throws IOException;

/**
 * Returns the timeout for the socket.  This is the value of so_timeout if it was set as
 * a connection parameter or the value of DEFAULT_TIMEOUT if none was set.
 */
public int getTimeout() {
	return timeout;
}

}
