package com.ibm.oti.crypto;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.ByteArrayOutputStream;
import java.io.IOException;

public class Key {

	public static final int OPERATION_ENCRYPT = 1;
	public static final int OPERATION_DECRYPT = 2;

	public static final int PAD_NONE = 1;
	public static final int PAD_PKCS5 = 2;
	public static final int PAD_SSL = 3;
	public static final int PAD_TLS = 4;

	// The highest and lowest padding values
	public static final int PAD_MIN = 1;
	public static final int PAD_MAX = 4;

	Provider provider;
	byte[] key;
	int operation;
	int padtype;

	private boolean cryptInit = false;

	private byte[] buffInput;
	private int buffInputCount = 0;
	private ByteArrayOutputStream buffOutput = new ByteArrayOutputStream();

Key(Provider provider, byte[] key) throws IOException {
	if (provider.isDestroyed()) {
		// K01f8 = provider is destroyed
		throw new IOException(com.ibm.oti.util.Msg.getString("K01f8")); //$NON-NLS-1$
	}
	this.provider = provider;

	if (key.length != provider.getKeyLength()) {
		// K01f9 = incorrect key length
		throw new IOException(com.ibm.oti.util.Msg.getString("K01f9")); //$NON-NLS-1$
	}
	this.key = key;

	if (provider.getBlockLength()==0)
		buffInput = null;
	else
		buffInput = new byte[key.length];
}

public void cryptInit(int operation, int padtype, byte[] iv) throws IOException {
	if (operation != OPERATION_ENCRYPT && operation != OPERATION_DECRYPT) {
		// K01fb = invalid operation
		throw new IOException(com.ibm.oti.util.Msg.getString("K01fb")); //$NON-NLS-1$
	}
	this.operation = operation;

	if (padtype < PAD_MIN || padtype > PAD_MAX) {
		// K01f7 = invalid padding
		throw new IOException(com.ibm.oti.util.Msg.getString("K01f7")); //$NON-NLS-1$
	}
	this.padtype = padtype;

	provider.cryptInit(this, operation, padtype, iv);
	cryptInit = true;

	buffInputCount = 0;
	buffOutput.reset();
}

public void cryptUpdate(byte[] bytes, int offset, int length) throws IOException {
	if (!cryptInit)
		throw new IllegalStateException();

	if (buffInput != null && buffInputCount > 0) {
		int buffRemaining = buffInput.length - buffInputCount;
		if (length >= buffRemaining) {
			System.arraycopy(bytes, offset, buffInput, buffInputCount, buffRemaining);
			buffOutput.write(provider.cryptUpdate(this, buffInput, 0, buffInput.length, false));
			offset += buffRemaining;
			length -= buffRemaining;
			buffInputCount = 0;
		}
	}

	if (provider.getBlockLength() > 0) {
		// use length-1 to ensure that there is always something remaining in the buffer (for the final call)
		int blockcount = (length-1) / provider.getBlockLength();
		if (blockcount > 0) {
			buffOutput.write(provider.cryptUpdate(this, bytes, offset, blockcount * provider.getBlockLength() - offset, false));
			offset += blockcount * provider.getBlockLength();
			length -= blockcount * provider.getBlockLength();
		}
	} else {
		buffOutput.write(provider.cryptUpdate(this, bytes, offset, length - offset, false));
		offset += length;
		length -= length;
	}

	if (length > 0) {
		// add the remainder to the buffer
		System.arraycopy(bytes, offset, buffInput, buffInputCount, length);
		buffInputCount += length;
	}
}

public byte[] cryptFinish() throws IOException {
	if (buffInputCount > 0) {
		// finish the last block
		buffOutput.write(provider.cryptUpdate(this, buffInput, 0, buffInputCount, true));
	}

	cryptInit = false;
	return buffOutput.toByteArray();
}

}
