package com.ibm.oti.crypto;

import java.io.IOException;

/**
 * Implements functions supporting cryptographic operations
 *
 * <p>
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 * </p>
 *
 * @author MKingston IBM OTI Labs
 */
public final class Util {

	/**
	 * Method padPKCS5 pads a data block to a PKCS5 padded 8-byte block
	 *
	 * @param 		data			the data to be paddded
	 * @param 		offset			the data offset
	 * @param 		numDataBytes	the number of bytes to pad
	 * @return 		PCKS5 padded 8-byte data block
	 */
	static /*package*/ byte[] padPKCS5( byte[] data, int offset, int numDataBytes, int blockSize ){

		// Create 8-byte block to contain padding
		byte[] block = new byte[ blockSize ];

		// Copy valid data into the padding block
		if ( data.length > 0 )
			System.arraycopy( data, offset, block, 0, numDataBytes );

		// PKCS5 padding requires all padding blocks to contain the number of padding bytes
		int padlen = blockSize - numDataBytes;
		for( int i = numDataBytes; i < blockSize; i++ )
			block[ i ] = (byte) padlen;

		return block;

	}

	/**
	 * Method unpadPKCS5 unpads block assuming PKCS5 padding scheme
	 *
	 * @param 		block 		padded data block
	 * @return 		unpadded array of bytes (0-byte array if input was padding only)
	 *
	 * @throws		IOException		if an IO error occurs while copying the data
	 */
	static /*package*/ byte[] unpadPKCS5( byte[] block )
		throws IOException
	{
		int blockSize = block.length;

		// PKCS5 requires all padding bytes to be set to number of padding bytes.  We know that the
		// last byte must be padding, so we read the length of padding from it.
		int padlen = (int) block[ blockSize - 1 ];

		// PKCS5 padding must be between 1 and blockSize bytes (inclusive)
		if( ( padlen < 1 ) || ( padlen > blockSize ) ){

			/* [MSG "K01f7", "invalid padding"] */
			throw new IOException( com.ibm.oti.util.Msg.getString( "K01f7" ) );

		}

		//  All padding?  Return 0-byte array
		if ( padlen == blockSize )
			return new byte[ 0 ];

		// Create new byte array without padding
		byte[] unpadded = new byte[ blockSize - padlen ];
		System.arraycopy( block, 0, unpadded, 0, blockSize - padlen );

		return unpadded;

	}

	/**
	 * Pads a data block by appending the byte (blockSize-numDataBytes) to each unused byte in the datablock.
	 *
	 * @param 		data			the data to be paddded
	 * @param 		offset			the data offset
	 * @param 		numDataBytes	the number of actual databyes in the block
	 * @return 		The padded block
	 */
	static byte[] padTLS10(byte[] data, int offset, int numDataBytes, int dataBlockSize) {

		//TODO: The TLS standard allows for and recomends the adding of a random number of padding bytes over and
		//		above those needed to completely fill a partial record.  The only requirement is that complete blocks
		//      are returned.

		// Create a block to contain padded data
		byte[] block = new byte[dataBlockSize];

		// Copy valid data into the padding block
		System.arraycopy(data, offset, block, 0, numDataBytes);

		int padlen = dataBlockSize - numDataBytes;
		for(int i = numDataBytes; i < dataBlockSize; i++ ) {
			block[i] = (byte) (padlen - 1);
		}
		return block;
	}

	/**
	 * Unpads a block assuming TLS10 padding scheme
	 *
	 * @param 		block 		padded data
	 * @return 		unpadded array of bytes (0-byte array if input was padding only)
	 *
	 * @throws		IOException		if an IO error occurs while copying the data
	 */
	static byte[] unpadTLS10(byte[] block) throws IOException {

		// PKCS5 requires all padding bytes to be set to number of padding bytes.  We know that the
		// last byte must be padding, so we read the length of padding from it.
		int padlen = block[block.length - 1];

		// PKCS5 padding must be between 1 256 (inclusive)
		if ((padlen < 1 ) || (padlen > 255)) {
			/* [MSG "K01f7", "invalid padding"] */
			throw new IOException( com.ibm.oti.util.Msg.getString( "K01f7" ) ); //$NON-NLS-1$
		}

		//  All padding?  Return 0-byte array
		if (padlen == block.length) {
			return new byte[0];
		}

		// Create new byte array without padding
		byte[] unpadded = new byte[block.length - padlen - 1];
		System.arraycopy(block, 0, unpadded, 0, unpadded.length);

		return unpadded;
	}

	/**
	 * Method padSSL pads a data block
	 *
	 * @param 		data			the data to be padded
	 * @param 		offset			the data offset
	 * @param 		numDataBytes	the number of bytes to pad
	 * @return 		8-byte padded data block
	 *
	 * @throws		IOException		if an IO error occurs while copying the data
	 */
	static byte[] padSSL(byte[] data, int offset, int numDataBytes, int dataBlockSize) throws IOException {

		byte[] tmpBlock = new byte[dataBlockSize];
		int padlen = dataBlockSize - numDataBytes;

		// Copy valid data into padded block
		if (numDataBytes > 0) {
			System.arraycopy(data, offset, tmpBlock, 0, numDataBytes);
		}

		// SSL padding requires that all non-data bytes contain 0
		for (int i = numDataBytes; i < dataBlockSize - 1; i++) {
			tmpBlock[ i ] = 0;
		}

		// SSL padding requires last byte be set to number of padding bytes that precede it
		tmpBlock[dataBlockSize - 1] = (byte)(padlen - 1);

		return tmpBlock;
	}

	/**
	 * Method unpadSSL unpads block assuming SSL padding scheme
	 *
	 * @param 		block		padded 8-byte array
	 * @return 		unpadded array of bytes (0-byte array if input was padding only)
	 *
	 * @throws		IOException		if the padding scheme is invalid
	 */
	static byte[] unpadSSL(byte[] block) throws IOException {

		// SSL padding scheme sets last byte to contain number of padding bytes that precede it
		int padlen = (block[block.length - 1]) + 1;

		// SSL padding must be between 1 and 8 bytes (inclusive)
		if ((padlen < 1) || (padlen > block.length)) {
			/* [MSG "K01f7", "invalid padding"] */
			throw new IOException(com.ibm.oti.util.Msg.getString("K01f7")); //$NON-NLS-1$
		}

		//  All padding?  Return 0-byte array
		if (padlen == block.length) {
			return new byte[0];
		}

		// Return unpadded block
		byte[] unpadded = new byte[block.length - padlen];
		System.arraycopy(block, 0, unpadded, 0, unpadded.length);
		return unpadded;
	}

	/**
	 * Contencates two byte arrays.
	 */
	public static byte[] concatenate(byte[] b1, byte[] b2) {
		byte[] result = new byte[b1.length + b2.length];
		System.arraycopy(b1, 0, result, 0, b1.length);
		System.arraycopy(b2, 0, result, b1.length, b2.length);
		return result;
	}
}
