package com.ibm.oti.midlet.help;

import javax.microedition.rms.RecordStoreException;

import java.io.*;
import java.util.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 */

public final class MidletLoader {

private MidletLoader(){
}

	static private Object lock = new Object();
	static private boolean hasMutex = false;
	private static MidletSuiteId uniqueID;

	/**
	 * The MIDlet metadata class for storing permission mode information.
	 * For more information on MIDlet metadata, see getMetadataEntry().
	 */
	private static final String METADATA_CLASS_PERMISSIONMODE = "perm";

/**
 * StoreList is a class representing MidletSuite record store names mapped to actual file names.
 */

	private static StoreList stores;

public static class StoreList {
	private Hashtable suiteStores;
	private Hashtable suiteMetadata;

private void openStoreList()
{
	com.ibm.oti.connection.file.FileInputStream in = null;
	try {
		in = new com.ibm.oti.connection.file.FileInputStream(getStorePath() + "storelist.ini");
		byte buf[] = new byte[in.available()];
		int read = 0;
		while (read < buf.length)
			read += in.read(buf, read, buf.length - read);
		in.close();
		Reader reader;
		try {
			final String contents = com.ibm.oti.util.Util.convertFromUTF8(buf, 0, buf.length);
			reader = new Reader() {
				int pos = 0, size = contents.length();
				public int read() throws IOException {
					if (pos == size) return -1;
					return contents.charAt(pos++);
				}
				public int read(char[] buf, int offset, int len) throws IOException {
					// Not implemented, not required
					return -1;
				}
				public boolean ready() {
					return pos < size;
				}
				public void close() throws IOException {
				}
			};
		} catch (UTFDataFormatException e) {
			ByteArrayInputStream buffStream = new ByteArrayInputStream(buf);
			reader = new InputStreamReader(buffStream);
		}
		readFileData(reader);
		if (suiteStores.isEmpty()) {
			// This is not good, trigger the exception case below
			throw new Exception();
		}
	} catch (Exception e) {
		try {
			if (in != null) in.close();
		} catch (IOException ee) {}
		resetGlobals();
		recreateFile();
	}
}

protected void resetGlobals() {
		suiteStores = new Hashtable(5);
		Hashtable globals = new Hashtable(5);
		globals.put("dbname", "0");
		suiteStores.put("globals", globals);
		suiteMetadata = new Hashtable(5);
		suiteMetadata.put("globals", new Hashtable(5));
}

private static class ResuableStringBuffer {
	private char[] value;
	private int count;

	public ResuableStringBuffer(int capacity) {
		count = 0;
		value = new char[capacity];
	}

	private void ensureCapacityImpl(int min) {
		int twice = (value.length << 1) + 2;
		char[] newData = new char[min > twice ? min : twice];
		System.arraycopy(value, 0, newData, 0, count);
		value = newData;
	}

	public ResuableStringBuffer append(char ch) {
		if (count >= value.length) {
			ensureCapacityImpl(count + 1);
		}

		value[count] = ch;
		count++;

		return this;
	}

	public void renew() {
		count = 0;
	}

	public String toString() {
		return new String(value, 0, count);
	}
}

private String readLine(Reader in, ResuableStringBuffer sb) throws IOException {
	int c;
	sb.renew();
	while ((c = in.read()) != '\n') {
		sb.append((char)c);
	}
	return sb.toString();
}

private void readFileData(Reader in)
{
	Hashtable currentSection;
	Hashtable currentMetadata;
	Vector currentVector;
	String sectionName = null;
	String entry, entryKey, entryValue;
	int index;
	int ch;
	ResuableStringBuffer rsb = new ResuableStringBuffer(64);

	suiteStores = new Hashtable(5);
	suiteMetadata = new Hashtable(5);
	currentMetadata = null;
	currentSection = null;
	currentVector = null;
	if(in == null)
		// Could not open file.
		return;
	try
	{
		while(in.ready())
		{
			// Skip whitespace
			while ((ch = in.read()) != -1) {
					if ((ch >= 0x1c && ch <= 0x20) || (ch >= 0x9 && ch <= 0xd))
						continue;
					else
						break;
			}

			if(ch == -1)
			{
				// Flush the old
				if((currentSection != null)&&(currentVector != null))
				{
					currentSection.put("", currentVector);
				}
				return;
			}
			else if(ch == ';')
			{
				// Comment
				readLine(in, rsb);
			}
			else if(ch == '[')
			{
				// New section
				sectionName = readLine(in, rsb).trim();
				if(!sectionName.endsWith("]"))
				{
					// Not a section header!
					return;
				}
				sectionName = sectionName.substring(0, sectionName.length() - 1);
				// Flush the old
				if((currentSection != null)&&(currentVector != null))
				{
					currentSection.put("", currentVector);
				}
				currentVector = null;
				currentSection = new Hashtable(5);
				suiteStores.put(sectionName, currentSection);
				currentMetadata = new Hashtable(5);
				suiteMetadata.put(sectionName, currentMetadata);
			}
			else
			{
				if(currentSection == null)
				{
					// Entry with no section
					return;
				}

				// Section entry.
				entry = (((char)ch) + readLine(in, rsb)).trim();
				index = entry.indexOf('=');
				if(index == -1)
				{
					// Format is not xxx = yyy
					if(currentVector == null) currentVector = new Vector();
					currentVector.addElement(entry);
				}
				else
				{
					if (entry.startsWith("\"")) {
						// It's a RMS database
						entryKey = entry.substring(1, index - 2);
						entryValue = entry.substring(index + 1).trim();
						currentSection.put(entryKey, entryValue);
					} else {
						// It's a metadata entry
						int posColon = entry.indexOf(':');
						String entryClass = posColon>0 ? entry.substring(0, posColon) : "";
						entryKey = entry.substring(posColon+1, index-1);
						entryValue = entry.substring(index+1).trim();
						Metadata currentMetadataClass = (Metadata)currentMetadata.get(entryClass);
						if (currentMetadataClass==null) {
							currentMetadataClass = new Metadata(entryClass, sectionName);
							currentMetadataClass.metadataTable = new Hashtable(15);
							currentMetadata.put(entryClass, currentMetadataClass);
						}
						currentMetadataClass.metadataTable.put(entryKey, entryValue);
					}
				}
			}
		}
		// Flush the old
		if((currentSection != null)&&(currentVector != null))
		{
			currentSection.put("", currentVector);
		}
		return;
	}
	catch(IOException ex)
	{
		return;
	}

	finally
	{
		try
		{
			in.close();
		}
		catch(IOException ex){};
	}
}
/**
 * Constructs an approximation of the original INI file from
 * the Hashtable of Hashtables returned by <code>readFile</code>.
 *
 * @return the reconstructed file text
 */
private String recreateFileText()
{
	StringBuffer buf = new StringBuffer();
	Enumeration enum1;

	enum1 = suiteStores.keys();
	while(enum1.hasMoreElements())
	{
		String sectionName = (String)(enum1.nextElement());
		Hashtable sectionEntries = (Hashtable)(suiteStores.get(sectionName));
		Hashtable metadataEntries = (Hashtable)(suiteMetadata.get(sectionName));
		if (sectionEntries.size() == 0) {
			boolean empty = true;
			Enumeration enum2 = metadataEntries.keys();
			while (enum2.hasMoreElements()) {
				Metadata metadata = (Metadata)metadataEntries.get(enum2.nextElement());
				if (metadata.metadataTable.size() > 0) {
					empty = false;
					break;
				}
			}
			if (empty) continue;
		}

		buf.append("[");
		buf.append(sectionName);
		buf.append("]\n");

		if (sectionEntries != null) {
			Enumeration enum2 = sectionEntries.keys();
			while(enum2.hasMoreElements())
			{
				String entryKey = (String)(enum2.nextElement());
				String entryValue;
				entryValue = (String)sectionEntries.get(entryKey);
				buf.append("\"");
				buf.append(entryKey);
				buf.append("\"");
				buf.append(" = ");
				buf.append(entryValue);
				buf.append("\n");
			}
		}
		if (metadataEntries != null) {
			Enumeration enum2 = metadataEntries.keys();
			while(enum2.hasMoreElements())
			{
				String entryClass = (String)enum2.nextElement();
				Metadata metadataClass = (Metadata)metadataEntries.get(entryClass);
				Enumeration enum3 = metadataClass.keys();
				while(enum3.hasMoreElements())
				{
					String entryKey = (String)(enum3.nextElement());
					String entryValue = (String)(metadataClass.metadataTable.get(entryKey));
					if (entryValue != null) {
						buf.append(entryClass);
						buf.append(":");
						buf.append(entryKey);
						buf.append(" = ");
						buf.append(entryValue);
						buf.append("\n");
					}
				}
			}
		}
		buf.append("\n");
	}
	return buf.toString();
}
/**
 * Constructs an approximation of the original INI file from
 * the Hashtable of Hashtables returned by <code>readFile</code>.
 */
protected void recreateFile()
{
	String text;
	try {
		com.ibm.oti.connection.file.FileOutputStream out = new com.ibm.oti.connection.file.FileOutputStream(getStorePath() + "storelist.ini", false);
		text = recreateFileText();
		out.write(text.getBytes("UTF8"));
		out.close();
	} catch (IOException e) {
		return;
	}
}
}

private static String nextDBName() {
	synchronized (lock) {
		Hashtable globalList = (Hashtable) getStores().suiteStores.get("globals");
		String dbName = (String) globalList.get("dbname");
		globalList.put("dbname", Integer.toString(Integer.parseInt(dbName) + 1));
		/**
		 * format of name is
		 * 		Y/N accessible from other midlet suites
		 * 		Y/N writable from other midlet suites
		 * 		INT integer of database name
		 * 		So by default they are not accessible and not writable
		 */
		dbName = "NN" + dbName;
		return dbName;
	}
}

/**
 * Answer a boolean indicating whether or not a store exists
 */
public static boolean storeExists(MidletSuiteId midletSuiteId, String recordStoreName) {
	synchronized (lock) {
		Hashtable suiteList = (Hashtable) getStores().suiteStores.get(midletSuiteId.getId());
		if (suiteList == null) return false;
		return suiteList.containsKey(recordStoreName);
	}
}

/**
 * Answer a boolean indicating whether or not a store is accessible
 */
public static boolean storeAccessible(MidletSuiteId midletSuiteId, String recordStoreName) {
	synchronized (lock) {
		Hashtable suiteList = (Hashtable) getStores().suiteStores.get(midletSuiteId.getId());
		if (suiteList == null) return false;
		String value = (String) suiteList.get(recordStoreName);
		if (value == null) return false;
		if (midletSuiteId.equals(getMidletSuiteId()))
			return true;
		return value.charAt(0) == 'Y';
	}
}

/**
 * Get the named Metadata from the persistent store. The metadata
 * store is created if required.
 *
 * @param midletSuiteId
 * 				The midlet suite identifier. For Palm only
 * 				the databaseName must be initialized. Pass
 * 				null for the running suite.
 * @param metadataClassName
 * 				The name of the metadata class. For Palm, this
 * 				name must be registered in the knownClassName table.
 *
 * @return	A Metadata instance.
 */
public static Metadata getMetadataClass(MidletSuiteId midletSuiteId, String metadataClassName) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	// acquire mutex to access storelist.ini file
	MidletLoader.acquireMetadataStore();
	synchronized(lock) {
		if (midletSuiteId == null) midletSuiteId = getMidletSuiteId();
		String suiteId = midletSuiteId.getId();
		Hashtable suiteMetadata = (Hashtable) getStores().suiteMetadata.get(suiteId);
		if (suiteMetadata == null) {
			if (getStores().suiteStores.get(suiteId) == null)
				getStores().suiteStores.put(suiteId, new Hashtable(5));
			getStores().suiteMetadata.put(suiteId, suiteMetadata = new Hashtable(5));
		}
		Metadata metadataClass = (Metadata)suiteMetadata.get(metadataClassName);
		if (metadataClass == null) {
			metadataClass = new Metadata(metadataClassName, suiteId);
			metadataClass.metadataTable = new Hashtable(15);
			suiteMetadata.put(metadataClassName, metadataClass);
		}
		// release mutex so that other midlets can access storelist.ini file
		MidletLoader.releaseStore();
		return metadataClass;
	}
}

private static void saveMetadata(Metadata metadata) {
	MidletLoader.acquireMetadataStore();
	synchronized(lock) {
		StoreList list = getStores();
		Hashtable suiteMetadata = (Hashtable) list.suiteMetadata.get(metadata.midletSuiteId);
		if (suiteMetadata == null) {
			if (list.suiteStores.get(metadata.midletSuiteId) == null)
				list.suiteStores.put(metadata.midletSuiteId, new Hashtable(5));
			list.suiteMetadata.put(metadata.midletSuiteId, suiteMetadata = new Hashtable(5));
		}
		suiteMetadata.put(metadata.metaClassName, metadata);
		list.recreateFile();
		MidletLoader.releaseStore();
	}

}

/**
 * Answer a boolean indicating whether or not a store is accessible
 */
public static boolean storeWritable(MidletSuiteId midletSuiteId, String recordStoreName) {
	synchronized (lock) {
		Hashtable suiteList = (Hashtable) getStores().suiteStores.get(midletSuiteId.getId());
		if (suiteList == null) return false;
		String value = (String) suiteList.get(recordStoreName);
		if (value == null) return false;
		if (midletSuiteId.equals(getMidletSuiteId()))
			return true;
		return value.charAt(1) == 'Y';
	}
}

public static String[] listStores(MidletSuiteId midletSuiteId) {
	return listStores(midletSuiteId, true);
}

private static String[] listStores(MidletSuiteId midletSuiteId, boolean useMutex) {
	if (useMutex) {
		try {
			// acquire mutex to access storelist.ini file
			MidletLoader.acquireStore();
		} catch (RecordStoreException e) {
			return new String[0];
		}
	}
	synchronized (lock) {
		Hashtable suiteList = (Hashtable) getStores().suiteStores.get(midletSuiteId.getId());
		if (useMutex) {
			// release mutex so that other midlets can access storelist.ini file
			MidletLoader.releaseStore();
		}
		if (suiteList == null || suiteList.size() == 0) return null;
		String storeNames[] = new String[suiteList.size()];
		Enumeration keys = suiteList.keys();
		for (int i = 0; i < storeNames.length; i++) {
			storeNames[i] = (String) keys.nextElement();
		}
		return storeNames;
	}
}

private static native void deleteRecordStoreImpl(byte[] path)
	throws RecordStoreException;

private static native boolean acquireRMSMutex();

private static native boolean releaseRMSMutex();

public static void releaseStore() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	synchronized(lock) {
		if (!hasMutex)
			throw new Error("Mutex was not acquired.");
		stores = null;
		if (!releaseRMSMutex())
			throw new Error("Mutex was not released.");
		hasMutex = false;
	}
}

public static void acquireStore() throws RecordStoreException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	if (!acquireRMSMutex())
		throw new RecordStoreException("Unable to acquire Mutex.");
	synchronized(lock) {
		if (hasMutex)
			throw new Error("Mutex already acquired.");
		hasMutex = true;
	}
}

public static void acquireMetadataStore() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	if (!acquireRMSMutex())
		throw new Error("Unable to acquire Mutex.");
	synchronized(lock) {
		if (hasMutex)
			throw new Error("Mutex already acquired.");
		hasMutex = true;
	}
}

/**
 * Remove the store name from this Midlet Suite
 */
public static void removeStore(MidletSuiteId midletSuiteId, String recordStoreName) throws RecordStoreException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	synchronized (lock) {
		deleteRecordStoreImpl(com.ibm.oti.util.Util.getBytes(getStorePath(midletSuiteId, recordStoreName)));

		String suiteId = midletSuiteId.getId();
		Hashtable suiteList = (Hashtable) getStores().suiteStores.get(suiteId);
		if (suiteList == null) return;
		suiteList.remove(recordStoreName);
		Hashtable permList = (Hashtable) getStores().suiteMetadata.get(suiteId);
		if (suiteList.size() == 0 && (permList == null || permList.size() == 0)) {
			getStores().suiteStores.remove(suiteId);
		}
		if (getStores().suiteStores.size() == 1) {
			// there are no record stores at all, reset the counter
			getStores().resetGlobals();
		}
		getStores().recreateFile();
	}
}

/**
 * Delete the RMS databases for the named midletSuite.
 */
public static void deleteMidletSuiteRMS(MidletSuiteId midletSuiteId) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	// acquire mutex to access storelist.ini file
	MidletLoader.acquireMetadataStore();
	synchronized(lock) {
		String[] stores = listStores(midletSuiteId, false);
		if (stores != null) {
			for (int i=0; i<stores.length; i++) {
				try {
					String path = getStorePath(midletSuiteId, stores[i]);
					deleteRecordStoreImpl(com.ibm.oti.util.Util.getBytes(path));
				} catch (RecordStoreException e) {
					e.printStackTrace();
				}
			}
		}
		StoreList list = getStores();
		list.suiteStores.remove(midletSuiteId.getId());
		list.recreateFile();
		// release mutex so that other midlets can access storelist.ini file
		MidletLoader.releaseStore();
	}
}

/**
 * Delete the metadata for the named midletSuite.
 */
public static void deleteMidletSuiteMetadata(MidletSuiteId midletSuiteId) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	// acquire mutex to access storelist.ini file
	MidletLoader.acquireMetadataStore();
	synchronized(lock) {
		StoreList list = getStores();
		list.suiteMetadata.remove(midletSuiteId.getId());
		list.recreateFile();
		// release mutex so that other midlets can access storelist.ini file
		MidletLoader.releaseStore();
	}
}

//Set the flags for a specific RecordStore
public static void setFlags(MidletSuiteId midletSuiteId, String recordStoreName, int authMode, boolean writable) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	synchronized (lock) {
		Hashtable suiteList = (Hashtable) getStores().suiteStores.get(midletSuiteId.getId());
		if (suiteList == null) return;
		String value = (String) suiteList.get(recordStoreName);
		if (value == null) return;
		char[] valueC = value.toCharArray();
		valueC[0] = authMode == 1 ? 'Y' : 'N';
		valueC[1] = writable ? 'Y' : 'N';
		suiteList.put(recordStoreName, new String(valueC));
		getStores().recreateFile();
	}
}

private static StoreList getStores() {
	synchronized(lock) {
		if (!hasMutex)
			throw new Error("Mutex was not acquired.");
	}
	if (stores == null) {
		// Ensure that a unique ID has been assigned
		getMidletSuiteId();
		// Init and try to open the store list, this will create if necessary
		stores = new StoreList();
		stores.openStoreList();
	}
	return stores;
}

/**
 * Refresh the metadata cache.
 */
public static void refresh() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	stores = null;
}

/**
 * Answers a file path that represents a unique file location for this record Store.
 */
public static String getStorePath(MidletSuiteId midletSuiteId, String recordStoreName) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	synchronized (lock) {
		String suiteId = midletSuiteId.getId();
		boolean recreate = false;
		Hashtable suiteList = (Hashtable) getStores().suiteStores.get(suiteId);
		if (suiteList == null) {
			suiteList = new Hashtable(5);
			getStores().suiteStores.put(suiteId, suiteList);
			if (getStores().suiteStores.get(suiteId)==null)
				getStores().suiteStores.put(suiteId, new Hashtable(5));
			recreate = true;
		}
		String dbName = (String) suiteList.get(recordStoreName);
		if (dbName == null) {
			dbName = nextDBName();
			suiteList.put(recordStoreName, dbName);
			recreate = true;
		}
		if (recreate)
			getStores().recreateFile();
		dbName = dbName.substring(2);
		return getStorePath() + dbName;
	}
}

/**
 * Answers a file path that represents a directory location for this record Store.
 */
public static String getStorePath() {
	String separator = System.getProperty("file.separator");
	StringBuffer storePath = new StringBuffer();
	String userHome = System.getProperty("user.home");
	storePath.append(userHome);
	if (!userHome.endsWith(separator))
		storePath.append(separator);
	storePath.append("recordStores");
	storePath.append(separator);
	return storePath.toString();
}

public static void init(String midletName, String midletVendor) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	if (midletName == null) {
		midletName = "Unknown";
		midletVendor = "IBM";
	}
	synchronized (lock) {
		uniqueID = new MidletSuiteId(midletName, midletVendor);
	}
}

/**
 * Answers a unique identifier for the Midlet suite of the caller: a RecordStore
 * object. The unique id's are stored in a special RMS DB
 *
 * @see		#init(String, String)
 */
public static MidletSuiteId getMidletSuiteId() {
	synchronized(lock) {
		if (uniqueID==null) init(null, null);
	}
	return uniqueID;
}

/**
 * Answer an integer representing the current mode of a particular permission.
 * If the permission has not been associated with this suite, 0 is returned.
 * The legitimate values return will be one of:
 *
 * public static final int MODE_DENY_ALWAYS = 1;
 * public static final int MODE_DENY_ONCE = 2
 * public static final int MODE_GRANT_ONCE = 3;
 * public static final int MODE_GRANT_SESSION = 4;
 * public static final int MODE_GRANT_ALWAYS = 5;
 *
 * @param permissionName a J2ME value like javax.microedition.io.http
 *
 * @return an int of 0 if not there or one of the value listed above.
 */
public static int getPermissionMode(String permissionName) {
	return getPermissionMode(null, permissionName);
}

/**
 * Answer an integer representing the current mode of a particular permission
 * for the given MIDlet name and vendor.
 * If the permission has not been associated with this suite, 0 is returned.
 * The legitimate values return will be one of:
 *
 * public static final int MODE_DENY_ALWAYS = 1;
 * public static final int MODE_DENY_ONCE = 2
 * public static final int MODE_GRANT_ONCE = 3;
 * public static final int MODE_GRANT_SESSION = 4;
 * public static final int MODE_GRANT_ALWAYS = 5;
 *
 * @param midletSuiteId
 * 				The midlet suite identifier. For Palm only
 * 				the databaseName must be initialized. Pass
 * 				null for the running suite.
 * @param permissionName a J2ME value like javax.microedition.io.http
 *
 * @return an int of 0 if not there or one of the value listed above.
 */
public static int getPermissionMode(MidletSuiteId midletSuiteId, String permissionName) {
	String valueString = getMetadataEntry(midletSuiteId, METADATA_CLASS_PERMISSIONMODE, permissionName);
	if (valueString==null) return 0;
	return Integer.parseInt(valueString);
}

/**
 * Answers the metadata String for the given class and key for a particular midlet.
 *
 * Each midlet can store arbitrary metadata associated with it that is persisted across sessions.  Each midlet can have
 * different classes of metadata, such that each class may have keys that are unique within the class.  For example,
 * permission mode metadata is stored in the "perm" metadata class for midlets.  This can be represented as a heirarchy
 * such that a midlet contains a set of metadata classes which contain a set of keys which correspond to metadata Strings.
 *
 * @param midletSuiteId
 * 				The midlet suite identifier. For Palm only
 * 				the databaseName must be initialized. Pass
 * 				null for the running suite.
 * @param metadataClassName
 * 				The name of the metadata class. For Palm, this
 * 				name must be registered in the knownClassName table.
 *
 * @param key the metadata key
 *
 * @return the metadata value for the particular midlet, class, key triplet or null if the key, class, or midlet does
 * not exist.
 */
public static String getMetadataEntry(MidletSuiteId midletSuiteId, String metadataClassName, String key) {
	synchronized (lock) {
		Metadata metadataClass = getMetadataClass(midletSuiteId, metadataClassName);
		return metadataClass.getEntry(key);
	}
}

/**
 * Set an integer representing the current mode of a particular permission.
 * If the permission should no longer be associated with the suite use 0.
 * The legitimate values will be one of:
 *
 * public static final int MODE_DENY_ALWAYS = 1;
 * public static final int MODE_DENY_ONCE = 2
 * public static final int MODE_GRANT_ONCE = 3;
 * public static final int MODE_GRANT_SESSION = 4;
 * public static final int MODE_GRANT_ALWAYS = 5;
 *
 * @param permissionName a J2ME value like javax.microedition.io.http
 * @param value an int of 0 if not there or one of the value listed above.
 */
public static void setPermissionMode(String permissionName, int value) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	setPermissionMode(null, permissionName, value);
}

/**
 * Set an integer representing the current mode of a particular permission
 * for the given MIDlet name and vendor.
 * If the permission should no longer be associated with the suite use 0.
 * The legitimate values will be one of:
 *
 * public static final int MODE_DENY_ALWAYS = 1;
 * public static final int MODE_DENY_ONCE = 2
 * public static final int MODE_GRANT_ONCE = 3;
 * public static final int MODE_GRANT_SESSION = 4;
 * public static final int MODE_GRANT_ALWAYS = 5;
 *
 * @param midletSuiteId		The midlet suite identifier. For Palm only
 * 							the databaseName must be initialized. Pass
 * 							null for the running suite.
 * @param permissionName	a J2ME value like javax.microedition.io.http
 * @param value 			an int of 0 if not there or one of the value listed above.
 */
public static void setPermissionMode(MidletSuiteId midletSuiteId, String permissionName, int value) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	if (value==0) {
		// no permission mode should be set
		removeMetadataEntry(midletSuiteId, METADATA_CLASS_PERMISSIONMODE, permissionName);
	} else {
		// store the permission mode
		setMetadataEntry(midletSuiteId, METADATA_CLASS_PERMISSIONMODE, permissionName, Integer.toString(value));
	}
}

/**
 * Sets the metadata for a given midlet, class, and key.
 *
 * @param midletSuiteId
 * 				The midlet suite identifier. For Palm only
 * 				the databaseName must be initialized. Pass
 * 				null for the running suite.
 * @param metadataClassName
 * 				The name of the metadata class. For Palm, this
 * 				name must be registered in the knownClassName table.
 * @param key the metadata key
 * @param value the metadata value
 */
public static void setMetadataEntry(MidletSuiteId midletSuiteId, String metadataClassName, String key, String value) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	Metadata metadataClass = getMetadataClass(midletSuiteId, metadataClassName);
	metadataClass.setEntry(key, value);
	try {
		metadataClass.save();
	} catch (IOException e) {
		System.err.println("MidletLoader.setMetadataEntry: " + e);
	}
}

/**
 * Removes the metadata for a given midlet, class, and key.
 *
 * @param midletSuiteId
 * 				The midlet suite identifier. For Palm only
 * 				the databaseName must be initialized. Pass
 * 				null for the running suite.
 * @param metadataClassName
 * 				The name of the metadata class. For Palm, this
 * 				name must be registered in the knownClassName table.
 * @param key the metadata key
 */
public static void removeMetadataEntry(MidletSuiteId midletSuiteId, String metadataClassName, String key) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
	Metadata metadataClass = getMetadataClass(midletSuiteId, metadataClassName);
	metadataClass.removeEntry(key);
	try {
		metadataClass.save();
	} catch (IOException e) {
		System.err.println("MidletLoader.removeMetadataEntry: " + e);
	}
}

public static final class Metadata {
	private String metaClassName, midletSuiteId;
	private Hashtable metadataTable;
	private boolean dirty = false;
	private Metadata(String name, String id) {
		metaClassName = name;
		midletSuiteId = id;
	}
	public String getEntry(String key) {
		return (String)metadataTable.get(key);
	}
	public void setEntry(String key, String value) {
		synchronized(lock) {
			String oldValue = (String)metadataTable.get(key);
			if (!value.equals(oldValue)) {
				metadataTable.put(key, value);
				dirty = true;
			}
		}
	}
	public void removeEntry(String key) {
		synchronized(lock) {
			if (metadataTable.remove(key) != null)
				dirty = true;
		}
	}
	public Enumeration keys() {
		return metadataTable.keys();
	}
	public void clear() {
		synchronized(lock) {
			metadataTable.clear();
		}
	}
	public void save() throws IOException {
		synchronized(lock) {
			if (dirty) {
				MidletLoader.saveMetadata(this);
				dirty = false;
			}
		}
	}
}
}
