package com.ibm.oti.security.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import java.util.*;
import com.ibm.oti.midlet.help.MidletLoader;

public class PermissionManager {

	// the permission manager instance
	private static PermissionManager manager = null;

	private String suiteName;
	// the permissions defined in the JAD file
	private String[] jadPermissions;
	private ProtectionDomain protectionDomain;
	private UserPermissionAgent userAgent;
	private Hashtable cachedModes = new Hashtable();

	// the permission checking lock
	private Object permCheckLock = new Object();

/**
 * Sets the permission manager instance.
 */
public static void setManager(PermissionManager newManager) {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	manager = newManager;
}

/**
 * Returns the permission manager instance.
 */
public static PermissionManager getManager() {
	return manager;
}

public PermissionManager(String suiteName, String[] jadPermissions, String protectionDomainId, UserPermissionAgent userAgent) {
	this(suiteName, jadPermissions, protectionDomainId, userAgent, SecurityPolicy.getInstance());
}

public PermissionManager(String suiteName, String[] jadPermissions, String protectionDomainId, UserPermissionAgent userAgent, SecurityPolicy policy) {
	this.suiteName = suiteName;
	this.jadPermissions = jadPermissions!=null ? jadPermissions : new String[0];
	this.protectionDomain = policy.getProtectionDomain(protectionDomainId);
	if (protectionDomain==null) {
		// if it is untrusted and this does not exist in the policy file, use the definition from the default policy file
		if (protectionDomainId.equals("untrusted")) {
			protectionDomain = SecurityPolicy.getDefaultPolicy().getProtectionDomain("untrusted");
		}
	}
	if (protectionDomain==null) throw new IllegalArgumentException();
	this.userAgent = userAgent!=null ? userAgent : UserPermissionAgent.DEFAULT_AGENT;
}

/**
 * Checks a permission for the current MIDlet.  The following conditions must be true for a permission to be granted:
 * 		1. It must have been requested in the JAD.
 * 		2. It must be allowed explicitly or through user interaction for the current MIDlet's protection domain.
 * 		3. If user interaction is required, the user must grant the permission.
 *
 * Based on the user's response, user interaction may or may not be necessary the next time the permission is checked.
 *
 * @param	permissionName	String	The name of the permission that is being requested (see the MIDP 2.0 spec
 * 									for details of permission names).
 */
public void checkPermission(String permissionName) throws SecurityException {

	// make sure that the MIDlet requested this permission (except if we are bound to the untrusted domain)
	if (!protectionDomain.getId().equals("untrusted") && !isJadPermission(permissionName))
		// K01e1 = access to {0} was not requested in JAD
		throw new SecurityException(com.ibm.oti.util.Msg.getString("K01e1", permissionName));

	int domainPermission = protectionDomain.getHighestPermissionLevel(permissionName);
	if (domainPermission==ProtectionDomain.PERMISSION_LEVEL_ALLOW)
		return;
	if (domainPermission==ProtectionDomain.PERMISSION_LEVEL_DENY)
		// K01e2 = access to {0} denied by protection domain
		throw new SecurityException(com.ibm.oti.util.Msg.getString("K01e2", permissionName));

	// determine the set of allowed user modes
	int[] allowedUserModes = protectionDomain.getAllowedUserPermissionModes(permissionName);

	// To defend against a MIDlet using the same protected resource in multiple threads simultaneously,
	// make checking the cache, asking the user and caching the new permission atomic
	int userMode = 0;
	synchronized (permCheckLock) {

		// check for a cached user mode (a previous decision by the user)
		int cachedUserMode = 0;
		Integer cachedMode = (Integer)cachedModes.get(permissionName);
		if (cachedMode == null) {
			if (this == manager)
				cachedUserMode = MidletLoader.getPermissionMode(permissionName);
		} else {
			cachedUserMode = cachedMode.intValue();
		}
		if (cachedUserMode != 0) {
			// make sure this is in the set of allowed user modes
			for(int i=0; i<allowedUserModes.length; i++) {
				if (allowedUserModes[i]==cachedUserMode) {
					userMode=cachedUserMode;
					break;
				}
			}
		}

		// Check if the cached permission is a denial of access
		if (userMode==UserPermissionAgent.MODE_DENY_ALWAYS||userMode==UserPermissionAgent.MODE_DENY_SESSION)
			// K01e3 = access to {0} denied by user
			throw new SecurityException(com.ibm.oti.util.Msg.getString("K01e3", permissionName));

		// Check if the cached permission is a grant of access
		if (userMode==UserPermissionAgent.MODE_GRANT_ALWAYS||userMode==UserPermissionAgent.MODE_GRANT_SESSION)
			return;

		// check with the user
		if (userMode==0) {

			// determine the default user mode
			int domainDefaultPermission = protectionDomain.getDefaultPermissionLevel(permissionName);
			int defaultUserMode;
			if (domainDefaultPermission==ProtectionDomain.PERMISSION_LEVEL_USER_ONESHOT) {
				defaultUserMode = UserPermissionAgent.MODE_GRANT_ONCE;
			} else if (domainDefaultPermission==ProtectionDomain.PERMISSION_LEVEL_USER_SESSION) {
				defaultUserMode = UserPermissionAgent.MODE_GRANT_SESSION;
			} else { // domainDefaultPermission==ProtectionDomain.PERMISSION_LEVEL_USER_BLANKET
				defaultUserMode = UserPermissionAgent.MODE_GRANT_ALWAYS;
			}

			// check with the user
			userMode = userAgent.checkPermission(suiteName, permissionName, allowedUserModes, defaultUserMode);

		}

		// cache the user mode (don't cache if it only has it's effect once)
		if (userMode != cachedUserMode
			&& userMode!=UserPermissionAgent.MODE_GRANT_ONCE
				&& userMode!=UserPermissionAgent.MODE_DENY_ONCE)
		{
			// cache the user mode in the active (session) store
			cachedModes.put(permissionName, new Integer(userMode));

			// cache the user mode in the persistent store (don't cache session modes)
			if (this == manager && userMode != UserPermissionAgent.MODE_GRANT_SESSION &&
				userMode != UserPermissionAgent.MODE_DENY_SESSION)
					MidletLoader.setPermissionMode(permissionName, userMode);
		}
	}

	if (userMode==UserPermissionAgent.MODE_DENY_ALWAYS||userMode==UserPermissionAgent.MODE_DENY_ONCE ||
		userMode == UserPermissionAgent.MODE_DENY_SESSION)
			// K01e3 = access to {0} denied by user
			throw new SecurityException(com.ibm.oti.util.Msg.getString("K01e3", permissionName));

	return;
}

/**
 * Remove all session permissions. Used for testing.
 */
public void removeSessionPermissions() {
	synchronized (cachedModes) {
		Enumeration ke = cachedModes.keys();
		Enumeration ve = cachedModes.elements();
		while (ve.hasMoreElements()) {
			String key = (String)ke.nextElement();
			int mode = ((Integer)ve.nextElement()).intValue();
			if (mode == UserPermissionAgent.MODE_GRANT_SESSION ||
				mode == UserPermissionAgent.MODE_DENY_SESSION)
					cachedModes.remove(key);
		}
	}
}

private boolean isJadPermission(String permissionName) {
	for(int i=0; i<jadPermissions.length; i++) {
		if (jadPermissions[i].equals(permissionName)) return true;
	}
	return false;
}

}
