package com.ibm.oti.security.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2004  All Rights Reserved
 */

/**
 * Abstract class to prompt the user about permissions.  Subclasses should implement checkPermission() such that
 * the user is prompted.
 */
public abstract class UserPermissionAgent {

	/**
	 * This permission should be denied for all requests.
	 */
	public static final int MODE_DENY_ALWAYS = 1;

	/**
	 * This permission should be denied for this request.  The user will be prompted again if the same permission
	 * is requested again.
	 */
	public static final int MODE_DENY_ONCE = 2;

	/**
	 * This permission should be granted for this request.  The user will be prompted again if the same permission
	 * is requested again.
	 */
	public static final int MODE_GRANT_ONCE = 3;

	/**
	 * This permission should be granted all requests in this session.  The user will be prompted again if the same
	 * permission is requested again in a subsequent session.  A session is defined as from the invocation of a
	 * MIDlet suite until it terminates.
	 */
	public static final int MODE_GRANT_SESSION = 4;

	/**
	 * This permission should be denied all requests in this session.  The user will be prompted again if the same
	 * permission is requested again in a subsequent session.  A session is defined as from the invocation of a
	 * MIDlet suite until it terminates.
	 */
	public static final int MODE_DENY_SESSION = 6;

	/**
	 * This permission should be granted for all requests.
	 */
	public static final int MODE_GRANT_ALWAYS = 5;

	/**
	 * The default user permission agent.  The behaviour of this agent is to deny all requests.
	 */
	static final UserPermissionAgent DEFAULT_AGENT = new UserPermissionAgent() {
		public int checkPermission(String suiteName, String permissionName, int[] allowedModes, int defaultMode) {
			return MODE_DENY_ONCE;
		}
	};

/**
 * Prompts the user to grant or deny a permission.  There are five modes that a user can select: MODE_DENY_ALWAYS,
 * MODE_DENY_ONCE, MODE_GRANT_ONCE, MODE_GRANT_SESSION, or MODE_GRANT_ALWAYS.  For more information on the
 * semantics of these modes, see their javadoc above.
 *
 * @param	suiteName		String	The name of the MIDlet suite requesting the permission.
 * @param	permissionName	String	The name of the permission that is being requested (see the MIDP 2.0 spec
 * 										for details of permission names).
 * @param	allowedModes	int[]	The constants representing the allowed modes that the user may select for
 * 										this permission.  These values are based on the protection domain
 * 										that the MIDlet suite is bound to and the permission that is being
 * 										requested.
 * @param	defaultMode		int		The constant representing the mode that should be presented to the user
 * 										as the default choice.  This value is based on the protection domain
 * 										that the MIDlet suite is bound to and the permission that is being
 * 										requested.
 *
 * @return int	The constant representing the mode that the user selected.  This must be one of the elements
 * 		in the allowedModes array.
 */
public abstract int checkPermission(String suiteName, String permissionName, int[] allowedModes, int defaultMode);

}
