
/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

package com.ibm.oti.security.provider;

import java.util.*;

/**
 *	This is the class represents ASN.1 Object Identifiers.
 *
 * @author		OTI
 * @version		initial
 */
final public class ASN1OID {
	private int size = 0;
	private int[] representation = null;
	private int hashCode;
		// We cache hashCode for speed. We can do this because an ID should
		// be immutable. Senders of 'representation()' must not tamper with the
		// byte array itself

/**
 * Constructs an instance of this class.
 */
private ASN1OID() {
	super();
}

/**
 * Constructs an instance of this class given the representation.
 *
 * @param idValue	byte[]	The actual value of this ASN.1 OID, represented as a byte[]
 */
public ASN1OID(int[] idValue) {
	this(idValue.length);
	this.representation = idValue;
	cacheHashCode();
}

public ASN1OID( String idValue ){

	this();
	int[] elements = stringToIntOID( idValue );
	this.representation = elements;

	cacheHashCode();

}

/**
 * Constructs an instance of this class given the representation size.
 *
 * @param size	int	The size of the representation of this ASN.1 OID object
 */
private ASN1OID(int size) {
	this();
	this.size = size;
}

/**
 * Return the pre-computed hashCode value for the receiver.
 */
private void cacheHashCode() {
	hashCode = 0;
	if (this.representation != null) {
		for (int i = 0; i < this.representation.length; i++) {
			hashCode += (this.representation[i] & 0xff);
		}
	}
}
/**
 * Compares two objects for equality. Returns a boolean that indicates
 * whether this object is equivalent to the specified object. This method
 * is used when an object is stored in a hashtable.
 * @param obj the Object to compare with
 * @return true if these Objects are equal; false otherwise.
 * @see java.util.Hashtable
 */
public boolean equals(Object obj) {
	if (obj == null) return false;
	if (this == obj) return true;
	if ( ! (obj instanceof ASN1OID) ) return false;

	ASN1OID other = (ASN1OID) obj;
	if (this.size != other.size) return false;

	// Catches the case where both are null or same byte array
	if ( this.representation == other.representation) return true;

	// If only one of them is null, they are different
	if ( (this.representation == null ) ||
			(other.representation == null ) ) return false;

	// Now it is safe to compare both byte arrays... and they have the same size
	for (int i = 0; i < this.representation.length; i++) {
		if (this.representation[i] != other.representation[i]) {
			return false;
		}
	}
	return true;

}

/**
 * Generates a hash code for the receiver.
 * This method is supported primarily for
 * hash tables, such as those provided in java.util.
 *
 * @return an integer hash code for the receiver
 * @see java.util.Hashtable
 */
public int hashCode() {
	return hashCode;
}

/**
 * Returns the actual representation for the receiver.
 * For instance, if the receiver is 1.2.840.9, it would
 * return a byte[] { 1 , 2, 840, 9}.
 *
 * @return byte[]	The actual representation for this ASN.1 OID
 * @see java.util.Hashtable
 */
public int[] representation() {
	return representation;
}

/**
 * Converts a int[] with the representation for an ASN.1 OID into
 * its dot-separated, String representation. For instance, if the
 * representation were { 1 , 2, 840, 9} this method would return "1.2.840.9".
 *
 * @return String	A dot-separated representation of the receiver.
 */
public static String OIDToString (int[] oid) {
	StringBuffer result = new StringBuffer();
	for (int i = 0; i < oid.length - 1; i++) {
		result.append (oid[i]);
		result.append ('.');
	}
	result.append (oid[oid.length-1]);
	return result.toString();
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @return		String	a printable representation for the receiver.
 *
 */
public String toString() {
	return OIDToString(representation);
}

/**
 * Converts a String dot-separated with the representation for an ASN.1 OID into
 * its int[] with the representation. For instance, if the
 * representation were "1.2.840.9" this method would return { 1 , 2, 840, 9}.
 *
 * @param 	oid	String	A String dot-separated representation.
 *
 * @return	int[]	the representation for an ASN.1 OID
 *
 */
public static int[] stringToIntOID(String oid) {
	Vector list = new Vector();
 	int start = 0;
 	int end = 0;
 	int size = 0;
 	while (start < oid.length()) {
 		end = oid.indexOf(".", start);
 		if (end == -1 ) end = oid.length();
 		list.addElement(oid.substring(start, end));
 		size++;
 		start = end + 1;
 	}
	Enumeration listEnum = list.elements();
	int[] res = new int[size];
	for (int i = 0; listEnum.hasMoreElements(); i++) {
		res[i] = Integer.parseInt((String)listEnum.nextElement());
	}

	return res;
}

}
