
package com.ibm.oti.security.provider;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

import java.io.InputStream;
import com.ibm.oti.util.math.BigInteger;

import java.util.Date;
import java.util.Vector;

import com.ibm.oti.util.ASN1Decoder.Node;

import javax.microedition.pki.CertificateException;

public class X509Certificate implements javax.microedition.pki.Certificate {

	private final X509CertImpl impl;

	static final String extendedKeyUsageOID = "2.5.29.37";
	static final String issuerAltNamesOID = "2.5.29.18";
	static final String subjectAltNamesOID = "2.5.29.17";

	public static final java.util.Hashtable OID_DATABASE = new java.util.Hashtable(5);
	static {
		// encryption algorithms
		OID_DATABASE.put("1.2.840.113549.1.1.2", "MD2withRSA");
		OID_DATABASE.put("1.2.840.113549.1.1.4", "MD5withRSA");
		OID_DATABASE.put("1.2.840.113549.1.1.5", "SHA1withRSA");
		OID_DATABASE.put("1.2.840.10040.4.3", "SHA1withDSA");

		// message digest algorithms
		OID_DATABASE.put("1.3.14.3.2.26", "SHA");
		OID_DATABASE.put("1.2.840.113549.2.5", "MD5");
		OID_DATABASE.put("1.2.840.113549.2.2", "MD2");

		// key factories
		OID_DATABASE.put("1.2.840.113549.1.1.1", "RSA");
		OID_DATABASE.put("1.2.840.10040.4.1", "DSA");
		OID_DATABASE.put("1.3.14.3.2.12", "DSA");
		OID_DATABASE.put("1.2.840.10046.2.1", "DiffieHellman");
	}

	public X509Certificate( X509CertImpl impl ){
		this.impl = impl;
	}

	/**
	 * Checks if the certificate represented by the receiver is still valid.
	 * Throws an exception if the certificate is not valid.
	 *
	 * @throws	CertificateExpiredException		The certificate is no longer valid, it has expired
	 * @throws	CertificateNotYetValidException	The certificate is not valid yet
	 */
	public void checkValidity()
		throws CertificateException {
		impl.checkValidity (new Date());
	}
	/**
	 * Checks if the certificate represented by the receiver is valid for the
	 * date provided. Throws an exception if the certificate is not valid.
	 *
	 * @param		date	Date	date to use when testing the validity of the certificate
	 * @throws	CertificateExpiredException		The certificate is no longer valid, it has expired
	 * @throws	CertificateNotYetValidException	The certificate is not valid yet
	 */
	public void checkValidity(Date date)
		throws CertificateException {
		impl.checkValidity( date );
	}

	private static final String OID_BasicConstraints = "2.5.29.19";

	/**
	 * Return the path length constraint for the BasicConstraints for the receiver.
	 * The value comes from the BasicConstraints extension (OID = 2.5.29.19)
	 *
	 * @return		int		the path length constraint for the receiver if BasicConstraints is defined.
	 * 						Return -1 otherwise.
	 *
	 */
	public int getBasicConstraints() {
		return impl.getBasicConstraints();
	}

	/**
	 * Return the issuer distinguished name for the receiver
	 *
	 * @return		Principal	the issuer distinguished name for the receiver
	 */
	public String getIssuer() {
		return impl.getIssuer();
	}

	/**
	 * Return the issuer unique ID for the receiver
	 *
	 * @return		boolean[]	the issuer unique ID for the receiver. null if not present.
	 */
	public boolean[] getIssuerUniqueID() {
		return impl.getIssuerUniqueID();
	}

	private static final String OID_KeyUsage = "2.5.29.15";
	/**
	 * Return a bit array representation for the key usage for the receiver. The
	 * value comes from the KeyUsage extension (OID = 2.5.29.15)
	 *
	 * @return		boolean[]	 a bit array representation for the key usage for the receiver.
	 */
	public boolean[] getKeyUsage() {
		return impl.getKeyUsage();
	}

	/**
	 * Return the Date until when the certificate is valid
	 *
	 * @return		Date		the Date until when the certificate is valid
	 */
	public long getNotAfter() {
		return impl.getNotAfter();
	}

	/**
	 * Return the Date when the certificate starts to be valid
	 *
	 * @return		Date		the Date when the certificate starts to be valid
	 */
	public long getNotBefore() {
		return impl.getNotBefore();
	}

	/**
	 * Return the serial number for the receiver
	 *
	 * @return		BigInteger	the serial number for the receiver
	 */
	public String getSerialNumber() {
		return impl.getSerialNumber();
	}
	/**
	 * Return the certificate signature algorithm name for the receiver.
	 *
	 * @return		String		the certificate signature algorithm name for the receiver.
	 */
	public String getSigAlgName() {
		return impl.getSigAlgName();
	}
	/**
	 * Return the certificate signature OID for the receiver.
	 *
	 * @return		String		the certificate signature OID for the receiver.
	 */
	public String getSigAlgOID() {
		return impl.getSigAlgOID();
	}

	/**
	 * Return the signature algorithm parameters for the receiver, DER-encoded.
	 *
	 * @return		byte[]		the signature algorithm parameters for the receiver, DER-encoded. null if not present.
	 */
	public byte[] getSigAlgParams() {
		return impl.getSigAlgParams();
	}
	/**
	 * Return the actual signature value for the receiver
	 *
	 * @return		byte[]		the actual signature value for the receiver
	 */
	public byte[] getSignature() {
		return impl.getSignature();
	}

	/**
	 * Return the subject distinguished name for the receiver
	 *
	 * @return		Principal	the subject distinguished name for the receiver
	 */
	public String getSubject() {
		return impl.getSubject();
	}

	/**
	 * Return the subject unique ID for the receiver
	 *
	 * @return		boolean[]	the subject unique ID for the receiver
	 */
	public boolean[] getSubjectUniqueID() {
		return impl.getSubjectUniqueID();
	}
	/**
	 * Return the tbsCertificate for the receiver, DER-encoded
	 *
	 * @return		byte[]	the tbsCertificate for the receiver, DER-encoded
	 */
	public byte[] getTBSCertificate()
		throws CertificateException {
		return impl.getTBSCertificate();
	}
	/**
	 * Return the version number for this certificate (1 for v1, 2 for v2, 3 for v3)
	 *
	 * @return		int		the version number for this certificate (1 for v1, 2 for v2, 3 for v3)
	 */
	public String getVersion() {
		return impl.getVersion();
	}

	private static final String[] SUPPORTED_CRITICAL_EXTENSIONS = {
		OID_BasicConstraints,
		OID_KeyUsage,
	};

	/**
	 * Return the DER-encoded OCTECT String for the extension value defined by oid in the receiver.
	 *
	 * @return		byte[]	the DER-encoded OCTECT String for the extension value defined by oid in the receiver.
	 */
	public byte[] getExtensionValue(String oid){
		return impl.getExtensionValue( oid );
	}

	/**
	 * Answers the encoded representation for this certificate.
	 *
	 * @return		the encoded representation for this certificate.
	 * @throws	CertificateEncodingException	if encoding errors are detected
	 */
	public byte[] getEncoded()
	{
		return impl.getEncoded();
	}

	/**
	 * Answers the public key corresponding to this certificate.
	 *
	 * @return		the public key corresponding to this certificate.
	 */
	public byte[] getPublicKey(){
		return impl.getPublicKey();
	}

	/**
	 * Answers a string containing a concise, human-readable
	 * description of the receiver.
	 *
	 * @return		String
	 *					a printable representation for the receiver.
	 */
	public String toString () {
		StringBuffer result = new StringBuffer();
		result.append ("X.509 Certificate v");
		result.append (getVersion());
		result.append ("\nSubjectDN: ");
		result.append (getSubject());
		result.append ("\n\tNot Before: ");
		result.append (getNotBefore());
		result.append ("\n\tNot After: ");
		result.append (getNotAfter());
		result.append ("\nIssuerDN: ");
		result.append (getIssuer());
		result.append ("\nSerialNumber: ");
		result.append (getSerialNumber());
		result.append ("\nSigAlgName: ");
		result.append (getSigAlgName());
		result.append (", SigAlgOID: ");
		result.append (getSigAlgOID());
		return result.toString();
	}

	/**
	 * Lookup the OID alias for the specified key in the list of providers.
	 * Fill in the provider name, if found. Look in the OID database
	 * table if no alias is found in the providers.
	 */
	static String getAlias(String key, String oid, String[] providerName) {
		return X509CertImpl.getAlias( key, oid, providerName );
	}

	/**
	 * Generates and initializes a Certificate from a java byte[] object representing
	 * an ASN.1 DER certificate
	 *
	 * @param		certificateBytes	byte[]	The byte[] representing the ASN.1 certificate
	 *
	 * @return	Certificate 	an initialized Certificate
	 * @throws	CertificateException 	if parsing problems are detected
	 */
	public static X509Certificate certificateFromASN1Object(byte[] certificateBytes) throws CertificateException {
			X509CertImpl impl = X509CertImpl.certificateFromASN1Object( certificateBytes );
			return new X509Certificate( impl );
	}

	/**
	 * Generates and initializes a Certificate from an ASN.1 Node object
	 *
	 * @param		node	the ASN.1 Node structure
	 * @param		rawBytes	byte[]	bytes where the node was read from
	 * @return	Certificate 	an initialized Certificate
	 *
	 * @throws	CertificateException 	if parsing problems are detected
	 */
	public static X509Certificate certificateFromASN1Object(Node node, byte[] rawBytes) throws CertificateException {
		X509CertImpl impl = X509CertImpl.certificateFromASN1Object( node, rawBytes );
		return new X509Certificate( impl );
	}

	/**
	 * Generates and initializes a Certificate from a stream with bytes representing
	 * an ASN.1 DER certificate
	 *
	 * @param		in	InputStream stream with bytes representing the ASN.1 certificate
	 *
	 * @return	Certificate 	an initialized Certificate
	 * @throws	CertificateException 	if parsing problems are detected
	 */
	public static X509Certificate certificateFromASN1Object(InputStream in) throws CertificateException {
		X509CertImpl impl = X509CertImpl.certificateFromASN1Object( in );
		return new X509Certificate( impl );
	}

	/**
	 * Set the issuer distinguished name
	 *
	 * @param		principal Principal	the subject distinguished
	*/
	public void setIssuer(String principal) {
		impl.setIssuer( principal );
	}

	/**
	 * Set the subject distinguished name
	 *
	 * @param		principal Principal	the subject distinguished
	*/
	public void setSubject(String principal) {
		impl.setSubject( principal );
	}

	public String getType() {
		return "X.509";
	}

	private String toPrintableString(BigInteger val) {
		if (val==null) return null;
		StringBuffer buff = new StringBuffer();
		byte[] byteval = val.toByteArray();
		// skip the sign byte (all values will be positive)
		int i = (byteval[0]==0 && byteval.length>1) ? 1 : 0;
		for(; i < byteval.length; i++) {
			// high order digit
			int ch = (byteval[i]&0xF0)>>4;
			if (ch > 9)
				ch = ch - 10 + (int) 'A';
			else
				ch += (int) '0';
			buff.append((char)ch);
			// low order digit
			ch = byteval[i]&0x0F;
			if (ch > 9)
				ch = ch - 10 + (int) 'A';
			else
				ch += (int) '0';
			buff.append((char)ch);
			if (i < byteval.length-1) buff.append(":");
		}
		return buff.toString();
	}

}
