package com.ibm.oti.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * Used to parse a string and return either a single
 * or double precision floating point number.
 *
 * @author		OTI
 * @version		initial
 */
public final class FloatingPointParser {

	private static final class StringExponentPair {
		String s;
		int e;
		boolean negative;
		StringExponentPair(String s, int e, boolean negative) {
			this.s = s;
			this.e = e;
			this.negative = negative;
		}
	}

/**
 * Takes a String and an integer exponent.  The String should
 * hold a positive integer value (or zero).  The exponent will
 * be used to calculate the floating point number by taking the
 * positive integer the  String represents and multiplying by 10
 * raised to the power of the of the exponent.  Returns the
 * closest double value to the real number
 *
 * @author		OTI
 * @version		initial
 *
 * @param		s	the String that will be parsed to a floating point
 * @param		e	an int represent the 10 to part
 * @return		the double closest to the real number
 *
 * @exception	NumberFormatException if the String doesn't represent
 *				a positive integer value
 */
private static native double parseDblImpl(String s, int e);

/**
 * Takes a String and an integer exponent.  The String should
 * hold a positive integer value (or zero).  The exponent will
 * be used to calculate the floating point number by taking the
 * positive integer the  String represents and multiplying by 10
 * raised to the power of the of the exponent.  Returns the
 * closest float value to the real number
 *
 * @author		OTI
 * @version		initial
 *
 * @param		s	the String that will be parsed to a floating point
 * @param		e	an int represent the 10 to part
 * @return		the float closest to the real number
 *
 * @exception	NumberFormatException if the String doesn't represent
 *				a positive integer value
 */
private static native float parseFltImpl(String s, int e);

/**
 * Takes a String and does some initial parsing.  Should return
 * a StringExponentPair containing a String with no leading or
 * trailing white space and trailing zeroes eliminated.  The
 * exponent of the StringExponentPair will be used to calculate
 * the floating point number by taking the positive integer the
 * String represents and multiplying by 10 raised to the power
 * of the of the exponent.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		s	the String that will be parsed to a floating point
 * @param 		length the length of s
 * @return		a StringExponentPair with necessary values
 *
 * @exception	NumberFormatException if the String doesn't pass
 *				basic tests
 */
private static StringExponentPair initialParse(String s, int length) {
	boolean negative = false;
	char c;
	int start, end, decimal;
	int e = 0;

	start = 0;
	if (length == 0)
		throw new NumberFormatException(s);

	c = s.charAt(length - 1);
	if (c == 'D' || c == 'd' || c == 'F' || c == 'f') {
		length--;
		if (length == 0)
			throw new NumberFormatException(s);
	}

	end = Math.max (s.indexOf('E'), s.indexOf('e'));
	if (end > -1) {
		if (end + 1 == length)
			throw new NumberFormatException(s);

		if (s.charAt(end + 1) == '+')
			e = Integer.parseInt(s.substring(end + 2, length));
		else
			e = Integer.parseInt(s.substring(end + 1, length));
	} else {
		end = length;
	}
	if (length == 0)
		throw new NumberFormatException(s);

	c = s.charAt(start);
	if (c == '-') {
		++start; --length;
		negative = true;
	} else if (c == '+') {
		++start; --length;
	}
	if (length == 0)
		throw new NumberFormatException(s);

	decimal = s.indexOf('.');
	if (decimal > -1) {
		e -= end - decimal - 1;
		s = s.substring(start, decimal) + s.substring(decimal + 1, end);
	} else {
		s = s.substring(start, end);
	}

	if ((length = s.length()) == 0)
		throw new NumberFormatException();

	end = length;
	while (end > 1 && s.charAt(end - 1) == '0')
		--end;

	start = 0;
	while (start < end - 1 && s.charAt(start) == '0')
		start ++;

	if (end != length || start != 0) {
		e += length - end;
		s = s.substring(start, end);
	}

	return new StringExponentPair(s, e, negative);
}

/**
 * Returns the closest double value to the real number in the
 * string.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		s	the String that will be parsed to a floating point
 * @return		the double closest to the real number
 *
 * @exception	NumberFormatException if the String doesn't represent
 *				a double
 */
public static double parseDouble(String s) {
	s = s.trim();
	int length = s.length();

	if (length == 0) {
		throw new NumberFormatException(s);
	}

	StringExponentPair info = initialParse(s, length);

	double result = parseDblImpl(info.s, info.e);
	if (info.negative)
		result = -result;

	return result;
}

/**
 * Returns the closest float value to the real number in the
 * string.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		s	the String that will be parsed to a floating point
 * @return		the float closest to the real number
 *
 * @exception	NumberFormatException if the String doesn't represent
 *				a float
 */
public static float parseFloat(String s) {
	s = s.trim();
	int length = s.length();

	if (length == 0) {
		throw new NumberFormatException(s);
	}

	StringExponentPair info = initialParse(s, length);

	float result = parseFltImpl(info.s, info.e);
	if (info.negative)
		result = -result;

	return result;
}
}
