package com.ibm.oti.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

import java.util.Calendar;
import java.util.TimeZone;
import java.io.UTFDataFormatException;

public final class Util {

	private static String[] WEEKDAYS = new String[] {"", "Sunday", "Monday", "Tuesday",
					"Wednesday", "Thursday", "Friday", "Saturday"};
	private static String[] MONTHS = new String[] {"January", "February", "March", "April",
					"May", "June", "July", "August", "September", "October",
					"November", "December"};

	private static final boolean useNative = com.ibm.oti.vm.VM.useNatives();
	private static final String defaultEncoding;

	static {
		String encoding = System.getProperty("os.encoding");
		if (encoding != null) {
			try {
				"".getBytes(encoding);
			} catch (java.io.UnsupportedEncodingException e) {
				encoding = null;
			}
		}
		defaultEncoding = encoding;
	}

public static byte[] getBytes(String name) {
	if (defaultEncoding != null) {
		try {
			return name.getBytes(defaultEncoding);
		} catch (java.io.UnsupportedEncodingException e) {}
	}
	return name.getBytes();
}

public static String toString(byte[] bytes) {
	if (defaultEncoding != null) {
		try {
			return new String(bytes, 0, bytes.length, defaultEncoding);
		} catch (java.io.UnsupportedEncodingException e) {}
	}
	return new String(bytes, 0, bytes.length);
}

public static String toString(byte[] bytes, int offset, int length) {
	if (defaultEncoding != null) {
		try {
			return new String(bytes, offset, length, defaultEncoding);
		} catch (java.io.UnsupportedEncodingException e) {}
	}
	return new String(bytes, offset, length);
}

/**
 * Answers the millisecond value of the date and time parsed from
 * the specified String. Many date/time formats are recognized
 *
 * @param	string	the String to parse
 * @return	the millisecond value parsed from the String
 */
public static long parseDate(String string) {
	int offset = 0, length = string.length(), state = 0;
	int year = -1, month = -1, date = -1;
	int hour = -1, minute = -1, second = -1;
	final int PAD=0, LETTERS=1, NUMBERS=2;
	StringBuffer buffer = new StringBuffer();

	while (offset <= length) {
		char next = offset < length ? string.charAt(offset) : '\r';
		offset++;

		int nextState;
		if ((next >= 'a' && next <= 'z') || (next >= 'A' && next <= 'Z')) nextState = LETTERS;
		else if (next >= '0' && next <= '9') nextState = NUMBERS;
		else if (" ,-:\r\t".indexOf(next) == -1)
			throw new IllegalArgumentException();
		else nextState = PAD;

		if (state == NUMBERS && nextState != NUMBERS) {
			int digit = Integer.parseInt(buffer.toString());
			buffer.setLength(0);
			if (digit >= 70) {
				if (year != -1 || (next != ' ' && next != ',' && next != '\r'))
					throw new IllegalArgumentException();
				year = digit;
			} else if (next == ':') {
				if (hour == -1) hour = digit;
				else if (minute == -1) minute = digit;
				else throw new IllegalArgumentException();
			} else if (next == ' ' || next == ',' || next == '-' || next == '\r') {
				if (hour != -1 && minute == -1) minute = digit;
				else if (minute != -1 && second == -1) second = digit;
				else if (date == -1) date = digit;
				else if (year == -1) year = digit;
				else throw new IllegalArgumentException();
			} else if (year == -1 && month != -1 && date != -1)
				year = digit;
			else throw new IllegalArgumentException();
		} else if (state == LETTERS && nextState != LETTERS) {
			String text = buffer.toString().toUpperCase();
			buffer.setLength(0);
			if (text.length() < 3) throw new IllegalArgumentException();
			if (parse(text, WEEKDAYS) != -1) ;
			else if (month == -1 && (month = parse(text, MONTHS)) != -1) ;
			else if (text.equals("GMT")) ;
			else throw new IllegalArgumentException();
		}

		if (nextState == LETTERS || nextState == NUMBERS) buffer.append(next);
		state = nextState;
	}

	if (year != -1 && month != -1 && date != -1) {
		if (hour == -1) hour = 0;
		if (minute == -1) minute = 0;
		if (second == -1) second = 0;
		Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
		int current = cal.get(Calendar.YEAR) - 80;
		if (year < 100) {
			year += current / 100 * 100;
			if (year < current) year += 100;
		}
		// First set a valid date for all months since an update occurs after every set()
		cal.set(Calendar.DATE, 1);
		cal.set(Calendar.YEAR, year);
		cal.set(Calendar.MONTH, month);
		cal.set(Calendar.DATE, date);
		cal.set(Calendar.HOUR_OF_DAY, hour);
		cal.set(Calendar.MINUTE, minute);
		cal.set(Calendar.SECOND, second);
		cal.set(Calendar.MILLISECOND, 0);
		return cal.getTime().getTime();
	}
	throw new IllegalArgumentException();
}

private static int parse(String string, String[] array) {
	int length = string.length();
	for (int i=0; i < array.length; i++) {
		if (string.regionMatches(true, 0, array[i], 0, length)) return i;
	}
	return -1;
}

private static native String convertUTF8(byte[] buf, int offset, int utfSize);

public static String convertFromUTF8(byte[] buf, int offset, int utfSize) throws UTFDataFormatException {
	if (useNative) {
		String result = convertUTF8(buf, offset, utfSize);
		if (result != null) return result;
		throw new UTFDataFormatException();
	}
	return convertUTF8WithBuf(buf, new char[utfSize], offset, utfSize);
}

public static String convertUTF8WithBuf(byte[] buf, char[] out, int offset, int utfSize) throws UTFDataFormatException {
	int count = 0, s = 0, a;
	while (count < utfSize) {
		if ((out[s] = (char)buf[offset + count++]) < '\u0080') s++;
		else if (((a = out[s]) & 0xe0) == 0xc0) {
			if (count >= utfSize)
				// K0062 = Second byte at {0} does not match UTF8 Specification
				throw new UTFDataFormatException(com.ibm.oti.util.Msg.getString("K0062", count));
			int b = buf[count++];
			if ((b & 0xC0) != 0x80)
				// K0062 = Second byte at {0} does not match UTF8 Specification
				throw new UTFDataFormatException(com.ibm.oti.util.Msg.getString("K0062", (count-1)));
			out[s++] = (char) (((a & 0x1F) << 6) | (b & 0x3F));
		} else if ((a & 0xf0) == 0xe0) {
			if (count+1 >= utfSize)
				// K0063 = Third byte at {0} does not match UTF8 Specification
				throw new UTFDataFormatException(com.ibm.oti.util.Msg.getString("K0063", (count+1)));
			int b = buf[count++];
			int c = buf[count++];
			if (((b & 0xC0) != 0x80) || ((c & 0xC0) != 0x80))
				// K0064 = Second or third byte at {0} does not match UTF8 Specification
				throw new UTFDataFormatException(com.ibm.oti.util.Msg.getString("K0064", (count-2)));
			out[s++] = (char) (((a & 0x0F) << 12) | ((b & 0x3F) << 6) | (c & 0x3F));
		} else {
			// K0065 = Input at {0} does not match UTF8 Specification
			throw new UTFDataFormatException(com.ibm.oti.util.Msg.getString("K0065", (count-1)));
		}
	}
	return new String(out, 0, s);
}

/**
 * Answers whether a given string matches a pattern that may contain a wildcard ("*") in any position.
 * There can only be one wildcard in a pattern.  The given string can not contain the wildcard character.
 *
 * @return boolean	True if the given string matches the pattern.
 */
public static boolean matchesPattern(String testString, String pattern) {
	// test string cannot contain wildcards
	if (testString.indexOf('*') != -1) {
		return false;
	}

	// test equality (works if there are no wildcards in pattern)
	if (testString.equals(pattern)) {
		return true;
	}

	// get index of wildcard in pattern string
	int idxWildcard = pattern.indexOf('*');
	if (idxWildcard==-1) {
		// no wildcard and strings are not equal
		return false;
	}

	// test the pattern string region before the wildcard
	if (idxWildcard > 0) {
		if (!testString.regionMatches(false, 0, pattern, 0, idxWildcard)) {
			return false;
		}
	}

	// test the pattern string region after the wildcard
	if (idxWildcard < pattern.length()-1) {
		int testlen = pattern.length()-idxWildcard-1;
		if (!testString.regionMatches(false, testString.length()-testlen, pattern, idxWildcard+1, testlen)) {
			return false;
		}
	}

	return true;
}

}
